## Tabs 


### Basic Usage 

The flex props of tabs is used for `Box` wrapper of `Tabs`, so you can specify border between tabs and tab panels by flexbox props of `Tabs`.

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout
class App:
    @mark_create_layout
    def my_layout(self):
        return mui.VBox([
            mui.Tabs([
                mui.TabDef("Tab 1", "1", mui.Markdown("Tab 1 Content")),
                mui.TabDef("Tab 2", "2", mui.Markdown("Tab 2 Content")),
                mui.TabDef("Tab 3", "3", mui.Markdown("Tab 3 Content")),
            ]).prop(borderBottom=1, borderColor='divider')
        ]).prop(width="100%")

```


### Wrapped labels

Long labels will automatically wrap on tabs. If the label is too long for the tab, it will overflow, and the text will not be visible.


```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout
class App:
    @mark_create_layout
    def my_layout(self):
        return mui.VBox([
            mui.Tabs([
                mui.TabDef("New Arrivals in the Longest Text of Nonfiction that should appear in the next line", "1", mui.Markdown("Tab 1 Content"), wrapped=True),
                mui.TabDef("Tab 2", "2", mui.Markdown("Tab 2 Content")),
                mui.TabDef("Tab 3", "3", mui.Markdown("Tab 3 Content")),
            ])
        ]).prop(width="100%")

```

### Colored tab


```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout
class App:
    @mark_create_layout
    def my_layout(self):
        return mui.VBox([
            mui.Tabs([
                mui.TabDef("Tab 1", "1", mui.Markdown("Tab 1 Content")),
                mui.TabDef("Tab 2", "2", mui.Markdown("Tab 2 Content")),
                mui.TabDef("Tab 3", "3", mui.Markdown("Tab 3 Content")),
            ]).prop(indicatorColor="secondary", textColor="secondary")
        ]).prop(width="100%")

```

### Disabled tab

A tab can be disabled by setting the ```disabled``` prop.

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout
class App:
    @mark_create_layout
    def my_layout(self):
        return mui.VBox([
            mui.Tabs([
                mui.TabDef("Tab 1", "1", mui.Markdown("Tab 1 Content")),
                mui.TabDef("Tab 2", "2", mui.Markdown("Tab 2 Content"), disabled=True),
                mui.TabDef("Tab 3", "3", mui.Markdown("Tab 3 Content")),
            ]).prop(indicatorColor="secondary", textColor="secondary")
        ]).prop(width="100%")

```


### Full width

The ```variant="fullWidth"``` prop should be used for smaller views. This demo also uses react-swipeable-views to animate the Tab transition, and allowing tabs to be swiped on touch devices.

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout
class App:
    @mark_create_layout
    def my_layout(self):
        return mui.VBox([
            mui.Tabs([
                mui.TabDef("Tab 1", "1", mui.Markdown("Tab 1 Content")),
                mui.TabDef("Tab 2", "2", mui.Markdown("Tab 2 Content")),
                mui.TabDef("Tab 3", "3", mui.Markdown("Tab 3 Content")),
            ]).prop(variant="fullWidth")
        ]).prop(width="100%")

```

### Centered

The centered prop should be used for larger views.

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout
class App:
    @mark_create_layout
    def my_layout(self):
        return mui.VBox([
            mui.Tabs([
                mui.TabDef("Tab 1", "1", mui.Markdown("Tab 1 Content")),
                mui.TabDef("Tab 2", "2", mui.Markdown("Tab 2 Content")),
                mui.TabDef("Tab 3", "3", mui.Markdown("Tab 3 Content")),
            ]).prop(centered=True)
        ]).prop(width="100%")

```


### Automatic scroll buttons

By default, left and right scroll buttons are automatically presented on desktop and hidden on mobile. (based on viewport width)

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout
class App:
    @mark_create_layout
    def my_layout(self):
        return mui.VBox([
            mui.Tabs([
                mui.TabDef("Tab 1", "1", mui.Markdown("Tab 1 Content")),
                mui.TabDef("Tab 2", "2", mui.Markdown("Tab 2 Content")),
                mui.TabDef("Tab 3", "3", mui.Markdown("Tab 3 Content")),
                mui.TabDef("Tab 4", "4", mui.Markdown("Tab 4 Content")),
                mui.TabDef("Tab 5", "5", mui.Markdown("Tab 5 Content")),
                mui.TabDef("Tab 6", "6", mui.Markdown("Tab 6 Content")),
                mui.TabDef("Tab 7", "7", mui.Markdown("Tab 7 Content")),
                mui.TabDef("Tab 8", "8", mui.Markdown("Tab 8 Content")),
                mui.TabDef("Tab 9", "9", mui.Markdown("Tab 9 Content")),
                mui.TabDef("Tab 10", "10", mui.Markdown("Tab 10 Content")),
            ]).prop(variant="scrollable")
        ]).prop(width="100%")

```

### Vertical tabs

To make vertical tabs instead of default horizontal ones, there is orientation="vertical":


```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout
class App:
    @mark_create_layout
    def my_layout(self):
        return mui.HBox([
            mui.Tabs([
                mui.TabDef("Tab 1", "1", mui.Markdown("Tab 1 Content")),
                mui.TabDef("Tab 2", "2", mui.Markdown("Tab 2 Content")),
                mui.TabDef("Tab 3", "3", mui.Markdown("Tab 3 Content")),
            ]).prop(orientation="vertical")
        ]).prop(width="100%")

```

## Icon tabs

Tab labels may be either all icons or all text.

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

svg = """
<?xml version="1.0" encoding="utf-8"?><svg version="1.1" id="Layer_1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" viewBox="0 0 122.88 122.88" style="enable-background:new 0 0 122.88 122.88" xml:space="preserve"><style type="text/css">.st0{fill-rule:evenodd;clip-rule:evenodd;}</style><g><path class="st0" d="M86.71,104.87c-16.58,8.49-33.27,8.72-50.08-0.02L54.62,74.2c4.09,2.43,8.89,2.45,14.39,0.01L86.71,104.87 L86.71,104.87z M61.44,0c16.97,0,32.33,6.88,43.44,18c11.12,11.12,18,26.48,18,43.44c0,16.97-6.88,32.33-18,43.44 c-11.12,11.12-26.48,18-43.44,18S29.11,116,18,104.88C6.88,93.77,0,78.41,0,61.44C0,44.47,6.88,29.11,18,18 C29.11,6.88,44.47,0,61.44,0L61.44,0z M101.53,21.35C91.27,11.09,77.1,4.74,61.44,4.74c-15.66,0-29.83,6.35-40.09,16.61 C11.09,31.61,4.74,45.78,4.74,61.44c0,15.66,6.35,29.83,16.61,40.09c10.26,10.26,24.43,16.61,40.09,16.61 c15.66,0,29.83-6.35,40.09-16.61c10.26-10.26,16.61-24.43,16.61-40.09C118.14,45.78,111.79,31.61,101.53,21.35L101.53,21.35z M86.45,17.73c15.64,10.11,24.19,24.45,25.02,43.38l-35.55-0.25c0.06-4.76-2.32-8.92-7.18-12.46L86.45,17.73L86.45,17.73z M61.5,52.41c5.33,0,9.65,4.32,9.65,9.65c0,5.33-4.32,9.65-9.65,9.65c-5.33,0-9.65-4.32-9.65-9.65 C51.85,56.73,56.17,52.41,61.5,52.41L61.5,52.41z M11.53,61.33c0.93-18.6,9.08-33.17,25.06-43.36l17.56,30.91 c-4.15,2.33-6.56,6.47-7.2,12.45H11.53L11.53,61.33z"/></g></svg>
"""

class App:
    @mark_create_layout
    def my_layout(self):
        return mui.HBox([
            mui.Tabs([
                mui.TabDef("Tab 1", "1", mui.Markdown("Tab 1 Content"), icon=mui.IconType.Delete),
                mui.TabDef("Tab 2", "2", mui.Markdown("Tab 2 Content"), icon=mui.IconType.Terminal),
                mui.TabDef("Tab 3", "3", mui.Markdown("Tab 3 Content"),
                icon=mui.Icon.encode_svg(svg)),
            ])
        ]).prop(width="100%")

```

vertical icon tabs

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        return mui.HBox([
            mui.Tabs([
                mui.TabDef("Tab 1", "1", mui.Markdown("Tab 1 Content"), icon=mui.IconType.Delete),
                mui.TabDef("Tab 2", "2", mui.Markdown("Tab 2 Content"), icon=mui.IconType.Terminal),
            ]).prop(orientation="vertical")
        ]).prop(height="100%")

```


### Customized tabs 

We can use `mui.ThemeProvider` to control the style of the component. e.g. remove the padding of tabs.


```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout
class App:
    @mark_create_layout
    def my_layout(self):
        return mui.VBox([
            mui.ThemeProvider([
                mui.Tabs([
                    mui.TabDef("Tab 1", "1", mui.Markdown("Tab 1 Content")),
                    mui.TabDef("Tab 2", "2", mui.Markdown("Tab 2 Content")),
                    mui.TabDef("Tab 3", "3", mui.Markdown("Tab 3 Content")),
                ])
            ], mui.Theme(components={
                "MuiTab": {
                    "styleOverrides": {
                        "root": {
                            "padding": "0",
                            "border": "1px solid red",
                        }
                    }
                }
            }))
        ]).prop(width="100%")

```

* tab with small icon

we can use `panelProps` to control panels directly, and use `ThemeProvider` to control the style of nested mui tab components.

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        return mui.HBox([
            mui.ThemeProvider([
                mui.Tabs([
                    mui.TabDef("", "1", mui.Markdown("Tab 1 Content"), icon=mui.IconType.Delete),
                    mui.TabDef("", "2", mui.Markdown("Tab 2 Content"), icon=mui.IconType.Terminal),
                ]).prop(panelProps=mui.FlexBoxProps(width="100%"), orientation="vertical", borderRight=1, borderColor='divider')
            ], mui.Theme(components={
                "MuiTab": {
                    "styleOverrides": {
                        "root": {
                            "padding": "0",
                            "border": "1px solid red",
                            "minWidth": "28px",
                            "minHeight": "28px",
                        }
                    }
                }
            }))
        ]).prop(height="100%", width="100%")

```

* tab with custom label component

In this example, we provide a custom mui component as a tab label. you can use this to implement tab contextmenu or close button.

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        tab1_content = mui.Markdown("Tab 1 Content")
        tab1_context_menu_handler = lambda x: tab1_content.write(f"Tab 1 Content click {x}")
        return mui.VBox([
            mui.Tabs([
                mui.TabDef("", "1", 
                    tab1_content, 
                    labelComponent=mui.MenuList([
                        mui.MenuItem(id="0", label="item 0"),
                        mui.MenuItem(id="1", label="item 1"),
                        mui.MenuItem(id="2", label="item 2"),
                    ], mui.Typography("Tab 1"),
                    tab1_context_menu_handler).prop(triggerMethod="contextmenu")),
                mui.TabDef("", "2", 
                    mui.Markdown("Tab 2 Content"), 
                    labelComponent=mui.Typography("Tab 2")),
            ])
        ]).prop(height="100%")


```