## Button 

mui button has two types: ```mui.Button``` and ```mui.IconButton```. 

each button have a ```click``` event, which will be triggered when the button is clicked.

### Basic Usage

basic props: 

* variant: ```contained```, ```outlined```, ```text```. Default is ```contained```.

* size: ```small```, ```medium```, ```large```. Default is ```medium```.

* fullWidth: if true, the button will take up the full width of its container. Default is ```False```.

* disabled: if true, the button will be disabled. Default is ```False```.

* loading: if true, the button will show a loading indicator. Default is ```False```.

```Python

from tensorpc.flow import mui, three, plus, appctx, mark_create_layout
class App:
    @mark_create_layout
    def my_layout(self):
        return mui.VBox([
            mui.Button("A Text Button"),
            mui.Button("A Outlined Button").prop(variant="outlined"),
            mui.Button("A Contained Button").prop(variant="contained"),
            mui.Button("A Small Button").prop(size="small"),
            mui.Button("A Large Button").prop(size="large"),
            mui.HBox([
                mui.Button("A Button").prop(fullWidth=False),
                mui.Button("A Full Width Button").prop(fullWidth=True),
            ]),
            mui.Button("A Disabled Button").prop(disabled=True),
            mui.Button("A Loading Button").prop(loading=True),
            mui.ButtonGroup([
                mui.Button("🚀").prop(size="small"),
                mui.Button("🔥").prop(size="small"),
                mui.Button("✨").prop(size="small"),
            ])

        ]).prop(width="100%")
```

### Event

Button support ```click``` event. 

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout
class App:
    @mark_create_layout
    def my_layout(self):
        btn_standard_event = mui.Button("A Standard Event Btn")
        btn_standard_event.event_click.on_standard(self._on_standard_clicked)
        return mui.VBox([
            mui.Button("A Simple Event Button", lambda: print("Button Clicked")),
        ]).prop(width="100%")

    async def _on_standard_clicked(self, ev: mui.Event):
        print(ev)

```

## IconButton 

```IconButton``` is a button with icon. currently only support a limited set of material icons, see enums in ```mui.IconType``` for all supported icons.

You can also use custom svg in icon, see example for more details.

**NOTE** all icon prop in this library will be sanitized by dompurify before rendering.

**NOTE** always use `currentColor` in svg string instead of hard coded color, so that the mui components can control theme of icon via standard way.

### Basic Usage

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

svg_string = """
<svg
    xmlns="http://www.w3.org/2000/svg"
    fill="none"
    viewBox="0 0 24 24"
    stroke-width="1.5"
    stroke="currentColor"
>
    <path
    stroke-linecap="round"
    stroke-linejoin="round"
    d="M11.42 15.17L17.25 21A2.652 2.652 0 0021 17.25l-5.877-5.877M11.42 15.17l2.496-3.03c.317-.384.74-.626 1.208-.766M11.42 15.17l-4.655 5.653a2.548 2.548 0 11-3.586-3.586l6.837-5.63m5.108-.233c.55-.164 1.163-.188 1.743-.14a4.5 4.5 0 004.486-6.336l-3.276 3.277a3.004 3.004 0 01-2.25-2.25l3.276-3.276a4.5 4.5 0 00-6.336 4.486c.091 1.076-.071 2.264-.904 2.95l-.102.085m-1.745 1.437L5.909 7.5H4.5L2.25 3.75l1.5-1.5L7.5 4.5v1.409l4.26 4.26m-1.745 1.437l1.745-1.437m6.615 8.206L15.75 15.75M4.867 19.125h.008v.008h-.008v-.008z"
    />
</svg>
"""

class App:
    @mark_create_layout
    def my_layout(self):
        return mui.HBox([
            mui.IconButton(mui.IconType.Add),
            mui.IconButton(mui.IconType.Add).prop(size="small"),
            mui.IconButton(mui.IconType.Add).prop(size="large"),
            mui.IconButton(mui.IconType.Add).prop(disabled=True),
            mui.IconButton(mui.IconType.Settings),
            mui.IconButton(mui.IconType.Terminal).prop(muiColor="success"),
            mui.IconButton(mui.IconType.Terminal).prop(muiColor="info"),
            mui.IconButton(svg_string).prop(muiColor="info"),
            mui.IconButton(svg_string).prop(muiColor="success"),

        ]).prop(alignItems="flex-start", flexWrap="wrap")
```

### Additional Features

* tooltip: if set, the button will show a tooltip when mouse hover on it.

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout
class App:
    @mark_create_layout
    def my_layout(self):
        return mui.HBox([
            mui.IconButton(mui.IconType.Add).prop(tooltip="Hello Tooltip!!!"),
            mui.IconButton(mui.IconType.Add).prop(tooltip="""
multiple line
tooltip
            """, tooltipMultiline=True),
            mui.IconButton(mui.IconType.Add).prop(tooltip="Hello Tooltip Right !!!", tooltipPlacement="right"),
        ]).prop(alignItems="flex-start", flexWrap="wrap")
```

* Progress: if set, the button will show a progress indicator when clicked (running).

```Python
import asyncio
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout
class App:
    @mark_create_layout
    def my_layout(self):
        return mui.HBox([
            mui.IconButton(mui.IconType.PlayArrow, lambda: asyncio.sleep(1)).prop(progressColor="primary")
        ]).prop(alignItems="flex-start", flexWrap="wrap")
```

* Confirm Dialog: if set, the button will show a confirm dialog when clicked. usually used to confirm a dangerous operation.

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout
import asyncio
svg = """
<?xml version="1.0" encoding="utf-8"?><svg version="1.1" id="Layer_1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" viewBox="0 0 122.88 122.88" style="enable-background:new 0 0 122.88 122.88" xml:space="preserve"><style type="text/css">.st0{fill-rule:evenodd;clip-rule:evenodd;}</style><g><path class="st0" d="M86.71,104.87c-16.58,8.49-33.27,8.72-50.08-0.02L54.62,74.2c4.09,2.43,8.89,2.45,14.39,0.01L86.71,104.87 L86.71,104.87z M61.44,0c16.97,0,32.33,6.88,43.44,18c11.12,11.12,18,26.48,18,43.44c0,16.97-6.88,32.33-18,43.44 c-11.12,11.12-26.48,18-43.44,18S29.11,116,18,104.88C6.88,93.77,0,78.41,0,61.44C0,44.47,6.88,29.11,18,18 C29.11,6.88,44.47,0,61.44,0L61.44,0z M101.53,21.35C91.27,11.09,77.1,4.74,61.44,4.74c-15.66,0-29.83,6.35-40.09,16.61 C11.09,31.61,4.74,45.78,4.74,61.44c0,15.66,6.35,29.83,16.61,40.09c10.26,10.26,24.43,16.61,40.09,16.61 c15.66,0,29.83-6.35,40.09-16.61c10.26-10.26,16.61-24.43,16.61-40.09C118.14,45.78,111.79,31.61,101.53,21.35L101.53,21.35z M86.45,17.73c15.64,10.11,24.19,24.45,25.02,43.38l-35.55-0.25c0.06-4.76-2.32-8.92-7.18-12.46L86.45,17.73L86.45,17.73z M61.5,52.41c5.33,0,9.65,4.32,9.65,9.65c0,5.33-4.32,9.65-9.65,9.65c-5.33,0-9.65-4.32-9.65-9.65 C51.85,56.73,56.17,52.41,61.5,52.41L61.5,52.41z M11.53,61.33c0.93-18.6,9.08-33.17,25.06-43.36l17.56,30.91 c-4.15,2.33-6.56,6.47-7.2,12.45H11.53L11.53,61.33z"/></g></svg>
"""
class App:
    @mark_create_layout
    def my_layout(self):
        md = mui.Markdown(":red[click to launch nuclear missle.]")
        self.md = md
        return mui.VBox([
            mui.IconButton(svg, self._on_launch_nuclear_missle).prop(
                muiColor="error", 
                confirmTitle="Are You Sure?", 
                confirmMessage="To Do This operation?",
                # enable progress when running
                progressColor="primary"),
            md,
        ]).prop(alignItems="flex-start", flexWrap="wrap")

    async def _on_launch_nuclear_missle(self):
        await self.md.write(":blue[nuclear missle launching...]")
        await asyncio.sleep(1.0)
        await self.md.write(":green[nuclear missle launched.]")
```
