#TODO: Change to automated unit tests
#TODO: Change to pip package
#TODO: Comment code fully
#TODO: Update documentation/README

import copy
import json
import csv
import matplotlib.pyplot as plt
import pandas
import random

class compartmentModel:
    def __init__(self, tmax=10, compartments=[], parameters=[], linkages={}, initCompartments={}, initParameters={}, parameterPoints=[]):
        """
        Parameters
        ----------
        tmax : int, optional
            Number of time steps in generated model, default = 10
        compartments : list(dict(str)), optional
            A formatted list of compartment dictionaries for each time
            step. Will be auto-generated by class. Only exists for programmatic
            creation of compartmentModel objects.
        parameters : list(dict(str)), optional
            A formatted list of parameter dictionaries for each time
            step. Will be auto-generated by class. Only exists for programmatic
            creation of compartmentModel objects.
        linkages : dict(tuple), optional
            A formatted dictionary of linkage formulas. Requires specific formatting.
            Only exists for programmatic creation of compartmentModel objects.
        initCompartments : dict(str), optional
            A formatted dictionary of compartments and initial values. 
            Only exists for programmatic creation of compartmentModel objects.
        initParameters : dict(str), optional
            A formatted dictionary of parameters and initial values.
            Only exists for programmatic creation of compartmentModel objects.
        parameterPoints : list(tuple), optional
            A list of formatted tuples that represent paramters, values, and time points.
            Only exists for programmatic creation of compartmentalModel objects
        """
        self.runSinceChange = False #Flag to give indication of model has had changes made to it since being run last 
        self.tmax = tmax #The number of time steps in generated model 
        self.compartments = compartments #List of dicts, {'compartmentName': valueAtIndex}
        self.parameters = parameters #List of dicts, {'parameterName': valueAtIndex}
        self.linkages = linkages #Dict of change functions, {('from', 'to'): lambda function string
        self.initCompartments = initCompartments #Dict of compartments and their initial values at t = 0
        self.initParameters = initParameters #Dict of parameters and their initial values at t = 0
        self.parameterPoints = parameterPoints #List of tuples, ('parameterName', newValue, timeStart)
        self.comparisonData = {} #Holder variable for observed data to compare model to

    #Private Methods
    def _generate_model(self):

        self.compartments = self._generate_compartment_list()
        self.parameters = self._generate_parameters_list()        
    def _generate_compartment_list(self):
        return [copy.deepcopy(self.initCompartments) for t in range(self.tmax)]    
    def _generate_parameters_list(self): #Adds all parameters to parameters list and updates all parameters that have change points
        self.parameters = [copy.deepcopy(self.initParameters) for t in range(self.tmax)]
        tempParameters = copy.deepcopy(self.parameters)

        for name in tempParameters[0]:
            tempParameters = self._update_parameter(name)
            self.parameters = copy.deepcopy(tempParameters)

        return tempParameters 
    def _update_parameter(self, name:str): #Given parameter list and change point tuple, update all points necessary in parameter list      
        if name in self.parameters[0]:
            self.runSinceChange = False  
            tempParametersList = copy.deepcopy(self.parameters)
            
            sortedChanges = sorted(self.parameterPoints, key = lambda x: x[2])
            for change in sortedChanges:
                if change[0] == name:
                    for t in range(self.tmax):
                        if t >= change[2]:
                            tempParametersList[t][name] = change[1]

            return tempParametersList      
        else:
            raise Exception("Model Generation Error: parameter (" + name + ") not found") 
    def _evalLink(self, time:int, key):
        if time == 0:
            return 0
        else:
            fullFun = 'lambda t, compartments, parameters, self: ' + self.linkages[key]
            return eval(fullFun)(time-1, self.compartments, self.parameters, self)
    def _stepEval(self, time:int, verbose=False):
        if time == 0:
            return self.compartments[time]
        else:
            tempCompartments = copy.deepcopy(self.compartments[time-1])
            for link in self.linkages:
                delta = self._evalLink(time, link)
                if verbose:
                    print('TIME = ', time)
                    print('Change Function: ', self.linkages[link])
                    print(link, ' delta: ', delta)
                tempCompartments[link[0]] -= delta
                tempCompartments[link[1]] += delta
            return tempCompartments

    #Public Methods
    def add_compartment(self, name:str, initVal):
        """
        Adds a compartment to initCompartments and updates compartments if 
        compartment name is not taken.

        Parameters 
        ----------
        name : str
            The name of the compartment to be added.
        initVal : int or float
            The initial value of the compartment at t = 0

        Raises
        ------
        Model Generation Error
            Occurs if the name provided is already in use by initCompartments.
        """
        initVal = float(initVal)
        if name in self.initCompartments:
            raise Exception("Model Generation Error: compartment name (" + name + ") already in use")
        else:
            self.runSinceChange = False
            self.initCompartments[name] = initVal
            self.compartments = self._generate_compartment_list()   
    def del_compartment(self, name:str):
        #TODO: handle removing compartment from linkages 
        if name in self.initCompartments:
            self.runSinceChange = False
            del self.initCompartments[name]
            self.compartments = self._generate_compartment_list()
        else:
            raise Exception("Deletion Error: compartment (" + name + ") not found")
    def edit_compartment(self, name:str, initVal):
        initVal = float(initVal)
        if name in self.initCompartments:
            self.runSinceChange = False
            self.initCompartments[name] = initVal
            self.compartments = self._generate_compartment_list()
        else:
            raise Exception("Edit Error: compartment (" + name + ") not found")
    def sum_compartments(self, t:int, names:list):
        total = 0
        for name in names:
            total += self.compartments[t][name]
        return total
    def add_parameter(self, name:str, initVal):
        initVal = float(initVal)
        if name in self.initParameters:
            raise Exception("Model Generation Error: parameter name (" + name + ") already in use")
        else:
            self.runSinceChange = False
            self.initParameters[name] = initVal
            self.parameters = self._generate_parameters_list()  
    def edit_parameter(self, name:str, initVal):
        initVal = float(initVal)
        if name in self.initParameters:
            self.runSinceChange = False
            self.initParameters[name] = initVal
            self.parameters = self._generate_parameters_list()
        else:
            raise Exception("Edit Error: parameter (" + name + ") not found")
    def del_parameter(self, name:str):    #Deletes parameter from initParameters and all change points associated with it
        #TODO: handle removing parameter from linkages
        if name in self.initParameters:
            self.runSinceChange = False
            del self.initParameters[name]
            for change in self.parameterPoints:
                if change[0] == name:
                    self.parameterPoints.remove(change)
            self.parameters = self._generate_parameters_list()
        else:
            raise Exception("Deletion Error: parameter (" + name + ") not found")       
    def add_parameter_change(self, name:str, newVal, timeStart:int):    #Adds parameter change point and updates parameters if name matches existing parameter. If change already exists at time point, its replaced
        timeStart = int(timeStart)
        newVal = float(newVal)
        if self.tmax < timeStart:
            raise Exception("Model Generation Error: Parameter change timestamp (" + str(timeStart) + ") exceeds model tmax")
        if name not in self.parameters[0]:
            raise Exception("Model Generation Error: Parameter name (" + name + ") not found in parameters list")   
        
        self.runSinceChange = False  
        pointCheck = [x for x in self.parameterPoints if x[0] == name and x[2] == timeStart]
        if len(pointCheck) > 0:
            self.parameterPoints.remove(pointCheck[0])
        self.parameterPoints.append((name, newVal, timeStart))

        self.parameters = self._update_parameter(name)
    def del_parameter_change(self, name:str, timeStart:int):    #Removes parameter change point at time point if exists
        timeStart = int(timeStart)
        pointCheck = [x for x in self.parameterPoints if x[0] == name and x[2] == timeStart]
        if len(pointCheck) > 0:
            self.runSinceChange = False 
            self.parameterPoints.remove(pointCheck[0])
            self.parameters = self._generate_parameters_list()
        else:
            raise Exception("Deletion Error: change point (" + name + ", " + str(timeStart) + ") not found")
    def del_all_parameter_changes(self, name:str):    #Removes all parameter changes for single parameter if name matches existing parameter
        if(name in self.parameters[0]):
            self.runSinceChange = False  
            pointCheck = [x for x in self.parameterPoints if x[0] == name]
            for point in pointCheck:
                self.parameterPoints.remove(point)
            self.parameters = self._generate_parameters_list()
        else:
            raise Exception("Deletion Error: parameter (" + name + ") not found")
    def reset_parameters(self):    #Removes all changes for all parameters and resets to init values
        self.runSinceChange = False  
        self.parameterPoints = []
        self.parameters = self._generate_parameters_list()
        return True        
    def add_linkage(self, fromCompartment:str, toCompartment:str, changeFunction:str):
        #TODO: add parsing for more intuitive change function writing, for now just making literal eval functions
        #TODO: add way to check if linkage is valid
        if (fromCompartment, toCompartment) in self.linkages:
            raise Exception("Model Generation Error: Linkage (" + fromCompartment + ", " + toCompartment + ") is already in use")
        elif(fromCompartment not in self.compartments[0] or toCompartment not in self.compartments[0]):
            raise Exception("Model Generation Error: linkage key (" + fromCompartment + ", " + toCompartment + ") not valid")
        else:
            self.runSinceChange = False  
            self.linkages[(fromCompartment, toCompartment)] = changeFunction   
    def del_linkage(self, fromCompartment:str, toCompartment:str):
        if (fromCompartment, toCompartment) in self.linkages:
            self.runSinceChange = False  
            del self.linkages[(fromCompartment, toCompartment)]
        else:
            raise Exception("Deletion Error: linkage (" + fromCompartment + ", " + toCompartment + ") not found")            
    def edit_linkage(self, fromCompartment:str, toCompartment:str, changeFunction:str):
        if (fromCompartment, toCompartment) in self.linkages:
            self.runSinceChange = False  
            self.linkages[(fromCompartment, toCompartment)] = changeFunction
            #TODO: add parsing for more intuitive change function writing, for now just making literal eval functions
            #TODO: add way to check if linkage is valid
        else:
            raise Exception("Edit Error: linkage (" + fromCompartment + ", " + toCompartment + ") not found")
    def run(self, verbose = False):
        self._generate_model()
        for t in range(self.tmax):
            self.compartments[t] = self._stepEval(t, verbose=verbose)
            if verbose: 
                print(self.get_all_compartments())
    def update_tmax(self, newMax:int):
        self.tmax = int(newMax)
        tempChangePoints = []
        for change in self.parameterPoints:
            if change[2] < self.tmax:
                tempChangePoints.append(change)
        self.parameterPoints = tempChangePoints
        self._generate_model()


    #Direct getters
    def get_tmax(self):
        return self.tmax
    def get_all_compartments(self):
        return self.compartments
    def get_initCompartments(self):
        return self.initCompartments
    def get_compartment(self, name:str):
        if name in self.compartments[0]:
            allPoints = [x[name] for x in self.compartments]
            return allPoints
        else:
            return None
    def get_initParameters(self):
        return self.initParameters
    def get_all_parameters(self):
        return self.parameters
    def get_parameter(self, name:str):
        if name in self.parameters[0]:
            allPoints = [x[name] for x in self.parameters]
            return allPoints
        else:
            return None
    def get_parameter_changes(self):
        return self.parameterPoints
    def get_linkages(self):
        return self.linkages
    def get_comparison_data(self):
        return self.comparisonData
    def get_compartment_names(self):
        return list(self.initCompartments.keys())
    def get_parameter_names(self):
        return list(self.initParameters.keys())
    def get_dates_for_parameter(self, parameter):
        return [x[2] for x in self.parameterPoints if x[0] == parameter]
    def get_linkage_names(self):
        return list(self.linkages.keys())

    #Data Frame Functions
    def get_initCompartments_df(self):
        if(len(self.initCompartments) == 0):
            return pandas.DataFrame(columns=['Compartment Name', 'Initial Value'])
        df = pandas.DataFrame(list(self.initCompartments.items()), columns=['Compartment Name', 'Initial Value'])   
        df = df.sort_values(by = 'Compartment Name', ascending = True, na_position = 'first')
        return df    
    def get_parameterValues_df(self):
        initParametersList = self.initParameters.items()
        initParametersList = [(x[0], x[1], 0) for x in initParametersList]
        allParameters = initParametersList + self.parameterPoints
        df = pandas.DataFrame(allParameters, columns=['Parameter Name', 'Value', 'Time'])
        df = df.sort_values(by = ['Parameter Name', 'Time'], ascending = [True, True], na_position = 'first')
        return df
    def get_linkage_df(self):
        if len(self.linkages) == 0:
            return pandas.DataFrame(columns=['From/To', 'Expression'])
        df = pandas.DataFrame(self.linkages.items(), columns=['From/To', 'Expression'])
        df = df.sort_values(by = ['From/To'], ascending=True, na_position = 'first')
        return df
    def get_compartments_df(self):
        if(len(self.initCompartments) == 0):
            return {}
        colNames = list(self.compartments[0].keys())
        csvData = []
        for t in range(self.tmax):
            csvData.append([])
            for key in colNames:
                csvData[t].append(self.compartments[t][key])
        df = pandas.DataFrame(csvData, columns=colNames)
        return df
    def get_compartment_as_df(self, name:str):
        return pandas.DataFrame(self.get_compartment(name), columns=[name])
    def get_compartment_delta_as_df(self, name:str):
        temp = self.get_compartment(name)
        delta = []
        for i in range(len(temp)):
            if i == 0:
                delta.append(0)
            else:
                delta.append(temp[i]-temp[i-1])
        return pandas.DataFrame(delta, columns=[name+'_delta'])
    def get_comparison_as_df(self, name:str):
        return pandas.DataFrame(self.comparisonData[name], columns=[name])
    def get_comparison_delta_as_df(self, name:str):
        temp = self.comparisonData[name]
        delta = []
        for i in range(len(temp)):
            if i == 0:
                delta.append(0)
            else:
                delta.append(temp[i]-temp[i-1])
        return pandas.DataFrame(delta, columns=[name+'_delta'])

    #Direct Setters (DO NOT USE UNLESS YOU KNOW WHAT YOU'RE DOING, ERROR CHECKING LACKING)
    def set_compartments(self, compartmentMatrix):
        self.compartments = compartmentMatrix
        return True   
    def set_initCompartments(self, compartmentDict):
        self.initCompartments = compartmentDict
        return True
    def set_initParameters(self, parametersDict):
        self.initParameters = parametersDict
        return True
    def set_all_parameters(self, parametersMatrix):
        self.parameters = parametersMatrix
        return True
    def set_parameter(self, name:str, parameterList):
        for t in range(self.tmax):
            self.parameters[t][name] = parameterList[t]
        return True
    def set_parameter_changes(self, changeList):
        self.parameterPoints = changeList
        return True
    def set_linkages(self, linkageDict):
        self.linkages = linkageDict
        return True

    #Model import/export functions
    def create_json(self):
        modelDict = {}
        modelDict['tmax'] = self.tmax
        modelDict['initCompartments'] = self.initCompartments
        modelDict['initParameters'] = self.initParameters
        modelDict['parameterPoints'] = self.parameterPoints

        #convert tuple to string for json dump requirements
        modelDict['linkages'] = {}
        for key in self.linkages:
            modelDict['linkages'][key[0] + "###" + key[1]] = self.linkages[key]

        modelJson = json.dumps(modelDict, indent=4)

        return modelJson
    def write_json(self, fileName='model.json', filepath=''):
        with open(filepath + fileName, "w") as outfile:
            outfile.write(self.create_json())
    def load_from_json(self, jsonFile=None, jsonObject=None):
        if jsonFile == None and jsonObject == None:
            raise Exception("Model Generation Error: No JSON file or object provided")
        if jsonFile != None and jsonObject != None:
            raise Exception("Model Generation Error: Only provide JSON file or JSON object, not both")
        if jsonFile != None and jsonObject == None:
            with open(jsonFile, 'r') as f:
                self.load_from_json(jsonObject = json.load(f))
        if jsonFile == None and jsonObject != None:
            self.tmax = jsonObject['tmax']
            self.initCompartments = jsonObject['initCompartments']
            self.initParameters = jsonObject['initParameters']
            self.parameterPoints = jsonObject['parameterPoints']
            
            tempLinkages = {}
            for link in jsonObject['linkages']:
                splitCompString = link.split('###')
                tempLinkages[(splitCompString[0], splitCompString[1])] = jsonObject['linkages'][link]
            self.linkages = tempLinkages
            self._generate_model()
    def create_csv_string(self):
        if(len(self.compartments) > 0):
            header = list(self.compartments[0].keys())
            csvData = []
            csvData.append(header)
            for t in range(self.tmax):
                csvData.append([])
                for key in header:
                    csvData[t+1].append(str(self.compartments[t][key]))
            
            csvString = ""
            for line in csvData:
                csvString += ','.join(line) + '\r\n'
            return csvString

        else:
            return []
        
    def write_compartments_to_csv(self, filename='model.csv', filepath=''):
        header = list(self.compartments[0].keys())
        csvData = []
        csvData.append(header)
        for t in range(self.tmax):
            csvData.append([])
            for key in header:
                csvData[t].append(self.compartments[t][key])

        with open(filepath + filename, 'w', newline='') as f:
            writer = csv.writer(f)
            for row in csvData:
                writer.writerow(row)

    #Comparison Data import functions
    def extract_columns(self, csvList):
        headers = csvList[0]
        columns = [[] for i in range(len(headers))]
        for row in csvList[1:]:
            for i, item in enumerate(row):
                columns[i].append(float(item))

        csvDict = {}
        for i, item in enumerate(headers):
            csvDict[item] = columns[i]

        return csvDict
    def import_comparison_data(self, filepath):
        comparisonData = []
        with open(filepath, 'r') as csvFile:
            for line in csvFile:
                temp = line.strip().split(',')
                temp = [x.strip() for x in temp]
                comparisonData.append(temp)

        self.comparisonData = self.extract_columns(comparisonData)
    def import_comparison_data_from_obj(self, fileObj):
        comparisonData = []
        for line in fileObj:
            temp = line.decode("utf-8")
            temp = temp.strip().split(',')
            temp = [x.strip() for x in temp]
            comparisonData.append(temp)

        self.comparisonData = self.extract_columns(comparisonData)

    #DONE: Define methods for generating performance metrics (not all required!)
    #Mean Absolute Error
    def MAE(self, predicted, observed):
        predData = self.get_compartment(predicted)
        obsData = self.comparisonData[observed]
        errTotal = 0
        for i in range(len(predData)):
            errTotal += abs(predData[i] - obsData[i])
        return errTotal/len(obsData)
    def MAE_overtime(self, predicted, observed):
        predData = self.get_compartment(predicted)
        obsData = self.comparisonData[observed]
        errTotal = 0
        errTotalList = []
        for i in range(len(predData)):
            errTotal += abs(predData[i] - obsData[i])
            errTotalList.append(errTotal/(i+1))
        return errTotalList
    #Mean Percentage Error
    def MAPE(self, predicted, observed):
        predData = self.get_compartment(predicted)
        obsData = self.comparisonData[observed]
        errTotal = 0
        for i in range(len(predData)):
            errTotal += abs((predData[i] - obsData[i])/obsData[i])
        return errTotal*(100/len(predData))
    def MAPE_overtime(self, predicted, observed):
        predData = self.get_compartment(predicted)
        obsData = self.comparisonData[observed]
        errTotal = 0
        errTotalList = []
        for i in range(len(predData)):
            errTotal += abs((predData[i] - obsData[i])/obsData[i])
            errTotalList.append(errTotal*(100/(i+1)))
        return errTotalList
    #Mean Squared Error
    def MSE(self, predicted, observed):
        predData = self.get_compartment(predicted)
        obsData = self.comparisonData[observed]
        errTotal = 0
        for i in range(len(predData)):
            errTotal += (predData[i] - obsData[i])**2
        return errTotal/len(obsData)
    def MSE_overtime(self, predicted, observed):
        predData = self.get_compartment(predicted)
        obsData = self.comparisonData[observed]
        errTotal = 0
        errTotalList = []
        for i in range(len(predData)):
            errTotal += (predData[i] - obsData[i])**2
            errTotalList.append(errTotal/(i+1))
        return errTotalList
    #Root Mean Squared Error
    def RMSE(self, predicted, observed):
        return (self.MSE(predicted, observed))**0.5
    def RMSE_overtime(self, predicted, observed):
        return [x**0.5 for x in self.MSE_overtime(predicted, observed)]
    #For dfs
    def RMSE_independent(self, predictedDf, observedDf):
        predData = predictedDf[predictedDf.columns[0]]
        obsData = observedDf[observedDf.columns[0]]
        errTotal = 0
        for i in range(len(predData)):
            errTotal += (float(predData[i]) - float(obsData[i]))**2
        return (errTotal/len(obsData))**0.5
    def RMSE_independent_overtime(self, predictedDf, observedDf):
        predData = predictedDf[predictedDf.columns[0]]
        obsData = observedDf[observedDf.columns[0]]
        errTotal = 0
        errTotalList = []
        for i in range(len(predData)):
            errTotal += (float(predData[i]) - float(obsData[i]))**2
            errTotalList.append((errTotal/(i+1))**0.5)
        return errTotalList
    #Normalized Root Mean Squared Error
    def NRMSE(self, predicted, observed):
        obsData = self.comparisonData[observed]
        obsDelta = abs(max(obsData) - min(obsData))
        return self.RMSE(predicted, observed)/obsDelta
    def NRMSE_overtime(self, predicted, observed):
        obsData = self.comparisonData[observed]
        obsDelta = []
        for i in range(len(obsData)):
            temp = abs(max(obsData[:i+1]) - min(obsData[:i+1]))
            if temp > 0: 
                obsDelta.append(temp)
            else: 
                obsDelta.append(1.0)
        rmseBase = self.RMSE_overtime(predicted, observed)
        return  [rmseBase[i]/obsDelta[i] for i in range(len(obsDelta))]
    #Graphing Model
    def plot_model(self, nameList): #Plots models based on name of item, this requires that your model and import data have unique names
        dataDict = {}
        for name in nameList:
            if name in self.comparisonData.keys():
                dataDict[name] = self.comparisonData[name]
            else:
                dataDict[name] = self.get_compartment(name)

        for key in dataDict.keys():
            plt.plot(dataDict[key], label = key)
        plt.legend()
        plt.show()


    #DONE: Genetic Algorithm
    def genetic_algorithm(self, predicted, observed, excludedParams, numGenerations, numChromosomes, mutationChance, genePercent):
        population = []
        for i in range(numChromosomes):
            population.append(self.generate_random_chromosome(excludedParams, genePercent))
        
        for step in range(numGenerations):
            print('Generation: ', step)
            #Selection
            evaluatedPopulation = self.population_selection(population, predicted, observed, excludedParams)
            population = evaluatedPopulation
            #Crossover
            for i in range(0,len(evaluatedPopulation),2):
                if i+1 < len(evaluatedPopulation):
                    population += self.crossover_func(evaluatedPopulation[i], evaluatedPopulation[i+1])
            #Mutate
            for i in range(int(mutationChance*numChromosomes)):
                mutationChoice = random.randint(5, len(population)-1)
                population[mutationChoice] = self.mutate_chromosome(population[mutationChoice])

        #Adding excluded parameters back into chromosome
        finalSelection = self.population_selection(population, predicted, observed, excludedParams)
        for parameter in excludedParams:
            for chromosome in finalSelection:
                chromosome += [x for x in self.parameterPoints if x[0] == parameter]
        return finalSelection

    #Define chromosome
    def generate_random_chromosome(self, excludedParams, genePercent = 0.20):
        numGenes = int(genePercent*self.tmax)
        chromosome = []
        for parameter in self.initParameters.keys():
            if parameter not in excludedParams:
                for i in range(numGenes):
                    chromosome.append((parameter, self.randParamValue(parameter), self.randParamDate()))
        return random.Random(12345).sample(chromosome, k=len(chromosome))
    def randParamValue(self, parameter, variance=5):
        return random.random() * (self.initParameters[parameter]*variance)
    def randParamDate(self):
        return random.choice(range(self.tmax))
    #Define fitness function
    def fitness_func(self, predicted, observed):
        return self.RMSE(predicted, observed)
    #Define selection function
    def population_selection(self, population, predicted, observed, excludedParams):
        evaluatedPopulation = []
        excludedPoints = []
        temporaryModel = copy.deepcopy(self)
        for parameter in excludedParams:
            for point in self.parameterPoints:
                if point[0] == parameter:
                    excludedPoints.append(tuple(point))
        for chromosome in population:
            temporaryModel.parameterPoints = chromosome[:]
            temporaryModel.parameterPoints += excludedPoints
            temporaryModel._generate_model()
            temporaryModel.run()
            evaluatedPopulation.append((chromosome, temporaryModel.fitness_func(predicted, observed)))
        evaluatedPopulation.sort(key = lambda x: x[1])
        evaluatedPopulation = evaluatedPopulation[:len(evaluatedPopulation)//2]
        popWithoutEval = [x[0] for x in evaluatedPopulation]
        return popWithoutEval
    #Define crossover function
    def crossover_func(self, chrom1, chrom2):
        crossoverPoint = random.randint(1, len(chrom1)-1)
        return chrom1[:crossoverPoint] + chrom2[crossoverPoint:], chrom2[:crossoverPoint] + chrom1[crossoverPoint:]
    #Define mutation
    def mutate_chromosome(self, chromosome):
        chosen = random.randint(0,len(chromosome)-1)
        chosenName = chromosome[chosen][0]
        chromosome[chosen] = (chosenName, self.randParamValue(chosenName, variance=2), self.randParamDate())
        return chromosome

if __name__ == "__main__":
    pass

