# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['cachetory',
 'cachetory.backends',
 'cachetory.backends.async_',
 'cachetory.backends.sync',
 'cachetory.caches',
 'cachetory.decorators',
 'cachetory.interfaces',
 'cachetory.interfaces.backends',
 'cachetory.private',
 'cachetory.serializers',
 'cachetory.serializers.compressors']

package_data = \
{'': ['*']}

install_requires = \
['pydantic>=1.10.4,<2.0.0', 'typing-extensions>=4.4.0,<5.0.0']

extras_require = \
{'redis': ['redis>=4.4.2,<5.0.0'], 'zstd': ['zstd>=1.5.2.6,<2.0.0.0']}

setup_kwargs = {
    'name': 'cachetory',
    'version': '2.0.1',
    'description': 'Caching library with support for multiple cache backends',
    'long_description': '# `cachetory`\n\n[![PyPI](https://img.shields.io/pypi/v/cachetory)](https://pypi.org/project/cachetory/)\n[![Checks](https://github.com/kpn/cachetory/actions/workflows/check.yml/badge.svg)](https://github.com/kpn/cachetory/actions/workflows/check.yml)\n[![Coverage](https://codecov.io/gh/kpn/cachetory/branch/main/graph/badge.svg?token=UNYTTvxiWk)](https://codecov.io/gh/kpn/cachetory)\n[![Code style](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/kpn/cachetory)\n\n## Quickstart\n\n```python\nfrom cachetory import serializers\nfrom cachetory.backends import async_ as async_backends\nfrom cachetory.caches.async_ import Cache\n\n\ncache = Cache[int, bytes](\n    serializer=serializers.from_url("pickle://?pickle-protocol=4"),\n    backend=async_backends.from_url("redis://localhost:6379"),\n)\nasync with cache:\n    await cache.set("foo", 42)\n    assert await cache.get("foo") == 42\n```\n\n### Non-async\n\n```python\nfrom cachetory import serializers\nfrom cachetory.backends import sync as sync_backends\nfrom cachetory.caches.sync import Cache\n\n\ncache = Cache[int, bytes](\n    serializer=serializers.from_url("pickle://"),\n    backend=sync_backends.from_url("redis://localhost:6379"),\n)\nwith cache:\n    cache.set("foo", 42)\n    assert cache.get("foo") == 42\n```\n\n## Tutorial\n\n### Supported operations\n\n| Operation                                         |                                                              |\n|:--------------------------------------------------|:-------------------------------------------------------------|\n| `get(key, default)`                               | **Retrieve** a value (or return a default one)               |\n| `__getitem__(key)`                                | **Retrieve** a value or raise `KeyError` (only sync `Cache`) |\n| `get_many(*keys)`                                 | **Retrieve** many values as a dictionary                     |\n| `set(key, value, *, time_to_live, if_not_exists)` | **Set** a value and return if the value has been changed     |\n| `__setitem__(key, value)`                         | **Set** a value (only sync `Cache`)                          |\n| `set_many(items)`                                 | **Set** many values                                          |\n| `expire_in(key, time_to_live)`                    | **Set** an expiration duration on a key                      |\n| `delete(key)`                                     | **Delete** a key and return whether the key existed          |\n| `__delitem__(key)`                                | **Delete** a key (only sync `Cache`)                         |\n\n### Instantiating a `Cache`\n\nBoth sync and async `Cache`s requires at least these parameters to work:\n\n- `backend`: functions as a storage\n- `serializer`: is responsible for converting actual values from and to something that a backend would be able to store\n\n`Cache` may be annotated with a value type, like this: `Cache[ValueT]`, which provides type hints for the cache methods.\n\n### Instantiating a backend\n\nThere is a few ways to instantiate a backend:\n\n- By **directly instantiating** a backend class via its `__init__`\n- By instantiating a specific backend class **via its `from_url` class method**. In that case the URL is forwarded to underlying client (if any)\n- **By using `cachetory.[sync|async_].from_url` function.** In that case specific backend class is chosen by the URL\'s scheme (see the scheme badges below), and the URL is forwarded to its `from_url` class method. This is especially useful to configure an arbitrary backend from a single configuration option – instead of hard-coding a specific backend class.\n\n#### Examples\n\n```python\nimport redis\nimport cachetory.backends.sync\nimport cachetory.backends.async_\n\nbackend = cachetory.backends.sync.from_url("memory://")\nbackend = cachetory.backends.async_.from_url("dummy://")\nbackend = cachetory.backends.sync.RedisBackend(redis.Redis(...))\nbackend = cachetory.backends.async_.from_url("redis://localhost:6379/1")\n```\n\n### Instantiating a serializer\n\nInstantiating of a serializer is very much similar to that of a backend. To instantiate it by a URL use `cachetory.serializers.from_url` – unlike the back-end case there are no separate sync and async versions.\n\n`cachetory.serializers.from_url` supports scheme joining with `+`, as in `pickle+zlib://`. In that case multiple serializers are instantiated and applied sequentially (in the example a value would be serialized by `pickle` and the serialized value is then compressed by `zlib`). Deserialization order is, of course, the opposite.\n\n#### Examples\n\n```python\nimport pickle\n\nimport cachetory.serializers\n\nserializer = cachetory.serializers.from_url("pickle+zstd://")\nserializer = cachetory.serializers.from_url("pickle+zstd://?pickle-protocol=4&compression-level=3")\nserializer = cachetory.serializers.from_url("null://")\nserializer = cachetory.serializers.NoopSerializer()\nserializer = cachetory.serializers.PickleSerializer(pickle_protocol=pickle.DEFAULT_PROTOCOL)\n```\n\n### Decorators\n\n#### Decorate a function with `@cached`\n\n`@cached` performs [memoization](https://en.wikipedia.org/wiki/Memoization) of a wrapped function:\n\n```python\nfrom cachetory.caches.sync import Cache\nfrom cachetory.decorators.shared import make_default_key\nfrom cachetory.decorators.sync import cached\n\ncache = Cache[int](backend=..., serializer=...)\nanother_cache = Cache[int](backend=..., serializer=...)\n\n\n@cached(\n    cache,  # may also be a callable that returns a specific cache for each call, e.g.:\n    # `cache=lambda wrapped_callable, *args, **kwargs: cache if … else another_cache`\n\n    # The following parameters are optional (shown the defaults):\n    make_key=make_default_key,  # cache key generator\n    time_to_live=None,  # forwarded to `Cache.set`\n    if_not_exists=False,  # forwarded to `Cache.set`\n)\ndef expensive_function() -> int:\n    return 42\n```\n\nThere\'s a few `make_key` functions provided by default:\n\n- `cachetory.decorators.shared.make_default_key` builds a human-readable cache key out of decorated function fully-qualified name and stringified arguments. The length of the key depends on the argument values.\n- `cachetory.decorators.shared.make_default_hashed_key` calls `make_default_key` under the hood but hashes the key and returns a hash hex digest – making it a fixed-length key and not human-readable.\n\n## Supported backends\n\nThe badges indicate which schemes are supported by a particular backend, and which package extras are required for it – if any:\n\n### Redis\n\n![scheme: redis](https://img.shields.io/badge/scheme-redis://-important)\n![scheme: rediss](https://img.shields.io/badge/scheme-rediss://-important)\n![scheme: redis+unix](https://img.shields.io/badge/scheme-redis+unix://-important)\n![extra: redis](https://img.shields.io/badge/extra-redis-blue)\n\n| Sync                                   | Async                                    |\n|:---------------------------------------|:-----------------------------------------|\n| `cachetory.backends.sync.RedisBackend` | `cachetory.backends.async_.RedisBackend` |\n\nThe URL is forwarded to the underlying client, which means one can use whatever options the client provides. The only special case is `redis+unix://`: the leading `redis+` is first stripped and the rest is forwarded to the client.\n\nAll the cache operations are **atomic** in both sync and async, including `get_many` and `set_many`.\n\nConsider explicitly adding [`hiredis`](https://github.com/redis/hiredis) to your dependencies for faster performance.\n\n### Memory\n\n![scheme: memory](https://img.shields.io/badge/scheme-memory://-important)\n\n| Sync                                    | Async                                     |\n|:----------------------------------------|:------------------------------------------|\n| `cachetory.backends.sync.MemoryBackend` | `cachetory.backends.async_.MemoryBackend` |\n\nSimple memory backend that stores values in a plain dictionary.\n\nNote the following **caveats**:\n\n- This backend does **not** copy values. Meaning that mutating a stored value mutates it in the backend too. If this is not desirable, consider using another serializer or making up your own serializer which copies values in its `serialize` method.\n- Expired items actually get deleted **only** when accessed. If you put a value into the backend and never try to retrieve it – it\'ll stay in memory forever.\n\n### Dummy\n\n![scheme: dummy](https://img.shields.io/badge/scheme-dummy://-important)\n\n| Sync                                    | Async                                     |\n|:----------------------------------------|:------------------------------------------|\n| `cachetory.backends.sync.DummyBackend`  | `cachetory.backends.async_.DummyBackend`  |\n\nDummy backend that always succeeds but never stores anything. Any values get forgotten immediately,\nand operations behave as if the cache always is empty.\n\n## Supported serializers\n\n### Pickle\n\n![scheme: pickle](https://img.shields.io/badge/scheme-pickle://-important)\n\nSerializes using the standard [`pickle`](https://docs.python.org/3/library/pickle.html) module.\n\n| Class                                    |\n|:-----------------------------------------|\n| `cachetory.serializers.PickleSerializer` |\n\n| URL parameter     |                                                                                                  |\n|:------------------|--------------------------------------------------------------------------------------------------|\n| `pickle-protocol` | Version of [`pickle` protocol](https://docs.python.org/3/library/pickle.html#data-stream-format) |\n\n### No-operation\n\n![scheme: noop](https://img.shields.io/badge/scheme-noop://-important)\n![scheme: null](https://img.shields.io/badge/scheme-null://-important)\n\n| Class                                   |\n|:----------------------------------------|\n| `cachetory.serializers.NoopSerializer`  |\n\n`NoopSerializer` does nothing and just bypasses value back and forth. Most of the backends don\'t support that and require some kind of serialization.\n\nHowever, it is possible to use `NoopSerializer` with `MemoryBackend`, because the latter just stores all values in a Python dictionary and doesn\'t necessarily require values to be serialized.\n\n## Supported compressors\n\n**Compressor** is basically just a partial case of serializer: it\'s a serializer from `bytes` to and from `bytes`, which by definition provides some kind of data compression.\n\nIt also means that you can use a compressor alone, effectively making a cache of compressed blobs:\n\n```python\nfrom datetime import timedelta\n\nfrom cachetory.caches.sync import Cache\nfrom cachetory.serializers.compressors import ZlibCompressor\nfrom cachetory.backends.sync import RedisBackend\n\ncache = Cache[bytes](\n    serializer=ZlibCompressor(),\n    backend=RedisBackend(...),\n)\ncache.set(\n    "my-blob",\n    b"this huge blob will be compressed and stored in Redis for an hour",\n    time_to_live=timedelta(hours=1),\n)\n```\n\n## Zlib\n\n![scheme: zlib](https://img.shields.io/badge/scheme-zlib://-important)\n\nUses the built-in [`zlib`](https://docs.python.org/3/library/zlib.html) module.\n\n| Class                                              |\n|:---------------------------------------------------|\n| `cachetory.serializers.compressors.ZlibCompressor` |\n\n| URL parameter         |                                                     |\n|:----------------------|-----------------------------------------------------|\n| `compression-level`   | From `0` (no compression) to `9` (best compression) |\n\n## Zstandard\n\n![scheme: zstd](https://img.shields.io/badge/scheme-zstd://-important)\n![scheme: zstandard](https://img.shields.io/badge/scheme-zstandard://-important)\n![extra: zstd](https://img.shields.io/badge/extra-zstd-blue)\n\nUses [`python-zstd`](https://github.com/sergey-dryabzhinsky/python-zstd) for [Zstandard](https://facebook.github.io/zstd/) compression.\n\n| Class                                              |\n|:---------------------------------------------------|\n| `cachetory.serializers.compressors.ZstdCompressor` |\n\n| URL parameter         |                                                             |\n|:----------------------|-------------------------------------------------------------|\n| `compression-level`   | See: https://github.com/sergey-dryabzhinsky/python-zstd#api |\n| `compression-threads` | See: https://github.com/sergey-dryabzhinsky/python-zstd#api |\n',
    'author': 'Pavel Perestoronin',
    'author_email': 'pavel.perestoronin@kpn.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/kpn/cachetory',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7.0,<4.0.0',
}


setup(**setup_kwargs)
