from typing import ClassVar
import pluggy
import importlib.metadata

from dlt.common.configuration.specs.base_configuration import ContainerInjectableContext

hookspec = pluggy.HookspecMarker("dlt")
hookimpl = pluggy.HookimplMarker("dlt")


class PluginContext(ContainerInjectableContext):
    global_affinity: ClassVar[bool] = True

    manager: pluggy.PluginManager

    def __init__(self) -> None:
        super().__init__()
        self.manager = pluggy.PluginManager("dlt")

        # TODO: we need to solve circular deps somehow

        # run_context
        from dlt.common.runtime import run_context

        self.manager.add_hookspecs(run_context)
        self.manager.register(run_context)

        # cli
        from dlt.cli import plugins

        self.manager.add_hookspecs(plugins)
        self.manager.register(plugins)

        load_setuptools_entrypoints(self.manager)


def manager() -> pluggy.PluginManager:
    """Returns current plugin context"""
    from .container import Container

    return Container()[PluginContext].manager


def load_setuptools_entrypoints(m: pluggy.PluginManager) -> None:
    """Scans setuptools distributions that are path or have name starting with `dlt-`
    loads entry points in group `dlt` and instantiates them to initialize contained plugins
    """

    for dist in list(importlib.metadata.distributions()):
        # skip named dists that do not start with dlt-
        if hasattr(dist, "name") and (dist.name is None or not dist.name.startswith("dlt-")):
            continue
        for ep in dist.entry_points:
            if (
                ep.group != "dlt"
                # already registered
                or m.get_plugin(ep.name)
                or m.is_blocked(ep.name)
            ):
                continue
            plugin = ep.load()
            m.register(plugin, name=ep.name)
            m._plugin_distinfo.append((plugin, pluggy._manager.DistFacade(dist)))
