import { IResource, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
/**
 * Represents the function's source code.
 *
 * @stability stable
 */
export declare abstract class FunctionCode {
    /**
     * Inline code for function.
     *
     * @param code The actual function code.
     * @returns `InlineCode` with inline code.
     * @stability stable
     */
    static fromInline(code: string): FunctionCode;
    /**
     * renders the function code.
     *
     * @stability stable
     */
    abstract render(): string;
}
/**
 * Represents a CloudFront Function.
 *
 * @stability stable
 */
export interface IFunction extends IResource {
    /**
     * The name of the function.
     *
     * @stability stable
     * @attribute true
     */
    readonly functionName: string;
    /**
     * The ARN of the function.
     *
     * @stability stable
     * @attribute true
     */
    readonly functionArn: string;
}
/**
 * Attributes of an existing CloudFront Function to import it.
 *
 * @stability stable
 */
export interface FunctionAttributes {
    /**
     * The name of the function.
     *
     * @stability stable
     */
    readonly functionName: string;
    /**
     * The ARN of the function.
     *
     * @stability stable
     */
    readonly functionArn: string;
}
/**
 * Properties for creating a CloudFront Function.
 *
 * @stability stable
 */
export interface FunctionProps {
    /**
     * A name to identify the function.
     *
     * @default - generated from the `id`
     * @stability stable
     */
    readonly functionName?: string;
    /**
     * A comment to describe the function.
     *
     * @default - same as `functionName`
     * @stability stable
     */
    readonly comment?: string;
    /**
     * The source code of the function.
     *
     * @stability stable
     */
    readonly code: FunctionCode;
}
/**
 * A CloudFront Function.
 *
 * @stability stable
 * @resource AWS::CloudFront::Function
 */
export declare class Function extends Resource implements IFunction {
    /**
     * Imports a function by its name and ARN.
     *
     * @stability stable
     */
    static fromFunctionAttributes(scope: Construct, id: string, attrs: FunctionAttributes): IFunction;
    /**
     * the name of the CloudFront function.
     *
     * @stability stable
     * @attribute true
     */
    readonly functionName: string;
    /**
     * the ARN of the CloudFront function.
     *
     * @stability stable
     * @attribute true
     */
    readonly functionArn: string;
    /**
     * the deployment stage of the CloudFront function.
     *
     * @stability stable
     * @attribute true
     */
    readonly functionStage: string;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: FunctionProps);
    private generateName;
}
/**
 * The type of events that a CloudFront function can be invoked in response to.
 *
 * @stability stable
 */
export declare enum FunctionEventType {
    /**
     * The viewer-request specifies the incoming request.
     *
     * @stability stable
     */
    VIEWER_REQUEST = "viewer-request",
    /**
     * The viewer-response specifies the outgoing response.
     *
     * @stability stable
     */
    VIEWER_RESPONSE = "viewer-response"
}
/**
 * Represents a CloudFront function and event type when using CF Functions.
 *
 * The type of the {@link AddBehaviorOptions.functionAssociations} property.
 *
 * @stability stable
 */
export interface FunctionAssociation {
    /**
     * The CloudFront function that will be invoked.
     *
     * @stability stable
     */
    readonly function: IFunction;
    /**
     * The type of event which should invoke the function.
     *
     * @stability stable
     */
    readonly eventType: FunctionEventType;
}
