"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * The generations of AWS load balancing solutions.
 */
var LoadBalancerGeneration;
(function (LoadBalancerGeneration) {
    /**
     * The first generation (ELB Classic).
     */
    LoadBalancerGeneration[LoadBalancerGeneration["FIRST"] = 0] = "FIRST";
    /**
     * The second generation (ALB and NLB).
     */
    LoadBalancerGeneration[LoadBalancerGeneration["SECOND"] = 1] = "SECOND";
})(LoadBalancerGeneration = exports.LoadBalancerGeneration || (exports.LoadBalancerGeneration = {}));
/**
 * An interface of an abstract load balancer, as needed by CodeDeploy.
 * Create instances using the static factory methods:
 * {@link #classic}, {@link #application} and {@link #network}.
 */
class LoadBalancer {
    /**
     * Creates a new CodeDeploy load balancer from a Classic ELB Load Balancer.
     *
     * @param loadBalancer a classic ELB Load Balancer
     */
    static classic(loadBalancer) {
        class ClassicLoadBalancer extends LoadBalancer {
            constructor() {
                super(...arguments);
                this.generation = LoadBalancerGeneration.FIRST;
                this.name = loadBalancer.loadBalancerName;
            }
        }
        return new ClassicLoadBalancer();
    }
    /**
     * Creates a new CodeDeploy load balancer from an Application Load Balancer Target Group.
     *
     * @param albTargetGroup an ALB Target Group
     */
    static application(albTargetGroup) {
        class AlbLoadBalancer extends LoadBalancer {
            constructor() {
                super(...arguments);
                this.generation = LoadBalancerGeneration.SECOND;
                this.name = albTargetGroup.targetGroupName;
            }
        }
        return new AlbLoadBalancer();
    }
    /**
     * Creates a new CodeDeploy load balancer from a Network Load Balancer Target Group.
     *
     * @param nlbTargetGroup an NLB Target Group
     */
    static network(nlbTargetGroup) {
        class NlbLoadBalancer extends LoadBalancer {
            constructor() {
                super(...arguments);
                this.generation = LoadBalancerGeneration.SECOND;
                this.name = nlbTargetGroup.targetGroupName;
            }
        }
        return new NlbLoadBalancer();
    }
}
exports.LoadBalancer = LoadBalancer;
//# sourceMappingURL=data:application/json;base64,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