"""Contains interfaces for package tools executor."""
import os
import sys
import click
from termcolor import colored
from pypans.file import Template
from pypans.project import Project, User  # noqa: I100


def _colored_output(string: str, color: str) -> int:
    """Writes data into colored output.

    Returns number of characters to write.
    """
    return sys.stdout.write(f"{colored(string, color)}\n")


class _Environment:
    """Representation of project environment."""

    def __init__(self, name: str, user: User) -> None:
        self._name = name
        self._user = user
        self._project: Project = Project(name, user)

    def setup_project(self) -> None:
        """Builds given project."""
        self._project.build_package()
        self._project.build_tests()
        self._project.build_meta()

    def setup_git(self) -> None:
        """Sets up git for a project."""

        def prepare(to_repo: str) -> None:
            os.system("git init")
            os.system(f"git config --local user.name {self._user.name}")
            os.system(f"git config --local user.email {self._user.email}")
            os.system(f"git remote add origin {to_repo}")

        git: str = input(colored(">>> Would you like to setup git for a project? (yes/no): ", "green"))
        if git == "yes":
            prepare(to_repo=input(colored(">>> Please enter github repo (e.g git@github:user/project.git): ", "green")))
        else:
            _colored_output(string=f"Project setup with git is skipped for `{self._name}` app", color="red")

    def install_requirements(self) -> None:
        """Installs project requirements."""

        def install_from(file: Template) -> None:  # noqa: VNE002
            os.system(f"pip install -r {file}")

        install: str = input(colored(">>> Would you like to install project dependencies? (yes/no): ", "green"))
        if install == "yes":
            install_from(file=Template.REQUIREMENTS)
            install_from(file=Template.DEV_REQUIREMENTS)
        else:
            _colored_output(string=f"Dependencies installation is skipped for `{self._name}` app", color="red")


def __build_environment(name: str, user: User) -> None:
    """Builds fully-fledged environment."""
    environment: _Environment = _Environment(name, user)
    environment.setup_project()
    environment.setup_git()
    environment.install_requirements()


@click.command()
@click.option("--start", "-s", show_default=True, is_flag=True, help="Starts python project composer")
def easypan(start: bool) -> None:
    """Runs `pypan` command line utility.

    Program allows to interactively compose fresh python project from template.
    """
    if start:
        __build_environment(
            name=input(colored(">>> Please name your application (e.g bomber): ", "green")).lower().replace("-", "_"),
            user=User(
                name=input(colored(">>> Please enter your username (e.g John Udot): ", "green")),
                email=input(colored(">>> Please enter your email (e.g user@gmail.com): ", "green")),
            ),
        )
        _colored_output(string=f"🐍 Successfully created fresh python project 🐍", color="magenta")
    else:
        click.echo(click.get_current_context().get_help())


if __name__ == "__main__":
    easypan()  # pylint:disable=no-value-for-parameter
