import PIL.Image
import PIL.ImageDraw
import PIL.ImageEnhance
import PIL.ImageFilter
import PIL.ImageFont
import PIL.ImageOps
from genflow.metadata.types import Tensor
from genflow.workflows.processing_context import ProcessingContext
from genflow.metadata.types import ImageRef
from genflow.workflows.genflow_node import GenflowNode
import numpy as np
from pydantic import Field
from typing import Literal


# class SaveImageNode(Node):
#     """
#     ## Save Image Node
#     ### Category: Image

#     #### Description
#     This node saves an image to the workspace.

#     The Save Image Node is used to save an image to the workspace. It can be used to save images that are generated by other nodes in the workflow. The node can be configured to save the image in a specific format, such as PNG or JPEG.

#     #### Applications
#     - Saving images generated by other nodes in the workflow.
#     - Saving images for later use in other workflows.
#     - Saving images for use in other applications.

#     #### Example
#     Suppose you have a workflow that generates an image. You want to save this image for later use in another workflow. You can use the Save Image Node to save the image to the workspace. Then, you can use the Image Load Node to load the image into another workflow.


#     """

#     image: ImageRef = ImageRef()
#     folder: FolderRef = Field(
#         default=FolderRef(), description="The folder to save the image in."
#     )
#     name: str = Field(default="%Y-%m-%d_%H-%M-%S.png")

#     async def process(self, context: ProcessingContext) -> ImageRef:
#         image = await context.to_pil(self.image)

#         name = datetime.now().strftime(self.name)

#         return await context.image_from_pil(
#             image=image, name=name, parent_id=self.folder.asset_id
#         )


class ImageToTensorNode(GenflowNode):
    """
    This node converts an image into a tensor.

    A tensor is a type of data structure used in AI. This particular node is used to transform digital image data into a tensor format, which can then be processed further in various AI workflows. Image data often comes in grayscale or color (RGB/RGBA), and this node is capable of handling these different forms.

    #### Applications
    - Pre-processing image data for use in machine learning models.
    - Transforming images to perform image analysis.
    - Preparing image data for deep learning algorithms.
    """

    image: ImageRef = Field(
        description="The input image to convert to a tensor. The image should have either 1 (grayscale), 3 (RGB), or 4 (RGBA) channels."
    )

    async def process(self, context: ProcessingContext) -> Tensor:
        image = await context.to_pil(self.image)
        image_data = np.array(image)
        tensor_data = image_data / 255.0
        if len(tensor_data.shape) == 2:
            tensor_data = tensor_data[:, :, np.newaxis]
        return Tensor.from_numpy(tensor_data)


class TensorToImageNode(GenflowNode):
    """
    This node converts a tensor (a generic geometric object) into an image.

    The Tensor To Image node is a special tool designed to transform tensor data structures into standard images. This is particularly useful for visualising and processing tensor data in an image format. The node can handle tensors with 1, 3, or 4 channels.

    #### Applications
    - Transforming machine learning model outputs (often tensors) into images for visualization.
    - Converting tensors generated from various image processing algorithms back into images.
    """

    tensor: Tensor = Field(
        description="The input tensor to convert to an image. Should have either 1, 3, or 4 channels."
    )

    async def process(self, context: ProcessingContext) -> ImageRef:
        tensor_data = self.tensor.to_numpy()
        if tensor_data.ndim not in [2, 3]:
            raise ValueError("The tensor should have 2 or 3 dimensions (HxW or HxWxC).")
        if (tensor_data.ndim == 3) and (tensor_data.shape[2] not in [1, 3, 4]):
            raise ValueError("The tensor channels should be either 1, 3, or 4.")
        if (tensor_data.ndim == 3) and (tensor_data.shape[2] == 1):
            tensor_data = tensor_data.reshape(
                tensor_data.shape[0], tensor_data.shape[1]
            )
        tensor_data = (tensor_data * 255).astype(np.uint8)
        output_image = PIL.Image.fromarray(tensor_data)
        return await context.image_from_pil(output_image)


class PasteNode(GenflowNode):
    """
    The PasteNode is used to paste one image onto another.

    The purpose of this node is to allow you to modify a base image by adding another image at a specified location. This node pastes an image ("paste") at the given coordinates ("left" and "top") onto another image ("image"). The result is a combination of two images.

    #### Applications
    - Creating collages: You can repeatedly paste different images onto a base image to create a collage.
    - Watermarking: You can paste a watermark image onto your main image at a desired location.
    - Overlaying Images: You can overlay one image onto another to create visual effects or combine information.
    """

    image: ImageRef = Field(default=ImageRef(), description="The image to paste into.")
    paste: ImageRef = Field(default=ImageRef(), description="The image to paste.")
    left: int = Field(default=0, ge=0, le=4096, description="The left coordinate.")
    top: int = Field(default=0, ge=0, le=4096, description="The top coordinate.")

    async def process(self, context: ProcessingContext) -> ImageRef:
        image = await context.to_pil(self.image)
        paste = await context.to_pil(self.paste)
        image.paste(paste, (self.left, self.top))
        return await context.image_from_pil(image)


class BlendNode(GenflowNode):
    """
    BlendNode is a tool for perfectly merging two images.

    This node simplifies the procedure of combining two images into one, maintaining the features of both. The achieved image balance can be controlled by adjusting the mix ratio. A unique aspect of this node is its ability to automatically resize images to make them identical, enabling smooth blending of the images.

    #### Applications
    - Creating a blended collage of favourite pictures.
    - Designing nuanced graphics by blending two images for artistic effect.
    - In building models for computer vision, generating enriched training data by blending different images.
    """

    image1: ImageRef = Field(
        default=ImageRef(), description="The first image to blend."
    )
    image2: ImageRef = Field(
        default=ImageRef(), description="The second image to blend."
    )
    alpha: float = Field(default=0.5, ge=0.0, le=1.0, description="The mix ratio.")

    async def process(self, context: ProcessingContext) -> ImageRef:
        image1 = await context.to_pil(self.image1)
        image2 = await context.to_pil(self.image2)
        if image1.size != image2.size:
            image2 = PIL.ImageOps.fit(image2, image1.size)
        image = PIL.Image.blend(image1, image2, self.alpha)
        return await context.image_from_pil(image)


class CompositeNode(GenflowNode):
    """
    The Composite Node is used to combine two images into a single output image.

    The Composite Node is a tool for combining two separate images together, using a mask to specify how they should be merged. The mask controls how much of each image is shown in the final result, and it can be any image with the same dimensions as the input images. The result is a new image that contains elements of the first and second images in accordance with the mask.

    #### Applications
    - Combining elements from multiple photos into a single image.
    - Overlaying an image with a semi-transparent texture.
    - Creating a "before and after" image that shows the effects of some process or change.
    """

    image1: ImageRef = Field(
        default=ImageRef(), description="The first image to composite."
    )
    image2: ImageRef = Field(
        default=ImageRef(), description="The second image to composite."
    )
    mask: ImageRef = Field(
        default=ImageRef(), description="The mask to composite with."
    )

    async def process(self, context: ProcessingContext) -> ImageRef:
        image1 = await context.to_pil(self.image1)
        image2 = await context.to_pil(self.image2)
        mask = await context.to_pil(self.mask)
        image1 = image1.convert("RGBA")
        image2 = image2.convert("RGBA")
        mask = mask.convert("RGBA")
        if image1.size != image2.size:
            image2 = PIL.ImageOps.fit(image2, image1.size)
        if image1.size != mask.size:
            mask = PIL.ImageOps.fit(mask, image1.size)
        image = PIL.Image.composite(image1, image2, mask)
        return await context.image_from_pil(image)
