import * as kms from 'aws-cdk-lib/aws-kms';
import * as cdk from 'aws-cdk-lib';
import { Construct } from 'constructs';
import { AcceleratorType } from './accelerator-type';
import { InstanceType } from './instance-type';
import { IModel } from './model';
/**
 * The interface for a SageMaker EndpointConfig resource.
 */
export interface IEndpointConfig extends cdk.IResource {
    /**
     * The ARN of the endpoint configuration.
     *
     * @attribute
     */
    readonly endpointConfigArn: string;
    /**
     * The name of the endpoint configuration.
     *
     * @attribute
     */
    readonly endpointConfigName: string;
}
/**
 * Common construction properties for all production variant types (e.g., instance, serverless).
 */
interface ProductionVariantProps {
    /**
     * Determines initial traffic distribution among all of the models that you specify in the
     * endpoint configuration. The traffic to a production variant is determined by the ratio of the
     * variant weight to the sum of all variant weight values across all production variants.
     *
     * @default 1.0
     */
    readonly initialVariantWeight?: number;
    /**
     * The model to host.
     */
    readonly model: IModel;
    /**
     * Name of the production variant.
     */
    readonly variantName: string;
}
/**
 * Construction properties for an instance production variant.
 */
export interface InstanceProductionVariantProps extends ProductionVariantProps {
    /**
     * The size of the Elastic Inference (EI) instance to use for the production variant. EI instances
     * provide on-demand GPU computing for inference.
     *
     * @default - none
     */
    readonly acceleratorType?: AcceleratorType;
    /**
     * Number of instances to launch initially.
     *
     * @default 1
     */
    readonly initialInstanceCount?: number;
    /**
     * Instance type of the production variant.
     *
     * @default InstanceType.T2_MEDIUM
     */
    readonly instanceType?: InstanceType;
}
/**
 * Represents common attributes of all production variant types (e.g., instance, serverless) once
 * associated to an EndpointConfig.
 */
interface ProductionVariant {
    /**
     * Determines initial traffic distribution among all of the models that you specify in the
     * endpoint configuration. The traffic to a production variant is determined by the ratio of the
     * variant weight to the sum of all variant weight values across all production variants.
     */
    readonly initialVariantWeight: number;
    /**
     * The name of the model to host.
     */
    readonly modelName: string;
    /**
     * The name of the production variant.
     */
    readonly variantName: string;
}
/**
 * Represents an instance production variant that has been associated with an EndpointConfig.
 *
 * @internal
 */
export interface InstanceProductionVariant extends ProductionVariant {
    /**
     * The size of the Elastic Inference (EI) instance to use for the production variant. EI instances
     * provide on-demand GPU computing for inference.
     *
     * @default - none
     */
    readonly acceleratorType?: AcceleratorType;
    /**
     * Number of instances to launch initially.
     */
    readonly initialInstanceCount: number;
    /**
     * Instance type of the production variant.
     */
    readonly instanceType: InstanceType;
}
/**
 * Construction properties for a SageMaker EndpointConfig.
 */
export interface EndpointConfigProps {
    /**
     * Name of the endpoint configuration.
     *
     * @default - AWS CloudFormation generates a unique physical ID and uses that ID for the endpoint
     * configuration's name.
     */
    readonly endpointConfigName?: string;
    /**
     * Optional KMS encryption key associated with this stream.
     *
     * @default - none
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * A list of instance production variants. You can always add more variants later by calling
     * `EndpointConfig#addInstanceProductionVariant`.
     *
     * @default - none
     */
    readonly instanceProductionVariants?: InstanceProductionVariantProps[];
}
/**
 * Defines a SageMaker EndpointConfig.
 */
export declare class EndpointConfig extends cdk.Resource implements IEndpointConfig {
    /**
     * Imports an EndpointConfig defined either outside the CDK or in a different CDK stack.
     * @param scope the Construct scope.
     * @param id the resource id.
     * @param endpointConfigArn the ARN of the endpoint configuration.
     */
    static fromEndpointConfigArn(scope: Construct, id: string, endpointConfigArn: string): IEndpointConfig;
    /**
     * Imports an EndpointConfig defined either outside the CDK or in a different CDK stack.
     * @param scope the Construct scope.
     * @param id the resource id.
     * @param endpointConfigName the name of the endpoint configuration.
     */
    static fromEndpointConfigName(scope: Construct, id: string, endpointConfigName: string): IEndpointConfig;
    /**
     * The ARN of the endpoint configuration.
     */
    readonly endpointConfigArn: string;
    /**
     * The name of the endpoint configuration.
     */
    readonly endpointConfigName: string;
    private readonly instanceProductionVariantsByName;
    constructor(scope: Construct, id: string, props?: EndpointConfigProps);
    /**
     * Add production variant to the endpoint configuration.
     *
     * @param props The properties of a production variant to add.
     */
    addInstanceProductionVariant(props: InstanceProductionVariantProps): void;
    /**
     * Get instance production variants associated with endpoint configuration.
     *
     * @internal
     */
    get _instanceProductionVariants(): InstanceProductionVariant[];
    /**
     * Find instance production variant based on variant name
     * @param name Variant name from production variant
     *
     * @internal
     */
    _findInstanceProductionVariant(name: string): InstanceProductionVariant;
    private validateProductionVariants;
    private validateInstanceProductionVariantProps;
    /**
     * Render the list of instance production variants.
     */
    private renderInstanceProductionVariants;
}
export {};
