"""This file contains metadata describing the results from Gaussian"""

metadata = {}

methods = {
    "AM1: AM1 semiempirical HF": {
        "method": "AM1",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "analytic",
        "nobasis": True,
        "citations": ((1, "doi:10.1021/ja00299a024"),),
    },
    "PM3: PM3 semiempirical HF": {
        "method": "PM3",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "analytic",
        "nobasis": True,
        "citations": (
            (1, "https://doi.org/10.1002/jcc.540100208"),
            (1, "https://doi.org/10.1002/jcc.540100209"),
        ),
    },
    "PM3MM: PM3 semiempirical HF with MM corrections": {
        "method": "PM3MM",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "analytic",
        "nobasis": True,
        "citations": (
            (1, "https://doi.org/10.1002/jcc.540100208"),
            (1, "https://doi.org/10.1002/jcc.540100209"),
        ),
    },
    "PM6: PM6 semiempirical HF": {
        "method": "PM6",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "analytic",
        "nobasis": True,
        "citations": ((1, "Stewart_2007"),),
    },
    "PDDG: PDDG variant of PM6 semiempirical HF": {
        "method": "PDDG",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "analytic",
        "nobasis": True,
        "citations": (
            (1, "https://doi.org/10.1002/jcc.10162"),
            (2, "doi:10.1021/ct0500287"),
        ),
    },
    "PM7: modified PM7 semiempirical HF": {
        "method": "PM7",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "analytic",
        "nobasis": True,
        "citations": ((1, "Stewart_2012"),),
    },
    "PM7MOPAC: PM7 semiempirical HF as in MOPAC": {
        "method": "PM7MOPAC",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "analytic",
        "nobasis": True,
        "citations": ((1, "Stewart_2012"),),
    },
    "DFT: Kohn-Sham density functional theory": {
        "method": "DFT",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "analytic",
        "citations": (
            (1, "PhysRev.136.B864"),
            (1, "PhysRev.140.A1133"),
        ),
    },
    "HF: Hartree-Fock self consistent field (SCF)": {
        "method": "HF",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "analytic",
        "citations": (
            (1, "RevModPhys.23.69"),
            (1, "berthier1954extension"),
            (1, "10.1063/1.1740120"),
        ),
    },
    "MP2: 2nd-order Møller–Plesset perturbation theory": {
        "method": "MP2",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "analytic",
        "freeze core?": True,
        "citations": (
            (1, "PhysRev.46.618"),
            (2, "FRISCH1990275"),
        ),
    },
    "MP3: 3rd-order Møller–Plesset perturbation theory": {
        "method": "MP3",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "analytic",
        "freeze core?": True,
        "citations": (
            (1, "PhysRev.46.618"),
            (2, "https://doi.org/10.1002/qua.560100802"),
            (2, "https://doi.org/10.1002/qua.560120820"),
        ),
    },
    "MP4(SDTQ): 4th-order Møller–Plesset perturbation theory": {
        "method": "MP4",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "numeric",
        "freeze core?": True,
        "citations": (
            (1, "PhysRev.46.618"),
            (2, "https://doi.org/10.1002/qua.560140109"),
            (2, "10.1063/1.439657"),
        ),
    },
    "MP4(SDQ): 4th-order Møller–Plesset perturbation theory": {
        "method": "MP4(SDQ)",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "analytic",
        "freeze core?": True,
        "citations": (
            (1, "PhysRev.46.618"),
            (2, "https://doi.org/10.1002/qua.560140109"),
        ),
    },
    "MP4(DQ): 4th-order Møller–Plesset perturbation theory": {
        "method": "MP4(DQ)",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "analytic",
        "freeze core?": True,
        "citations": (
            (1, "PhysRev.46.618"),
            (2, "https://doi.org/10.1002/qua.560140109"),
        ),
    },
    "MP5: 5th-order Møller–Plesset perturbation theory": {
        "method": "MP5",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "analytic",
        "freeze core?": True,
        "citations": (
            (1, "PhysRev.46.618"),
            (2, "doi:10.1021/j100377a033"),
        ),
    },
    "QCISD: Quadratic CI": {
        "method": "QCI",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "analytic",
        "freeze core?": True,
        "citations": ((1, "10.1063/1.453520"),),
    },
    "QCISD(T): Quadratic CI with triples correction": {
        "method": "QCISD(T)",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "numerical",
        "freeze core?": True,
        "citations": ((1, "10.1063/1.453520"),),
    },
    "CCD: coupled cluster doubles": {
        "method": "CCD",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "analytic",
        "freeze core?": True,
        "citations": (
            (1, "https://doi.org/10.1002/qua.560140504"),
            (2, "https://doi.org/10.1002/qua.560140503"),
        ),
    },
    "CCSD: coupled cluster singles & doubles": {
        "method": "CCSD",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "numeric",
        "freeze core?": True,
        "citations": (
            (1, "https://doi.org/10.1002/qua.560140504"),
            (1, "doi:https://doi.org/10.1002/9780470143599.ch2"),
            (2, "https://doi.org/10.1002/qua.560140503"),
        ),
    },
    "CCSD(T): coupled cluster singles & doubles plus triples": {
        "method": "CCSD(T)",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "numeric",
        "freeze core?": True,
        "citations": (
            (1, "https://doi.org/10.1002/qua.560140504"),
            (2, "https://doi.org/10.1002/qua.560140503"),
            (2, "10.1063/1.443164"),
        ),
    },
    "CBS-4M: Complete Basis Set method of Petersson, et al.": {
        "method": "CBS-4M",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "none",
        "nobasis": True,
        "citations": (
            (1, "10.1063/1.470985"),
            (1, "10.1063/1.481224"),
        ),
    },
    "CBS-QB3: Complete Basis Set method of Petersson, et al.": {
        "method": "CBS-QB3",
        "calculation": ["energy", "gradients"],
        "level": "normal",
        "gradients": "none",
        "nobasis": True,
        "citations": (
            (1, "10.1063/1.477924"),
            (1, "10.1063/1.481224"),
        ),
    },
    "CBS-APNO: Complete Basis Set method of Petersson, et al.": {
        "method": "CBS-APNO",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "none",
        "nobasis": True,
        "citations": ((1, "10.1063/1.470985"),),
    },
    "G1: Gaussian-1 composite method": {
        "method": "G1",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "none",
        "nobasis": True,
        "citations": (
            (1, "10.1063/1.456415"),
            (1, "10.1063/1.458892"),
        ),
    },
    "G2: Gaussian-2 composite method": {
        "method": "G2",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "none",
        "nobasis": True,
        "citations": ((1, "10.1063/1.460205"),),
    },
    "G2MP2: Gaussian-2 MP2 composite method": {
        "method": "G2MP2",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "none",
        "nobasis": True,
        "citations": ((1, "10.1063/1.464297"),),
    },
    "G3: Gaussian-3 composite method": {
        "method": "G3",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "none",
        "nobasis": True,
        "citations": ((1, "10.1063/1.477422"),),
    },
    "G3MP2: Gaussian-3 MP2 composite method": {
        "method": "G3MP2",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "none",
        "nobasis": True,
        "citations": ((1, "10.1063/1.478385"),),
    },
    "G3B3: Gaussian-3 B3LYP composite method": {
        "method": "G3B3",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "none",
        "nobasis": True,
        "citations": ((1, "10.1063/1.478676"),),
    },
    "G3MP2B3: Gaussian-3 MP2 B3LYP composite method": {
        "method": "G3MP2B3",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "none",
        "nobasis": True,
        "citations": (
            (1, "10.1063/1.478385"),
            (1, "10.1063/1.478676"),
        ),
    },
    "G4: Gaussian-4 composite method": {
        "method": "G4",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "none",
        "nobasis": True,
        "citations": ((1, "10.1063/1.2436888"),),
    },
    "G4MP2: Gaussian-4 MP2 composite method": {
        "method": "G4MP2",
        "calculation": ["energy", "gradients"],
        "level": "advanced",
        "gradients": "none",
        "nobasis": True,
        "citations": ((1, "10.1063/1.2770701"),),
    },
}

dft_functionals = {
    "PBE : GGA of Perdew, Burke and Ernzerhof (revised)": {
        "name": "PBEhPBE",
        "dispersion": ["none", "GD3BJ", "GD3", "GD2", "PFD"],
        "level": "normal",
        "citations": [
            (1, "10.1063/1.476928"),
            (1, "PhysRevLett.77.3865"),
            (1, "PhysRevLett.78.1396"),
        ],
    },
    "B3LYP : hybrid functional of Becke and Lee, Yang, and Parr": {
        "name": "B3LYP",
        "dispersion": ["none", "GD3BJ", "GD3", "GD2", "PFD"],
        "level": "normal",
        "citations": [
            (1, "10.1063/1.464913"),
            (1, "doi:10.1021/j100096a001"),
        ],
    },
    "HSE06 : hybrid functional of Heyd, Scuseria and Ernzerhof": {
        "name": "HSEH1PBE",
        "dispersion": ["none", "PFD"],
        "level": "normal",
        "citations": [
            (2, "10.1063/1.1760074"),
            (2, "10.1063/1.1668634"),
            (2, "10.1063/1.2085170"),
            (2, "10.1063/1.2204597"),
            (2, "10.1063/1.3185673"),
            (2, "10.1063/1.2347713"),
            (1, "10.1063/1.2404663"),
        ],
    },
    "PBE0 : hybrid functional of Perdew, Burke and Ernzerhof, and Adamo (revised)": {
        "name": "PBEh1PBE",
        "dispersion": ["none", "GD3BJ", "GD3", "PFD"],
        "level": "normal",
        "citations": [
            (1, "10.1063/1.478522"),
            (2, "10.1063/1.476928"),
        ],
    },
    "wB97XD : hybrid functional of Chai and Head-Gordon, with dispersion": {
        "name": "wB97XD",
        "dispersion": ["none"],
        "level": "normal",
        "citations": [
            (1, "10.1039/B810189B"),
        ],
    },
    "M062X : hybrid functional of Zhao and Truhlar, for nonmetals": {
        "name": "M062x",
        "dispersion": ["none"],
        "level": "normal",
        "citations": [
            (1, "10.1007/s00214-007-0310-x"),
        ],
    },
}

optimization_convergence = {
    "default": "",
    "tight": "Tight",
    "very tight": "VeryTight",
    "loose": "Loose",
}

"""Properties that Gaussian produces.
`metadata["results"]` describes the results that this step can produce. It is a
dictionary where the keys are the internal names of the results within this step, and
the values are a dictionary describing the result. For example::

    metadata["results"] = {
        "total_energy": {
            "calculation": [
                "energy",
                "optimization",
            ],
            "description": "The total energy",
            "dimensionality": "scalar",
            "methods": [
                "ccsd",
                "ccsd(t)",
                "dft",
                "hf",
            ],
            "property": "total energy#QuickMin#{model}",
            "type": "float",
            "units": "E_h",
        },
    }

Fields
______

calculation : [str]
    Optional metadata describing what subtype of the step produces this result.
    The subtypes are completely arbitrary, but often they are types of calculations
    which is why this is name `calculation`. To use this, the step or a substep
    define `self._calculation` as a value. That value is used to select only the
    results with that value in this field.

description : str
    A human-readable description of the result.

dimensionality : str
    The dimensions of the data. The value can be "scalar" or an array definition
    of the form "[dim1, dim2,...]". Symmetric tringular matrices are denoted
    "triangular[n,n]". The dimensions can be integers, other scalar
    results, or standard parameters such as `n_atoms`. For example, '[3]',
    [3, n_atoms], or "triangular[n_aos, n_aos]".

methods : str
    Optional metadata like the `calculation` data. `methods` provides a second
    level of filtering, often used for the Hamiltionian for *ab initio* calculations
    where some properties may or may not be calculated depending on the type of
    theory.

property : str
    An optional definition of the property for storing this result. Must be one of
    the standard properties defined either in SEAMM or in this steps property
    metadata in `data/properties.csv`.

type : str
    The type of the data: string, integer, or float.

units : str
    Optional units for the result. If present, the value should be in these units.
"""
Gn_composite_methods = (
    "G1",
    "G2",
    "G3",
    "G4",
    "G2MP2",
    "G3B3",
    "G3MP2",
    "G3MP2B3",
    "G4MP2",
)
CBS_composite_methods = (
    "CBS-4M",
    "CBS-QB3",
    "CBS-APNO",
)
composite_methods = Gn_composite_methods + CBS_composite_methods

metadata["translation"] = {
    "Alpha Orbital Energies": "alpha orbital energies",
    "Cartesian Force Constants": "force constants",
    "Cartesian Gradient": "gradients",
    "Cluster Energy with triples": "E ccsd_t",
    "Cluster Energy": "E cc",
    "Composite/DE(2DF)": "delta E 2df",
    "Composite/DE(CBS)": "deltaE cbs",
    "Composite/DE(CCSD)": "delta E ccsd",
    "Composite/DE(HF)": "delta E hf",
    "Composite/DE(MP2)": "deltaE mp2",
    "Composite/DE(MP34)": "deltaE mp34",
    "Composite/DE(Plus)": "delta E diffuse",
    "Composite/DE(Empirical)": "delta E empirical",
    "Composite/DE(int)": "delta E internals",
    "Composite/(0 K)": "H 0",
    "Composite/E(CCSD(T))": "E ccsd_t",
    "Composite/E(Delta-G3XP)": "delta E g3xp",
    "Composite/E(SCF)": "E scf",
    "Composite/E(Thermal)": "E thermal",
    "Composite/E(ZPE)": "ZPE",
    "Composite/Enthalpy": "H",
    "Composite/Free Energy": "G",
    "Composite/Pressure": "P",
    "Composite/Temperature": "T",
    "Dipole Derivatives": "dipole derivatives",
    "Gaussian Version": "code version",
    "Geometry Optimization Converged": "optimization is converged",
    "HyperPolarizability": "hyperpolarizability",
    "MP2 Energy": "E mp2",
    "MP3 Energy": "E mp3",
    "MP4 Energy": "E mp4",
    "MP4DQ Energy": "E mp4dq",
    "MP4SDQ Energy": "E mp4sdq",
    "MP5 Energy": "E mp5",
    "Mulliken Charges": "Mulliken charges",
    "Number of alpha electrons": "N alpha electrons",
    "Number of beta electrons": "N beta electrons",
    "Number of electrons": "N electrons",
    "Polarizability Derivatives": "polarizability derivatives",
    "Polarizability": "polarizability",
    "RMS Density": "RMS density difference",
    "Real atomic weights": "atomic masses",
    "SCF Energy": "E scf",
    "Total Energy": "energy",
    "Virial Ratio": "virial ratio",
    "atomcharges/apt": "atomic polar tensor charges",
    "atomcharges/mulliken": "mulliken charges",
    "coreelectrons": "core electrons per atom",
    "enthalpy": "H",
    "entropy": "S",
    "freeenergy": "G",
    "metadata/basis_set": "basis set name",
    "metadata/cpu_time": "cpu time",
    "metadata/functional": "density functional",
    "metadata/methods": "methods",
    "metadata/package": "code name",
    "metadata/package_version": "code version",
    "metadata/platform": "platform",
    "metadata/success": "success",
    "metadata/symmetry_detected": "symmetry group",
    "metadata/symmetry_used": "symmetry group used",
    "metadata/wall_time": "elapsed time",
    "moenergies": "MO energies",
    "mosyms": "MO symmetries",
    "nbasis": "NBF",
    "nmo": "NMO",
    "pressure": "P",
    "rotconstants": "rotational constants",
    "scfenergies": "SCF energies",
    "temperature": "T",
    "ts frequency": "transition frequency",
    "vibdisps": "vibrational displacements",
    "vibfconsts": "vibrational force constants",
    "vibfreqs": "vibrational frequencies",
    "vibirs": "vibrational IR intensities",
    "vibramans": "vibrational Raman intensities",
    "vibrmasses": "vibrational reduced masses",
    "vibsyms": "vibrational symmetries",
    "zpve": "ZPE",
}

metadata["results"] = {
    "model": {
        "description": "The model string",
        "dimensionality": "scalar",
        "type": "string",
    },
    "density functional": {
        "description": "The density functional",
        "dimensionality": "scalar",
        "methods": ["DFT"],
        "type": "string",
        "format": "",
    },
    "basis set name": {
        "description": "The name of the basis set",
        "dimensionality": "scalar",
        "type": "string",
        "format": "",
    },
    "basis set": {
        "description": "The basis set",
        "dimensionality": "scalar",
        "type": "string",
        "format": "",
    },
    "DfE0": {
        "description": "the energy of formation",
        "dimensionality": "scalar",
        "property": "DfE0#Gaussian#{model}",
        "type": "float",
        "units": "kJ/mol",
        "format": ".2f",
    },
    "E atomization": {
        "description": "the energy of formation from atoms",
        "dimensionality": "scalar",
        "property": "E atomization#Gaussian#{model}",
        "type": "float",
        "units": "kJ/mol",
        "format": ".2f",
    },
    "DfH0": {
        "calculation": ["thermodynamics"],
        "description": "the enthalpy of formation",
        "dimensionality": "scalar",
        "property": "DfH0#Gaussian#{model}",
        "type": "float",
        "units": "kJ/mol",
        "format": ".2f",
    },
    "H atomization": {
        "calculation": ["thermodynamics"],
        "description": "the enthalpy of formation from atoms",
        "dimensionality": "scalar",
        "property": "H atomization#Gaussian#{model}",
        "type": "float",
        "units": "kJ/mol",
        "format": ".2f",
    },
    "energy": {
        "description": "total energy including all terms",
        "dimensionality": "scalar",
        "property": "E#Gaussian#{model}",
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "H 0": {
        "calculation": ["thermodynamics"],
        "description": "the electronic energy plus ZPE",
        "dimensionality": "scalar",
        "property": "H(0 K)#Gaussian#{model}",
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "U": {
        "calculation": ["thermodynamics"],
        "description": "the internal energy",
        "dimensionality": "scalar",
        "property": "U#Gaussian#{model}",
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "H": {
        "calculation": ["thermodynamics"],
        "description": "The enthalpy of the system",
        "dimensionality": "scalar",
        "property": "H#Gaussian#{model}",
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "S": {
        "calculation": ["thermodynamics"],
        "description": "The entropy of the system",
        "dimensionality": "scalar",
        "property": "S#Gaussian#{model}",
        "type": "float",
        "units": "E_h/K",
        "format": ".6f",
    },
    "G": {
        "calculation": ["thermodynamics"],
        "description": "The free energy of the system",
        "dimensionality": "scalar",
        "property": "G#Gaussian#{model}",
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "ZPE": {
        "calculation": ["thermodynamics"],
        "description": "the zero point energy",
        "dimensionality": "scalar",
        "property": "ZPE#Gaussian#{model}",
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "P": {
        "calculation": ["thermodynamics"],
        "description": "Pressure",
        "dimensionality": "scalar",
        "property": "P#Gaussian#{model}",
        "type": "float",
        "units": "atm",
        "format": ".2f",
    },
    "T": {
        "calculation": ["thermodynamics"],
        "description": "Temperature",
        "dimensionality": "scalar",
        "property": "T#Gaussian#{model}",
        "type": "float",
        "units": "K",
        "format": ".2f",
    },
    "transition state frequency": {
        "calculation": ["thermodynamics"],
        "description": "Transition state curvature",
        "dimensionality": "scalar",
        "property": "transition state frequency#Gaussian#{model}",
        "type": "float",
        "units": "cm^-1",
        "format": ".1f",
    },
    "N saddle modes": {
        "calculation": ["thermodynamics"],
        "description": "Number of saddle point modes",
        "dimensionality": "scalar",
        "property": "number of saddle modes#Gaussian#{model}",
        "type": "integer",
        "format": "",
    },
    "gradients": {
        "calculation": ["energy", "optimization", "thermodynamics"],
        "description": "the Cartesian gradients",
        "dimensionality": ["natoms", 3],
        "property": "gradients#Gaussian#{model}",
        "type": "float",
        "units": "E_h/a0",
        "format": ".6f",
    },
    "force constants": {
        "calculation": ["thermodynamics"],
        "description": "the Cartesian force constants",
        "dimensionality": "[natoms*(natoms+1)]",
        "property": "force constants#Gaussian#{model}",
        "type": "float",
        "units": "kcal/mol/Å^2",
        "format": ".2f",
    },
    "E scf": {
        "description": "the SCF energy from the CBS/Gn extrapolation",
        "dimensionality": "scalar",
        "type": "float",
        "property": "E scf#Gaussian#{model}",
        "units": "E_h",
        "format": ".6f",
    },
    "E ccsd_t": {
        "description": "the CCSD(T) energy",
        "dimensionality": "scalar",
        "methods": ["CCSD(T)"],
        "property": "E ccsd_t#Gaussian#{model}",
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "E cc": {
        "description": "the CCD or CCSD energy",
        "dimensionality": "scalar",
        "methods": ["CCD", "CCSD", "CCSD(T)"],
        "property": "E cc#Gaussian#{model}",
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "E qcisd_t": {
        "description": "the CCSD(T) energy",
        "dimensionality": "scalar",
        "methods": ["QCISD(T)"],
        "property": "E qcisd_t#Gaussian#{model}",
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "E qcisd": {
        "description": "the QCISD energy",
        "dimensionality": "scalar",
        "methods": ["QCISD", "QCISD(T)"],
        "property": "E qcisd#Gaussian#{model}",
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "E mp2": {
        "description": "the MP2 energy",
        "dimensionality": "scalar",
        "methods": [
            "MP2",
            "MP3",
            "MP4",
            "MP4SDQ",
            "MP5",
            "QCISD",
            "QCISD(T)",
            "CCSD",
            "CCSD(T)",
        ],
        "type": "float",
        "property": "E mp2#Gaussian#{model}",
        "units": "E_h",
        "format": ".6f",
    },
    "E mp3": {
        "description": "the MP3 energy",
        "dimensionality": "scalar",
        "methods": [
            "MP3",
            "MP4",
            "MP4SDQ",
            "MP5",
            "QCISD",
            "QCISD(T)",
            "CCSD",
            "CCSD(T)",
        ],
        "type": "float",
        "property": "E mp3#Gaussian#{model}",
        "units": "E_h",
        "format": ".6f",
    },
    "E mp4": {
        "description": "the MP4 energy",
        "dimensionality": "scalar",
        "methods": ["MP4", "MP4SDQ", "MP5", "CCSD", "CCSD(T)"],
        "property": "E mp4#Gaussian#{model}",
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "E mp4dq": {
        "description": "the MP4DQ energy",
        "dimensionality": "scalar",
        "methods": ["MP4", "MP4SDQ", "MP5", "QCISD", "QCISD(T)", "CCSD", "CCSD(T)"],
        "property": "E mp4dq#Gaussian#{model}",
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "E mp4sdq": {
        "description": "the MP4SDQ energy",
        "dimensionality": "scalar",
        "methods": ["MP4", "MP4SDQ", "MP5", "QCISD", "QCISD(T)", "CCSD", "CCSD(T)"],
        "property": "E mp4sdq#Gaussian#{model}",
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "E mp5": {
        "description": "the MP5 energy",
        "dimensionality": "scalar",
        "methods": ["MP5"],
        "property": "E mp5#Gaussian#{model}",
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "S**2": {
        "description": "<S\N{SUPERSCRIPT TWO}>",
        "dimensionality": "scalar",
        "property": "S**2#Gaussian#{model}",
        "type": "string",
        "format": ".4f",
    },
    "S**2 after annihilation": {
        "description": "<S\N{SUPERSCRIPT TWO}> after annihilation",
        "dimensionality": "scalar",
        "property": "S**2 after annihilation#Gaussian#{model}",
        "type": "string",
        "format": ".4f",
    },
    "ideal S**2": {
        "description": "ideal <S\N{SUPERSCRIPT TWO}>",
        "dimensionality": "scalar",
        "type": "string",
        "format": ".4f",
    },
    "symmetry group": {
        "description": "The symmetry of the system",
        "dimensionality": "scalar",
        "property": "symmetry group#Gaussian#{model}",
        "type": "string",
        "format": "",
    },
    "symmetry group used": {
        "description": "The symmetry used in the calculation",
        "dimensionality": "scalar",
        "type": "string",
        "format": "",
    },
    "state": {
        "description": "The electronic state of the system",
        "dimensionality": "scalar",
        "type": "string",
        "format": "",
    },
    "virial ratio": {
        "calculation": ["energy", "optimization", "thermodynamics"],
        "description": "the virial ratio",
        "dimensionality": "scalar",
        "property": "virial ratio#Gaussian#{model}",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "format": ".4f",
    },
    "RMS density difference": {
        "calculation": ["energy", "optimization", "thermodynamics"],
        "description": "the RMS density difference in the SCF",
        "dimensionality": "scalar",
        "methods": ["DFT", "HF", "MP2", "MP3", "MP4"],
        "type": "float",
        "format": ".2e",
    },
    "N electrons": {
        "description": "Number of electrons",
        "dimensionality": "scalar",
        "property": "N electrons#Gaussian#{model}",
        "type": "integer",
        "format": "",
    },
    "N alpha electrons": {
        "description": "Number of α electrons",
        "dimensionality": "scalar",
        "property": "N alpha electrons#Gaussian#{model}",
        "type": "integer",
        "format": "",
    },
    "N beta electrons": {
        "description": "Number of β electrons",
        "dimensionality": "scalar",
        "property": "N beta electrons#Gaussian#{model}",
        "type": "integer",
        "format": "",
    },
    "E alpha gap": {
        "description": (
            "The gap between the α-LUMO and α-HOMO or conduction and valence bands"
        ),
        "dimensionality": "scalar",
        "property": "E alpha gap#Gaussian#{model}",
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "E alpha homo": {
        "description": "The energy of highest occupied MO (α-HOMO)",
        "dimensionality": "scalar",
        "property": "E alpha homo#Gaussian#{model}",
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "E alpha nhomo": {
        "description": "The energy of the next-to-highest occupied orbital (α-NHOMO)",
        "dimensionality": "scalar",
        "property": "E alpha nhomo#Gaussian#{model}",
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "E alpha lumo": {
        "description": "The energy of lowest unoccupied MO (α-LUMO)",
        "dimensionality": "scalar",
        "property": "E alpha lumo#Gaussian#{model}",
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "E alpha slumo": {
        "description": "The energy of the second lowest unoccupied orbital (α-SLUMO)",
        "dimensionality": "scalar",
        "property": "E alpha slumo#Gaussian#{model}",
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "alpha HOMO orbital number": {
        "description": "The orbital number of the α-HOMO (starting at 1)",
        "dimensionality": "scalar",
        "type": "integer",
        "format": "",
    },
    "alpha HOMO symmetry": {
        "description": "The symmetry of the highest occupied MO (α-HOMO)",
        "dimensionality": "scalar",
        "property": "alpha HOMO symmetry#Gaussian#{model}",
        "type": "string",
        "format": "",
    },
    "alpha NHOMO symmetry": {
        "description": "The symmetry of the next-to-highest occupied orbital (α-NHOMO)",
        "dimensionality": "scalar",
        "property": "alpha NHOMO symmetry#Gaussian#{model}",
        "type": "string",
        "format": "",
    },
    "alpha LUMO symmetry": {
        "description": "The symmetry of the lowest unoccupied MO (α-LUMO)",
        "dimensionality": "scalar",
        "property": "alpha LUMO symmetry#Gaussian#{model}",
        "type": "string",
        "format": "",
    },
    "alpha SLUMO symmetry": {
        "description": "The symmetry of the second lowest unoccupied orbital (α-SLUMO)",
        "dimensionality": "scalar",
        "property": "alpha SLUMO symmetry#Gaussian#{model}",
        "type": "string",
        "format": "",
    },
    "E beta gap": {
        "description": (
            "The gap between the β-LUMO and β-HOMO or conduction and valence bands"
        ),
        "dimensionality": "scalar",
        "property": "E beta gap#Gaussian#{model}",
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "E beta homo": {
        "description": "The energy of highest occupied MO (β-HOMO)",
        "dimensionality": "scalar",
        "property": "E beta homo#Gaussian#{model}",
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "E beta nhomo": {
        "description": "The energy of the next-to-highest occupied orbital (β-NHOMO)",
        "dimensionality": "scalar",
        "property": "E beta nhomo#Gaussian#{model}",
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "E beta lumo": {
        "description": "The energy of lowest unoccupied MO (β-LUMO)",
        "dimensionality": "scalar",
        "property": "E beta lumo#Gaussian#{model}",
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "E beta slumo": {
        "description": "The energy of the second lowest unoccupied orbital (β-SLUMO)",
        "dimensionality": "scalar",
        "property": "E beta slumo#Gaussian#{model}",
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "beta HOMO orbital number": {
        "description": "The orbital number of the β-HOMO (starting at 1)",
        "dimensionality": "scalar",
        "type": "integer",
        "format": "",
    },
    "beta HOMO symmetry": {
        "description": "The symmetry of the highest occupied MO (β-HOMO)",
        "dimensionality": "scalar",
        "property": "beta HOMO symmetry#Gaussian#{model}",
        "type": "string",
        "format": "",
    },
    "beta NHOMO symmetry": {
        "description": "The symmetry of the next-to-highest occupied orbital (β-NHOMO)",
        "dimensionality": "scalar",
        "property": "beta NHOMO symmetry#Gaussian#{model}",
        "type": "string",
        "format": "",
    },
    "beta LUMO symmetry": {
        "description": "The symmetry of the lowest unoccupied MO (β-LUMO)",
        "dimensionality": "scalar",
        "property": "beta LUMO symmetry#Gaussian#{model}",
        "type": "string",
        "format": "",
    },
    "beta SLUMO symmetry": {
        "description": "The symmetry of the second lowest unoccupied orbital (β-SLUMO)",
        "dimensionality": "scalar",
        "property": "beta SLUMO symmetry#Gaussian#{model}",
        "type": "string",
        "format": "",
    },
    "multipole reference point": {
        "calculation": ["energy", "optimization", "thermodynamics"],
        "description": "The reference point for the multipole expansions",
        "dimensionality": "[3]",
        "property": "multipole reference point#Gaussian#{model}",
        "type": "float",
        "units": "Å",
        "format": ".6f",
    },
    "dipole moment": {
        "calculation": ["energy", "optimization", "thermodynamics"],
        "description": "The dipole moment",
        "dimensionality": "[3]",
        "property": "dipole moment vector#Gaussian#{model}",
        "type": "float",
        "units": "debye",
        "format": ".6f",
    },
    "dipole moment magnitude": {
        "calculation": ["energy", "optimization", "thermodynamics"],
        "description": "The magnitude of the dipole moment",
        "dimensionality": "scalar",
        "property": "dipole moment#Gaussian#{model}",
        "type": "float",
        "units": "debye",
        "format": ".6f",
    },
    "dipole derivatives": {
        "calculation": ["thermodynamics"],
        "description": (
            "The derivatives of the dipole moment with respect to the nuclear "
            "coordinates"
        ),
        "dimensionality": "[natoms, 3]",
        "property": "dipole derivatives#Gaussian#{model}",
        "type": "float",
        "units": "debye/Å",
        "format": ".6f",
    },
    "quadrupole moment": {
        "calculation": ["energy", "optimization", "thermodynamics"],
        "description": "The quadrupole moment",
        "dimensionality": "[6]",
        "property": "quadrupole moment#Gaussian#{model}",
        "type": "float",
        "units": "debye*Å",
        "format": ".6f",
    },
    "octapole moment": {
        "calculation": ["energy", "optimization", "thermodynamics"],
        "description": "The octapole moment",
        "dimensionality": "[10]",
        "property": "octapole moment#Gaussian#{model}",
        "type": "float",
        "units": "debye*Å^2",
        "format": ".6f",
    },
    "hexadecapole moment": {
        "calculation": ["energy", "optimization", "thermodynamics"],
        "description": "The hexadecapole moment",
        "dimensionality": "[15]",
        "property": "hexadecapole moment#Gaussian#{model}",
        "type": "float",
        "units": "debye*Å^3",
        "format": ".6f",
    },
    "polarizability": {
        "calculation": ["thermodynamics"],
        "description": "The polarizability",
        "dimensionality": "[6]",
        "property": "polarizability#Gaussian#{model}",
        "type": "float",
        "units": "a_0^3",
        "format": ".6f",
    },
    "polarizability derivatives": {
        "calculation": ["thermodynamics"],
        "description": "The derivatives of the polarizability",
        "dimensionality": "[natoms, 6]",
        "property": "polarizability derivatives#Gaussian#{model}",
        "type": "float",
        "units": "a_0^2",
        "format": ".6f",
    },
    "hyperpolarizability": {
        "calculation": ["thermodynamics"],
        "description": "The hyperpolarizability",
        "dimensionality": "[10]",
        "property": "hyperpolarizability#Gaussian#{model}",
        "type": "float",
        "units": "a_0^5/e",
        "format": ".6f",
    },
    "NBF": {
        "description": "Number of basis functions",
        "dimensionality": "scalar",
        "property": "NBF#Gaussian#{model}",
        "type": "integer",
        "format": "",
    },
    "NMO": {
        "description": "Number of molecular orbitals",
        "dimensionality": "scalar",
        "property": "NMO#Gaussian#{model}",
        "type": "integer",
        "format": "",
    },
    "SEAMM elapsed time": {
        "description": "total elapsed time for Gaussian",
        "dimensionality": "scalar",
        "type": "float",
        "units": "s",
        "format": ".1f",
    },
    "SEAMM np": {
        "description": "number of processors for Gaussian",
        "dimensionality": "scalar",
        "type": "integer",
        "format": "d",
    },
    "optimization is converged": {
        "calculation": ["optimization", "thermodynamics"],
        "description": "whether the geometry optimization converged",
        "dimensionality": "scalar",
        "type": "boolean",
        "format": "s",
    },
    "N steps optimization": {
        "calculation": ["optimization", "thermodynamics"],
        "description": "number of optimization steps",
        "dimensionality": "scalar",
        "property": "N steps optimization#Gaussian#{model}",
        "type": "integer",
    },
    "RMS atom force": {
        "calculation": ["optimization", "thermodynamics"],
        "description": "the RMS force on the atoms",
        "dimensionality": "scalar",
        "type": "float",
        "units": "E_h/Å",
        "format": ".6f",
    },
    "RMS atom force threshold": {
        "calculation": ["optimization", "thermodynamics"],
        "description": "the RMS force threshold for the atoms",
        "dimensionality": "scalar",
        "type": "float",
        "units": "E_h/Å",
        "format": ".6f",
    },
    "RMS atom force trajectory": {
        "calculation": ["optimization", "thermodynamics"],
        "description": "trajectory of the RMS force on the atoms",
        "dimensionality": "[N steps optimization]",
        "type": "float",
        "units": "E_h/Å",
        "format": ".6f",
    },
    "RMS atom displacement": {
        "calculation": ["optimization", "thermodynamics"],
        "description": "the RMS displacement of the atoms",
        "dimensionality": "scalar",
        "type": "float",
        "units": "Å",
        "format": ".6f",
    },
    "RMS atom displacement threshold": {
        "calculation": ["optimization", "thermodynamics"],
        "description": "the RMS displacement threshold for the atoms",
        "dimensionality": "scalar",
        "type": "float",
        "units": "Å",
        "format": ".6f",
    },
    "RMS atom displacement trajectory": {
        "calculation": ["optimization", "thermodynamics"],
        "description": "trajectory of the RMS displacement of the atoms",
        "dimensionality": "[N steps optimization]",
        "type": "float",
        "units": "Å",
        "format": ".6f",
    },
    "maximum atom force": {
        "calculation": ["optimization", "thermodynamics"],
        "description": "the maximum force on an atom",
        "dimensionality": "scalar",
        "type": "float",
        "units": "E_h/Å",
        "format": ".6f",
    },
    "maximum atom force threshold": {
        "calculation": ["optimization", "thermodynamics"],
        "description": "the maximum force threshold for an atom",
        "dimensionality": "scalar",
        "type": "float",
        "units": "E_h/Å",
        "format": ".6f",
    },
    "maximum atom force trajectory": {
        "calculation": ["optimization", "thermodynamics"],
        "description": "trajectory of the maximum force on an atom",
        "dimensionality": "[N steps optimization]",
        "type": "float",
        "units": "E_h/Å",
        "format": ".6f",
    },
    "maximum atom displacement": {
        "calculation": ["optimization", "thermodynamics"],
        "description": "the maximum displacement of an atom",
        "dimensionality": "scalar",
        "type": "float",
        "units": "Å",
        "format": ".6f",
    },
    "maximum atom displacement threshold": {
        "calculation": ["optimization", "thermodynamics"],
        "description": "the maximum displacement threshold for an atom",
        "dimensionality": "scalar",
        "type": "float",
        "units": "Å",
        "format": ".6f",
    },
    "maximum atom displacement trajectory": {
        "calculation": ["optimization", "thermodynamics"],
        "description": "trajectory of the maximum displacement of an atom",
        "dimensionality": "[N steps optimization]",
        "type": "float",
        "units": "Å",
        "format": ".6f",
    },
    "mulliken charges": {
        "description": "The Mulliken charges of the atoms",
        "dimensionality": "[natoms]",
        "property": "Mulliken charges#Gaussian#{model}",
        "type": "float",
        "format": ".6f",
    },
    "atomic polar tensor charges": {
        "description": "The atomic polar tensor charges of the atoms",
        "dimensionality": "[natoms]",
        "property": "atomic polar tensor charges#Gaussian#{model}",
        "type": "float",
        "format": ".6f",
    },
    "atomic masses": {
        "description": "The atomic masses of the atoms",
        "dimensionality": "[natoms]",
        "property": "atomic masses#Gaussian#{model}",
        "type": "float",
        "units": "Da",
        "format": "",
    },
    "core electrons per atom": {
        "description": "The number of core electrons per atom",
        "dimensionality": "[natoms]",
        "type": "integer",
        "format": "",
    },
    "cpu time": {
        "description": "The cpu time for the calculation",
        "dimensionality": "scalar",
        "type": "string",
        "format": "",
    },
    "methods": {
        "description": (
            "The method used in each optimization step or step of the calculation"
        ),
        "dimensionality": "[nsteps]",
        "type": "string",
        "format": "",
    },
    "code name": {
        "description": "The name of the code (Gaussian)",
        "dimensionality": "scalar",
        "type": "string",
        "format": "",
    },
    "code version": {
        "description": "The version of the Gaussian code (year+revision)",
        "dimensionality": "scalar",
        "type": "string",
        "format": "",
    },
    "platform": {
        "description": "The platform used for the calculation",
        "dimensionality": "scalar",
        "type": "string",
        "format": "",
    },
    "success": {
        "description": "Whether the calculation was successful",
        "dimensionality": "scalar",
        "type": "boolean",
        "format": "",
    },
    "elapsed time": {
        "description": "The wall-clock time for the calculation",
        "dimensionality": "scalar",
        "type": "string",
        "format": "",
    },
    "MO energies": {
        "calculation": ["energy", "optimization", "thermodynamics"],
        "description": "The energy of the molecular orbitals (MOs)",
        "dimensionality": "[[nmo]*nspin]",
        "property": "MO energies#Gaussian#{model}",
        "type": "float",
        "units": "eV",
        "format": ".3f",
    },
    "MO symmetries": {
        "calculation": ["energy", "optimization", "thermodynamics"],
        "description": "The symmetry of the MOs",
        "dimensionality": "[nmo]",
        "property": "MO symmetries#Gaussian#{model}",
        "type": "string",
        "format": "",
    },
    "symmetry number": {
        "calculation": ["thermodynamics"],
        "description": "Symmetry number",
        "dimensionality": "scalar",
        "type": "integer",
        "format": "",
    },
    "rotational constants": {
        "calculation": ["energy", "optimization", "thermodynamics"],
        "description": "Rotational constants",
        "dimensionality": "[[3]*nsteps]",
        "type": "float",
        "units": "GHz",
        "format": ".3f",
    },
    "vibrational displacements": {
        "calculation": ["thermodynamics"],
        "description": "Vibrational displacements",
        "dimensionality": "[nvibs, natoms, 3]",
        "type": "float",
        "format": ".2f",
    },
    "vibrational frequencies": {
        "calculation": ["thermodynamics"],
        "description": "Vibrational frequencies",
        "dimensionality": "[nvibs]",
        "property": "vibrational frequencies#Gaussian#{model}",
        "type": "float",
        "units": "cm^-1",
        "format": ".1f",
    },
    "vibrational IR intensities": {
        "calculation": ["thermodynamics"],
        "description": "Vibrational IR intensities",
        "dimensionality": "[nvibs]",
        "property": "vibrational IR intensities#Gaussian#{model}",
        "type": "float",
        "units": "km/mol",
        "format": ".4f",
    },
    "vibrational Raman intensities": {
        "calculation": ["thermodynamics"],
        "description": "Vibrational Raman intensities",
        "dimensionality": "[nvibs]",
        "property": "vibrational Raman intensities#Gaussian#{model}",
        "type": "float",
        "units": "Å^4/Da",
        "format": ".4f",
    },
    "vibrational reduced masses": {
        "calculation": ["thermodynamics"],
        "description": "Vibrational reduced masses",
        "dimensionality": "[nvibs]",
        "type": "float",
        "units": "Da",
        "format": ".4f",
    },
    "vibrational symmetries": {
        "calculation": ["thermodynamics"],
        "description": "Vibrational symmetries",
        "dimensionality": "[nvibs]",
        "property": "vibrational symmetries#Gaussian#{model}",
        "type": "string",
        "format": "",
    },
    "E thermal": {
        "calculation": ["thermodynamics"],
        "description": "the thermal correction to the electronic energy",
        "dimensionality": "scalar",
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "H thermal": {
        "calculation": ["thermodynamics"],
        "description": "the enthalpy correction to the electronic energy",
        "dimensionality": "scalar",
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "G thermal": {
        "calculation": ["thermodynamics"],
        "description": "the free energy correction to the electronic energy",
        "dimensionality": "scalar",
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "deltaE mp2": {
        "calculation": ["thermodynamics"],
        "description": "dE from MP2 from the CBS/Gn extrapolation",
        "dimensionality": "scalar",
        "methods": composite_methods,
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "deltaE cbs": {
        "calculation": ["thermodynamics"],
        "description": "dE from the basis extrapolation from the CBS/Gn extrapolation",
        "dimensionality": "scalar",
        "methods": composite_methods,
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "deltaE mp34": {
        "calculation": ["thermodynamics"],
        "description": "dE of MP3-4 from the CBS/Gn extrapolation",
        "dimensionality": "scalar",
        "methods": composite_methods,
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "delta E ccsd": {
        "calculation": ["thermodynamics"],
        "description": "dE of CCSD from the CBS/Gn extrapolation",
        "dimensionality": "scalar",
        "methods": ["CBS-4M", "CBS-QB3", "CBS-APNO"],
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "delta E internals": {
        "calculation": ["thermodynamics"],
        "description": "dE from internals from the CBS/Gn extrapolation",
        "dimensionality": "scalar",
        "methods": composite_methods,
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "delta E empirical": {
        "calculation": ["thermodynamics"],
        "description": "Empirical correction to the CBS/Gn extrapolation",
        "dimensionality": "scalar",
        "methods": composite_methods,
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "delta E diffuse": {
        "calculation": ["thermodynamics"],
        "description": "Plus term for the Gn extrapolation",
        "dimensionality": "scalar",
        "methods": Gn_composite_methods,
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "delta E 2df": {
        "calculation": ["thermodynamics"],
        "description": "The delta energy for 2D's and F term for the Gn extrapolation",
        "dimensionality": "scalar",
        "methods": Gn_composite_methods,
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "delta E g3xp": {
        "calculation": ["thermodynamics"],
        "description": "Delta G3XP term for the Gn extrapolation",
        "dimensionality": "scalar",
        "methods": Gn_composite_methods,
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "delta E hf": {
        "calculation": ["thermodynamics"],
        "description": "delta E(HF) term for the Gn extrapolation",
        "dimensionality": "scalar",
        "methods": Gn_composite_methods,
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
    "Wiberg bond order matrix": {
        "description": "the Wiberg bond order matrix",
        "dimensionality": "[natoms * [natoms]]",
        "type": "float",
        "format": ".3f",
    },
    "SCF energies": {
        "description": "The energies per step",
        "dimensionality": "[nsteps]",
        "type": "float",
        "units": "E_h",
        "format": ".6f",
    },
}
