import os
import sys
import json
from types import ModuleType
from contextlib import contextmanager


def env_exists(envname):
    """Returns True if a conda environment already exists and False otherwise"""
    envs = json.loads($(conda env list --json))["envs"]
    for env in envs:
        if os.path.split(env)[1] == envname:
            return True
    else:
        return False


def conda_init():
    """Initializes conda, if needed. This may safely be called many times."""
    if "xontrib.conda" in sys.modules:
        return
    mod = ModuleType("xontrib.conda",
                     "Autogenerated from $(conda shell.xonsh hook)")
    __xonsh__.execer.exec(
        $(conda shell.xonsh hook),
        glbs=mod.__dict__,
        filename="$(conda shell.xonsh hook)"
    )
    sys.modules["xontrib.conda"] = mod


@contextmanager
def run_in_conda_env(packages, envname='rever-env'):
    """
    Context manager to run in a conda environment

    Examples
    --------

    >>> with run_in_conda_env(['python=3']):
    ...     ./setup.py test

    """
    try:
        if env_exists(envname):
            conda remove -y -n @(envname) --all
        conda create -y -n @(envname) @(packages)
        conda_init()
        conda activate @(envname)
        yield
    finally:
        conda deactivate
        conda remove -y -n @(envname) --all
