"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StaticWebsite = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const iam = require("aws-cdk-lib/aws-iam");
const route53 = require("aws-cdk-lib/aws-route53");
const patterns = require("aws-cdk-lib/aws-route53-patterns");
const targets = require("aws-cdk-lib/aws-route53-targets");
const s3 = require("aws-cdk-lib/aws-s3");
const cr = require("aws-cdk-lib/custom-resources");
const constructs_1 = require("constructs");
const origin_request_function_1 = require("./origin-request-function");
/**
 * A CloudFront static website hosted on S3
 */
class StaticWebsite extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        this.bucket = new s3.Bucket(this, 'Bucket', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
        });
        const certificate = new acm.DnsValidatedCertificate(this, 'Certificate', {
            domainName: props.domainName,
            hostedZone: props.hostedZone,
            region: 'us-east-1',
        });
        this.distribution = new cloudfront.Distribution(this, 'Distribution', {
            defaultBehavior: {
                origin: new origins.S3Origin(this.bucket),
                viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                edgeLambdas: [
                    {
                        eventType: cloudfront.LambdaEdgeEventType.ORIGIN_REQUEST,
                        functionVersion: new origin_request_function_1.OriginRequestFunction(this, 'OriginRequest'),
                    },
                ],
                responseHeadersPolicy: (_b = props.responseHeadersPolicy) !== null && _b !== void 0 ? _b : new cloudfront.ResponseHeadersPolicy(this, 'ResponseHeadersPolicy', {
                    securityHeadersBehavior: StaticWebsite.defaultSecurityHeadersBehavior,
                }),
            },
            defaultRootObject: 'index.html',
            domainNames: [props.domainName],
            certificate,
            minimumProtocolVersion: cloudfront.SecurityPolicyProtocol.TLS_V1_2_2021,
        });
        new route53.ARecord(this, 'ARecord', {
            recordName: props.domainName,
            zone: props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(this.distribution)),
        });
        new route53.AaaaRecord(this, 'AaaaRecord', {
            recordName: props.domainName,
            zone: props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(this.distribution)),
        });
        if (props.backendConfiguration) {
            // Save backend config to bucket, can be queried by the frontend
            new cr.AwsCustomResource(this, 'PutConfig', {
                policy: cr.AwsCustomResourcePolicy.fromStatements([
                    new iam.PolicyStatement({
                        actions: ['s3:PutObject'],
                        resources: [this.bucket.arnForObjects('config.json')],
                    }),
                ]),
                onUpdate: {
                    service: 'S3',
                    action: 'putObject',
                    parameters: {
                        Bucket: this.bucket.bucketName,
                        Key: 'config.json',
                        Body: aws_cdk_lib_1.Stack.of(this).toJsonString(props.backendConfiguration),
                        ContentType: 'application/json',
                        CacheControl: 'max-age=0, no-cache, no-store, must-revalidate',
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('config'),
                },
            });
        }
        if (shouldAddRedirect(props)) {
            const httpsRedirect = new patterns.HttpsRedirect(this, 'HttpsRedirect', {
                targetDomain: props.domainName,
                zone: props.hostedZone,
                recordNames: props.redirects,
            });
            // Force minimum protocol version
            const redirectDistribution = httpsRedirect.node.tryFindChild('RedirectDistribution');
            const cfnDistribution = redirectDistribution.node.tryFindChild('CFDistribution');
            cfnDistribution.addPropertyOverride('DistributionConfig.ViewerCertificate.MinimumProtocolVersion', 'TLSv1.2_2021');
        }
    }
}
exports.StaticWebsite = StaticWebsite;
_a = JSII_RTTI_SYMBOL_1;
StaticWebsite[_a] = { fqn: "cloudstructs.StaticWebsite", version: "0.4.31" };
/**
* Best practice security headers used as default
*/
StaticWebsite.defaultSecurityHeadersBehavior = {
    contentTypeOptions: {
        override: true,
    },
    frameOptions: {
        frameOption: cloudfront.HeadersFrameOption.DENY,
        override: true,
    },
    referrerPolicy: {
        referrerPolicy: cloudfront.HeadersReferrerPolicy.STRICT_ORIGIN_WHEN_CROSS_ORIGIN,
        override: true,
    },
    strictTransportSecurity: {
        accessControlMaxAge: aws_cdk_lib_1.Duration.seconds(31536000),
        includeSubdomains: true,
        preload: true,
        override: true,
    },
    xssProtection: {
        protection: true,
        modeBlock: true,
        override: true,
    },
};
function shouldAddRedirect(props) {
    if (props.redirects && props.redirects.length === 0) {
        return false;
    }
    if (!props.redirects && !aws_cdk_lib_1.Token.isUnresolved(props.domainName)
        && !aws_cdk_lib_1.Token.isUnresolved(props.hostedZone.zoneName)
        && props.domainName === props.hostedZone.zoneName) {
        return false;
    }
    return true;
}
//# sourceMappingURL=data:application/json;base64,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