'''
    configpilot
    ~~~~~~~~~~~

        https://github.com/ValentinBELYN/configpilot

    :copyright: Copyright 2017-2020 Valentin BELYN.
    :license: MIT, see the LICENSE for details.

    ~~~~~~~~~~~

    Permission is hereby granted, free of charge, to any person
    obtaining a copy of this software and associated documentation
    files (the 'Software'), to deal in the Software without
    restriction, including without limitation the rights to use, copy,
    modify, merge, publish, distribute, sublicense, and/or sell copies
    of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be
    included in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED 'AS IS', WITHOUT WARRANTY OF ANY KIND,
    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
    OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
    NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
    HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
    WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
    OR OTHER DEALINGS IN THE SOFTWARE.
'''

from .exceptions import NoOptionError, IllegalValueError


class Section:
    '''
    A class that represents a section.
    Generated by a ConfigPilot object.

    :type name: str
    :param name: The name of the section.

    '''
    def __init__(self, name):
        self._name = name
        self._options = {}

    def __repr__(self):
        return f'<Section [{self._name}]>'

    def __len__(self):
        '''
        Return the number of options contained in the section.

        '''
        return len(self._options)

    def __getattr__(self, name):
        '''
        Get the value of the specified option.

        :raises NoOptionError: If the option is not found in the
            configuration file.
        :raises IllegalValueError: If the option does not meet the
            constraints defined in its specification.

        '''
        if name[:2] == '__':
            return super().__getattr__(name)

        return self[name]

    def __getitem__(self, key):
        '''
        Get the value of the specified option.

        :raises NoOptionError: If the option is not found in the
            configuration file.
        :raises IllegalValueError: If the option does not meet the
            constraints defined in its specification.

        '''
        if key not in self._options:
            raise NoOptionError(self._name, key)

        return self._options[key].value

    def add(self, option):
        '''
        Add an option to the section.

        '''
        self._options[option.name] = option


class Option:
    '''
    A class that represents an option.
    Generated by a ConfigPilot object.

    :type name: str
    :param name: The name of the option.

    :type value: object
    :param value: The (casted) value of the option.

    :type specification: OptionSpec
    :param specification: The specification corresponding to the
        option.

    '''
    def __init__(self, name, value, specification):
        self._name = name
        self._value = value
        self._specification = specification

    def __repr__(self):
        return f'<Option [{self._name}]>'

    @property
    def name(self):
        '''
        The name of the option.

        '''
        return self._name

    @property
    def value(self):
        '''
        The (casted) value of the option.

        :raises IllegalValueError: If the option does not meet the
            constraints defined in its specification.

        '''
        if self._value is None:
            raise IllegalValueError(
                section=self._specification.section,
                option=self._name)

        return self._value

    @property
    def specification(self):
        '''
        The specification corresponding to the option.

        '''
        return self._specification


class OptionSpec:
    '''
    A user-created object that represents the constraints that an
    option must meet to be considered valid.

    :type section: str
    :param section: The name of a section in the file.

    :type option: str
    :param option: The name of an option in the specified section.

    :type allowed: object that supports the 'in' operator (membership).
    :param allowed: (Optional) The list or range of allowed values.

    :type default: object
    :param default: (Optional) The default value of the option if it
        does not exist. Must be an object of the same type as the value
        obtained after the cast (see the 'type' parameter).

    :type type: type or list
    :param type: (Optional) The expected value type for this option.
        Set it to 'int', 'float', 'bool', 'str' (default) or any other
        type of object. If you expect a list of values, use instead
        '[int]', '[float]', '[bool]', '[str]' (equivalent of 'list') or
        even '[MyClass]'.

    '''
    def __init__(self, section, option, allowed=None, default=None,
            type=str):

        self._section = section
        self._option = option
        self._allowed = allowed
        self._default = default
        self._type = type

    def __repr__(self):
        return f'<OptionSpec [{self._section}:{self._option}]>'

    @property
    def section(self):
        '''
        The name of a section in the file.

        '''
        return self._section

    @property
    def option(self):
        '''
        The name of an option in the specified section.

        '''
        return self._option

    @property
    def allowed(self):
        '''
        The list or range of allowed values.

        '''
        return self._allowed

    @property
    def default(self):
        '''
        The default value of the option if it does not exist.

        '''
        return self._default

    @property
    def type(self):
        '''
        The expected value type for this option.

        '''
        return self._type
