# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['cryton_worker',
 'cryton_worker.etc',
 'cryton_worker.lib',
 'cryton_worker.lib.triggers',
 'cryton_worker.lib.util']

package_data = \
{'': ['*'], 'cryton_worker.etc': ['systemd-service/*']}

install_requires = \
['PyYAML>=6.0,<6.1',
 'amqpstorm>=2.10.4,<2.11.0',
 'bottle>=0.12.19,<0.13.0',
 'click>=8.1.2,<8.2.0',
 'paramiko>=2.11.0,<2.12.0',
 'pyfiglet>=0.8.post1,<0.9',
 'pymetasploit3>=1.0.3,<1.1.0',
 'python-dotenv>=0.20.0,<0.21.0',
 'requests>=2.27.0,<2.28.0',
 'schema>=0.7.5,<0.8.0',
 'structlog>=21.5.0,<21.6.0',
 'utinni>=0.5.0,<0.6.0']

entry_points = \
{'console_scripts': ['cryton-worker = cryton_worker.lib.cli:cli']}

setup_kwargs = {
    'name': 'cryton-worker',
    'version': '2022.2.2',
    'description': 'Attack scenario orchestrator for Cryton',
    'long_description': '[[_TOC_]]\n\n![Coverage](https://gitlab.ics.muni.cz/cryton/cryton-worker/badges/master/coverage.svg)\n\n# Cryton Worker\n\n## Description\nCryton Worker is used for executing attack modules remotely. It utilizes [RabbitMQ](https://www.rabbitmq.com/) \nas its asynchronous remote procedures call protocol. It connects to the Rabbit MQ server and consumes messages from \nthe Core component or any other app that implements its [RabbitMQ API](#rabbit-api).\n\nTo be able to execute attack scenarios, you also need to install **[Cryton Core](https://gitlab.ics.muni.cz/cryton/cryton-core)** \n(or your custom tool that implements Worker\'s API). \nModules provided by Cryton can be found [here](https://gitlab.ics.muni.cz/cryton/cryton-modules).\n\nCryton toolset is tested and targeted primarily on **Debian** and **Kali Linux**. Please keep in mind that \n**only the latest version is supported** and issues regarding different OS or distributions may **not** be resolved.\n\n[Link to the documentation](https://cryton.gitlab-pages.ics.muni.cz/cryton-documentation/).\n\n## Settings\nCryton Worker uses environment variables for its settings. Please update them to your needs.\n\n| name                               | value   | example                          | description                                                                                                                                                              |\n|------------------------------------|---------|----------------------------------|--------------------------------------------------------------------------------------------------------------------------------------------------------------------------|\n| CRYTON_WORKER_NAME                 | string  | my_worker1                       | Unique name used to identify the Worker.                                                                                                                                 |\n| CRYTON_WORKER_MODULES_DIR          | string  | /path/to/cryton-modules/modules/ | Path to the directory containing the modules.                                                                                                                            |\n| CRYTON_WORKER_DEBUG                | boolean | false                            | Make Worker run with debug output.                                                                                                                                       |\n| CRYTON_WORKER_INSTALL_REQUIREMENTS | boolean | true                             | Install requirements.txt for each module on startup.                                                                                                                     |\n| CRYTON_WORKER_CONSUMER_COUNT       | int     | 7                                | Number of consumers used for Rabbit communication <br> (more equals faster request processing and heavier processor usage).                                              |\n| CRYTON_WORKER_PROCESSOR_COUNT      | int     | 7                                | Number of processors used for internal requests <br> (more equals faster internal requests processing, but heavier processor usage).                                     |\n| CRYTON_WORKER_MAX_RETRIES          | int     | 3                                | How many times to try to re-connect when the connection is lost.                                                                                                         |\n| CRYTON_WORKER_MSFRPCD_HOST         | str     | localhost                        | Metasploit Framework RPC host.                                                                                                                                           |\n| CRYTON_WORKER_MSFRPCD_PORT         | int     | 55553                            | Metasploit Framework RPC port.                                                                                                                                           |\n| CRYTON_WORKER_MSFRPCD_SSL          | boolean | true                             | Use SSL to connect to Metasploit Framework RPC.                                                                                                                          |\n| CRYTON_WORKER_MSFRPCD_USERNAME     | string  | msf                              | Username for Metasploit Framework RPC login.                                                                                                                             |\n| CRYTON_WORKER_MSFRPCD_PASSWORD     | string  | toor                             | Password for Metasploit Framework RPC login.                                                                                                                             |\n| CRYTON_WORKER_RABBIT_HOST          | string  | 127.0.0.1                        | RabbitMQ server host.                                                                                                                                                    |\n| CRYTON_WORKER_RABBIT_PORT          | int     | 5672                             | RabbitMQ server port.                                                                                                                                                    |\n| CRYTON_WORKER_RABBIT_USERNAME      | string  | admin                            | Username for RabbitMQ server login.                                                                                                                                      |\n| CRYTON_WORKER_RABBIT_PASSWORD      | string  | mypass                           | Password for RabbitMQ server login.                                                                                                                                      |\n| CRYTON_WORKER_EMPIRE_HOST          | string  | 127.0.0.1                        | Empire server host.                                                                                                                                                      |\n| CRYTON_WORKER_EMPIRE_PORT          | int     | 1337                             | Empire server port.                                                                                                                                                      |\n| CRYTON_WORKER_EMPIRE_USERNAME      | string  | empireadmin                      | Username for Empire server login.                                                                                                                                        |\n| CRYTON_WORKER_EMPIRE_PASSWORD      | string  | password123                      | Password for Empire server login.                                                                                                                                        |\n| CRYTON_WORKER_APP_DIRECTORY        | string  | ~/.local/cryton-worker/          | Path to the Cryton Worker directory. **(do not change/set/export, if you don\'t know what you\'re doing)** <br> If changed, update the commands in this guide accordingly. |\n\nTo save the settings **create an app directory**:\n```shell\nmkdir ~/.local/cryton-worker/\n```\n\nThe directory will be also used to store logs and other data created by Cryton Worker.  \n**This doesn\'t apply to the Docker installation.** It will be available in the same directory as the Dockerfile \n(`/path/to/cryton-worker/cryton-worker`).\n\nNext, we download example settings (**change the version to match the app version - versions can be found [here](https://gitlab.ics.muni.cz/cryton/cryton-worker/-/tags)**):\n```shell\ncurl -o ~/.local/cryton-worker/.env https://gitlab.ics.muni.cz/cryton/cryton-worker/-/raw/<version>/.env\n```\nUpdate these settings to your needs.\n\n### Overriding the settings\n**NOTICE: This doesn\'t apply to the Docker Compose installation.**\n\nTo override the persistent settings, you can set/export the variables yourself using the **export** command \n(use **unset** to remove the variable). For example:\n```shell\nexport CRYTON_WORKER_NAME=my_worker1\n```\n\nSome environment variables can be overridden in CLI. Try using `cryton-worker --help`.\n\n### Setting up modules\nTo be able to **execute** (validate) **attack modules** you must download them into one directory. Then update \n`CRYTON_WORKER_MODULES_DIR` environment variable to point to the correct location. If you\'re using the provided modules \nfrom the [modules\' repository](https://gitlab.ics.muni.cz/cryton/cryton-modules), then the variable \nwill look similar to this `CRYTON_WORKER_MODULES_DIR=/path/to/cryton-modules/modules/`.\n\nModules are hot-swappable, which means the modules don\'t have to be present at startup. \nThis is especially useful for development but **not recommended for production**.\n\nModules directory example:\n```\ntree $CRYTON_WORKER_MODULES_DIR\nCRYTON_WORKER_MODULES_DIR/\n├── mod_hydra\n│   └── mod.py\n└── mod_cmd\n    └── mod.py\n```\n\n## Prerequisites\nWorker can run without these prerequisites. However, they are **highly recommended** since they allow Worker to use all of its functionality.\n- [Metasploit Framework](https://docs.metasploit.com/docs/using-metasploit/getting-started/nightly-installers.html) allows using Metasploit sessions and MSF listeners.\n- [Empire post-exploitation framework](https://bc-security.gitbook.io/empire-wiki/quickstart/installation) allows deployment and interaction with Empire agents.\n\nAdditionally, to start the MSF as a service follow [this guide](https://docs.rapid7.com/metasploit/running-metasploit-remotely/) or simply use:\n```shell\nmsfrpcd -U <CRYTON_WORKER_MSFRPCD_USERNAME> -P <CRYTON_WORKER_MSFRPCD_PASSWORD>\n```\n\n## Installation (using pip/pipx)\nCryton Worker is available in the [PyPI](https://pypi.org/project/cryton-worker/) and can be installed using *pip* (`pip install --user cryton-worker`). \nHowever, we **highly recommend** installing the app in an isolated environment using [pipx](https://pypa.github.io/pipx/).\n\n### Requirements\nInstall the following requirements:\n- [Python](https://www.python.org/about/gettingstarted/) >=3.8\n- [pipx](https://pypa.github.io/pipx/)\n\n### Installing with pipx\nOnce you have *pipx* ready on your system, you can start the installation:\n```shell\npipx install cryton-worker\n```\n\nMake sure you\'ve correctly set the [settings](#settings).\n\nOptionally, you can set up [shell completion](#shell-completion).\n\nEverything should be set. Check out the [usage section](#usage).\n\n## Installation (using Docker Compose)\nCryton Worker can be installed using Docker Compose.\n\nTo allow the Worker to start listeners, the container has raw access to the host’s network interface.\n\n**This guide won\'t describe how to install or mount the tools/applications used by the (attack) modules.** \nMore information can be found in the [Docker documentation](https://docs.docker.com/storage/volumes/).\n\n### Requirements\n- [Docker Compose](https://docs.docker.com/compose/install/)\n\nAdd yourself to the group *docker*, so you can work with Docker CLI without *sudo*:\n```shell\nsudo groupadd docker\nsudo usermod -aG docker $USER\nnewgrp docker\ndocker run hello-world\n```\n\n### Installing and running with Docker Compose\nFirst, we have to clone the repo and switch to the correct version.\n```shell\ngit clone https://gitlab.ics.muni.cz/cryton/cryton-worker.git\ncd cryton-worker\ngit checkout <version>\n```\n\nMake sure you\'ve correctly set the [settings](#settings). You can\'t change the settings on a running container.\n\nFinally, copy your settings:\n```shell\ncp ~/.local/cryton-worker/.env .env\n```\n\nWe are now ready to build and start the Worker:\n```shell\ndocker compose up -d --build\n```\n\nAfter a while you should see a similar output:\n```\n[+] Running 1/1\n ⠿ Container cryton_worker  Started\n```\n\nEverything should be set. Check if the installation was successful and the Worker is running:\n```shell\ndocker compose logs\n```\nYou should see `[*] Waiting for messages.` in the output.\n\nDocker can sometimes create dangling (`<none>:<none>`) images which can result in high disk space usage. You can remove them using: \n```shell\ndocker image prune\n```\n\n## Development\nTo install Cryton Worker for development, you must install [Poetry](https://python-poetry.org/docs/).\n\nClone the repository:\n```shell\ngit clone https://gitlab.ics.muni.cz/cryton/cryton-worker.git\n```\n\nThen go to the correct directory and install the project:\n```shell\ncd cryton-worker\npoetry install\n```\n\nTo spawn a shell use:\n```shell\npoetry shell\n```\n\nMake sure you\'ve correctly set the [settings](#settings).  \nTo override the settings quickly, you can use this handy one-liner:\n```shell\nexport $(grep -v \'^#\' .env | xargs)\n```\n\nOptionally, you can set up [shell completion](#shell-completion)\n\nEverything should be set, check out the [usage section](#usage).\n\n## Usage\n**NOTICE: If you\'re using Docker Compose to install the app, you can skip this section.**\n\nUse the following to invoke the app:\n```shell\ncryton-worker\n```\n\nYou should see a help page:\n```\nUsage: cryton-worker [OPTIONS] COMMAND [ARGS]...\n\n  Cryton Worker CLI.\n\nOptions:\n  ...\n```\n\n**To learn about each command\'s options use**:\n```shell\ncryton-worker <your command> --help\n```\n\nTo start Worker use `cryton-worker start` and you should see something like:\n```\nStarting Worker <Worker name>..\nTo exit press CTRL+C\nConnection does not exist. Retrying..\nConnection to RabbitMQ server established.\n[*] Waiting for messages.\n```\n\n## Executing modules\nTo be able to execute a module (Python script), it must have the following structure and IO arguments.\n\n### Modules\' structure\n- Each module must have its own directory with its name.\n- Script (module) must be called `mod.py`.\n- Module must contain an `execute` function that takes a dictionary and returns a dictionary. It\'s an entry point for executing it.\n- Module should contain a `validate` function that takes a dictionary, validates it, and returns 0 if it\'s okay, else raises an exception.\n\nPath example:  \n`/CRYTON_WORKER_MODULES_DIR/my-module-name/mod.py`\n\nWhere:  \n- **CRYTON_WORKER_MODULES_DIR** has to be the same path as is defined in the *CRYTON_WORKER_MODULES_DIR* variable.\n- **my-module-name** is the directory containing your module.\n- **mod.py** is the module file.\n\nModule (`mod.py`) example:  \n```python\ndef validate(arguments: dict) -> int:\n    if arguments != {}:\n        return 0  # If arguments are valid.\n    raise Exception("No arguments")  # If arguments aren\'t valid.\n\ndef execute(arguments: dict) -> dict:\n    # Do stuff.\n    return {"return_code": 0, "serialized_output": ["x", "y"]}\n\n```\n\n### Input parameters\nEvery module has its own input parameters. These input parameters are given as a dictionary to the \nmodule `execute` (when executing the module) or `validate` (when validating the module parameters) function. \n\n### Output parameters\nEvery attack module (its `execute` function) returns a dictionary with the following keys:\n\n| Parameter name      | Parameter meaning                                                                                                                                                                                                                                 |\n|---------------------|---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|\n| `return_code`       | Numeric representation of result (0, -1, -2). <br />0 (OK) means the module finished successfully.<br />-1 (FAIL) means the module finished unsuccessfully.<br />-2 (ERROR) means the module finished with an unhandled error.                    |\n| `serialized_output` | Parsed output of the module. Eg. for a bruteforce module, this might be a list of found usernames and passwords.                                                                                                                                  |                                                                                                                                                                           |\n| `output`            | Raw output of the module                                                                                                                                                                                                                          |\n\n## Prebuilt functionality for modules\nWorker provides prebuilt functionality to make building modules easier. Import it using:\n```python\nfrom cryton_worker.lib.util import module_util\n```\n\nIt gives you access to:\n### Metasploit\nWrapper for *MsfRpcClient* from *[pymetasploit3](https://github.com/DanMcInerney/pymetasploit3)*.\nExamples:\n```python\n# Check if the connection to msfrpcd is OK before doing anything.\nfrom cryton_worker.lib.util.module_util import Metasploit\nmsf = Metasploit()\nif msf.is_connected():\n    msf.do_stuff()\n```\n```python\nfrom cryton_worker.lib.util.module_util import Metasploit\nsearch_criteria = {"via_exploit": "my/exploit"}\nfound_sessions = Metasploit().get_sessions(**search_criteria)\n```\n```python\nfrom cryton_worker.lib.util.module_util import Metasploit\noutput = Metasploit().execute_in_session("my_command", "session_id")\n```\n\n```python\nfrom cryton_worker.lib.util.module_util import Metasploit\n\noptions = {"exploit_arguments": {}, "payload_arguments": {}}\nMetasploit().execute_exploit("my_exploit", "my_payload", **options)\n```\n```python\nfrom cryton_worker.lib.util.module_util import Metasploit\ntoken = Metasploit().client.add_perm_token()\n```\n```python\nfrom cryton_worker.lib.util.module_util import Metasploit\noutput = Metasploit().get_parameter_from_session("session_id", "my_param")\n```\n\n### get_file_binary\nFunction to get a file as binary.  \nExample:\n```python\nfrom cryton_worker.lib.util.module_util import get_file_binary\nmy_file_content = get_file_binary("/path/to/my/file")\n```\n\n### File\nClass used with *[schema](https://pypi.org/project/schema/)* for validation if file exists.  \nExample:\n```python\nfrom schema import Schema\nfrom cryton_worker.lib.util.module_util import File\nschema = Schema(File(str))\nschema.validate("/path/to/file")\n```\n\n### Dir\nClass used with *[schema](https://pypi.org/project/schema/)* for validation if directory exists.  \nExample:\n```python\nfrom schema import Schema\nfrom cryton_worker.lib.util.module_util import Dir\nschema = Schema(Dir(str))\nschema.validate("/path/to/directory")\n```\n\n## Rabbit API\nWorker is able to process any request sent through RabbitMQ to its Queues (`cryton_worker.WORKER_NAME.attack.request`, \n`cryton_worker.WORKER_NAME.control.request`, `cryton_worker.WORKER_NAME.agent.request`)\ndefined using *WORKER_NAME* (can be changed using CLI or in the settings).\n\nThe response is sent to the queue defined using the `reply_to` parameter in a *message.properties*.\n\n### Attack requests\nRequests to execute a command or a module are being processed in the `cryton_worker.WORKER_NAME.attack.request` queue.  \nList of supported requests:\n\n#### Execute attack module\nTo execute an attack module, send a message to `cryton_worker.WORKER_NAME.attack.request` queue in a format \n```json lines\n{"ack_queue": "confirmation_queue", "step_type": "worker/execute", "module": module_name, "module_arguments": module_arguments}\n```\n\nACK response format:\n```json\n{"return_code": 0, "correlation_id": "id"}\n```\n\nResponse format:\n```json\n{"return_code": 0, "output": "", "serialized_output": ""}\n```\n\n#### Execute command on agent\nTo execute a command on a deployed agent, send a message to the `cryton_worker.WORKER_NAME.attack.request` queue in a format \n```json\n{"step_type": "empire/execute", "arguments": {"shell_command": "whoami", "use_agent": "MyAgent"}}\n```\n\nACK response format:\n```json\n{"return_code": 0, "correlation_id": "id"}\n```\n\nResponse format:\n```json\n{"return_code": 0, "output": "", "serialized_output": ""}\n```\n\n#### Execute empire module on agent\nTo execute an empire module on a deployed agent, send a message to the `cryton_worker.WORKER_NAME.attack.request` queue in a format \n```json\n{"step_type": "empire/execute", "arguments": { "empire_module": "python/collection/linux/pillage_user", "use_agent": "MyAgent"}}\n```\n\nACK response format:\n```json\n{"return_code": 0, "correlation_id": "id"}\n```\n\nResponse format: \n```json\n{"return_code": 0, "output": "", "serialized_output": ""}\n```\n\n### Agent requests\nRequests to control empire agents are being processed in `cryton_worker.WORKER_NAME.agent.request` queue.  \nList of supported requests:\n\n#### Deploy agent\nDeploy an agent and send a response containing the result.  \nExample: \n```json\n{"step_type": "empire/agent-deploy", "arguments": {"stager_type": "multi/bash", "agent_name": "MyAgent", "listener_name": "TestListener", "listener_port": 80, "session_id": "MSF_SESSION_ID"}}\n```\n\nResponse example: \n```json\n{"return_code": 0, "output": "Agent \'MyAgent\' deployed on target 192.168.33.12."}\n```\n### Control requests\nTo perform a control event send a message to `cryton_worker.WORKER_NAME.control.request` queue in a format \n```json lines\n{"event_t": type, "event_v": value}\n```\n\nResponse format:\n```json lines\n{"event_t": type, "event_v": value}\n```\n\n**List of supported requests:**\n\n#### Validate module\nValidate a module and send a response containing the result.  \nExample: \n```json lines\n{"event_t": "VALIDATE_MODULE", "event_v": {"module": module_name, "module_arguments": module_arguments}}\n```\n\nResponse example: \n```json\n{"event_t": "VALIDATE_MODULE", "event_v": {"return_code": 0, "output": "output"}}\n```\n\n#### List modules\nList available modules and send a response containing the result.  \n\nRequest example: \n```json\n{"event_t": "LIST_MODULES", "event_v": {}}\n```\n\nResponse example: \n```json\n{"event_t": "LIST_MODULES", "event_v": {"module_list": ["module_name"]}}\n```\n\n#### List sessions\nList available Metasploit sessions and send a response containing the result.\n\nRequest example:\n```json lines\n{"event_t": "LIST_SESSIONS", "event_v": {"target_host": target_ip}}\n```\n\nResponse example: \n```json\n{"event_t": "LIST_SESSIONS", "event_v": {"session_list": ["session_id"]}}\n```\n\n#### Kill Step execution\nKill running Step (module) and send a response containing the result.  \nExample:\n```json lines\n{"event_t": "KILL_STEP_EXECUTION", "event_v": {"correlation_id": correlation_id}}\n```\n\nResponse example:\n```json\n{"event_t": "KILL_STEP_EXECUTION", "event_v": {"return_code": -2, "output": "exception"}}\n```\n\n#### Health check\nCheck if Worker is alive and send a response containing the result.  \nExample: \n```json\n{"event_t": "HEALTH_CHECK", "event_v": {}}\n```\n\nResponse example: \n```json\n{"event_t": "HEALTH_CHECK", "event_v": {"return_code": 0}}\n```\n\n#### Add trigger for HTTPListener\nAdd trigger with parameters and start listener with `host` and `port` if it doesn\'t already exists, send a response containing the result afterwards.  \n\nRequest example: \n```json lines\n{"event_t": "ADD_TRIGGER", "event_v": {"host": host, "port": port, "listener_type": "HTTP", "reply_to": reply_to_queue, \n  "routes": [{"path": path, "method": method, "parameters": [{"name": name, "value": value}]}]}}\n```\n\nResponse example:\n```json\n{"event_t": "ADD_TRIGGER", "event_v": {"return_code": 0, "trigger_id": "123"}}\n```\n#### Remove trigger for HTTPListener\nRemove trigger, optionally stop the  HTTPListener if there are no triggers left and send a response containing the result.  \n\nRequest example: \n```json\n{"event_t": "REMOVE_TRIGGER", "event_v": {"trigger_id": "123"}}\n```\n\n#### Add trigger for MSFListener\nAdd trigger with session identifiers and start MSFListener.\n\nRequest example:\n```json\n{"event_t": "ADD_TRIGGER", "event_v": {"listener_type": "MSF", "reply_to": "cryton_core.control.response", "identifiers": {"via_exploit": "auxiliary/scanner/ssh/ssh_login"}}}\n```\n\nResponse example: \n```json\n{"event_t": "ADD_TRIGGER", "event_v": {"return_code": 0, "trigger_id": "123"}}\n```\n\n#### Remove trigger for MSFListener\nThis will stop the MSFListener because it can\'t have multiple triggers.\n\nRequest example:\n```json\n{"event_t": "REMOVE_TRIGGER", "event_v": {"trigger_id": "123"}}\n```\n\nResponse example:\n```json\n{"event_t": "REMOVE_TRIGGER", "event_v": {"return_code": -2, "output": "exception"}}\n```\n\n#### List triggers\nList available triggers and send a response containing the result.  \n\nExample:\n```json\n{"event_t": "LIST_TRIGGERS", "event_v": {}}\n```\n\nResponse example:\n```json lines\n{"event_t": "LIST_TRIGGERS", "event_v": {"trigger_list": [{"id": "123", "trigger_param": "trigger_param_value", ...}]}}\n```\n\n#### Trigger Stage (Response only)\nSent when a trigger is activated.\n\nResponse example:\n```json lines\n{"event_t": "TRIGGER_STAGE", "event_v": {"stage_execution_id": stage_execution_id}}\n```\n\n## Shell completion\nShell completion is available for the *Bash*, *Zsh*, and *Fish* shell and has to be manually enabled (**the tool must be installed first**).\n\n### Bash\nFirst, **create an app directory** (if you haven\'t already):\n```shell\nmkdir ~/.local/cryton-worker/\n```\n\nGenerate and save the completion script:\n```shell\n_CRYTON_WORKER_COMPLETE=bash_source cryton-worker > ~/.local/cryton-worker/cryton-worker-complete.bash\n```\n\nSource the file in the `~/.bashrc` file:\n```shell\necho ". ~/.local/cryton-worker/cryton-worker-complete.bash" >> ~/.bashrc\n```\n\nYou may need to restart your shell for the changes to take effect.\n\n### Zsh\nFirst, **create an app directory** (if you haven\'t already):\n```shell\nmkdir ~/.local/cryton-worker/\n```\n\nGenerate and save the completion script:\n```shell\n_CRYTON_WORKER_COMPLETE=zsh_source cryton-worker > ~/.local/cryton-worker/cryton-worker-complete.zsh\n```\n\nSource the file in the `~/.zshrc` file:\n```shell\necho ". ~/.local/cryton-worker/cryton-worker-complete.zsh" >> ~/.zshrc\n```\n\nYou may need to restart your shell for the changes to take effect.\n\n### Fish\nGenerate and save the completion script:\n```shell\n_CRYTON_WORKER_COMPLETE=fish_source cryton-worker > ~/.config/fish/completions/cryton-worker-complete.fish\n```\n\nYou may need to restart your shell for the changes to take effect.\n',
    'author': 'Ivo Nutár',
    'author_email': 'nutar@ics.muni.cz',
    'maintainer': 'Jiří Rája',
    'maintainer_email': 'raja@ics.muni.cz',
    'url': 'https://gitlab.ics.muni.cz/cryton',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
