# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['excelcy']

package_data = \
{'': ['*'], 'excelcy': ['.git/*', '.git/hooks/*', '.git/info/*']}

install_requires = \
['attrs>=20.1.0,<21.0.0',
 'pyexcel-xlsx>=0.5.8,<0.6.0',
 'pyexcel>=0.6.4,<0.7.0',
 'pytest-cov>=2.0,<3.0',
 'pytest>=5.4.3,<6.0.0',
 'pyyaml>=5.3.1,<6.0.0',
 'spacy>=2.3.2,<3.0.0']

setup_kwargs = {
    'name': 'excelcy',
    'version': '0.4.1',
    'description': 'Excel Integration with SpaCy. Includes, Entity training, Entity matcher pipe.',
    'long_description': 'ExcelCy\n=======\n\n[![Travis (.org)](https://img.shields.io/travis/kororo/excelcy)](https://pypi.python.org/project/excelcy/)\n[![Coverage Status](https://coveralls.io/repos/github/kororo/excelcy/badge.svg?branch=master)](https://coveralls.io/github/kororo/excelcy?branch=master)\n[![MIT license](https://img.shields.io/badge/License-MIT-blue.svg)](https://lbesson.mit-license.org/)\n[![PyPI pyversions](https://img.shields.io/pypi/pyversions/excelcy.svg)](https://pypi.python.org/project/excelcy/)\n[![PyPI - Downloads](https://img.shields.io/pypi/dm/excelcy)](https://pypi.python.org/project/excelcy/)\n\n* * * * *\n\nExcelCy is a NER trainer from XLSX, PDF, DOCX, PPT, PNG or JPG. ExcelCy uses spaCy framework to match Entity with PhraseMatcher or Matcher in regular expression.\n\nExcelCy is convenience\n----------------------\n\nThis is example taken from spaCy documentation, [Simple Style Training](https://spacy.io/usage/training#training-simple-style). It demonstrates how to train NER using spaCy:\n\n```python\nimport spacy\nimport random\n\nTRAIN_DATA = [\n     ("Uber blew through $1 million a week", {\'entities\': [(0, 4, \'ORG\')]}), # note: it is required to supply the character position\n     ("Google rebrands its business apps", {\'entities\': [(0, 6, "ORG")]})] # note: it is required to supply the character position\n\nnlp = spacy.blank(\'en\')\noptimizer = nlp.begin_training()\nfor i in range(20):\n    random.shuffle(TRAIN_DATA)\n    for text, annotations in TRAIN_DATA:\n        nlp.update([text], [annotations], sgd=optimizer)\n\nnlp.to_disk(\'test_model\')\n```\n\nThe **TRAIN\\_DATA**, describes sentences and annotated entities to be trained. It is cumbersome to always count the characters. With ExcelCy, (start,end) characters can be omitted.\n\n```python\n# install excelcy\n# pip install excelcy\n\n# download the en model from spacy\n# python -m spacy download en"\n\n# run this inside python or file\nfrom excelcy import ExcelCy\n\n# Test: John is the CEO of this_is_a_unique_company_name\nexcelcy = ExcelCy()\n# by default it is assume the nlp_base using model en_core_web_sm\n# excelcy.storage.config = Config(nlp_base=\'en_core_web_sm\')\ndoc = excelcy.nlp(\'John is the CEO of this_is_a_unique_company_name\')\n# it will show no company entities\nprint([(ent.label_, ent.text) for ent in doc.ents])\n# run this in root of repo or https://github.com/kororo/excelcy/raw/master/tests/data/test_data_01.xlsx\nexcelcy = ExcelCy.execute(file_path=\'tests/data/test_data_01.xlsx\')\n# use the nlp object as per spaCy API\ndoc = excelcy.nlp(\'John is the CEO of this_is_a_unique_company_name\')\n# now it recognise the company name\nprint([(ent.label_, ent.text) for ent in doc.ents])\n# NOTE: if not showing, remember, it may be required to increase the "train_iteration" or lower the "train_drop", the "config" sheet in Excel\n```\n\nExcelCy is friendly\n-------------------\n\nBy default, ExcelCy training is divided into phases, the example Excel file can be found in [tests/data/test\\_data\\_01.xlsx](https://github.com/kororo/excelcy/raw/master/tests/data/test_data_01.xlsx):\n\n### 1. Discovery\n\nThe first phase is to collect sentences from data source in sheet "source". The data source can be either:\n\n-   Text: Direct sentence values.\n-   Files: PDF, DOCX, PPT, PNG or JPG will be parsed using\n    [textract](https://github.com/deanmalmgren/textract).\n\nNote: See textract source examples in [tests/data/test\\_data\\_03.xlsx](https://github.com/kororo/excelcy/raw/master/tests/data/test_data_03.xlsx)\nNote: Dependencies "textract" is not included in the ExcelCy, it is required to add manually\n\n### 2. Preparation\n\nNext phase, the Gold annotation needs to be defined in sheet "prepare", based on:\n\n-   Current Data Model: Using spaCy API of **nlp(sentence).ents**\n-   Phrase pattern: Robbie, Uber, Google, Amazon\n-   Regex pattern: \\^([0-1]?[0-9]|2[0-3]):[0-5][0-9]\\$\n\nAll annotations in here are considered as Gold annotations, which described in [here](https://spacy.io/usage/training#example-new-entity-type).\n\n### 3. Training\n\nMain phase of NER training, which described in [Simple Style Training](https://spacy.io/usage/training#training-simple-style).\nThe data is iterated from sheet "train", check sheet "config" to control the parameters.\n\n### 4. Consolidation\n\nThe last phase, is to test/save the results and repeat the phases if required.\n\nExcelCy is flexible\n-------------------\n\nNeed more specific export and phases? It is possible to control it using phase API.\nThis is the illustration of the real-world scenario:\n\n1.  Train from\n    [tests/data/test\\_data\\_05.xlsx](https://github.com/kororo/excelcy/raw/master/tests/data/test_data_05.xlsx)\n\n    ```shell script\n    # download the dataset\n    $ wget https://github.com/kororo/excelcy/raw/master/tests/data/test_data_05.xlsx\n    # this will create a directory and file "export/train_05.xlsx"\n    $ excelcy execute test_data_05.xlsx\n    ```\n\n2.  Open the result in "export/train\\_05.xlsx", it shows all identified sentences from source given. However, there is error in the "Himalayas" as identified as "PRODUCT".\n    \n3.  To fix this, add phrase matcher for "Himalayas = FAC". It is illustrated in\n    [tests/data/test\\_data\\_05a.xlsx](https://github.com/kororo/excelcy/raw/master/tests/data/test_data_05a.xlsx)\n    \n4.  Train again and check the result in "export/train\\_05a.xlsx"\n\n    ```shell script\n    # download the dataset\n    $ wget https://github.com/kororo/excelcy/raw/master/tests/data/test_data_05a.xlsx\n    # this will create a directory "nlp/data" and file "export/train_05a.xlsx"\n    $ excelcy execute test_data_05a.xlsx\n    ```\n\n5.  Check the result that there is backed up nlp data model in "nlp" and the result is corrected in "export/train\\_05a.xlsx"\n\n6.  Keep training the data model, if there is unexpected behaviour, there is backup data model in case needed.\n\nExcelCy is comprehensive\n------------------------\n\nUnder the hood, ExcelCy has strong and well-defined data storage. At any given phase above, the data can be inspected.\n\n```python\nfrom excelcy import ExcelCy\nfrom excelcy.storage import Config\n\n# Test: John is the CEO of this_is_a_unique_company_name\nexcelcy = ExcelCy()\nexcelcy.storage.config = Config(nlp_base=\'en_core_web_sm\', train_iteration=10, train_drop=0.2)\ndoc = excelcy.nlp(\'John is the CEO of this_is_a_unique_company_name\')\n# showing no ORG\nprint([(ent.label_, ent.text) for ent in doc.ents])\nexcelcy.storage.source.add(kind=\'text\', value=\'John is the CEO of this_is_a_unique_company_name\')\nexcelcy.discover()\nexcelcy.storage.prepare.add(kind=\'phrase\', value=\'this_is_a_unique_company_name\', entity=\'ORG\')\nexcelcy.prepare()\nexcelcy.train()\ndoc = excelcy.nlp(\'John is the CEO of this_is_a_unique_company_name\')\n# ORG now is recognised\nprint([(ent.label_, ent.text) for ent in doc.ents])\n# NOTE: if not showing, remember, it may be required to increase the "train_iteration" or lower the "train_drop", the "config" sheet in Excel\n```\n\nFeatures\n--------\n\n-   Load multiple data sources such as Word documents, PowerPoint presentations, PDF or images.\n-   Import/Export configuration with JSON, YML or Excel.\n-   Add custom Entity labels.\n-   Rule based phrase matching using [PhraseMatcher](https://spacy.io/usage/linguistic-features#adding-phrase-patterns)\n-   Rule based matching using [regex + Matcher](https://spacy.io/usage/linguistic-features#regex)\n-   Train Named Entity Recogniser with ease\n\nInstall\n-------\n\nEither use the famous pip or clone this repository and execute the\nsetup.py file.\n\n```shell script\n$ pip install excelcy\n# ensure you have the language model installed before\n$ spacy download en\n```\n\nTrain\n-----\n\nTo train the spaCy model:\n\n```python\nfrom excelcy import ExcelCy\nexcelcy = ExcelCy.execute(file_path=\'test_data_01.xlsx\')\n```\n\nNote: [tests/data/test\\_data\\_01.xlsx](https://github.com/kororo/excelcy/raw/master/tests/data/test_data_01.xlsx)\n\nCLI\n---\n\nExelCy has basic CLI command for execute:\n\n```shell script\n$ excelcy execute https://github.com/kororo/excelcy/raw/master/tests/data/test_data_01.xlsx\n```\n\nTest\n----\n\nRun test by installing packages and run tox\n\n```shell script\n$ pip install poetry tox\n$ tox\n$ tox -e py36 -- tests/test_readme.py\n```\n\nFor hot-reload development coding\n```shell script\n$ npm i -g nodemon\n$ nodemon\n```\n\nData Definition\n---------------\n\nExcelCy has data definition which expressed in [api.yml](https://github.com/kororo/excelcy/raw/master/data/api.yml).\nAs long as, data given in this specific format and structure, ExcelCy will able to support any type of data format.\nCheck out, the Excel file format in [api.xlsx](https://github.com/kororo/excelcy/raw/master/data/api.xlsx).\nData classes are defined with [attrs](https://github.com/python-attrs/attrs),\ncheck in [storage.py](https://github.com/kororo/excelcy/raw/master/excelcy/storage.py) for more detail.\n\nPublishing\n----------\n```shell script\n# this is note for contributors\n# ensure locally tests all running\nnpm run test\n\n# prepare for new version\npoetry version 0.4.1\nnpm run export\n\n# make changes in the git, especially release branch and check in the travis\n# https://travis-ci.com/github/kororo/excelcy\n\n# if all goes well, push to master\n\n```\nFAQ\n---\n\n**What is that idx columns in the Excel sheet?**\n\nThe idea is to give reference between two things. Imagine in sheet "train", like to know where the sentence generated\nfrom in sheet "source". And also, the nature of Excel, you can sort things, this is the safe guard to keep things in\nthe correct order.\n\n**Can ExcelCy import/export to X, Y, Z data format?**\n\nExcelCy has strong and well-defined data storage, thanks to [attrs](https://github.com/python-attrs/attrs).\nIt is possible to import/export data in any format.\n\n**Error: ModuleNotFoundError: No module named \'pip\'**\n\nThere are lots of possibility on this. Try to lower pip version (it was buggy for version 19.0.3).\n\n**ExcelCy accepts suggestions/ideas?**\n\nYes! Please submit them into new issue with label "enhancement".\n\nAcknowledgement\n---------------\n\nThis project uses other awesome projects:\n\n-   [attrs](https://github.com/python-attrs/attrs): Python Classes Without Boilerplate.\n-   [pyexcel](https://github.com/pyexcel/pyexcel): Single API for reading, manipulating and writing data in csv, ods, xls, xlsx and xlsm files.\n-   [pyyaml](https://github.com/yaml/pyyaml): The next generation YAML parser and emitter for Python.\n-   [spacy](https://github.com/explosion/spaCy): Industrial-strength Natural Language Processing (NLP) with Python and Cython.\n-   [textract](https://github.com/deanmalmgren/textract): extract text from any document. no muss. no fuss.\n\n',
    'author': 'kororo',
    'author_email': 'kororo@users.noreply.github.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/kororo/excelcy',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
