"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SubnetFilter = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const network_util_1 = require("./network-util");
/**
 * Contains logic which chooses a set of subnets from a larger list, in conjunction with SubnetSelection, to determine where to place AWS resources such as VPC endpoints, EC2 instances, etc.
 *
 * @stability stable
 */
class SubnetFilter {
    /**
     * Chooses subnets by id.
     *
     * @stability stable
     */
    static byIds(subnetIds) {
        return new SubnetIdSubnetFilter(subnetIds);
    }
    /**
     * Chooses subnets which are in one of the given availability zones.
     *
     * @stability stable
     */
    static availabilityZones(availabilityZones) {
        return new AvailabilityZoneSubnetFilter(availabilityZones);
    }
    /**
     * Chooses subnets such that there is at most one per availability zone.
     *
     * @stability stable
     */
    static onePerAz() {
        return new OnePerAZSubnetFilter();
    }
    /**
     * Chooses subnets which contain any of the specified IP addresses.
     *
     * @stability stable
     */
    static containsIpAddresses(ipv4addrs) {
        return new ContainsIpAddressesSubnetFilter(ipv4addrs);
    }
    /**
     * Chooses subnets which have the provided CIDR netmask.
     *
     * @stability stable
     */
    static byCidrMask(mask) {
        return new CidrMaskSubnetFilter(mask);
    }
    /**
     * Executes the subnet filtering logic, returning a filtered set of subnets.
     *
     * @stability stable
     */
    selectSubnets(_subnets) {
        throw new Error('Cannot select subnets with an abstract SubnetFilter. `selectSubnets` needs to be implmemented.');
    }
}
exports.SubnetFilter = SubnetFilter;
_a = JSII_RTTI_SYMBOL_1;
SubnetFilter[_a] = { fqn: "@aws-cdk/aws-ec2.SubnetFilter", version: "1.136.0" };
/**
 * Chooses subnets which are in one of the given availability zones.
 */
class AvailabilityZoneSubnetFilter extends SubnetFilter {
    constructor(availabilityZones) {
        super();
        this.availabilityZones = availabilityZones;
    }
    /**
     * Executes the subnet filtering logic.
     */
    selectSubnets(subnets) {
        return subnets.filter(s => this.availabilityZones.includes(s.availabilityZone));
    }
}
/**
 * Chooses subnets such that there is at most one per availability zone.
 */
class OnePerAZSubnetFilter extends SubnetFilter {
    constructor() {
        super();
    }
    /**
     * Executes the subnet filtering logic.
     */
    selectSubnets(subnets) {
        return this.retainOnePerAz(subnets);
    }
    retainOnePerAz(subnets) {
        const azsSeen = new Set();
        return subnets.filter(subnet => {
            if (azsSeen.has(subnet.availabilityZone)) {
                return false;
            }
            azsSeen.add(subnet.availabilityZone);
            return true;
        });
    }
}
/**
 * Chooses subnets which contain any of the specified IP addresses.
 */
class ContainsIpAddressesSubnetFilter extends SubnetFilter {
    constructor(ipAddresses) {
        super();
        this.ipAddresses = ipAddresses;
    }
    /**
     * Executes the subnet filtering logic.
     */
    selectSubnets(subnets) {
        return this.retainByIp(subnets, this.ipAddresses);
    }
    retainByIp(subnets, ips) {
        const cidrBlockObjs = ips.map(ip => {
            const ipNum = network_util_1.NetworkUtils.ipToNum(ip);
            return new network_util_1.CidrBlock(ipNum, 32);
        });
        return subnets.filter(s => {
            const subnetCidrBlock = new network_util_1.CidrBlock(s.ipv4CidrBlock);
            return cidrBlockObjs.some(cidr => subnetCidrBlock.containsCidr(cidr));
        });
    }
}
/**
 * Chooses subnets based on the subnetId
 */
class SubnetIdSubnetFilter extends SubnetFilter {
    constructor(subnetIds) {
        super();
        this.subnetIds = subnetIds;
    }
    /**
     * Executes the subnet filtering logic.
     */
    selectSubnets(subnets) {
        return subnets.filter(subnet => this.subnetIds.includes(subnet.subnetId));
    }
}
/**
 * Chooses subnets based on the CIDR Netmask
 */
class CidrMaskSubnetFilter extends SubnetFilter {
    constructor(mask) {
        super();
        this.mask = mask;
    }
    /**
     * Executes the subnet filtering logic.
     */
    selectSubnets(subnets) {
        return subnets.filter(subnet => {
            const subnetCidr = new network_util_1.CidrBlock(subnet.ipv4CidrBlock);
            return subnetCidr.mask === this.mask;
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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