"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const types_1 = require("../types");
/**
 * Base class for all other state classes
 */
class State extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.branches = [];
        this.retries = [];
        this.catches = [];
        this.choices = [];
        this.prefixes = [];
        /**
         * States with references to this state.
         *
         * Used for finding complete connected graph that a state is part of.
         */
        this.incomingStates = [];
        this.startState = this;
        this.comment = props.comment;
        this.inputPath = props.inputPath;
        this.parameters = props.parameters;
        this.outputPath = props.outputPath;
        this.resultPath = props.resultPath;
    }
    /**
     * Add a prefix to the stateId of all States found in a construct tree
     */
    static prefixStates(root, prefix) {
        const queue = [root];
        while (queue.length > 0) {
            const el = queue.splice(0, 1)[0];
            if (isPrefixable(el)) {
                el.addPrefix(prefix);
            }
            queue.push(...el.node.children);
        }
    }
    /**
     * Find the set of end states states reachable through transitions from the given start state
     */
    static findReachableEndStates(start, options = {}) {
        const visited = new Set();
        const ret = new Set();
        const queue = [start];
        while (queue.length > 0) {
            const state = queue.splice(0, 1)[0];
            if (visited.has(state)) {
                continue;
            }
            visited.add(state);
            const outgoing = state.outgoingTransitions(options);
            if (outgoing.length > 0) {
                // We can continue
                queue.push(...outgoing);
            }
            else {
                // Terminal state
                ret.add(state);
            }
        }
        return Array.from(ret);
    }
    /**
     * Return only the states that allow chaining from an array of states
     */
    static filterNextables(states) {
        return states.filter(isNextable);
    }
    get id() {
        return this.node.id;
    }
    /**
     * Tokenized string that evaluates to the state's ID
     */
    get stateId() {
        return this.prefixes.concat(this.id).join('');
    }
    /**
     * Add a prefix to the stateId of this state
     */
    addPrefix(x) {
        if (x !== '') {
            this.prefixes.splice(0, 0, x);
        }
    }
    /**
     * Register this state as part of the given graph
     *
     * Don't call this. It will be called automatically when you work
     * with states normally.
     */
    bindToGraph(graph) {
        if (this.containingGraph === graph) {
            return;
        }
        if (this.containingGraph) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`Trying to use state '${this.stateId}' in ${graph}, but is already in ${this.containingGraph}. Every state can only be used in one graph.`);
        }
        this.containingGraph = graph;
        this.whenBoundToGraph(graph);
        for (const incoming of this.incomingStates) {
            incoming.bindToGraph(graph);
        }
        for (const outgoing of this.outgoingTransitions({ includeErrorHandlers: true })) {
            outgoing.bindToGraph(graph);
        }
        for (const branch of this.branches) {
            branch.registerSuperGraph(this.containingGraph);
        }
        if (!!this.iteration) {
            this.iteration.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Add a retrier to the retry list of this state
     * @internal
     */
    _addRetry(props = {}) {
        this.retries.push({
            ...props,
            errors: props.errors ? props.errors : [types_1.Errors.ALL],
        });
    }
    /**
     * Add an error handler to the catch list of this state
     * @internal
     */
    _addCatch(handler, props = {}) {
        this.catches.push({
            next: handler,
            props: {
                errors: props.errors ? props.errors : [types_1.Errors.ALL],
                resultPath: props.resultPath
            }
        });
        handler.addIncoming(this);
        if (this.containingGraph) {
            handler.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Make the indicated state the default transition of this state
     */
    makeNext(next) {
        // Can't be called 'setNext' because of JSII
        if (this._next) {
            throw new Error(`State '${this.id}' already has a next state`);
        }
        this._next = next;
        next.addIncoming(this);
        if (this.containingGraph) {
            next.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Add a choice branch to this state
     */
    addChoice(condition, next) {
        this.choices.push({ condition, next });
        next.startState.addIncoming(this);
        if (this.containingGraph) {
            next.startState.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Add a paralle branch to this state
     */
    addBranch(branch) {
        this.branches.push(branch);
        if (this.containingGraph) {
            branch.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Add a map iterator to this state
     */
    addIterator(iteration) {
        this.iteration = iteration;
        if (this.containingGraph) {
            iteration.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Make the indicated state the default choice transition of this state
     */
    makeDefault(def) {
        // Can't be called 'setDefault' because of JSII
        if (this.defaultChoice) {
            throw new Error(`Choice '${this.id}' already has a default next state`);
        }
        this.defaultChoice = def;
    }
    /**
     * Render the default next state in ASL JSON format
     */
    renderNextEnd() {
        if (this._next) {
            return { Next: this._next.stateId };
        }
        else {
            return { End: true };
        }
    }
    /**
     * Render the choices in ASL JSON format
     */
    renderChoices() {
        return {
            Choices: renderList(this.choices, renderChoice),
            Default: this.defaultChoice ? this.defaultChoice.stateId : undefined,
        };
    }
    /**
     * Render InputPath/Parameters/OutputPath in ASL JSON format
     */
    renderInputOutput() {
        return {
            InputPath: renderJsonPath(this.inputPath),
            Parameters: this.parameters,
            OutputPath: renderJsonPath(this.outputPath),
        };
    }
    /**
     * Render parallel branches in ASL JSON format
     */
    renderBranches() {
        return {
            Branches: this.branches.map(b => b.toGraphJson())
        };
    }
    /**
     * Render map iterator in ASL JSON format
     */
    renderIterator() {
        if (!this.iteration) {
            throw new Error('Iterator must not be undefined !');
        }
        return {
            Iterator: this.iteration.toGraphJson()
        };
    }
    /**
     * Render error recovery options in ASL JSON format
     */
    renderRetryCatch() {
        return {
            Retry: renderList(this.retries, renderRetry),
            Catch: renderList(this.catches, renderCatch),
        };
    }
    /**
     * Called whenever this state is bound to a graph
     *
     * Can be overridden by subclasses.
     */
    whenBoundToGraph(graph) {
        graph.registerState(this);
    }
    /**
     * Add a state to the incoming list
     */
    addIncoming(source) {
        this.incomingStates.push(source);
    }
    /**
     * Return all states this state can transition to
     */
    outgoingTransitions(options) {
        const ret = new Array();
        if (this._next) {
            ret.push(this._next);
        }
        if (this.defaultChoice) {
            ret.push(this.defaultChoice);
        }
        for (const c of this.choices) {
            ret.push(c.next);
        }
        if (options.includeErrorHandlers) {
            for (const c of this.catches) {
                ret.push(c.next);
            }
        }
        return ret;
    }
}
exports.State = State;
/**
 * Render a choice transition
 */
function renderChoice(c) {
    return {
        ...c.condition.renderCondition(),
        Next: c.next.stateId
    };
}
/**
 * Render a Retry object to ASL
 */
function renderRetry(retry) {
    return {
        ErrorEquals: retry.errors,
        IntervalSeconds: retry.interval && retry.interval.toSeconds(),
        MaxAttempts: retry.maxAttempts,
        BackoffRate: retry.backoffRate
    };
}
/**
 * Render a Catch object to ASL
 */
function renderCatch(c) {
    return {
        ErrorEquals: c.props.errors,
        ResultPath: renderJsonPath(c.props.resultPath),
        Next: c.next.stateId,
    };
}
/**
 * Render a list or return undefined for an empty list
 */
function renderList(xs, fn) {
    if (xs.length === 0) {
        return undefined;
    }
    return xs.map(fn);
}
exports.renderList = renderList;
/**
 * Render JSON path, respecting the special value DISCARD
 */
function renderJsonPath(jsonPath) {
    if (jsonPath === undefined) {
        return undefined;
    }
    if (jsonPath === types_1.DISCARD) {
        return null;
    }
    if (!jsonPath.startsWith('$')) {
        throw new Error(`Expected JSON path to start with '$', got: ${jsonPath}`);
    }
    return jsonPath;
}
exports.renderJsonPath = renderJsonPath;
/**
 * Whether an object is a Prefixable
 */
function isPrefixable(x) {
    return typeof (x) === 'object' && x.addPrefix;
}
/**
 * Whether an object is INextable
 */
function isNextable(x) {
    return typeof (x) === 'object' && x.next;
}
//# sourceMappingURL=data:application/json;base64,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