from rich.console import Console
from pathlib import Path
import click
import os
import uuid

from flowui.cli.utils import platform, operators_repository


console = Console()

# Ref: https://patorjk.com/software/taag/
msg = """
==============================================
      ______ __                 __  __ ____
     / ____// /____  _      __ / / / //  _/
    / /_   / // __ \| | /| / // / / / / /  
   / __/  / // /_/ /| |/ |/ // /_/ /_/ /   
  /_/    /_/ \____/ |__/|__/ \____//___/         
  
=============================================="""


###############################################################################
# FLOWUI PLATFORM
###############################################################################

def validate_github_token_workflows(value):
    if value and value.startswith("ghp_"):
        return value
    return None


def validate_github_token_operators(value):
    if value and value.startswith("ghp_"):
        return value
    return None


def get_cluster_name_from_env():
    cluster_name = os.environ.get("FLOWUI_KIND_CLUSTER_NAME", None)
    if not cluster_name:
        cluster_name = "flowui-cluster"
    return cluster_name


def get_github_workflows_ssh_private_key_from_env():
    return os.environ.get("FLOWUI_GITHUB_WORKFLOWS_SSH_PRIVATE_KEY", "")

def get_github_token_operators_from_env():
    return os.environ.get("FLOWUI_GITHUB_ACCESS_TOKEN_OPERATORS", None)

def get_github_token_workflows_from_env():
    return os.environ.get("FLOWUI_GITHUB_ACCESS_TOKEN_WORKFLOWS", None)

def get_dockerhub_username_from_env():
    return os.environ.get("DOCKERHUB_USERNAME", "")

def get_dockerhub_password_from_env():
    return os.environ.get("DOCKERHUB_PASSWORD", "")

def get_workflows_repository_from_env():
    return os.environ.get("FLOWUI_WORKFLOWS_REPOSITORY", None)

@click.command()
@click.option(
    '--cluster-name', 
    prompt='Local cluster name', 
    default=get_cluster_name_from_env,
    help='Define the name for the local k8s cluster.'
)
@click.option(
    '--workflows-repository',
    prompt='Workflows repository',
    default=get_workflows_repository_from_env,
    help='Github repository where the FlowUI workflows will be stored.'
)
@click.option(
    '--github-workflows-ssh-private-key', 
    prompt='Github ssh private for Workflows repository. If none, it will create a ssh key pair to be used.',
    default=get_github_workflows_ssh_private_key_from_env,
    help='Github ssh key for GitSync read/write operations on Workflows repository. The private key will be used in the FlowUI cluster and the public key should be added to the Github repository deploy keys.'
)
@click.option(
    '--github-operators-token', 
    prompt='Github token for Operators repository',
    default=get_github_token_operators_from_env,
    help='Github token for read operations on Operators repositories.'
)
@click.option(
    '--github-workflows-token', 
    prompt='Github token for Workflows Repository',
    default=get_github_token_workflows_from_env,
    help='Github token for read/write operations on Workflows Repository.'
)
def cli_prepare_platform(cluster_name, workflows_repository, github_workflows_ssh_private_key, github_operators_token, github_workflows_token):
    """Prepare local folder for running a FlowUI platform."""
    platform.prepare_platform(
        cluster_name=cluster_name, 
        workflows_repository=workflows_repository,
        github_workflows_ssh_private_key=github_workflows_ssh_private_key, 
        github_operators_token=github_operators_token,
        github_workflows_token=github_workflows_token
    )


@click.command()
@click.option(
    "--flowui-frontend-image", 
    default=None,
    help="Load a local FlowUI frontend image to cluster."
)
@click.option(
    "--flowui-backend-image", 
    default=None,
    help="Load a local FlowUI backend image to cluster."
)
def cli_create_platform(flowui_frontend_image, flowui_backend_image):
    """Create cluster, install services and run FlowUI platform."""
    platform.create_platform(flowui_frontend_image, flowui_backend_image)


@click.group()
@click.pass_context
def cli_platform(ctx):
    """FlowUI platform actions"""
    if ctx.invoked_subcommand == "prepare":
        console.print("Let's get you started configuring a local FlowUI platform:")
    elif ctx.invoked_subcommand == "create":
        console.print("Your local FlowUI platform is being created. This might take a while...")    


cli_platform.add_command(cli_prepare_platform, name="prepare")
cli_platform.add_command(cli_create_platform, name="create")

###############################################################################
# OPERATORS REPOSITORY 
###############################################################################

def generate_random_repo_name():
    return f"new_repository_{str(uuid.uuid4())[0:8]}"


@click.command()
@click.option(
    '--name', 
    prompt="Repository's name",
    default=generate_random_repo_name,
    help="Repository's name"
)
@click.option(
    '--dockerhub-registry', 
    prompt="Dockerhub registry name",
    default="",
    help="Dockerhub registry name"
)
def cli_create_op_repository(name, dockerhub_registry):
    """Create a basic Operators repository with placeholder files."""
    operators_repository.create_operators_repository(repository_name=name, dockerhub_registry=dockerhub_registry)


@click.command()
@click.option(
    '--build-images', 
    is_flag=True,
    prompt='Build Docker images?',
    expose_value=True,
    default=False,
    help='If True (default), builds Docker images.'
)
@click.option(
    '--publish-images', 
    is_flag=True,
    prompt='Publish Docker images?',
    expose_value=True,
    default=False,
    help='If True (default), publishes Docker images to Dockerhub.'
)
@click.option(
    '--dockerhub-username', 
    prompt='Dockerhub username',
    default=get_dockerhub_username_from_env,
    help='Your username from DockerHub account.'
)
@click.option(
    '--dockerhub-password', 
    prompt='Dockerhub password',
    default=get_dockerhub_password_from_env,
    help='Your password from DockerHub account.'
)
def cli_organize_op_repository(build_images, publish_images, dockerhub_username, dockerhub_password):
    """Prepare local folder for running a FlowUI platform."""
    os.environ['DOCKERHUB_USERNAME'] = dockerhub_username
    os.environ['DOCKERHUB_PASSWORD'] = dockerhub_password
    operators_repository.organize_operators_repository(build_images, publish_images)


@click.command()
def cli_create_release():
    """
    Get release version for the Operators repository in github stdout format.
    Used by github actions to set the release version.
    Needs the following env vars:
        - GITHUB_TOKEN
        - GITHUB_REPOSITORY
    """
    operators_repository.create_release()
    

@click.group()
@click.pass_context
def cli_operator(ctx):
    """Operator repository actions"""
    if ctx.invoked_subcommand == "organize":
        console.print(f"Organizing Operators repository at: {Path('.').resolve()}")
    elif ctx.invoked_subcommand == "create":
        pass


cli_operator.add_command(cli_organize_op_repository, name="organize")
cli_operator.add_command(cli_create_op_repository, name="create")
cli_operator.add_command(cli_create_release, name="release")


###############################################################################
# PARENT GROUP
###############################################################################

@click.command()
def cli_run_operator_k8s():
    """Run Operator on Kubernetes Pod"""
    from flowui.scripts.run_operator_k8s import run_operator as run_operator_in_k8s
    
    console.print("Running Operator inside K8s pod...")
    run_operator_in_k8s()


@click.command()
def cli_run_operator_bash():
    """Run Operator on bash"""
    from flowui.scripts.run_operator_bash import run_operator as run_operator_in_bash

    console.print("Running bash Operator...")
    run_operator_in_bash()


###############################################################################
# PARENT GROUP
###############################################################################

@click.group()
@click.pass_context
def cli(ctx):
    console.print(msg, style="rgb(109,125,176)", highlight=False)
    console.print("Welcome to FlowUI! :red_heart-emoji:")
    console.print("")


cli.add_command(cli_platform, name="platform")
cli.add_command(cli_operator, name="operator")
cli.add_command(cli_run_operator_k8s, name="run-operator-k8s")
cli.add_command(cli_run_operator_bash, name="run-operator-bash")


if __name__ == '__main__':
    cli()
