# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['torchfl',
 'torchfl.datamodules',
 'torchfl.federated',
 'torchfl.federated.agents',
 'torchfl.federated.aggregators',
 'torchfl.federated.samplers',
 'torchfl.models',
 'torchfl.models.core',
 'torchfl.models.core.cifar',
 'torchfl.models.core.cifar.cifar10',
 'torchfl.models.core.cifar.cifar100',
 'torchfl.models.core.emnist',
 'torchfl.models.core.emnist.balanced',
 'torchfl.models.core.emnist.byclass',
 'torchfl.models.core.emnist.bymerge',
 'torchfl.models.core.emnist.digits',
 'torchfl.models.core.emnist.letters',
 'torchfl.models.core.emnist.mnist',
 'torchfl.models.core.fashionmnist',
 'torchfl.models.sota',
 'torchfl.models.wrapper']

package_data = \
{'': ['*']}

install_requires = \
['numpy>=1.24.2,<2.0.0',
 'pytest>=7.2.1,<8.0.0',
 'pytorch-lightning>=1.9,<1.10',
 'rich>=13.3.1,<14.0.0',
 'torch>=1.13,<1.14',
 'torchvision>=0.14,<0.15']

setup_kwargs = {
    'name': 'torchfl',
    'version': '0.1.9',
    'description': 'A Python library for rapid prototyping, experimenting, and logging of federated learning using state-of-the-art models and datasets. Built using PyTorch and PyTorch Lightning.',
    'long_description': '<div align="center">\n\t<img src="https://raw.githubusercontent.com/vivekkhimani/torchfl/master/docs/_static/images/torchfl-github.png" width="960px" height="480px">\n</div>\n<hr>\n<div align="center">\n\t<a href="https://pypi.org/project/torchfl"><img src="https://img.shields.io/pypi/v/torchfl"></a>\n\t<a href="https://github.com/vivekkhimani/torchfl"><img src="https://img.shields.io/github/v/release/vivekkhimani/torchfl"></a>\n\t<a href="https://pypi.org/project/torchfl"><img src="https://img.shields.io/pypi/pyversions/torchfl"></a>\n\t<a href="https://pypi.org/project/torchfl"><img src="https://img.shields.io/pypi/wheel/torchfl"></a>\n\t<a href="https://pypi.org/project/torchfl"><img src="https://img.shields.io/pypi/implementation/torchfl"></a>\n\t<a href="https://github.com/vivekkhimani/torchfl"><img src="https://img.shields.io/github/contributors/vivekkhimani/torchfl"></a>\n\t<a href="https://github.com/vivekkhimani/torchfl"><img src="https://img.shields.io/github/commit-activity/w/vivekkhimani/torchfl"></a>\n\t<a href="https://pypi.org/project/torchfl"><img src="https://img.shields.io/pypi/dm/torchfl"></a>\n\t<a href="https://github.com/vivekkhimani/torchfl"><img src="https://img.shields.io/github/directory-file-count/vivekkhimani/torchfl"></a>\n\t<a href="https://github.com/vivekkhimani/torchfl"><img src="https://img.shields.io/github/languages/count/vivekkhimani/torchfl"></a>\n\t<a href="https://github.com/vivekkhimani/torchfl"><img src="https://img.shields.io/github/languages/top/vivekkhimani/torchfl"></a>\n\t<a href="https://github.com/vivekkhimani/torchfl"><img src="https://img.shields.io/github/search/vivekkhimani/torchfl/torchfl"></a>\n\t<a href="https://torchfl.readthedocs.io"><img src="https://img.shields.io/readthedocs/torchfl"</a>\n\t<a href="https://github.com/vivekkhimani/torchfl/blob/master/LICENSE"><img src="https://img.shields.io/pypi/l/torchfl"></a>\n</div>\n<hr>\n\n## Table of Contents\n\n- [Key Features](#features)\n- [Installation](#installation)\n- [Examples and Usage](#examples-and-usage)\n- [Available Models](#available-models)\n- [Available Datasets](#available-datasets)\n- [Contributing](#contributing)\n- [Citation](#citation)\n\n## Features\n\n- Python 3.6+ support. Built using ```torch-1.10.1```, ```torchvision-0.11.2```, and ```pytorch-lightning-1.5.7```.\n- Customizable implementations for state-of-the-art deep learning [models](#available-models) which can be trained in federated or non-federated settings.\n- Supports finetuning of the pre-trained deep learning models, allowing for faster training using transfer learning.\n- PyTorch LightningDataModule wrappers for the most commonly used [datasets](#available-datasets) to reduce the boilerplate code before experiments.\n- Built using the bottom-up approach for the datamodules and models which ensures abstractions while allowing for customization.\n- Provides implementation of the federated learning (FL) samplers, aggregators, and wrappers, to prototype FL experiments on-the-go.\n- Backwards compatible with the PyTorch LightningDataModule, LightningModule, loggers, and DevOps tools.\n- More details about the examples and usage can be found [below](#examples-and-usage).\n- For more documentation related to the usage, visit - https://torchfl.readthedocs.io/.\n\n## Installation\n### Stable Release\nAs of now, ```torchfl``` is available on PyPI and can be installed using the following command in your terminal:\n```\n$ pip install torchfl\n```\nThis is the preferred method to install ```torchfl``` with the most stable release.\nIf you don\'t have [pip](https://pip.pypa.io/en/stable/) installed, this [Python installation guide](http://docs.python-guide.org/en/latest/starting/installation/) can guide you through the process.\n\n## Examples and Usage\nAlthough ```torchfl``` is primarily built for quick prototyping of federated learning experiments, the models, datasets, and abstractions can also speed up the non-federated learning experiments. In this section, we will explore examples and usages under both the settings.\n\n### Non-Federated Learning\nThe following steps should be followed on a high-level to train a non-federated learning experiment. We are using the ```EMNIST (MNIST)``` dataset and ```densenet121``` for this example.\n\n1. Import the relevant modules.\n\t```python\n\tfrom torchfl.datamodules.emnist import EMNISTDataModule\n\tfrom torchfl.models.wrapper.emnist import MNISTEMNIST\n\t```\n\n\t```python\n\timport pytorch_lightning as pl\n\tfrom pytorch_lightning.loggers import TensorBoardLogger\n\tfrom pytorch_lightning.callbacks import (\n\t\tModelCheckpoint,\n\t\tLearningRateMonitor,\n\t\tDeviceStatsMonitor,\n\t\tModelSummary,\n\t\tProgressBar,\n\t\t...\n\t)\n\t```\n\tFor more details, view the full list of PyTorch Lightning [callbacks](https://pytorch-lightning.readthedocs.io/en/stable/extensions/callbacks.html#callback) and [loggers](https://pytorch-lightning.readthedocs.io/en/latest/common/loggers.html#loggers) on the official website.\n2. Setup the PyTorch Lightning trainer.\n\t```python\n\ttrainer = pl.Trainer(\n\t\t...\n\t\tlogger=[\n\t\t\tTensorBoardLogger(\n\t\t\t\tname=experiment_name,\n\t\t\t\tsave_dir=os.path.join(checkpoint_save_path, experiment_name),\n\t\t\t)\n\t\t],\n\t\tcallbacks=[\n\t\t\tModelCheckpoint(save_weights_only=True, mode="max", monitor="val_acc"),\n\t\t\tLearningRateMonitor("epoch"),\n\t\t\tDeviceStatsMonitor(),\n\t\t\tModelSummary(),\n\t\t\tProgressBar(),\n\t\t],\n\t\t...\n\t)\n\t```\n\tMore details about the PyTorch Lightning [Trainer API](https://pytorch-lightning.readthedocs.io/en/latest/common/trainer.html#) can be found on their official website.\n\n3. Prepare the dataset using the wrappers provided by ```torchfl.datamodules```.\n\t```python\n\tdatamodule = EMNISTDataModule(dataset_name="mnist")\n\tdatamodule.prepare_data()\n\tdatamodule.setup()\n\t```\n\n4. Initialize the model using the wrappers provided by ```torchfl.models.wrappers```.\n\t```python\n\t# check if the model can be loaded from a given checkpoint\n\tif (checkpoint_load_path) and os.path.isfile(checkpoint_load_path):\n\t\tmodel = MNISTEMNIST(\n\t\t\t"densenet121", "adam", {"lr": 0.001}\n\t\t\t).load_from_checkpoint(checkpoint_load_path)\n\telse:\n\t\tpl.seed_everything(42)\n\t\tmodel = MNISTEMNIST("densenet121", "adam", {"lr": 0.001})\n\t\ttrainer.fit(model, datamodule.train_dataloader(), datamodule.val_dataloader())\n\t```\n\n5. Collect the results.\n\t```python\n\tval_result = trainer.test(\n\t\tmodel, test_dataloaders=datamodule.val_dataloader(), verbose=True\n\t)\n\ttest_result = trainer.test(\n\t\tmodel, test_dataloaders=datamodule.test_dataloader(), verbose=True\n\t)\n\t```\n\n6. The corresponding files for the experiment (model checkpoints and logger metadata) will be stored at ```default_root_dir``` argument given to the PyTorch Lightning ```Trainer``` object in Step 2. For this experiment, we use the [Tensorboard](https://www.tensorflow.org/tensorboard) logger. To view the logs (and related plots and metrics), go to the ```default_root_dir``` path and find the Tensorboard log files. Upload the files to the Tensorboard Development portal following the instructions [here](https://tensorboard.dev/#get-started). Once the log files are uploaded, a unique url to your experiment will be generated which can be shared with ease! An example can be found [here](https://tensorboard.dev/experiment/Q1tw19FySLSjLN6CW5DaUw/).\n\n7. Note that, ```torchfl``` is compatible with all the loggers supported by PyTorch Lightning. More information about the PyTorch Lightning loggers can be found [here](https://pytorch-lightning.readthedocs.io/en/latest/common/loggers.html#loggers).\n\nFor full non-federated learning example scripts, check [examples/trainers](https://github.com/vivekkhimani/torchfl/tree/master/examples/trainers).\n\n### Federated Learning\nThe following steps should be followed on a high-level to train a federated learning experiment.\n\n1. Pick a dataset and use the ```datamodules``` to create federated data shards with iid or non-iid distribution.\n\t```python\n\tdef get_datamodule() -> EMNISTDataModule:\n\t\tdatamodule: EMNISTDataModule = EMNISTDataModule(\n\t\t\tdataset_name=SUPPORTED_DATASETS_TYPE.MNIST, train_batch_size=10\n\t\t)\n\t\tdatamodule.prepare_data()\n\t\tdatamodule.setup()\n\t\treturn datamodule\n\n    agent_data_shard_map = get_agent_data_shard_map().federated_iid_dataloader(\n        num_workers=fl_params.num_agents,\n        workers_batch_size=fl_params.local_train_batch_size,\n    )\n\t```\n2. Use the TorchFL ```agents``` module and the  ```models``` module to initialize the global model, agents, and distribute their models.\n\t```python\n\tdef initialize_agents(\n\t\tfl_params: FLParams, agent_data_shard_map: Dict[int, DataLoader]\n\t) -> List[V1Agent]:\n\t\t"""Initialize agents."""\n\t\tagents = []\n\t\tfor agent_id in range(fl_params.num_agents):\n\t\t\tagent = V1Agent(\n\t\t\t\tid=agent_id,\n\t\t\t\tmodel=MNISTEMNIST(\n\t\t\t\t\tmodel_name=EMNIST_MODELS_ENUM.MOBILENETV3SMALL,\n\t\t\t\t\toptimizer_name=OPTIMIZERS_TYPE.ADAM,\n\t\t\t\t\toptimizer_hparams={"lr": 0.001},\n\t\t\t\t\tmodel_hparams={"pre_trained": True, "feature_extract": True},\n\t\t\t\t\tfl_hparams=fl_params,\n\t\t\t\t),\n\t\t\t\tdata_shard=agent_data_shard_map[agent_id],\n\t\t\t)\n\t\t\tagents.append(agent)\n\t\treturn agents\n\n\tglobal_model = MNISTEMNIST(\n        model_name=EMNIST_MODELS_ENUM.MOBILENETV3SMALL,\n        optimizer_name=OPTIMIZERS_TYPE.ADAM,\n        optimizer_hparams={"lr": 0.001},\n        model_hparams={"pre_trained": True, "feature_extract": True},\n        fl_hparams=fl_params,\n    )\n\n\tall_agents = initialize_agents(fl_params, agent_data_shard_map)\n\t```\n3. Initiliaze an ```FLParam``` object with the desired FL hyperparameters and pass it on to the ```Entrypoint``` object which will abstract the training.\n\t```python\n    fl_params = FLParams(\n        experiment_name="iid_mnist_fedavg_10_agents_5_sampled_50_epochs_mobilenetv3small_latest",\n        num_agents=10,\n        global_epochs=10,\n        local_epochs=2,\n        sampling_ratio=0.5,\n    )\n    entrypoint = Entrypoint(\n        global_model=global_model,\n        global_datamodule=get_agent_data_shard_map(),\n        fl_hparams=fl_params,\n        agents=all_agents,\n        aggregator=FedAvgAggregator(all_agents=all_agents),\n        sampler=RandomSampler(all_agents=all_agents),\n    )\n    entrypoint.run()\n\t```\n\nFor full federated learning example scripts, check [examples/federated](https://github.com/vivekkhimani/torchfl/tree/master/examples/federated).\n\n## Available Models\nFor the initial release, ```torchfl``` will only support state-of-the-art computer vision models. The following table summarizes the available models, support for pre-training, and the possibility of feature-extracting. Please note that the models have been tested with all the available datasets. Therefore, the link to the tests will be provided in the next section.\n\n<table>\n\t<thead>\n\t\t<tr>\n\t\t\t<th>Name</th>\n\t\t\t<th>Pre-Training</th>\n\t\t\t<th>Feature Extraction</th>\n\t\t</tr>\n\t</thead>\n\t<tbody>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/alexnet.py" target="_blank">AlexNet</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/densenet.py#L20" target="_blank">DenseNet121</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/densenet.py#L78" target="_blank">DenseNet161</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/densenet.py#L136" target="_blank">DenseNet169</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/densenet.py#L194" target="_blank">DenseNet201</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/lenet.py" target="_blank">LeNet</a></td>\n\t\t\t<td>:x:</td>\n\t\t\t<td>:x:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/mlp.py" target="_blank">MLP</a></td>\n\t\t\t<td>:x:</td>\n\t\t\t<td>:x:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/mobilenet.py#L23" target="_blank">MobileNetV2</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/mobilenet.py#L78" target="_blank">MobileNetV3Small</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/mobilenet.py#L140" target="_blank">MobileNetV3Large</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/resnet.py#L26" target="_blank">ResNet18</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/resnet.py#L76" target="_blank">ResNet34</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/resnet.py#L125" target="_blank">ResNet50</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/resnet.py#L174" target="_blank">ResNet101</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/resnet.py#L223" target="_blank">ResNet152</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/resnet.py#L272" target="_blank">ResNext50(32x4d)</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/resnet.py#L323" target="_blank">ResNext101(32x8d)</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/resnet.py#L374" target="_blank">WideResNet(50x2)</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/resnet.py#L425" target="_blank">WideResNet(101x2)</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/shufflenetv2.py#L20" target="_blank">ShuffleNetv2(x0.5)</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/shufflenetv2.py#L74" target="_blank">ShuffleNetv2(x1.0)</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/shufflenetv2.py#L128" target="_blank">ShuffleNetv2(x1.5)</a></td>\n\t\t\t<td>:x:</td>\n\t\t\t<td>:x:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/shufflenetv2.py#L83" target="_blank">ShuffleNetv2(x2.0)</a></td>\n\t\t\t<td>:x:</td>\n\t\t\t<td>:x:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/squeezenet.py#L19" target="_blank">SqueezeNet1.0</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/squeezenet.py#L83" target="_blank">SqueezeNet1.1</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/vgg.py#L45" target="_blank">VGG11</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/vgg.py#L95" target="_blank">VGG11_BatchNorm</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/vgg.py#L145" target="_blank">VGG13</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/vgg.py#L195" target="_blank">VGG13_BatchNorm</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/vgg.py#L245" target="_blank">VGG16</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/vgg.py#L295" target="_blank">VGG16_BatchNorm</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/vgg.py#L345" target="_blank">VGG19</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/models/sota/vgg.py#L395" target="_blank">VGG19_BatchNorm</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t</tr>\n\t</tbody>\n</table>\n\n\n## Available Datasets\nFollowing datasets have been wrapped inside a ```LightningDataModule``` and made available for the initial release of ```torchfl```. To add a new dataset, check the source code in ```torchfl.datamodules```, add tests, and create a PR with ```Features``` tag.\n\n\n<table>\n\t<thead>\n\t\t<tr>\n\t\t\t<th>Group</th>\n\t\t\t<th>Datasets</th>\n\t\t\t<th>IID Split</th>\n\t\t\t<th>Non-IID Split</th>\n\t\t\t<th>Datamodules Tests</th>\n\t\t\t<th>Models</th>\n\t\t\t<th>Models Tests</th>\n\t\t</tr>\n\t</thead>\n\t<tbody>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/datamodules/cifar.py">CIFAR</a></td>\n\t\t\t<td>\n\t\t\t\t<ul>\n\t\t\t\t\t<li><a href="http://pytorch.org/vision/main/generated/torchvision.datasets.CIFAR10.html">CIFAR10</a></li>\n\t\t\t\t\t<li><a href="http://pytorch.org/vision/main/generated/torchvision.datasets.CIFAR100.html">CIFAR100</a></li>\n\t\t\t\t</ul>\n\t\t\t</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>\n\t\t\t\t<a href="https://github.com/vivekkhimani/torchfl/actions/workflows/pytest-datamodules.yml"><img src="https://github.com/vivekkhimani/torchfl/actions/workflows/pytest-datamodules.yml/badge.svg"></a>\n\t\t\t</td>\n\t\t\t<td>\n\t\t\t\t<ul>\n\t\t\t\t\t<li>\n\t\t\t\t\t\t<a href="https://github.com/vivekkhimani/torchfl/tree/master/torchfl/models/core/cifar/cifar10">CIFAR10</a>\n\t\t\t\t\t</li>\n\t\t\t\t\t<li>\n\t\t\t\t\t\t<a href="https://github.com/vivekkhimani/torchfl/tree/master/torchfl/models/core/cifar/cifar100">CIFAR100</a>\n\t\t\t\t\t</li>\n\t\t\t\t</ul>\n\t\t\t</td>\n\t\t\t<td>\n\t\t\t\t<a href="https://github.com/vivekkhimani/torchfl/actions/workflows/pytest-models.yml"><img src="https://github.com/vivekkhimani/torchfl/actions/workflows/pytest-models.yml/badge.svg"></a>\n\t\t\t</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/datamodules/emnist.py">EMNIST</a></td>\n\t\t\t<td>\n\t\t\t\t<ul>\n\t\t\t\t\t<li><a href="https://pytorch.org/vision/main/generated/torchvision.datasets.EMNIST.html">By Class</a></li>\n\t\t\t\t\t<li><a href="https://pytorch.org/vision/main/generated/torchvision.datasets.EMNIST.html">By Merge</a></li>\n\t\t\t\t\t<li><a href="https://pytorch.org/vision/main/generated/torchvision.datasets.EMNIST.html">Balanced</a></li>\n\t\t\t\t\t<li><a href="https://pytorch.org/vision/main/generated/torchvision.datasets.EMNIST.html">Digits</a></li>\n\t\t\t\t\t<li><a href="https://pytorch.org/vision/main/generated/torchvision.datasets.EMNIST.html">Letters</a></li>\n\t\t\t\t\t<li><a href="https://pytorch.org/vision/main/generated/torchvision.datasets.EMNIST.html">MNIST</a></li>\n\t\t\t\t</ul>\n\t\t\t</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/actions/workflows/pytest-datamodules.yml"><img src="https://github.com/vivekkhimani/torchfl/actions/workflows/pytest-datamodules.yml/badge.svg"></a></td>\n\t\t\t<td>\n\t\t\t\t<ul>\n\t\t\t\t\t<li>\n\t\t\t\t\t\t<a href="https://github.com/vivekkhimani/torchfl/tree/master/torchfl/models/core/emnist/byclass">By Class</a>\n\t\t\t\t\t</li>\n\t\t\t\t\t<li>\n\t\t\t\t\t\t<a href="https://github.com/vivekkhimani/torchfl/tree/master/torchfl/models/core/emnist/bymerge">By Merge</a>\n\t\t\t\t\t</li>\n\t\t\t\t\t<li>\n\t\t\t\t\t\t<a href="https://github.com/vivekkhimani/torchfl/tree/master/torchfl/models/core/emnist/balanced">Balanced</a>\n\t\t\t\t\t</li>\n\t\t\t\t\t<li>\n\t\t\t\t\t\t<a href="https://github.com/vivekkhimani/torchfl/tree/master/torchfl/models/core/emnist/digits">Digits</a>\n\t\t\t\t\t</li>\n\t\t\t\t\t<li>\n\t\t\t\t\t\t<a href="https://github.com/vivekkhimani/torchfl/tree/master/torchfl/models/core/emnist/letters">Letters</a>\n\t\t\t\t\t</li>\n\t\t\t\t\t<li>\n\t\t\t\t\t\t<a href="https://github.com/vivekkhimani/torchfl/tree/master/torchfl/models/core/emnist/mnist">MNIST</a>\n\t\t\t\t\t</li>\n\t\t\t\t</ul>\n\t\t\t</td>\n\t\t\t<td>\n\t\t\t\t<a href="https://github.com/vivekkhimani/torchfl/actions/workflows/pytest-models.yml"><img src="https://github.com/vivekkhimani/torchfl/actions/workflows/pytest-models.yml/badge.svg"></a>\n\t\t\t</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/blob/master/torchfl/datamodules/fashionmnist.py">FashionMNIST</a></td>\n\t\t\t<td><a href="https://pytorch.org/vision/main/generated/torchvision.datasets.FashionMNIST.html">FashionMNIST</a></td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td>:white_check_mark:</td>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/actions/workflows/pytest-datamodules.yml"><img src="https://github.com/vivekkhimani/torchfl/actions/workflows/pytest-datamodules.yml/badge.svg"></a></td>\n\t\t\t<td><a href="https://github.com/vivekkhimani/torchfl/tree/master/torchfl/models/core/fashionmnist">FashionMNIST</a></td>\n\t\t\t<td>\n\t\t\t\t<a href="https://github.com/vivekkhimani/torchfl/actions/workflows/pytest-models.yml"><img src="https://github.com/vivekkhimani/torchfl/actions/workflows/pytest-models.yml/badge.svg"></a>\n\t\t\t</td>\n\t</tbody>\n</table>\n\n## Contributing\nContributions are welcome, and they are greatly appreciated! Every little bit helps, and credit will always be given.\n\nYou can contribute in many ways:\n\n### Types of Contributions\n#### Report Bugs\nReport bugs at [https://github.com/vivekkhimani/torchfl/issues](https://github.com/vivekkhimani/torchfl/issues).\n\nIf you are reporting a bug, please include:\n- Your operating system name and version.\n- Any details about your local setup that might be helpful in troubleshooting.\n- Detailed steps to reproduce the bug.\n\n#### Fix Bugs\nLook through the GitHub issues for bugs. Anything tagged with "bug" and "help wanted" is open to whoever wants to implement it.\n\n#### Implement Features\nLook through the GitHub issues for features. Anything tagged with "enhancement", "help wanted", "feature" is open to whoever wants to implement it.\n\n#### Write Documentation\n```torchfl``` could always use more documentation, whether as part of the official torchfl docs, in docstrings, or even on the web in blog posts, articles, and such.\n\n#### Submit Feedback\nThe best way to send feedback is to file an issue at [https://github.com/vivekkhimani/torchfl/issues](https://github.com/vivekkhimani/torchfl/issues).\nIf you are proposing a feature:\n- Explain in detail how it would work.\n- Keep the scope as narrow as possible, to make it easier to implement.\n- Remember that this is a volunteer-driven project, and that contributions are welcome :)\n\n### Get Started\nReady to contribute? Here\'s how to set up torchfl for local development.\n1. Fork the torchfl repo on GitHub.\n2. Clone your fork locally:\n```\n$ git clone git@github.com:<your_username_here>/torchfl.git\n```\n3. Install your local copy into a virtualenv. Assuming you have virtualenvwrapper installed, this is how you set up your fork for local development:\n```\n$ mkvirtualenv torchfl\n$ cd torchfl/\n$ python setup.py develop\n```\n4. Create a branch for local development:\n```\n$ git checkout -b name-of-your-bugfix-or-feature\n```\nNow you can make your changes locally and maintain them on your own branch.\n\n\n5. When you\'re done making changes, check that your changes pass flake8 and the tests, including testing other Python versions with tox:\n```\ntox\n```\nRun ```tox --help``` to explore more features for tox.\n\n\n6. Your changes need to pass the existing test cases and add the new ones if required under the ```tests``` directory. Following approaches can be used to run the test cases.\n- To run all the test cases.\n```\n$ coverage run -m pytest tests\n```\n- To run a specific file containing the test cases.\n```\n$ coverage run -m pytest <path-to-the-file>\n```\n\n7. Commit your changes and push your branch to GitHub:\n```\n$ git add --all\n$ git commit -m "Your detailed description of your changes."\n$ git push origin <name-of-your-bugfix-or-feature>\n```\n8. Submit a pull request through the Github web interface.\n9. Once the pull request has been submitted, the continuous integration pipelines on Github Actions will be triggered. Ensure that all of them pass before one of the maintainers can review the request.\n\n### Pull Request Guidelines\nBefore you submit a pull request, check that it meets these guidelines:\n1. The pull request should include tests.\n\t- Try adding new test cases for new features or enhancements and make changes to the CI pipelines accordingly.\n\t- Modify the existing tests (if required) for the bug fixes.\n2. If the pull request adds functionality, the docs should be updated. Put your new functionality into a function with a docstring, and add the feature to the list in ```README.md```.\n3. The pull request should pass all the existing CI pipelines (Github Actions) and the new/modified workflows should be added as required.\n4. Please note that the test cases should only be run in the CI pipeline if the direct/indirect dependencies of the tests have changed. Look at the [workflow YAML files](https://github.com/vivekkhimani/torchfl/tree/master/.github/workflows) for more details or reach out to one of the contributors.\n\n### Deploying\nA reminder for the maintainers on how to deploy. Make sure all your changes are committed (including an entry in HISTORY.rst). Then run:\n```\n$ bump2version patch # possible: major / minor / patch\n$ git push\n$ git push --tags\n```\n\n## Citation\nPlease cite the following article if you end up using this software:\n```\n@misc{https://doi.org/10.48550/arxiv.2211.00735,\n  doi = {10.48550/ARXIV.2211.00735},\n  url = {https://arxiv.org/abs/2211.00735},\n  author = {Khimani, Vivek and Jabbari, Shahin},\n  keywords = {Machine Learning (cs.LG), Distributed, Parallel, and Cluster Computing (cs.DC), Systems and Control (eess.SY), FOS: Computer and information sciences, FOS: Computer and information sciences, FOS: Electrical engineering, electronic engineering, information engineering, FOS: Electrical engineering, electronic engineering, information engineering, I.2.11},\n  title = {TorchFL: A Performant Library for Bootstrapping Federated Learning Experiments},\n  publisher = {arXiv},\n  year = {2022},\n  copyright = {Creative Commons Attribution Non Commercial Share Alike 4.0 International}\n}\n```\n',
    'author': 'vivekkhimani',
    'author_email': 'vivekkhimani07@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://torchfl.readthedocs.io/en/latest/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
