"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MicroAppsSvcs = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path = require("path");
const apigwy = require("@aws-cdk/aws-apigatewayv2-alpha");
const apigwyAuth = require("@aws-cdk/aws-apigatewayv2-authorizers-alpha");
const apigwyint = require("@aws-cdk/aws-apigatewayv2-integrations-alpha");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const lambdaNodejs = require("aws-cdk-lib/aws-lambda-nodejs");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const constructs_1 = require("constructs");
/**
 * Create a new MicroApps Services construct, including the Deployer
 * and Router Lambda Functions, and the DynamoDB Table used by both.
 */
class MicroAppsSvcs extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        if (props === undefined) {
            throw new Error('props cannot be undefined');
        }
        const { bucketApps, bucketAppsOAI, bucketAppsStaging, deployerTimeout = aws_cdk_lib_1.Duration.minutes(2), s3PolicyBypassAROAs = [], s3PolicyBypassPrincipalARNs = [], s3StrictBucketPolicy = false, appEnv, httpApi, removalPolicy, assetNameRoot, assetNameSuffix, rootPathPrefix = '', requireIAMAuthorization = true, } = props;
        if (s3StrictBucketPolicy === true) {
            if (s3PolicyBypassAROAs.length === 0 && s3PolicyBypassPrincipalARNs.length === 0) {
                throw new Error('s3StrictBucketPolicy cannot be true without specifying at least one s3PolicyBypassAROAs or s3PolicyBypassPrincipalARNs');
            }
        }
        //
        // DynamoDB Table
        //
        if (props.table === undefined) {
            // Create able if none passed
            this._ownedTable = new dynamodb.Table(this, 'table', {
                tableName: assetNameRoot
                    ? `${assetNameRoot}${assetNameSuffix}`
                    : aws_cdk_lib_1.PhysicalName.GENERATE_IF_NEEDED,
                billingMode: dynamodb.BillingMode.PAY_PER_REQUEST,
                partitionKey: {
                    name: 'PK',
                    type: dynamodb.AttributeType.STRING,
                },
                sortKey: {
                    name: 'SK',
                    type: dynamodb.AttributeType.STRING,
                },
                removalPolicy,
            });
            this._table = this._ownedTable;
        }
        else {
            this._table = props.table;
        }
        //
        // Deployer Lambda Function
        //
        // Create Deployer Lambda Function
        const iamRoleUploadName = assetNameRoot
            ? `${assetNameRoot}-deployer-upload${assetNameSuffix}`
            : undefined;
        const iamRoleDeployerName = assetNameRoot
            ? `${assetNameRoot}-deployer${assetNameSuffix}`
            : undefined;
        const iamRoleDeployer = new iam.Role(this, 'deployer-role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            roleName: iamRoleDeployerName,
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        const deployerFuncName = assetNameRoot
            ? `${assetNameRoot}-deployer${assetNameSuffix}`
            : undefined;
        const deployerFuncProps = {
            functionName: deployerFuncName,
            role: iamRoleDeployer,
            memorySize: 1769,
            logRetention: logs.RetentionDays.ONE_MONTH,
            runtime: lambda.Runtime.NODEJS_16_X,
            timeout: deployerTimeout,
            environment: {
                NODE_ENV: appEnv,
                ...(httpApi ? { APIGWY_ID: httpApi.httpApiId } : {}),
                DATABASE_TABLE_NAME: this._table.tableName,
                FILESTORE_STAGING_BUCKET: bucketAppsStaging.bucketName,
                FILESTORE_DEST_BUCKET: bucketApps.bucketName,
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                ROOT_PATH_PREFIX: rootPathPrefix,
                REQUIRE_IAM_AUTHORIZATION: requireIAMAuthorization ? 'true' : 'false',
            },
        };
        if (process.env.NODE_ENV === 'test' &&
            fs_1.existsSync(path.join(__dirname, '..', '..', 'microapps-deployer', 'dist', 'index.js'))) {
            // This is for local dev
            this._deployerFunc = new lambda.Function(this, 'deployer-func', {
                code: lambda.Code.fromAsset(path.join(__dirname, '..', '..', 'microapps-deployer', 'dist')),
                handler: 'index.handler',
                ...deployerFuncProps,
            });
        }
        else if (fs_1.existsSync(path.join(__dirname, 'microapps-deployer', 'index.js'))) {
            // This is for built apps packaged with the CDK construct
            this._deployerFunc = new lambda.Function(this, 'deployer-func', {
                code: lambda.Code.fromAsset(path.join(__dirname, 'microapps-deployer')),
                handler: 'index.handler',
                ...deployerFuncProps,
            });
        }
        else {
            this._deployerFunc = new lambdaNodejs.NodejsFunction(this, 'deployer-func', {
                entry: path.join(__dirname, '..', '..', 'microapps-deployer', 'src', 'index.ts'),
                handler: 'handler',
                bundling: {
                    minify: true,
                    sourceMap: true,
                },
                ...deployerFuncProps,
            });
        }
        if (removalPolicy !== undefined) {
            this._deployerFunc.applyRemovalPolicy(removalPolicy);
        }
        // Give the Deployer access to DynamoDB table
        this._table.grantReadWriteData(this._deployerFunc);
        this._table.grant(this._deployerFunc, 'dynamodb:DescribeTable');
        // Add Tags to Deployer
        aws_cdk_lib_1.Tags.of(this._deployerFunc).add('microapps-deployer', 'true');
        //
        // Deployer upload temp role
        // Deployer assumes this role with a limited policy to generate
        // an STS temp token to return to microapps-publish for the upload.
        //
        const iamRoleUpload = new iam.Role(this, 'deployer-upload-role', {
            roleName: iamRoleUploadName,
            inlinePolicies: {
                uploadPolicy: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            actions: ['s3:ListBucket'],
                            resources: [bucketAppsStaging.bucketArn],
                        }),
                        new iam.PolicyStatement({
                            actions: ['s3:PutObject', 's3:GetObject', 's3:AbortMultipartUpload'],
                            resources: [`${bucketAppsStaging.bucketArn}/*`],
                        }),
                    ],
                }),
            },
            assumedBy: this._deployerFunc.grantPrincipal,
        });
        this._deployerFunc.addEnvironment('UPLOAD_ROLE_NAME', iamRoleUpload.roleName);
        //
        // Update S3 permissions
        //
        // Create PrincipalARN List
        const s3PolicyBypassArnPrincipals = [];
        for (const arnPrincipal of s3PolicyBypassPrincipalARNs) {
            s3PolicyBypassArnPrincipals.push(new iam.ArnPrincipal(arnPrincipal));
        }
        // Create AROA List that matches assumed sessions
        const s3PolicyBypassAROAMatches = [];
        for (const aroa of s3PolicyBypassAROAs) {
            s3PolicyBypassAROAMatches.push(`${aroa}:*`);
        }
        // Deny apps from reading:
        // - If they are missing the microapp-name tag
        // - Anything outside of the folder that matches their microapp-name tag
        const policyDenyPrefixOutsideTag = new iam.PolicyStatement({
            sid: 'deny-prefix-outside-microapp-name-tag',
            effect: iam.Effect.DENY,
            actions: ['s3:*'],
            notPrincipals: [
                new iam.CanonicalUserPrincipal(bucketAppsOAI.cloudFrontOriginAccessIdentityS3CanonicalUserId),
                new iam.AccountRootPrincipal(),
                ...s3PolicyBypassArnPrincipals,
                this._deployerFunc.grantPrincipal,
            ],
            notResources: [
                `${bucketApps.bucketArn}/\${aws:PrincipalTag/microapp-name}/*`,
                bucketApps.bucketArn,
            ],
            conditions: {
                Null: { 'aws:PrincipalTag/microapp-name': 'false' },
            },
        });
        if (removalPolicy !== undefined) {
            policyDenyPrefixOutsideTag.addCondition(
            // Allows the DeletableBucket Lambda to delete items in the buckets
            'StringNotLike', { 'aws:PrincipalTag/application': `${aws_cdk_lib_1.Stack.of(this).stackName}-core*` });
        }
        const policyDenyMissingTag = new iam.PolicyStatement({
            sid: 'deny-missing-microapp-name-tag',
            effect: iam.Effect.DENY,
            actions: ['s3:*'],
            notPrincipals: [
                new iam.CanonicalUserPrincipal(bucketAppsOAI.cloudFrontOriginAccessIdentityS3CanonicalUserId),
                new iam.AccountRootPrincipal(),
                // Exclude the Deployer Function directly
                this._deployerFunc.grantPrincipal,
                // 2021-12-04 - Not 100% sure that this is actually needed...
                // Let's test this and remove if actually not necessary
                new iam.ArnPrincipal(`arn:aws:sts::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:assumed-role/${(_b = this._deployerFunc.role) === null || _b === void 0 ? void 0 : _b.roleName}/${this._deployerFunc.functionName}`),
                ...s3PolicyBypassArnPrincipals,
            ],
            resources: [`${bucketApps.bucketArn}/*`, bucketApps.bucketArn],
            conditions: {
                Null: { 'aws:PrincipalTag/microapp-name': 'true' },
                // Note: This AROA must be specified to prevent this policy from locking
                // out non-root sessions that have assumed the admin role.
                // The notPrincipals will only match the role name exactly and will not match
                // any session that has assumed the role since notPrincipals does not allow
                // wildcard matches and does not do them implicitly either.
                // The AROA must be used because there are only 3 Principal variables:
                //  https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_variables.html#principaltable
                //  aws:username, aws:userid, aws:PrincipalTag
                // For an assumed role, aws:username is blank, aws:userid is:
                //  [unique id AKA AROA for Role]:[session name]
                // Table of unique ID prefixes such as AROA:
                //  https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-prefixes
                // The name of the role is simply not available and if it was
                // we'd need to write a complicated comparison to make sure
                // that we didn't exclude the Deny tag from roles in other accounts.
                //
                // To get the AROA with the AWS CLI:
                //   aws iam get-role --role-name ROLE-NAME
                //   aws iam get-user -–user-name USER-NAME
                StringNotLike: { 'aws:userid': [aws_cdk_lib_1.Aws.ACCOUNT_ID, ...s3PolicyBypassAROAMatches] },
            },
        });
        if (removalPolicy !== undefined) {
            policyDenyMissingTag.addCondition(
            // Allows the DeletableBucket Lambda to delete items in the buckets
            'StringNotLike', { 'aws:PrincipalTag/application': `${aws_cdk_lib_1.Stack.of(this).stackName}-core*` });
        }
        const policyCloudFrontAccess = new iam.PolicyStatement({
            sid: 'cloudfront-oai-access',
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:ListBucket'],
            principals: [
                new iam.CanonicalUserPrincipal(bucketAppsOAI.cloudFrontOriginAccessIdentityS3CanonicalUserId),
            ],
            resources: [`${bucketApps.bucketArn}/*`, bucketApps.bucketArn],
        });
        if (bucketApps.policy === undefined) {
            const document = new s3.BucketPolicy(this, 's3-policy', {
                bucket: bucketApps,
            }).document;
            document.addStatements(policyCloudFrontAccess);
            if (s3StrictBucketPolicy) {
                document.addStatements(policyDenyPrefixOutsideTag);
                document.addStatements(policyDenyMissingTag);
            }
        }
        else {
            bucketApps.policy.document.addStatements(policyCloudFrontAccess);
            if (s3StrictBucketPolicy) {
                bucketApps.policy.document.addStatements(policyDenyPrefixOutsideTag);
                bucketApps.policy.document.addStatements(policyDenyMissingTag);
            }
        }
        // Allow the Lambda to read from the staging bucket
        const policyReadListStaging = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            // FIXME: Allow Deployer to delete from Staging bucket
            actions: ['s3:DeleteObject', 's3:GetObject', 's3:ListBucket'],
            resources: [`${bucketAppsStaging.bucketArn}/*`, bucketAppsStaging.bucketArn],
        });
        this._deployerFunc.addToRolePolicy(policyReadListStaging);
        // Allow the Lambda to write to the target bucket and delete
        const policyReadWriteListTarget = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:DeleteObject', 's3:GetObject', 's3:PutObject', 's3:ListBucket'],
            resources: [`${bucketApps.bucketArn}/*`, bucketApps.bucketArn],
        });
        this._deployerFunc.addToRolePolicy(policyReadWriteListTarget);
        // Allow the deployer to get a temporary STS token
        const policyGetSTSToken = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['sts:GetFederationToken'],
            resources: ['*'],
        });
        this._deployerFunc.addToRolePolicy(policyGetSTSToken);
        // Allow the deployer to assume the upload role
        const policyAssumeUpload = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['sts:AssumeRole'],
            resources: [iamRoleUpload.roleArn],
        });
        this._deployerFunc.addToRolePolicy(policyAssumeUpload);
        //
        // Give Deployer permissions to create routes and integrations
        // on the API Gateway API.
        //
        // Grant the ability to List all APIs (we have to find it)
        const policyAPIList = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['apigateway:GET'],
            resources: [`arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis`],
        });
        this._deployerFunc.addToRolePolicy(policyAPIList);
        if (httpApi) {
            // Grant full control over the API we created
            const policyAPIManage = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['apigateway:*'],
                resources: [
                    `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:${httpApi.httpApiId}/*`,
                    `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis/${httpApi.httpApiId}/integrations/*`,
                    `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis/${httpApi.httpApiId}/integrations`,
                    `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis/${httpApi.httpApiId}/routes`,
                    `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis/${httpApi.httpApiId}/routes/*`,
                ],
            });
            this._deployerFunc.addToRolePolicy(policyAPIManage);
        }
        // Grant full control over lambdas that indicate they are microapps
        const policyAPIManageLambdas = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['lambda:*'],
            resources: [
                `arn:aws:lambda:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:function:*`,
                `arn:aws:lambda:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:function:*:*`,
            ],
            conditions: {
                StringEqualsIfExists: { 'aws:ResourceTag/microapp-managed': 'true' },
            },
        });
        this._deployerFunc.addToRolePolicy(policyAPIManageLambdas);
        if (httpApi) {
            //
            // Router Lambda Function
            //
            // Create Router Lambda Function
            const routerFuncProps = {
                functionName: assetNameRoot ? `${assetNameRoot}-router${assetNameSuffix}` : undefined,
                memorySize: 1769,
                logRetention: logs.RetentionDays.ONE_MONTH,
                runtime: lambda.Runtime.NODEJS_16_X,
                timeout: aws_cdk_lib_1.Duration.seconds(15),
                environment: {
                    NODE_ENV: appEnv,
                    DATABASE_TABLE_NAME: this._table.tableName,
                    AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                    ROOT_PATH_PREFIX: rootPathPrefix,
                },
            };
            if (process.env.NODE_ENV === 'test' &&
                fs_1.existsSync(path.join(__dirname, '..', '..', 'microapps-router', 'dist', 'index.js'))) {
                // This is for local dev
                this._routerFunc = new lambda.Function(this, 'router-func', {
                    code: lambda.Code.fromAsset(path.join(__dirname, '..', '..', 'microapps-router', 'dist')),
                    handler: 'index.handler',
                    ...routerFuncProps,
                });
            }
            else if (fs_1.existsSync(path.join(__dirname, 'microapps-router', 'index.js'))) {
                // This is for built apps packaged with the CDK construct
                this._routerFunc = new lambda.Function(this, 'router-func', {
                    code: lambda.Code.fromAsset(path.join(__dirname, 'microapps-router')),
                    handler: 'index.handler',
                    ...routerFuncProps,
                });
            }
            else {
                // Create Router Lambda Layer
                const routerDataFiles = new lambda.LayerVersion(this, 'router-templates', {
                    code: lambda.Code.fromAsset(path.join(__dirname, '..', '..', 'microapps-router', 'templates')),
                    removalPolicy,
                });
                this._routerFunc = new lambdaNodejs.NodejsFunction(this, 'router-func', {
                    entry: path.join(__dirname, '..', '..', 'microapps-router', 'src', 'index.ts'),
                    handler: 'handler',
                    bundling: {
                        minify: true,
                        sourceMap: true,
                    },
                    layers: [routerDataFiles],
                    ...routerFuncProps,
                });
            }
            if (removalPolicy !== undefined) {
                this._routerFunc.applyRemovalPolicy(removalPolicy);
            }
            const policyReadTarget = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['s3:GetObject'],
                resources: [`${bucketApps.bucketArn}/*`],
            });
            for (const router of [this._routerFunc]) {
                router.addToRolePolicy(policyReadTarget);
                // Give the Router access to DynamoDB table
                this._table.grantReadData(router);
                this._table.grant(router, 'dynamodb:DescribeTable');
            }
            // Create alias for Router
            const routerAlias = this._routerFunc.addAlias('CurrentVersion');
            // This creates an integration and a router
            const route = new apigwy.HttpRoute(this, 'route-default', {
                httpApi,
                routeKey: apigwy.HttpRouteKey.DEFAULT,
                integration: new apigwyint.HttpLambdaIntegration('router-integration', routerAlias),
                authorizer: requireIAMAuthorization ? new apigwyAuth.HttpIamAuthorizer() : undefined,
            });
            let routeArn = route.routeArn;
            // Remove the trailing `/` on the ARN, which is not correct
            if (routeArn.endsWith('/')) {
                routeArn = routeArn.slice(0, routeArn.length - 1);
            }
            // Grant API Gateway permission to invoke the Lambda
            new lambda.CfnPermission(this, 'router-invoke', {
                action: 'lambda:InvokeFunction',
                functionName: this._routerFunc.functionName,
                principal: 'apigateway.amazonaws.com',
                sourceArn: routeArn,
            });
        }
    }
    get table() {
        return this._table;
    }
    get deployerFunc() {
        return this._deployerFunc;
    }
    get routerFunc() {
        return this._routerFunc;
    }
}
exports.MicroAppsSvcs = MicroAppsSvcs;
_a = JSII_RTTI_SYMBOL_1;
MicroAppsSvcs[_a] = { fqn: "@pwrdrvr/microapps-cdk.MicroAppsSvcs", version: "0.4.0-alpha.1" };
//# sourceMappingURL=data:application/json;base64,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