"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MicroAppsEdgeToOrigin = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/* eslint-disable @typescript-eslint/indent */
const crypto = require("crypto");
const fs_1 = require("fs");
const os = require("os");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cf = require("aws-cdk-lib/aws-cloudfront");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const lambdaNodejs = require("aws-cdk-lib/aws-lambda-nodejs");
const logs = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
class MicroAppsEdgeToOriginRoleStack extends aws_cdk_lib_1.Stack {
    constructor(scope, id, props) {
        var _b;
        super(scope, id, props);
        const { assetNameRoot, assetNameSuffix, setupApiGatewayPermissions, allowedFunctionUrlAccounts, } = props;
        // Create IAM Role for the Edge Function
        this._role = new iam.Role(this, 'edge-role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            roleName: assetNameRoot
                ? `${assetNameRoot}-edge-role${assetNameSuffix}`
                : aws_cdk_lib_1.PhysicalName.GENERATE_IF_NEEDED,
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
            inlinePolicies: {
                edgeToOriginPolicy: new iam.PolicyDocument({
                    statements: [
                        // This can't have a reference to the httpApi because it would mean
                        // the parent stack (this stack) has to be created before the us-east-1
                        // child stack for the Edge Lambda Function.
                        // That's why we use a tag-based policy to allow the Edge Function
                        // to invoke any API Gateway API that we apply a tag to
                        // We allow the edge function to sign for all regions since
                        // we may use custom closest region in the future.
                        ...(setupApiGatewayPermissions
                            ? [
                                new iam.PolicyStatement({
                                    actions: ['execute-api:Invoke'],
                                    resources: [`arn:aws:execute-api:*:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:*/*/*/*`],
                                }),
                            ]
                            : []),
                        //
                        // Grant permission to invoke tagged Function URLs (in same account)
                        //
                        new iam.PolicyStatement({
                            actions: ['lambda:InvokeFunctionUrl'],
                            resources: [`arn:aws:lambda:*:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:*`],
                            conditions: {
                                StringEquals: { 'aws:ResourceTag/microapp-managed': 'true' },
                            },
                        }),
                        //
                        // Grant permission to invoke Function URLs in listed accounts
                        //
                        ...(allowedFunctionUrlAccounts && allowedFunctionUrlAccounts.length > 0
                            ? [
                                new iam.PolicyStatement({
                                    actions: ['lambda:InvokeFunctionUrl'],
                                    resources: allowedFunctionUrlAccounts.map((accountId) => `arn:aws:lambda:*:${accountId}:*`),
                                }),
                            ]
                            : []),
                    ],
                }),
            },
        });
        (_b = this._role.assumeRolePolicy) === null || _b === void 0 ? void 0 : _b.addStatements(new iam.PolicyStatement({
            principals: [
                new iam.ServicePrincipal('edgelambda.amazonaws.com'),
                new iam.ServicePrincipal('lambda.amazonaws.com'),
            ],
            actions: ['sts:AssumeRole'],
            effect: iam.Effect.ALLOW,
        }));
        new aws_cdk_lib_1.CfnOutput(this, 'edge-to-origin-role-arn', {
            value: `${this.role.roleArn}`,
            exportName: `${this.stackName}-role-arn`,
        });
    }
    get role() {
        return this._role;
    }
}
/**
 * Create a new MicroApps Edge to Origin Function w/ `config.yml`
 */
class MicroAppsEdgeToOrigin extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props === undefined) {
            throw new Error('props must be set');
        }
        const { addXForwardedHostHeader = true, allowedFunctionUrlAccounts = [], assetNameRoot, assetNameSuffix, originRegion, setupApiGatewayPermissions = false, signingMode = 'sign', removalPolicy, rootPathPrefix, replaceHostHeader = true, tableRulesArn, } = props;
        // Create the edge function config file from the construct options
        const edgeToOriginConfigYaml = MicroAppsEdgeToOrigin.generateEdgeToOriginConfig({
            originRegion: originRegion || aws_cdk_lib_1.Aws.REGION,
            addXForwardedHostHeader,
            replaceHostHeader,
            signingMode: signingMode === 'none' ? '' : signingMode,
            rootPathPrefix,
            ...(tableRulesArn
                ? {
                    tableName: tableRulesArn,
                }
                : {}),
        });
        const roleStack = new MicroAppsEdgeToOriginRoleStack(this, 'role-stack', {
            stackName: `${aws_cdk_lib_1.Stack.of(this).stackName}-edge-role`,
            assetNameRoot,
            assetNameSuffix,
            allowedFunctionUrlAccounts,
            setupApiGatewayPermissions,
            env: {
                region: aws_cdk_lib_1.Stack.of(this).region,
                account: aws_cdk_lib_1.Stack.of(this).account,
            },
        });
        aws_cdk_lib_1.Stack.of(this).addDependency(roleStack);
        this._edgeToOriginRole = roleStack.role;
        new aws_cdk_lib_1.CfnOutput(this, 'role-stack-name', {
            value: `${roleStack.stackName}`,
            exportName: `${aws_cdk_lib_1.Stack.of(this).stackName}-role-stack`,
        });
        //
        // Create the Edge to Origin Function
        //
        const edgeToOriginFuncProps = {
            functionName: assetNameRoot ? `${assetNameRoot}-edge-to-origin${assetNameSuffix}` : undefined,
            role: this._edgeToOriginRole,
            memorySize: 1769,
            logRetention: logs.RetentionDays.ONE_MONTH,
            runtime: lambda.Runtime.NODEJS_16_X,
            timeout: aws_cdk_lib_1.Duration.seconds(5),
            ...(removalPolicy ? { removalPolicy } : {}),
        };
        const rootDistPath = path.join(__dirname, '..', '..', 'microapps-edge-to-origin', 'dist');
        const rootDistExists = fs_1.existsSync(path.join(rootDistPath, 'index.js'));
        const localDistPath = path.join(__dirname, 'microapps-edge-to-origin');
        const localDistExists = fs_1.existsSync(path.join(localDistPath, 'index.js'));
        if (process.env.NODE_ENV === 'test' && rootDistExists) {
            // This is for tests run under jest - Prefer root dist bundle
            // This is also for anytime when the edge function has already been bundled
            this._edgeToOriginFunction = this.createEdgeFunction(rootDistPath, edgeToOriginConfigYaml, edgeToOriginFuncProps);
        }
        else if (localDistExists) {
            // Prefer local dist above root dist if both exist (when building for distribution)
            this._edgeToOriginFunction = this.createEdgeFunction(localDistPath, edgeToOriginConfigYaml, edgeToOriginFuncProps);
        }
        else if (rootDistExists) {
            // Use local dist if it exists (when deploying from CDK in this repo)
            this._edgeToOriginFunction = this.createEdgeFunction(rootDistPath, edgeToOriginConfigYaml, edgeToOriginFuncProps);
        }
        else {
            // This is used when bundling the app and building the CDK module
            // for distribution.
            fs_1.writeFileSync(path.join(__dirname, '..', '..', 'microapps-edge-to-origin', 'config.yml'), edgeToOriginConfigYaml);
            // Copy the appFrame.html to the place where the bundling will find it
            fs_1.copyFileSync(path.join(__dirname, '..', '..', 'microapps-router', 'appFrame.html'), path.join(__dirname, '..', '..', 'microapps-edge-to-origin', 'appFrame.html'));
            // This builds the function for distribution with the CDK Construct
            // and will be used during local builds and PR builds of microapps-core
            // if the microapps-edge-to-origin function is not already bundled.
            // This will fail to deploy in any region other than us-east-1
            this._edgeToOriginFunction = new lambdaNodejs.NodejsFunction(this, 'edge-to-apigwy-func', {
                entry: path.join(__dirname, '..', '..', 'microapps-edge-to-origin', 'src', 'index.ts'),
                handler: 'handler',
                bundling: {
                    minify: true,
                    sourceMap: true,
                    commandHooks: {
                        beforeInstall: () => [],
                        beforeBundling: () => [],
                        afterBundling: (_inputDir, outputDir) => {
                            // 2022-10-02 - Note that this is ignoring the generated config
                            // file above and including the default template config file
                            return [
                                `${os.platform() === 'win32' ? 'copy' : 'cp'} ${path.join(__dirname, '..', '..', '..', 'configs', 'microapps-edge-to-origin', 'config.yml')} ${outputDir}`,
                                `${os.platform() === 'win32' ? 'copy' : 'cp'} ${path.join(__dirname, '..', '..', 'microapps-router', 'appFrame.html')} ${outputDir}`,
                            ];
                        },
                    },
                },
                ...edgeToOriginFuncProps,
            });
        }
        this._edgeToOriginLambdas = [
            {
                eventType: cf.LambdaEdgeEventType.ORIGIN_REQUEST,
                functionVersion: this._edgeToOriginFunction.currentVersion,
                includeBody: true,
            },
        ];
        // Grant access to the rules table
        if (tableRulesArn) {
            const tableRules = dynamodb.Table.fromTableName(this, 'tableRules', tableRulesArn);
            tableRules.grantReadData(this._edgeToOriginFunction);
        }
        this._edgeToOriginFunction.stack.stackName;
        new aws_cdk_lib_1.CfnOutput(this, 'edge-stack-name', {
            value: `${this._edgeToOriginFunction.stack.stackName}`,
            exportName: `${aws_cdk_lib_1.Stack.of(this).stackName}-edge-stack`,
        });
    }
    /**
     * Generate the yaml config for the edge lambda
     * @param props
     * @returns
     */
    static generateEdgeToOriginConfig(props) {
        return `originRegion: ${props.originRegion}
${props.signingMode === '' ? '' : `signingMode: ${props.signingMode}`}
addXForwardedHostHeader: ${props.addXForwardedHostHeader}
replaceHostHeader: ${props.replaceHostHeader}
${props.tableName ? `tableName: '${props.tableName}'` : ''}
${props.rootPathPrefix ? `rootPathPrefix: '${props.rootPathPrefix}'` : ''}`;
    }
    get edgeToOriginFunction() {
        return this._edgeToOriginFunction;
    }
    get edgeToOriginLambdas() {
        return this._edgeToOriginLambdas;
    }
    get edgeToOriginRole() {
        return this._edgeToOriginRole;
    }
    /**
     * Hash the stack name to make the EdgeFunction parameter name unique
     *
     * @param stack
     * @returns
     */
    hashStackName() {
        return crypto.createHash('sha1').update(aws_cdk_lib_1.Stack.of(this).stackName).digest('hex').substring(0, 8);
    }
    createEdgeFunction(distPath, edgeToOriginConfigYaml, edgeToOriginFuncProps) {
        var _b;
        fs_1.writeFileSync(path.join(distPath, 'config.yml'), edgeToOriginConfigYaml);
        // Skip the copy of appFrame.html on deployed modules
        if (!__dirname.includes('node_modules')) {
            fs_1.copyFileSync(path.join(__dirname, '..', '..', 'microapps-router', 'appFrame.html'), path.join(distPath, 'appFrame.html'));
        }
        // The exclude varying per stack name is a kludge to get the asset bundled
        // with the stack-specifc config.yml file, otherwise they all get the file
        // generated for the first instance of the construct within any stack
        // in the app.
        const code = lambda.Code.fromAsset(distPath, { exclude: [`**/${aws_cdk_lib_1.Stack.of(this)}`] });
        const stackHash = (_b = this.hashStackName()) !== null && _b !== void 0 ? _b : '';
        // EdgeFunction has a bug where it will generate the same parameter
        // name across multiple stacks in the same region if the id param is constant
        const edge = new cf.experimental.EdgeFunction(this, `edge-to-apigwy-func-${stackHash}`, {
            stackId: `microapps-edge-to-origin-${stackHash}`,
            code,
            functionName: `microapps-edge-to-origin-${stackHash}`,
            handler: 'index.handler',
            ...edgeToOriginFuncProps,
        });
        aws_cdk_lib_1.Tags.of(edge).add('Name', aws_cdk_lib_1.Stack.of(this).stackName);
        return edge;
    }
}
exports.MicroAppsEdgeToOrigin = MicroAppsEdgeToOrigin;
_a = JSII_RTTI_SYMBOL_1;
MicroAppsEdgeToOrigin[_a] = { fqn: "@pwrdrvr/microapps-cdk.MicroAppsEdgeToOrigin", version: "0.4.0-alpha.1" };
//# sourceMappingURL=data:application/json;base64,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