# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['request_token',
 'request_token.management',
 'request_token.management.commands',
 'request_token.migrations',
 'request_token.templatetags']

package_data = \
{'': ['*']}

install_requires = \
['django>=3.2,<5.0', 'pyjwt>=2.0,<3.0']

setup_kwargs = {
    'name': 'django-request-token',
    'version': '2.1',
    'description': 'JWT-backed Django app for managing querystring tokens.',
    'long_description': '## Supported versions\n\nThis project supports Django 3.2+ and Python 3.8+. The latest version\nsupported is Django 4.1 running on Python 3.11.\n\n## Django Request Token\n\nDjango app that uses JWT to manage one-time and expiring tokens to\nprotect URLs.\n\nThis app currently requires the use of PostgreSQL.\n\n### Background\n\nThis project was borne out of our experiences at YunoJuno with \'expiring\nlinks\' - which is a common use case of providing users with a URL that\nperforms a single action, and may bypass standard authentication. A\nwell-known use of this is the ubiquitous \'unsubscribe\' link you find\nat the bottom of newsletters. You click on the link and it immediately\nunsubscribes you, irrespective of whether you are already authenticated\nor not.\n\nIf you google "temporary url", "one-time link" or something similar you\nwill find lots of StackOverflow articles on supporting this in Django -\nit\'s pretty obvious, you have a dedicated token url, and you store the\ntokens in a model - when they are used you expire the token, and it\ncan\'t be used again. This works well, but it falls down in a number of\nareas:\n\n* Hard to support multiple endpoints (views)\n\nIf you want to support the same functionality (expiring links) for more\nthan one view in your project, you either need to have multiple models\nand token handlers, or you need to store the specific view function and\nargs in the model; neither of these is ideal.\n\n* Hard to debug\n\nIf you use have a single token url view that proxies view functions, you\nneed to store the function name, args and it then becomes hard to\nsupport - when someone claims that they clicked on\n`example.com/t/<token>`, you can\'t tell what that would resolve to\nwithout looking it up in the database - which doesn\'t work for customer\nsupport.\n\n* Hard to support multiple scenarios\n\nSome links expire, others have usage quotas - some have both. Links may\nbe for use by a single user, or multiple users.\n\nThis project is intended to provide an easy-to-support mechanism for\n\'tokenising\' URLs without having to proxy view functions - you can build\nwell-formed Django URLs and views, and then add request token support\nafterwards.\n\n### Use Cases\n\nThis project supports three core use cases, each of which is modelled\nusing the `login_mode` attribute of a request token:\n\n1. Public link with payload\n2. ~~Single authenticated request~~ (DEPRECATED: use `django-visitor-pass`)\n3. ~~Auto-login~~ (DEPRECATED: use `django-magic-link`)\n\n**Public Link** (`RequestToken.LOGIN_MODE_NONE`)\n\nIn this mode (the default for a new token), there is no authentication,\nand no assigned user. The token is used as a mechanism for attaching a\npayload to the link. An example of this might be a custom registration\nor affiliate link, that renders the standard template with additional\ninformation extracted from the token - e.g. the name of the affiliate,\nor the person who invited you to register.\n\n```python\n# a token that can be used to access a public url, without authenticating\n# as a user, but carrying a payload (affiliate_id).\ntoken = RequestToken.objects.create_token(\n    scope="foo",\n    login_mode=RequestToken.LOGIN_MODE_NONE,\n    data={\n        \'affiliate_id\': 1\n    }\n)\n\n...\n\n@use_request_token(scope="foo")\nfunction view_func(request):\n    # extract the affiliate id from an token _if_ one is supplied\n    affiliate_id = (\n        request.token.data[\'affiliate_id\']\n        if hasattr(request, \'token\')\n        else None\n    )\n```\n\n**Single Request** (`RequestToken.LOGIN_MODE_REQUEST`)\n\nIn Request mode, the request.user property is overridden by the user\nspecified in the token, but only for a single request. This is useful\nfor responding to a single action (e.g. RSVP, unsubscribe). If the user\nthen navigates onto another page on the site, they will not be\nauthenticated. If the user is already authenticated, but as a different\nuser to the one in the token, then they will receive a 403 response.\n\n```python\n# this token will identify the request.user as a given user, but only for\n# a single request - not the entire session.\ntoken = RequestToken.objects.create_token(\n    scope="foo",\n    login_mode=RequestToken.LOGIN_MODE_REQUEST,\n    user=User.objects.get(username="hugo")\n)\n\n...\n\n@use_request_token(scope="foo")\nfunction view_func(request):\n    assert request.user == User.objects.get(username="hugo")\n```\n**Auto-login** (`RequestToken.LOGIN_MODE_SESSION`)\n\nThis is the nuclear option, and must be treated with extreme care. Using\na Session token will automatically log the user in for an entire\nsession, giving the user who clicks on the link full access the token\nuser\'s account. This is useful for automatic logins. A good example of\nthis is the email login process on medium.com, which takes an email\naddress (no password) and sends out a login link.\n\nSession tokens have a default expiry of ten minutes.\n\n```python\n# this token will log in as the given user for the entire session -\n# NB use with caution.\ntoken = RequestToken.objects.create_token(\n    scope="foo",\n    login_mode=RequestToken.LOGIN_MODE_SESSION,\n    user=User.objects.get(username="hugo")\n)\n```\n\n### Implementation\n\nThe project contains middleware and a view function decorator that\ntogether validate request tokens added to site URLs.\n\n**request_token.models.RequestToken** - stores the token details\n\nStep 1 is to create a `RequestToken` - this has various attributes that\ncan be used to modify its behaviour, and mandatory property - `scope`.\nThis is a text value - it can be anything you like - it is used by the\nfunction decorator (described below) to confirm that the token given\nmatches the function being called - i.e. the `token.scope` must match\nthe function decorator scope kwarg:\n\n```python\ntoken = RequestToken(scope="foo")\n\n# this will raise a 403 without even calling the function\n@use_request_token(scope="bar")\ndef incorrect_scope(request):\n    pass\n\n# this will call the function as expected\n@use_request_token(scope="foo")\ndef correct_scope(request):\n    pass\n```\n\nThe token itself - the value that must be appended to links as a\nquerystring argument - is a JWT - and comes from the\n`RequestToken.jwt()` method. For example, if you were sending out an\nemail, you might render the email as an HTML template like this:\n\n```html\n{% if token %}\n    <a href="{{url}}?rt={{token.jwt}}>click here</a>\n{% else %}\n    <a href="{{url}}">click here</a>\n{% endif %}\n```\n\nIf you haven\'t come across JWT before you can find out more on the\n[jwt.io](https://jwt.io/) website. The token produced will include the\nfollowing JWT claims (available as the property `RequestToken.claims`:\n\n* `max`: maximum times the token can be used\n* `sub`: the scope\n* `mod`: the login mode\n* `jti`: the token id\n* `aud`: (optional) the user the token represents\n* `exp`: (optional) the expiration time of the token\n* `iat`: (optional) the time the token was issued\n* `ndf`: (optional) the not-before-time of the token\n\n**request_token.models.RequestTokenLog** - stores usage data for tokens\n\nEach time a token is used successfully, a log object is written to the\ndatabase. This provided an audit log of the usage, and it stores client\nIP address and user agent, so can be used to debug issues. This can be\ndisabled using the `REQUEST_TOKEN_DISABLE_LOGS` setting. The logs table\ncan be maintained using the management command as described below.\n\n**request_token.middleware.RequestTokenMiddleware** - decodes and verifies tokens\n\nThe `RequestTokenMiddleware` will look for a querystring token value\n(the argument name defaults to \'rt\' and can overridden using the\n`JWT_QUERYSTRING_ARG` setting), and if it finds one it will verify the\ntoken (using the JWT decode verification). If the token is verified, it\nwill fetch the token object from the database and perform additional\nvalidation against the token attributes. If the token checks out it is\nadded to the incoming request as a `token` attribute. This way you can\nadd arbitrary data (stored on the token) to incoming requests.\n\nIf the token has a user specified, then the `request.user` is updated to\nreflect this. The middleware must run after the Django auth middleware,\nand before any custom middleware that inspects / monkey-patches the\n`request.user`.\n\nIf the token cannot be verified it returns a 403.\n\n**request_token.decorators.use_request_token** - applies token\npermissions to views\n\nA function decorator that takes one mandatory kwargs (`scope`) and one\noptional kwargs (`required`). The `scope` is used to match tokens to\nview functions - it\'s just a straight text match - the value can be\nanything you like, but if the token scope is \'foo\', then the\ncorresponding view function decorator scope must match. The `required`\nkwarg is used to indicate whether the view **must** have a token in\norder to be used, or not. This defaults to False - if a token **is**\nprovided, then it will be validated, if not, the view function is called\nas is.\n\nIf the scopes do not match then a 403 is returned.\n\nIf required is True and no token is provided the a 403 is returned.\n\n### Installation\n\nDownload / install the app using pip:\n\n```shell\npip install django-request-token\n```\n\nAdd the app `request_token` to your `INSTALLED_APPS` Django setting:\n\n```python\n# settings.py\nINSTALLED_APPS = (\n    \'django.contrib.admin\',\n    \'django.contrib.auth\',\n    \'django.contrib.contenttypes\',\n    \'django.contrib.sessions\',\n    \'django.contrib.messages\',\n    \'django.contrib.staticfiles\',\n    \'request_token\',\n    ...\n)\n```\n\nAdd the middleware to your settings, **after** the standard\nauthentication middleware, and before any custom middleware that uses\nthe `request.user`.\n\n```python\nMIDDLEWARE_CLASSES = [\n    # default django middleware\n    \'django.contrib.sessions.middleware.SessionMiddleware\',\n    \'django.middleware.common.CommonMiddleware\',\n    \'django.middleware.csrf.CsrfViewMiddleware\',\n    \'django.contrib.auth.middleware.AuthenticationMiddleware\',\n    \'django.contrib.messages.middleware.MessageMiddleware\',\n    \'request_token.middleware.RequestTokenMiddleware\',\n]\n```\n\nYou can now add `RequestToken` objects, either via the shell (or within\nyour app) or through the admin interface. Once you have added a\n`RequestToken` you can add the token JWT to your URLs (using the `jwt()`\nmethod):\n\n```python\n>>> token = RequestToken.objects.create_token(scope="foo")\n>>> url = "https://example.com/foo?rt=" + token.jwt()\n```\n\nYou now have a request token enabled URL. You can use this token to\nprotect a view function using the view decorator:\n\n```python\n@use_request_token(scope="foo")\nfunction foo(request):\n    pass\n```\n\nNB The \'scope\' argument to the decorator is used to bind the function to\nthe incoming token - if someone tries to use a valid token on another\nURL, this will return a 403.\n\n**NB this currently supports only view functions - not class-based views.**\n\n### Management commands\n\nThere is a single management command, `truncate_request_token_log` which can\nbe used to manage the size of the log table (each token usage is logged to\nthe database). It supports two arguments - `--max-count` and `--max-days` which\nare self-explanatory:\n\n```\n$ python manage.py truncate_request_token_log --max-count=100\nTruncating request token log records:\n-> Retaining last 100 request token log records\n-> Truncating request token log records from 2021-08-01 00:00:00\n-> Truncating 0 request token log records.\n$\n```\n\n### Settings\n\n* `REQUEST_TOKEN_QUERYSTRING`\n\nThe querystring argument name used to extract the token from incoming\nrequests, defaults to **rt**.\n\n* `REQUEST_TOKEN_EXPIRY`\n\nSession tokens have a default expiry interval, specified in minutes. The\nprimary use case (above) dictates that the expiry should be no longer\nthan it takes to receive and open an email, defaults to **10**\n(minutes).\n\n* `REQUEST_TOKEN_403_TEMPLATE`\n\nSpecifying the 403-template so that for prettyfying the 403-response,\nin production with a setting like:\n\n```python\nFOUR03_TEMPLATE = os.path.join(BASE_DIR,\'...\',\'403.html\')\n```\n\n* `REQUEST_TOKEN_DISABLE_LOGS`\n\nSet to `True` to disable the creation of `RequestTokenLog` objects on\neach use of a token. This is not recommended in production, as the\nauditing of token use is a valuable part of the library.\n\n### Tests\n\nThere is a set of `tox` tests.\n\n### License\n\nMIT\n\n### Contributing\n\nThis is by no means complete, however, it\'s good enough to be of value, hence releasing it.\nIf you would like to contribute to the project, usual Github rules apply:\n\n1. Fork the repo to your own account\n2. Submit a pull request\n3. Add tests for any new code\n4. Follow coding style of existing project\n\n### Acknowledgements\n\n@jpadilla for [PyJWT](https://github.com/jpadilla/pyjwt/)\n',
    'author': 'YunoJuno',
    'author_email': 'code@yunojuno.com',
    'maintainer': 'YunoJuno',
    'maintainer_email': 'code@yunojuno.com',
    'url': 'https://github.com/yunojuno/django-request-token',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
