# This code is part of Qiskit.
#
# (C) Copyright IBM 2020, 2021.
#
# This code is licensed under the Apache License, Version 2.0. You may
# obtain a copy of this license in the LICENSE.txt file in the root directory
# of this source tree or at http://www.apache.org/licenses/LICENSE-2.0.
#
# Any modifications or derivative works of this code must retain this
# copyright notice, and modified files need to carry a notice indicating
# that they have been altered from the originals.

"""
Code inside the test is the optimization sample from the readme.
If this test fails and code changes are needed here to resolve
the issue then ensure changes are made to readme too.
"""

import unittest
from test import QiskitOptimizationTestCase
from qiskit.utils import algorithm_globals

# pylint: disable=import-outside-toplevel,redefined-builtin


class TestReadmeSample(QiskitOptimizationTestCase):
    """Test sample code from readme"""

    def _sample_code(self):
        def print(*args):
            """overloads print to log values"""
            if args:
                self.log.debug(args[0], *args[1:])

        # --- Exact copy of sample code ----------------------------------------

        import networkx as nx
        import numpy as np

        from qiskit_optimization import QuadraticProgram
        from qiskit_optimization.algorithms import MinimumEigenOptimizer

        from qiskit import BasicAer
        from qiskit.algorithms import QAOA
        from qiskit.algorithms.optimizers import SPSA

        # Generate a graph of 4 nodes
        n = 4
        graph = nx.Graph()
        graph.add_nodes_from(np.arange(0, n, 1))
        elist = [(0, 1, 1.0), (0, 2, 1.0), (0, 3, 1.0), (1, 2, 1.0), (2, 3, 1.0)]
        graph.add_weighted_edges_from(elist)

        # Compute the weight matrix from the graph
        w = nx.adjacency_matrix(graph)

        # Formulate the problem as quadratic program
        problem = QuadraticProgram()
        _ = [problem.binary_var("x{}".format(i)) for i in range(n)]  # create n binary variables
        linear = w.dot(np.ones(n))
        quadratic = -w
        problem.maximize(linear=linear, quadratic=quadratic)

        # Fix node 0 to be 1 to break the symmetry of the max-cut solution
        problem.linear_constraint([1, 0, 0, 0], "==", 1)

        # Run quantum algorithm QAOA on qasm simulator
        spsa = SPSA(maxiter=250)
        backend = BasicAer.get_backend("qasm_simulator")
        qaoa = QAOA(optimizer=spsa, reps=5, quantum_instance=backend)
        algorithm = MinimumEigenOptimizer(qaoa)
        result = algorithm.solve(problem)
        print(result)  # prints solution, x=[1, 0, 1, 0], the cost, fval=4
        # ----------------------------------------------------------------------

        return result

    def test_readme_sample(self):
        """readme sample test"""

        print("")
        import numpy as np

        # for now do this until test is fixed
        msg = None
        for idx in range(3):
            try:
                print(f"Trial number {idx+1}")
                # Fix the random seed of SPSA (Optional)
                algorithm_globals.random_seed = 123
                result = self._sample_code()
                np.testing.assert_array_almost_equal(result.x, [1, 0, 1, 0])
                self.assertAlmostEqual(result.fval, 4.0)
                msg = None
                break
            except Exception as ex:  # pylint: disable=broad-except
                msg = str(ex)

        if msg is not None:
            self.skipTest(msg)


if __name__ == "__main__":
    unittest.main()
