"use strict";
// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Deserialize and return the unpacked message.
 *
 * #### Notes
 * Handles JSON blob strings and binary messages.
 */
function deserialize(data) {
    let value;
    if (typeof data === 'string') {
        value = JSON.parse(data);
    }
    else {
        value = deserializeBinary(data);
    }
    return value;
}
exports.deserialize = deserialize;
/**
 * Serialize a kernel message for transport.
 *
 * #### Notes
 * If there is binary content, an `ArrayBuffer` is returned,
 * otherwise the message is converted to a JSON string.
 */
function serialize(msg) {
    let value;
    if (msg.buffers && msg.buffers.length) {
        value = serializeBinary(msg);
    }
    else {
        value = JSON.stringify(msg);
    }
    return value;
}
exports.serialize = serialize;
/**
 * Deserialize a binary message to a Kernel Message.
 */
function deserializeBinary(buf) {
    let data = new DataView(buf);
    // read the header: 1 + nbufs 32b integers
    let nbufs = data.getUint32(0);
    let offsets = [];
    if (nbufs < 2) {
        throw new Error('Invalid incoming Kernel Message');
    }
    for (let i = 1; i <= nbufs; i++) {
        offsets.push(data.getUint32(i * 4));
    }
    let jsonBytes = new Uint8Array(buf.slice(offsets[0], offsets[1]));
    let msg = JSON.parse(new TextDecoder('utf8').decode(jsonBytes));
    // the remaining chunks are stored as DataViews in msg.buffers
    msg.buffers = [];
    for (let i = 1; i < nbufs; i++) {
        let start = offsets[i];
        let stop = offsets[i + 1] || buf.byteLength;
        msg.buffers.push(new DataView(buf.slice(start, stop)));
    }
    return msg;
}
/**
 * Implement the binary serialization protocol.
 *
 * Serialize Kernel message to ArrayBuffer.
 */
function serializeBinary(msg) {
    let offsets = [];
    let buffers = [];
    let encoder = new TextEncoder();
    let origBuffers = [];
    if (msg.buffers !== undefined) {
        origBuffers = msg.buffers;
        delete msg['buffers'];
    }
    let jsonUtf8 = encoder.encode(JSON.stringify(msg));
    buffers.push(jsonUtf8.buffer);
    for (let i = 0; i < origBuffers.length; i++) {
        // msg.buffers elements could be either views or ArrayBuffers
        // buffers elements are ArrayBuffers
        let b = origBuffers[i];
        buffers.push(b instanceof ArrayBuffer ? b : b.buffer);
    }
    let nbufs = buffers.length;
    offsets.push(4 * (nbufs + 1));
    for (let i = 0; i + 1 < buffers.length; i++) {
        offsets.push(offsets[offsets.length - 1] + buffers[i].byteLength);
    }
    let msgBuf = new Uint8Array(offsets[offsets.length - 1] + buffers[buffers.length - 1].byteLength);
    // use DataView.setUint32 for network byte-order
    let view = new DataView(msgBuf.buffer);
    // write nbufs to first 4 bytes
    view.setUint32(0, nbufs);
    // write offsets to next 4 * nbufs bytes
    for (let i = 0; i < offsets.length; i++) {
        view.setUint32(4 * (i + 1), offsets[i]);
    }
    // write all the buffers at their respective offsets
    for (let i = 0; i < buffers.length; i++) {
        msgBuf.set(new Uint8Array(buffers[i]), offsets[i]);
    }
    return msgBuf.buffer;
}
//# sourceMappingURL=serialize.js.map