"""
WassersteinGAN
--------------
Implements the Wasserstein GAN[1].

Uses the Wasserstein loss to determine the realness of real and fake images.
The Wasserstein loss has several theoretical advantages over the Jensen-Shanon divergence
minimised by the original GAN. In this architecture the critic (discriminator) is often
trained multiple times for every generator step.
Lipschitz continuity is "enforced" by weight clipping.

Losses:
    - Generator: Wasserstein
    - Critic: Wasserstein
Default optimizer:
    - torch.optim.RMSprop
Custom parameter:
    - clip_val: Clip value for the critic to maintain Lipschitz continuity.

References
----------
.. [1] https://export.arxiv.org/pdf/1701.07875
"""

import torch

import numpy as np

from vegans.models.unconditional.AbstractGAN1v1 import AbstractGAN1v1
from vegans.utils.utils import WassersteinLoss


class WassersteinGAN(AbstractGAN1v1):
    """
    WassersteinGAN
    --------------
    Implements the Wasserstein GAN[1].

    Uses the Wasserstein loss to determine the realness of real and fake images.
    The Wasserstein loss has several theoretical advantages over the Jensen-Shanon divergence
    minimised by the original GAN. In this architecture the critic (discriminator) is often
    trained multiple times for every generator step.
    Lipschitz continuity is "enforced" by weight clipping.

    Losses:
        - Generator: Wasserstein
        - Critic: Wasserstein
    Default optimizer:
        - torch.optim.RMSprop
    Custom parameter:
        - clip_val: Clip value for the critic to maintain Lipschitz continuity.

    References
    ----------
    .. [1] https://export.arxiv.org/pdf/1701.07875

    Parameters
    ----------
    generator: nn.Module
        Generator architecture. Produces output in the real space.
    adversary: nn.Module
        Adversary architecture. Produces predictions for real and fake samples to differentiate them.
    x_dim : list, tuple
        Number of the output dimensions of the generator and input dimension of the discriminator / critic.
        In the case of images this will be [nr_channels, nr_height_pixels, nr_width_pixels].
    z_dim : int, list, tuple
        Number of the latent dimensions for the generator input. Might have dimensions of an image.
    optim : dict or torch.optim
        Optimizer used for each network. Could be either an optimizer from torch.optim or a dictionary with network
        name keys and torch.optim as value, i.e. {"Generator": torch.optim.Adam}.
    optim_kwargs : dict
        Optimizer keyword arguments used for each network. Must be a dictionary with network
        name keys and dictionary with keyword arguments as value, i.e. {"Generator": {"lr": 0.0001}}.
    clip_val: float
        Clip value for the critic to maintain Lipschitz continuity.
    feature_layer : torch.nn.*
        Output layer used to compute the feature loss. Should be from either the discriminator or critic.
        If `feature_layer` is not None, the original generator loss is replaced by a feature loss, introduced
        [here](https://arxiv.org/abs/1606.03498v1).
    fixed_noise_size : int
        Number of images shown when logging. The fixed noise is used to produce the images in the folder/images
        subdirectory, the tensorboard images tab and the samples in get_training_results().
    device : string
        Device used while training the model. Either "cpu" or "cuda".
    ngpu : int
        Number of gpus used during training if device == "cuda".
    folder : string
        Creates a folder in the current working directory with this name. All relevant files like summary, images, models and
        tensorboard output are written there. Existing folders are never overwritten or deleted. If a folder with the same name
        already exists a time stamp is appended to make it unique.
    """

    #########################################################################
    # Actions before training
    #########################################################################
    def __init__(
            self,
            generator,
            adversary,
            x_dim,
            z_dim,
            optim=None,
            optim_kwargs=None,
            clip_val=0.01,
            feature_layer=None,
            fixed_noise_size=32,
            device=None,
            ngpu=None,
            folder="./WassersteinGAN",
            secure=True):

        super().__init__(
            generator=generator, adversary=adversary,
            z_dim=z_dim, x_dim=x_dim, adv_type="Critic",
            optim=optim, optim_kwargs=optim_kwargs,
            feature_layer=feature_layer,
            fixed_noise_size=fixed_noise_size,
            device=device,
            folder=folder,
            ngpu=ngpu, secure=secure
        )
        self._clip_val = clip_val
        self.hyperparameters["clip_val"] = clip_val

    def _default_optimizer(self):
        return torch.optim.RMSprop

    def _default_optimizer(self):
        return torch.optim.RMSprop

    def _define_loss(self):
        loss_functions = {"Generator": WassersteinLoss(), "Adversary": WassersteinLoss()}
        return loss_functions


    #########################################################################
    # Actions during training
    #########################################################################
    def _step(self, who=None):
        if who is not None:
            self.optimizers[who].step()
            if who == "Adversary":
                for p in self.adversary.parameters():
                    p.data.clamp_(-self._clip_val, self._clip_val)
        else:
            [optimizer.step() for _, optimizer in self.optimizers.items()]