# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['confect']

package_data = \
{'': ['*']}

extras_require = \
{'click': ['click>=2.0'], 'pendulum': ['pendulum>=2.0.0,<3.0.0']}

setup_kwargs = {
    'name': 'confect',
    'version': '0.4.2',
    'description': 'a Python configuration library loads Python configuration files',
    'long_description': '\nConfect - a Python configuration library loads Python configuration files\n**************************************************************************\n\nIntroduction\n==============\n\nWhy you need a configuration library?\n-------------------------------------\n\n\n- **For storing secrets**\n\n  You have a project that needs to access database or other services with password or some secret keys.\n  Storing secrets and passwords in your code is not smart.\n  You need a configuration file and a library for loading and using it in the runtime.\n\n- **For different runtime environments**\n\n  For example, database IP addresses and passwords in development environment normally differs from production environment.\n  You need multiple configuration files for storing those information for different environment, and load one of them in the run time.\n\n- **For better parameter management**\n\n  You\'re running some experiments, e.g. working on Machine Learning projects.\n  There\'re a bunch of parameters needs to be changed in the run time.\n  And you want to manage them in a smarter and more elegant way.\n\nHow confect differs from others?\n-------------------------------------\n\n- **Python configuration files**\n\n  This makes it possible to\n\n  + have complex type objects as configuration values, like Decimal, timedelta\n    or any class instance\n  + dynamically handle complicated logic, you can use conditional statements\n    like ``if`` in it.\n  + read other TOML/YMAL/JSON/ini files or even environment variables in the\n    configuration file.\n\n- **Loads configuration file through module importing**\n\n  Confect loads configuration file through a given file path, or through module importing.\n  It\'s easy to control the source of configuration file through ``PYTHONPATH``.\n\n- **Loads configuration file multiple times**\n\n  Sometimes we need multiple configuration files — one for project,\n  one for team and one for personal use.\n  And we want that the personal configuration file has the highest priority.\n  If there\'s a configuration setting existing in that file, it would override values\n  from other files.\n\n- **Loads configuration properties from environment variable**\n\n  This feature is convenient if you want to change a single or some properties values,\n  and don\'t want to modify the configuration file.\n\n- **Attachs command line options to some click_ command**\n\n  You can change any configuration value through command line options, if your command is created by click_.\n\n- **Better maintainability**\n\n  Confect forces users to define configuration properties and set a default value before using them.\n  And the ``conf`` object is immutable for reducing the possibility of making errors.\n\n\nInstall\n========\n\n``confect`` is a Python package hosted on PyPI and works only with Python 3.6 up.\n\nJust like other Python packages, install it by pip_ into a virtualenv_\n, or use poetry_ to manage project dependencies and virtualenv.\n\n.. code:: console\n\n   $ pip install confect\n\n\nBasics\n===========\n\nConf Object\n-----------\n\nCalling ``conf = confect.Conf()`` creates a new configuration manager object.\n\nSuppose ``projx`` is your top-level package name. Put the following lines into\n``projx/core.py`` or ``projx/confspec.py``\n\n.. code:: python\n\n   import confect\n   conf = confect.Conf()\n\nIt is possible to create multiple ``Conf`` objects, but normally it\'s not what\nyou want. In most cases, initialize and manage only one ``Conf`` object in your\napplication, then import and use it anywhere.\n\n\nConfiguration Properties Declaration\n------------------------------------\n\n**Configuration properties should be declared before using it.** This feature\nmakes your code more readable and maintainable.\n\nTwo ways to declare properties.\n\n1.  context manager:\n\n    .. code:: python\n\n      with conf.declare_group(group_name) as group_name:\n          group_name.prop1 = \'default value\'\n          group_name.prop2 = 42\n\n2. function call\n\n   .. code:: python\n\n     conf.declare_group(group_name, prop1=\'default value\', prop2=42)\n\nGroup names and property names should be valid Python variable names, which\nconsist of letters (A-Z, a-z), digits (0-9), and the underscore character (_).\nNormally, the group name is your class name, module name or subpackage name.\n\nDefault Value and Parser\n^^^^^^^^^^^^^^^^^^^^^^^^^\n\nDefault values of all properties should be defined along with the declaration.\nUse ``confect.prop(default, desc=None, prop_type=None)`` to specify details other than the\ndefault value. ``desc`` is for commentary and the help message in CLI option.\nArgument of ``prop_type`` is an instance of confect.PropertyType which is\nresponsable for CLI argument and environment variable parsing. ``prop_type`` of\npopular Python types would be infered from default value automatically.\n\nDefault values don\'t have to be a workable value (e.g. fake secret keys or\npasswords). The true workable value can be defined in the configuration file.\nHowever, even if it\'s not a workable value, the mock default values still make\nthe declaration and the code more readable and maintainable. For instance:\n\n.. code:: python\n\n   with conf.declare_group(\'aws\') as aws:\n       aws.access_key_id = \'true-access-key\'\n       aws.secret_access_key = \'fake-key-plz-set-it-in-local_conf.py\'\n\nDeclaration Example\n^^^^^^^^^^^^^^^^^^^^^\n\n.. code:: python\n\n   import confect\n   conf = confect.Conf()\n\n   # declare properties with context manager\n   with conf.declare_group(\'api\') as api:\n       # default value only. confect would infer property type automatically\n       api.cache_prefix = \'projx_cache\'\n       api.cache_expire = confect.prop(\n           default=60 * 60 * 24,\n           desc="expire time in seconds")\n\n       # add description for CLI help message and commentary\n       api.url_base_path = confect.prop(\n           default=\'api/v2/\',\n           desc=\'URL base path of API\')\n\n\n   with conf.declare_group(\'db\') as db:\n       db.host = \'127.0.0.1\'\n       db.db_name = \'projx\'\n       db.username = \'projx_admin\'\n\n       # if default value has to be None, it\'d be better to assign property\n       # type manually for parsing\n       db.password = confect.prop(\n          default=None,\n          prop_type=confect.prop_type.String(),\n          desc=\'`None` for no password\')\n\n       db.port = confect.prop(\n           default=None,\n           prop_type=confect.prop_type.Integer(),\n           desc=\'`None` for db engine default port\')\n\n   # declare properties with function call\n   conf.declare_group(\n       \'ctr_predict_model\',\n       model_pickle_s3folder=\'s3://some-bucket/path/to/folder\',\n       model_version=confect.prop(default=\'v3\')\n   )\n\n\nDeclaration Location\n^^^^^^^^^^^^^^^^^^^^^\n\nProperty declarations can be put into the module where the ``conf`` object is\nlocated. Or, you can put them into those modules where you need these\nconfigurations, like ``projx/db.py`` or ``projx/api.py``. Just make sure your\napplication import all these modules eagerly, not lazily.\n\n\nConfiguration Access\n--------------------\n\nAfter the group and properties are declared, they are accessable through\nthe ``conf`` object directly, like ``conf.group_name.prop_name``.\n\n``projx/api.py``\n\n.. code:: python\n\n   from projx.core import conf\n\n   @routes(conf.api.url_base_path + \'add\')\n   @redis_cache(key=conf.api.cache_prefix, expire=conf.api.cache_expire)\n   def add(a, b)\n       return a + b\n\n``projx/db.py``\n\n.. code:: python\n\n   from projx.core import conf\n\n   engine = create_engine(\n        f\'mysql://{conf.db.username}:{conf.db.password}\'\n        f\'@{conf.db.host}/{conf.db.db_name}\')\n\n\nAccess Errors\n^^^^^^^^^^^^^^^^^^^^^^^^^\n\nMake sure that the configuration properties are **declared before access**. If not,\nexceptions would be raised.\n\n.. code:: python\n\n   >>> conf.unknown_group.unknown_prop\n   Traceback (most recent call last):\n     ...\n   UnknownConfError: "Unknown configuration group \'unknown_group\'"\n\n.. code:: python\n\n   >>> conf.api.unknown_prop\n   Traceback (most recent call last):\n     ...\n   UnknownConfError: "Unknown \'unknown_prop\' property in configuration group \'api\'"\n\n\n**Configuration properties and groups are immutable.** They are meant to be\naltered globally by loading configuration files, environment variables or CLI\nargument.\n\n.. code:: python\n\n   >>> conf.api.cache_expire = 60 * 60 * 3\n   Traceback (most recent call last):\n     ...\n   confect.error.FrozenConfPropError: Configuration properties are frozen.\n\n\nSetting Configuration Properties\n---------------------------------\n\nConfiguration properties are immutable in the application runtime. This feature make sure\n the runtime environment is stable without unexpected behavior.\n\nThe standard ways to change the configuration properties are:\n\n1. Load from Python file ``conf.load_module(module_name)`` and ``conf.load_file(file_path)``. (Check `Loading Configuration File`_)\n2. Load from environment variable ``conf.load_envvar(prefix)``. (Check `Loading Environment Variables`_)\n3. Override by CLI options ``conf.click_options(click_command)``. (Check `Command Line Options`_)\n\nConfect still provide a hacky way to change them in the runtime, but use them wisely.\n\n1. Alter configuration in the runtime(Check `Runtime Configuration Altering`_)\n\n\nLoading Configuration File\n--------------------------\n\nConfect loads Python configuration files. That makes your configuration file\nprogrammable and unrestricted as we described in the section `How confect\ndiffers from others?`_.\n\nTwo ways to load configuration file.\n\n1. Through Python module importing: ``conf.load_module(module_name)``\n2. Through Python file reading: ``conf.load_file(file_path)``\n\nNo matter the loading statement is located before or after properties\ndeclaration, property values in configuration file always override default\nvalues in the declarations. It\'s possible to load configuration file multiple times,\nthe latter one would replace values from former loading.\n\nBe aware, *you should access your configuration properties after load\nconfiguration files.* If not, you might get wrong/default value. Therefore, we\nusually load configuration file right after the statement of creating the\n``Conf`` object.\n\n.. code:: python\n\n   import confect\n   conf = confect.Conf()\n\n   # load configuration files through importing\n   try:\n       conf.load_module(\'local_conf\')\n   except ImportError:\n       pass\n\n   SYSTEM_CONF_PATH = Path(\'path/to/system_conf.py\')\n   if SYSTEM_CONF_PATH.exists():\n       conf.load_file(SYSTEM_CONF_PATH)\n\n\nUse ``PYTHONPATH`` environment varibale to control the source of configuration file.\n\n.. code:: console\n\n   $ vi local_conf.py\n   $ export PYTHONPATH=.\n   $ python your_application.py\n\n\nWrite Configuration File\n^^^^^^^^^^^^^^^^^^^^^^^^^\n\nConfiguration files are written in Python, but they are isolated from your application. The configuration declaration(check `Configuration Properties Declaration`_) use the ``conf`` object directly, and declare properties with default value with ``conf.declar_group(...)``. While configuration files use ``confect.c`` to override declared properties. Configuration files shouldn\'t be import directly, they can only be loaded with ``conf.load_module(module_name))`` or ``conf.load_file(file_path)``.\n\nIn configuration file, import ``confect.c`` object and set all properties on it\nas if ``c`` is the conf object. Here\'s an example of configuration file.\n\n``local_conf.py``\n\n.. code:: python\n\n   from confect import c\n\n   import os\n\n   DEBUG = True\n\n   if DEBUG:\n       c.cache.expire = 1\n\n   c.cache.key = os.environ[\'CACHE_KEY\']\n\n   # loading some secret file and set configuration\n   import json\n   with open(\'db_secret.json\') as f:\n       db_secret = json.load(f)\n\n   c.db.username = db_secret[\'username\']\n   c.db.password = db_secret[\'password\']\n\n\nIt\'s not necessary and is unusual to have all configuration properties be defined in the\nconfiguration file. *Put only those configuration properties that you want to override to the configuration file.*\n\nYou can set any property in any configuration group onto the ``c`` object.\nHowever, **they are only accessable if you declared it in the source code with**\n``Conf.declare_group(group_name)``. See `Configuration Properties Declaration`_ for details.\n\nThe ``c`` object only exits when loading a python configuration file, it\'s not\npossible to import it in your source code.\n\n\nAdvanced Usage\n===================\n\nLoading Environment Variables\n------------------------------\n\n   # overrides configuration with environment variables with the prefix `projx`\n   conf.load_envvars(\'projx\')\n\n``Conf.load_envvars(prefix: str)`` automatically searches environment variables\nin ``<prefix>__<group>__<prop>`` format. All of these three identifier are case\nsensitive. If you have a configuration property ``conf.cache.expire_time`` and\nyou call ``Conf.load_envvars(\'projx\')``. It will set that ``expire_time``\nproperty to the parsed value from ``projx__cache__expire_time`` environment\nvariable.\n\n>>> import os\n>>> os.environ[\'projx__cache__expire\'] = \'3600\'\n\n>>> conf = confect.Conf()\n>>> conf.load_envvars(\'projx\')  # doctest: +SKIP\n\nIf ``cache.expire`` has been declared, then\n\n>>> conf.cache.expire\n3600\n\nCommand Line Options\n-------------------------\n\n``conf.click_options`` decorator attachs all declared configuration to a click_\ncommand.\n\n\n``projx/__main__.py``\n\n.. code:: python\n\n   import click\n   from projx.core import conf\n\n   @click.command()\n   @conf.click_options\n   def cli():\n       click.echo(f\'cache_expire: {conf.api.cache_expire}\')\n\n   if __name__ == \'__main__\':\n       cli()\n\nIt automatically creates a comprehensive help message with all properties and default values.\n\n.. code:: console\n\n   $ python -m projx.cli --help\n   Usage: cli.py [OPTIONS]\n\n   Options:\n     --api-cache_expire INTEGER  [default: 86400]\n     --api-cache_prefix TEXT     [default: projx_cache]\n     --api-url_base_path TEXT    [default: api/v2/]\n     --db-db_name TEXT           [default: proj_x]\n     --db-username TEXT          [default: proj_x_admin]\n     --db-password TEXT          [default: your_password]\n     --db-host TEXT              [default: 127.0.0.1]\n     --help                      Show this message and exit.\n\n\nThe option do change the value of configuration property.\n\n.. code:: console\n\n   $ python -m projx.cli\n   cache_expire: 86400\n   $ python -m projx.cli --api-cache_expire 33\n   cache_expire: 33\n\n\nParser\n---------------\n\nConfect includes predefined parsers of these primitive types.\n\n- ``str``: ``s``\n- ``int``: ``ast.literal_eval(s)``\n- ``float``: ``ast.literal_eval(s)``\n- ``bytes``: ``s.encode(encoding)``\n- ``datetime.datetime`` : ``dt.datetime.strptime(s, fmt)``\n- ``datetime.date`` : ``dt.datetime.strptime(s, fmt).date()``\n- ``tuple`` : ``json.loads(s)``\n- ``dict``: ``json.loads(s)``\n- ``list``: ``json.loads(s)``\n\n\nComplex Configuration Loading\n-----------------------------\nThe code in the section `Conf Object`_ is a simple example that loads only through module importing.\nHere\'s an much more complex example that demostrates how to dynamically select and load configurations.\n\n.. code:: python\n\n   import sys\n   import confect\n\n   conf = confect.Conf()\n\n   # load configuration file\n   if len(sys.argv) == 2:\n       conf.load_file(sys.argv[1])\n   else:\n       try:\n          conf.load_file(\'path/to/team_conf.py\')\n       FileNotFoundError:\n          logger.warning(\'Unable to find team configuration file\')\n\n       try:\n          conf.load_file(\'path/to/personal_conf.py\')\n       FileNotFoundError:\n          logger.info(\'Unable to find personal configuration file\')\n\n   # load configuration file through importing\n   try:\n       conf.load_module(\'projx_conf\')\n   except ImportError:\n       logger.warning(\'Unable to load find configuration module %r\',\n                      \'proj_x_conf\')\n\n   # overrides configuration with environment variables\n   conf.load_envvars(\'projx\')\n\nRuntime Configuration Altering\n-------------------------------\n\n``Conf.mutate_locally()`` context manager creates an environment that makes\n``Conf`` object temporarily mutable. All changes would be restored when it\nleaves the block. It is usaful on writing test case or testing configuration\nproperties in Python REPL.\n\n>>> conf = Conf()\n>>> conf.declare_group(  # declare group through keyword arguments\n...      \'dummy\',\n...      prop1=3,\n...      prop2=\'some string\')\n...\n>>> with conf.mutate_locally():\n...      conf.dummy.prop1 = 5\n...      print(conf.dummy.prop1)\n5\n...     call_some_function_use_this_property()\n>>> print(conf.dummy.prop1)  # all configuration restored\n3\n\n\nTo-Dos\n======\n\n- A public interface for exporting a conf group into a dictionary\n- A plugin for `argparse <https://docs.python.org/3/library/argparse.html>`_  that adds command line options for altering configuration properties.\n- Copy-on-write mechenism in ``conf.mutate_locally()`` for better performance and memory usage.\n- API reference page\n\n.. _click: http://click.pocoo.org/\n.. _pip: https://pip.pypa.io/en/stable/\n.. _virtualenv: https://hynek.me/articles/virtualenv-lives/\n.. _poetry: https://poetry.eustace.io/\n',
    'author': 'Yen, Tzu-Hsi',
    'author_email': 'joseph.yen@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/d2207197/confect',
    'packages': packages,
    'package_data': package_data,
    'extras_require': extras_require,
    'python_requires': '>=3.6',
}


setup(**setup_kwargs)
