"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SchemaTransformer = void 0;
const fs = require("fs");
const path_1 = require("path");
const graphql_auth_transformer_1 = require("graphql-auth-transformer");
const graphql_connection_transformer_1 = require("graphql-connection-transformer");
const graphql_dynamodb_transformer_1 = require("graphql-dynamodb-transformer");
const graphql_http_transformer_1 = require("graphql-http-transformer");
const graphql_key_transformer_1 = require("graphql-key-transformer");
const graphql_transformer_core_1 = require("graphql-transformer-core");
const graphql_ttl_transformer_1 = require("graphql-ttl-transformer");
const graphql_versioned_transformer_1 = require("graphql-versioned-transformer");
const cdk_transformer_1 = require("./cdk-transformer");
// Import this way because FunctionTransformer.d.ts types were throwing an eror. And we didn't write this package so hope for the best :P
// eslint-disable-next-line
const { FunctionTransformer } = require('graphql-function-transformer');
class SchemaTransformer {
    constructor(props) {
        this.schemaPath = props.schemaPath || './schema.graphql';
        this.outputPath = props.outputPath || './appsync';
        this.isSyncEnabled = props.syncEnabled || false;
        this.outputs = {};
        this.resolvers = {};
        // TODO: Make this better?
        this.authTransformerConfig = {
            authConfig: {
                defaultAuthentication: {
                    authenticationType: 'AMAZON_COGNITO_USER_POOLS',
                    userPoolConfig: {
                        userPoolId: '12345xyz',
                    },
                },
                additionalAuthenticationProviders: [
                    {
                        authenticationType: 'API_KEY',
                        apiKeyConfig: {
                            description: 'Testing',
                            apiKeyExpirationDays: 100,
                        },
                    },
                    {
                        authenticationType: 'AWS_IAM',
                    },
                ],
            },
        };
    }
    transform() {
        var _a, _b;
        const transformConfig = this.isSyncEnabled ? this.loadConfigSync() : {};
        // Note: This is not exact as we are omitting the @searchable transformer as well as some others.
        const transformer = new graphql_transformer_core_1.GraphQLTransform({
            transformConfig: transformConfig,
            transformers: [
                new graphql_dynamodb_transformer_1.DynamoDBModelTransformer(),
                new graphql_ttl_transformer_1.default(),
                new graphql_versioned_transformer_1.VersionedModelTransformer(),
                new FunctionTransformer(),
                new graphql_key_transformer_1.KeyTransformer(),
                new graphql_connection_transformer_1.ModelConnectionTransformer(),
                new graphql_auth_transformer_1.ModelAuthTransformer(this.authTransformerConfig),
                new graphql_http_transformer_1.HttpTransformer(),
                new cdk_transformer_1.CdkTransformer(),
            ],
        });
        const schema = fs.readFileSync(this.schemaPath);
        const cfdoc = transformer.transform(schema.toString());
        // TODO: Get Unauth Role and Auth Role policies for authorization stuff
        this.unauthRolePolicy = ((_a = cfdoc.rootStack.Resources) === null || _a === void 0 ? void 0 : _a.UnauthRolePolicy01) || undefined;
        this.writeSchema(cfdoc.schema);
        this.writeResolversToFile(cfdoc.resolvers);
        // Outputs shouldn't be null but default to empty map
        this.outputs = (_b = cfdoc.rootStack.Outputs) !== null && _b !== void 0 ? _b : {};
        return this.outputs;
    }
    /**
       *
       */
    getResolvers() {
        const statements = ['Query', 'Mutation', 'Subscription'];
        const resolversDirPath = path_1.normalize('./appsync/resolvers');
        if (fs.existsSync(resolversDirPath)) {
            const files = fs.readdirSync(resolversDirPath);
            files.forEach(file => {
                // Example: Mutation.createChannel.response
                let args = file.split('.');
                let typeName = args[0];
                let name = args[1];
                let templateType = args[2]; // request or response
                let filepath = path_1.normalize(`${resolversDirPath}/${file}`);
                if (statements.indexOf(typeName) >= 0 || (this.outputs.noneResolvers && this.outputs.noneResolvers[name])) {
                    if (!this.resolvers[name]) {
                        this.resolvers[name] = {
                            typeName: typeName,
                            fieldName: name,
                        };
                    }
                    if (templateType === 'req') {
                        this.resolvers[name].requestMappingTemplate = filepath;
                    }
                    else if (templateType === 'res') {
                        this.resolvers[name].responseMappingTemplate = filepath;
                    }
                }
                else { // This is a GSI
                    if (!this.resolvers.gsi) {
                        this.resolvers.gsi = {};
                    }
                    let mapName = `${typeName}${name}`;
                    if (!this.resolvers.gsi[mapName]) {
                        this.resolvers.gsi[mapName] = {
                            typeName: typeName,
                            fieldName: name,
                            tableName: name.charAt(0).toUpperCase() + name.slice(1),
                        };
                    }
                    if (templateType === 'req') {
                        this.resolvers.gsi[mapName].requestMappingTemplate = filepath;
                    }
                    else if (templateType === 'res') {
                        this.resolvers.gsi[mapName].responseMappingTemplate = filepath;
                    }
                }
            });
        }
        return this.resolvers;
    }
    /**
       * Writes the schema to the output directory for use with @aws-cdk/aws-appsync
       * @param schema
       */
    writeSchema(schema) {
        if (!fs.existsSync(this.outputPath)) {
            fs.mkdirSync(this.outputPath);
        }
        fs.writeFileSync(`${this.outputPath}/schema.graphql`, schema);
    }
    /**
       * Writes all the resolvers to the output directory for loading into the datasources later
       * @param resolvers
       */
    writeResolversToFile(resolvers) {
        if (!fs.existsSync(this.outputPath)) {
            fs.mkdirSync(this.outputPath);
        }
        const resolverFolderPath = path_1.normalize(this.outputPath + '/resolvers');
        if (fs.existsSync(resolverFolderPath)) {
            const files = fs.readdirSync(resolverFolderPath);
            files.forEach(file => fs.unlinkSync(resolverFolderPath + '/' + file));
            fs.rmdirSync(resolverFolderPath);
        }
        if (!fs.existsSync(resolverFolderPath)) {
            fs.mkdirSync(resolverFolderPath);
        }
        Object.keys(resolvers).forEach((key) => {
            const resolver = resolvers[key];
            const fileName = key.replace('.vtl', '');
            const resolverFilePath = path_1.normalize(`${resolverFolderPath}/${fileName}`);
            fs.writeFileSync(resolverFilePath, resolver);
        });
    }
    /**
       * @returns {@link TransformConfig}
      */
    loadConfigSync(projectDir = 'resources') {
        // Initialize the config always with the latest version, other members are optional for now.
        let config = {
            Version: graphql_transformer_core_1.TRANSFORM_CURRENT_VERSION,
            ResolverConfig: {
                project: {
                    ConflictHandler: "OPTIMISTIC_CONCURRENCY" /* OPTIMISTIC */,
                    ConflictDetection: 'VERSION',
                },
            },
        };
        const configDir = path_1.join(__dirname, '..', '..', projectDir);
        try {
            const configPath = path_1.join(configDir, graphql_transformer_core_1.TRANSFORM_CONFIG_FILE_NAME);
            const configExists = fs.existsSync(configPath);
            if (configExists) {
                const configStr = fs.readFileSync(configPath);
                config = JSON.parse(configStr.toString());
            }
            return config;
        }
        catch (err) {
            return config;
        }
    }
}
exports.SchemaTransformer = SchemaTransformer;
//# sourceMappingURL=data:application/json;base64,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