"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const api = require("aws-cdk-lib/aws-apigateway");
const kms = require("aws-cdk-lib/aws-kms");
const sqs = require("aws-cdk-lib/aws-sqs");
const assertions_1 = require("aws-cdk-lib/assertions");
test('Test deployment w/o DLQ', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        deployDeadLetterQueue: false
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        AuthorizationType: "AWS_IAM"
    });
});
test('Test deployment w/o allowReadOperation', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        allowCreateOperation: true,
        allowReadOperation: false,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "AWS_IAM"
    });
});
test('Test deployment w/ allowReadOperation', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        allowReadOperation: true,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        AuthorizationType: "AWS_IAM"
    });
});
test('Test properties', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const pattern = new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        deployDeadLetterQueue: true,
        maxReceiveCount: 3
    });
    // Assertion 1
    expect(pattern.apiGateway).toBeDefined();
    // Assertion 2
    expect(pattern.sqsQueue).toBeDefined();
    // Assertion 3
    expect(pattern.apiGatewayRole).toBeDefined();
    expect(pattern.apiGatewayCloudWatchRole).toBeDefined();
    expect(pattern.apiGatewayLogGroup).toBeDefined();
    expect(pattern.deadLetterQueue).toBeDefined();
});
test('Test deployment ApiGateway AuthorizationType override', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.NONE
            }
        },
        queueProps: {},
        createRequestTemplate: "{}",
        allowCreateOperation: true,
        allowReadOperation: false,
        allowDeleteOperation: true,
        deployDeadLetterQueue: false
    });
    // Assertion 1
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "NONE"
    });
    // Assertion 2
    template.hasResourceProperties("AWS::ApiGateway::Method", {
        HttpMethod: "DELETE",
        AuthorizationType: "NONE"
    });
});
test('Test deployment for override ApiGateway createRequestTemplate', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        createRequestTemplate: "Action=SendMessage&MessageBody=$util.urlEncode(\"HelloWorld\")",
        allowCreateOperation: true
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        Integration: {
            RequestTemplates: {
                "application/json": "Action=SendMessage&MessageBody=$util.urlEncode(\"HelloWorld\")"
            }
        }
    });
});
test('Test deployment for override ApiGateway getRequestTemplate', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        readRequestTemplate: "Action=HelloWorld",
        allowReadOperation: true
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        Integration: {
            RequestTemplates: {
                "application/json": "Action=HelloWorld"
            }
        }
    });
});
test('Test deployment for override ApiGateway deleteRequestTemplate', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        deleteRequestTemplate: "Action=HelloWorld",
        allowDeleteOperation: true
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ApiGateway::Method", {
        HttpMethod: "DELETE",
        Integration: {
            RequestTemplates: {
                "application/json": "Action=HelloWorld"
            }
        }
    });
});
test('Test deployment for disallow delete operation', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        allowDeleteOperation: false
    });
    const template = assertions_1.Template.fromStack(stack);
    const resources = template.findResources('AWS::ApiGateway::Resource', {
        PathPart: "message"
    });
    expect(Object.keys(resources).length).toBe(0);
});
test('Test deployment for allow delete operation', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        allowDeleteOperation: true
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Resource', {
        PathPart: "message"
    });
});
test('Test deployment with existing queue object', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const existingQueueObj = new sqs.Queue(stack, 'existing-queue', {
        fifo: true
    });
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        existingQueueObj
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        FifoQueue: true
    });
    template.resourceCountIs("AWS::SQS::Queue", 1);
});
test('Queue is encrypted with imported CMK when set on encryptionKey prop', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const cmk = new kms.Key(stack, 'cmk');
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        encryptionKey: cmk
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "cmk01DE03DA",
                "Arn"
            ]
        }
    });
});
test('Queue is encrypted with imported CMK when set on queueProps.encryptionKeyProps prop', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const cmk = new kms.Key(stack, 'cmk');
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        queueProps: {
            encryptionMasterKey: cmk
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "cmk01DE03DA",
                "Arn"
            ]
        }
    });
});
test('Queue is encrypted with provided encryptionKeyProps', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        encryptionKeyProps: {
            alias: 'new-key-alias-from-props'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "apigatewaysqsqueueKeyEC2D27F3",
                "Arn"
            ]
        }
    });
    template.hasResourceProperties('AWS::KMS::Alias', {
        AliasName: 'alias/new-key-alias-from-props',
        TargetKeyId: {
            'Fn::GetAtt': [
                'apigatewaysqsqueueKeyEC2D27F3',
                'Arn'
            ]
        }
    });
});
test('Queue is encrypted by default with AWS-managed KMS key when no other encryption properties are set', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {});
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::SQS::Queue', {
        KmsMasterKeyId: "alias/aws/sqs"
    });
});
test('Queue is encrypted with customer managed KMS Key when enable encryption flag is true', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        enableEncryptionWithCustomerManagedKey: true
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "apigatewaysqsqueueKeyEC2D27F3",
                "Arn"
            ]
        }
    });
});
test('Construct accepts additional read request templates', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        enableEncryptionWithCustomerManagedKey: true,
        additionalReadRequestTemplates: {
            'text/plain': 'Hello'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'GET',
        Integration: {
            RequestTemplates: {
                'application/json': 'Action=ReceiveMessage',
                'text/plain': 'Hello'
            }
        }
    });
});
test('Construct accepts additional create request templates', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        enableEncryptionWithCustomerManagedKey: true,
        allowCreateOperation: true,
        additionalCreateRequestTemplates: {
            'text/plain': 'Hello'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'POST',
        Integration: {
            RequestTemplates: {
                'application/json': 'Action=SendMessage&MessageBody=$util.urlEncode(\"$input.body\")',
                'text/plain': 'Hello'
            }
        }
    });
});
test('Construct accepts additional delete request templates', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        enableEncryptionWithCustomerManagedKey: true,
        allowDeleteOperation: true,
        additionalDeleteRequestTemplates: {
            'text/plain': 'DeleteMe'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'DELETE',
        Integration: {
            RequestTemplates: {
                'application/json': `Action=DeleteMessage&ReceiptHandle=$util.urlEncode($input.params('receiptHandle'))`,
                'text/plain': 'DeleteMe'
            }
        }
    });
});
test('Construct can override default create request template type', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        enableEncryptionWithCustomerManagedKey: true,
        allowCreateOperation: true,
        createRequestTemplate: 'Hello'
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'POST',
        Integration: {
            RequestTemplates: {
                'application/json': 'Hello'
            }
        }
    });
});
test('Construct can override default read request template type', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        enableEncryptionWithCustomerManagedKey: true,
        allowReadOperation: true,
        readRequestTemplate: 'Hello'
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'GET',
        Integration: {
            RequestTemplates: {
                'application/json': 'Hello'
            }
        }
    });
});
test('Construct can override default delete request template type', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        enableEncryptionWithCustomerManagedKey: true,
        allowDeleteOperation: true,
        deleteRequestTemplate: 'Hello'
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'DELETE',
        Integration: {
            RequestTemplates: {
                'application/json': 'Hello'
            }
        }
    });
});
test('Construct uses default integration responses', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        allowCreateOperation: true,
        allowReadOperation: true,
        allowDeleteOperation: true,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'POST',
        Integration: {
            IntegrationResponses: [
                {
                    StatusCode: '200'
                },
                {
                    ResponseTemplates: {
                        'text/html': 'Error'
                    },
                    SelectionPattern: '500',
                    StatusCode: '500'
                }
            ]
        }
    });
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'GET',
        Integration: {
            IntegrationResponses: [
                {
                    StatusCode: '200'
                },
                {
                    ResponseTemplates: {
                        'text/html': 'Error'
                    },
                    SelectionPattern: '500',
                    StatusCode: '500'
                }
            ]
        }
    });
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'DELETE',
        Integration: {
            IntegrationResponses: [
                {
                    StatusCode: '200'
                },
                {
                    ResponseTemplates: {
                        'text/html': 'Error'
                    },
                    SelectionPattern: '500',
                    StatusCode: '500'
                }
            ]
        }
    });
});
test('Construct uses custom createIntegrationResponses property', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        allowCreateOperation: true,
        createIntegrationResponses: [
            {
                statusCode: '200',
                responseTemplates: {
                    'text/html': 'OK'
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'POST',
        Integration: {
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        'text/html': 'OK'
                    },
                    StatusCode: '200'
                }
            ]
        }
    });
});
test('Construct uses custom readIntegrationResponses property', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        allowCreateOperation: true,
        readIntegrationResponses: [
            {
                statusCode: '200',
                responseTemplates: {
                    'text/html': 'OK'
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'GET',
        Integration: {
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        'text/html': 'OK'
                    },
                    StatusCode: '200'
                }
            ]
        }
    });
});
test('Construct uses custom deleteIntegrationResponses property', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        allowDeleteOperation: true,
        deleteIntegrationResponses: [
            {
                statusCode: '200',
                responseTemplates: {
                    'text/html': 'OK'
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'DELETE',
        Integration: {
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        'text/html': 'OK'
                    },
                    StatusCode: '200'
                }
            ]
        }
    });
});
test('Construct throws error when createRequestTemplate is set and allowCreateOperation is not true', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        createRequestTemplate: '{}',
    });
    expect(app).toThrowError(/The 'allowCreateOperation' property must be set to true when setting any of the following: 'createRequestTemplate', 'additionalCreateRequestTemplates', 'createIntegrationResponses'/);
});
test('Construct throws error when additionalCreateRequestTemplates is set and allowCreateOperation is not true', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        additionalCreateRequestTemplates: {}
    });
    expect(app).toThrowError(/The 'allowCreateOperation' property must be set to true when setting any of the following: 'createRequestTemplate', 'additionalCreateRequestTemplates', 'createIntegrationResponses'/);
});
test('Construct throws error when createIntegrationResponses is set and allowCreateOperation is not true', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        createIntegrationResponses: []
    });
    expect(app).toThrowError(/The 'allowCreateOperation' property must be set to true when setting any of the following: 'createRequestTemplate', 'additionalCreateRequestTemplates', 'createIntegrationResponses'/);
});
test('Construct throws error when readRequestTemplate is set and allowReadOperation is false', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        allowReadOperation: false,
        readRequestTemplate: '{}',
    });
    expect(app).toThrowError(/The 'allowReadOperation' property must be set to true or undefined when setting any of the following: 'readRequestTemplate', 'additionalReadRequestTemplates', 'readIntegrationResponses'/);
});
test('Construct throws error when additionalReadRequestTemplates is set and allowReadOperation is false', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        allowReadOperation: false,
        additionalReadRequestTemplates: {},
    });
    expect(app).toThrowError(/The 'allowReadOperation' property must be set to true or undefined when setting any of the following: 'readRequestTemplate', 'additionalReadRequestTemplates', 'readIntegrationResponses'/);
});
test('Construct throws error when readIntegrationResponses is set and allowReadOperation is false', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        allowReadOperation: false,
        readIntegrationResponses: [],
    });
    expect(app).toThrowError(/The 'allowReadOperation' property must be set to true or undefined when setting any of the following: 'readRequestTemplate', 'additionalReadRequestTemplates', 'readIntegrationResponses'/);
});
test('Construct throws error when deleteRequestTemplate is set and allowDeleteOperation is not true', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        deleteRequestTemplate: '{}',
    });
    expect(app).toThrowError(/The 'allowDeleteOperation' property must be set to true when setting any of the following: 'deleteRequestTemplate', 'additionalDeleteRequestTemplates', 'deleteIntegrationResponses'/);
});
test('Construct throws error when additionalDeleteRequestTemplates is set and allowDeleteOperation is not true', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        additionalDeleteRequestTemplates: {}
    });
    expect(app).toThrowError(/The 'allowDeleteOperation' property must be set to true when setting any of the following: 'deleteRequestTemplate', 'additionalDeleteRequestTemplates', 'deleteIntegrationResponses'/);
});
test('Construct throws error when deleteIntegrationResponses is set and allowDeleteOperation is not true', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        deleteIntegrationResponses: []
    });
    expect(app).toThrowError(/The 'allowDeleteOperation' property must be set to true when setting any of the following: 'deleteRequestTemplate', 'additionalDeleteRequestTemplates', 'deleteIntegrationResponses'/);
});
test('Confirm the CheckSqsProps is being called', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => {
        new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
            queueProps: {
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            },
            existingQueueObj: new sqs.Queue(stack, 'test', {})
        });
    };
    expect(app).toThrowError(/Error - Either provide queueProps or existingQueueObj, but not both.\n/);
});
//# sourceMappingURL=data:application/json;base64,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