from __future__ import unicode_literals
import unittest
from pyoma.browser import locus_parser
import logging

logging.basicConfig(level=logging.DEBUG)


class LocusParserTest(unittest.TestCase):
    def verifyResult(self, res, eNr, nr_exons, min_loc, max_loc, strand):
        self.assertEqual(res.shape, (nr_exons,))
        for x in res["EntryNr"]:
            self.assertEqual(x, eNr)
        self.assertEqual(res["Start"].min(), min_loc)
        self.assertEqual(res["End"].max(), max_loc)
        self.assertEqual(res[0]["Strand"], strand)

    def test_singleExonForward(self):
        s, eNr = "422..522", 22
        lp = locus_parser.LocusParser()
        res = lp.parse(s, eNr)
        self.verifyResult(res, eNr, 1, 422, 522, 1)

    def test_singleExonReverse(self):
        s, eNr = "complement(22..6232)", 21
        lp = locus_parser.LocusParser()
        res = lp.parse(s, eNr)
        self.verifyResult(res, eNr, 1, 22, 6232, -1)

    def test_forward(self):
        s, eNr = "join(23..15555,22432..66666)", 5
        lp = locus_parser.LocusParser()
        res = lp.parse(s, eNr)
        self.verifyResult(res, eNr, 2, 23, 66666, 1)

    def test_reverse(self):
        s, eNr = "join(complement(23..15555),complement(22432..66666))", 5
        lp = locus_parser.LocusParser()
        res = lp.parse(s, eNr)
        self.verifyResult(res, eNr, 2, 23, 66666, -1)

    def test_before_and_after(self):
        s, eNr = "join(Before(10)..40,33..After(66))", 8
        lp = locus_parser.LocusParser()
        res = lp.parse(s, eNr)
        self.verifyResult(res, eNr, 2, 10, 66, 1)

    def test_complement_of_join(self):
        s, eNr = "complement(join(1653679..1653924,1654352..1654354))", 333
        lp = locus_parser.LocusParser()
        res = lp.parse(s, eNr)
        self.verifyResult(res, eNr, 2, 1653679, 1654354, -1)

    def test_huge(self):
        """parsing fails when too many exons are joined"""
        s, eNr = (
            "join(18057347..18057437, 18059684..18059887, 18061318..18061605, 18062072..18062157, 18062252..18062496, 18066578..18066893, 18067450..18067602, 18068246..18068383, 18069873..18070004, 18071247..18071384, 18071975..18072112, 18072588..18072725, 18076564..18076857, 18076954..18077076, 18078491..18078772, 18079026..18079091, 18079794..18080052, 18080242..18080305, 18080421..18080636, 18081557..18081699, 18082817..18083022, 18083572..18083805, 18083916..18084160, 18085260..18085501, 18085621..18085785, 18085885..18086053, 18086155..18087848, 18087991..18088272, 18089474..18089740, 18089836..18090108, 18090210..18090473, 18090566..18090826, 18092434..18092694, 18093446..18093709, 18093802..18094062, 18094172..18094432, 18094770..18095030, 18095538..18095679, 18095768..18095933, 18097032..18097263, 18098723..18099007, 18099238..18099363, 18105150..18105338, 18110887..18110931, 18122263..18125013, 18125797..18126075, 18127849..18128412, 18128947..18129228, 18129328..18129606, 18130180..18130461, 18130553..18130831, 18130949..18131236, 18131349..18131627, 18131713..18131994, 18132089..18132367, 18132477..18132755, 18132858..18133136, 18133251..18133538, 18133664..18133942, 18134077..18134358, 18134459..18134737, 18134857..18135135, 18135249..18135527, 18135629..18135916, 18136161..18136439, 18136653..18136934, 18138061..18138339, 18138461..18138733, 18139476..18139766, 18139880..18140167, 18140319..18140597, 18140690..18140968, 18141072..18141350, 18141468..18141755, 18141878..18142165, 18142709..18142990, 18143249..18143527, 18143644..18143928, 18144081..18144359, 18144454..18144741, 18144932..18145210, 18145305..18145583, 18145683..18145961, 18146101..18146388, 18146530..18146817, 18148170..18148460, 18148896..18149180, 18149741..18150022, 18150139..18150417, 18151111..18151404, 18151515..18151793, 18151907..18152185, 18152275..18152553, 18153289..18153576, 18153714..18154001, 18154523..18154813, 18156541..18156828, 18157556..18157648, 18157769..18158054, 18158829..18159012, 18159247..18159336, 18159430..18159697, 18159800..18160066, 18161643..18161852, 18162061..18162138, 18162244..18162270, 18162678..18162737, 18163282..18163365, 18165218..18165292, 18166440..18166493, 18167341..18167745, 18168204..18168266, 18168625..18168702, 18169306..18169383, 18169969..18170061, 18170290..18170370, 18171394..18171470, 18171867..18171951, 18174191..18174274, 18174499..18174579, 18174725..18174805, 18174980..18175063, 18175312..18175413, 18175834..18175944, 18177746..18177826, 18178003..18178083, 18178313..18178396, 18178555..18178638, 18179198..18179281, 18179504..18179587, 18180340..18180420, 18180723..18180938, 18181857..18181934, 18182139..18182213, 18182421..18182513, 18183292..18183369, 18183567..18183725, 18183926..18184018, 18184392..18184469, 18184973..18185077, 18185334..18185417, 18185858..18185941, 18186136..18186230, 18186267..18186510, 18188002..18188076, 18188852..18188926, 18189567..18189659, 18190514..18190591, 18191554..18191622, 18191776..18191802, 18191833..18191880, 18192021..18192314, 18193622..18193702, 18194477..18194554, 18200945..18201028, 18201140..18201223, 18201307..18201390, 18204571..18204654, 18204959..18205042, 18205320..18205403, 18205537..18205626, 18205757..18205840, 18205965..18206048, 18206182..18206259, 18206648..18206731, 18207248..18207361, 18207636..18207713, 18210681..18210794, 18212271..18212339, 18214196..18214276, 18215178..18215252, 18215854..18215943, 18216382..18216444, 18216775..18216864, 18218599..18218649, 18219200..18219601, 18219771..18220049, 18220292..18220567, 18220698..18220837, 18221361..18221487, 18221580..18221843, 18222284..18222550, 18222684..18222947, 18223037..18223176, 18223343..18223469, 18223588..18223854, 18223966..18224232, 18225169..18225435, 18225527..18225666, 18226076..18226202, 18227010..18227152, 18230912..18231035, 18231753..18232019, 18233879..18233976, 18234929..18235097, 18235187..18235453, 18235555..18235821, 18235911..18236189, 18236281..18236689, 18236796..18236920, 18237025..18237291, 18238035..18238304, 18238395..18238697, 18238796..18239098, 18239376..18239563, 18239653..18239767, 18239874..18240158, 18240249..18240400, 18241092..18241239, 18241519..18241696, 18241988..18242109, 18242204..18242491, 18242602..18242898, 18243020..18243206, 18243664..18243779, 18243871..18244170, 18244286..18244585, 18244680..18244785, 18244874..18245070, 18245154..18245459, 18245555..18245833, 18246377..18246676, 18246786..18247088, 18247419..18247781, 18248065..18248367, 18248498..18248797, 18249290..18249586, 18249687..18249974, 18250068..18250361, 18251430..18251729, 18251839..18252147, 18252236..18252426, 18252819..18253248, 18254780..18255088, 18255222..18255370, 18255457..18255489, 18255586..18255715, 18255805..18256104, 18256192..18256509, 18257562..18257858, 18257968..18258267, 18258353..18258667, 18258754..18258902, 18260109..18260259, 18260348..18260629, 18261855..18262157, 18262652..18262954, 18263058..18263339, 18263444..18263743, 18263837..18264139, 18264235..18264543, 18264658..18264939, 18265061..18265360, 18265444..18265737, 18265845..18268811, 18269139..18269459, 18269555..18269839, 18269937..18270236, 18270547..18270849, 18271279..18271554, 18271658..18271957, 18272050..18272352, 18272512..18272811, 18273296..18273583, 18273945..18274241, 18274330..18274632, 18274729..18275034, 18275981..18276268, 18276363..18276653, 18276745..18277032, 18277386..18277973, 18278074..18278178, 18278606..18278803, 18278910..18279206, 18279306..18279893, 18279985..18280287, 18280389..18297494, 18298084..18298380, 18298485..18299072, 18299165..18299467, 18299581..18299877, 18301197..18301484, 18301596..18301895, 18302581..18302883, 18302973..18303278, 18303388..18305154, 18305459..18305752, 18306442..18306729, 18306849..18307148, 18307244..18309310, 18309434..18309736, 18309834..18310139, 18310238..18310528, 18310624..18310920, 18311057..18311362, 18312656..18312961, 18313048..18313329, 18313514..18314107, 18314231..18314518, 18314617..18314919, 18315393..18315695, 18316272..18316574, 18316676..18317260, 18317389..18317694, 18317783..18318082, 18318710..18319285, 18319387..18319692, 18320058..18320651, 18320767..18326212, 18326327..18326525, 18326665..18326821, 18327558..18328249, 18328342..18328495, 18329070..18329372, 18329509..18329666, 18329671..18329692)",
            7928749,
        )
        lp = locus_parser.LocusParser()
        res = lp.parse(s, eNr)
        self.verifyResult(res, eNr, 316, 18057347, 18329692, 1)

    def test_exon_with_single_nuc(self):
        s, eNr = (
            "join(complement(351224..351421), complement(350821..350825), complement(350585..350789), complement(350552), complement(349876..350415), complement(348297..349785), complement(347837..348173))",
            6584010,
        )
        lp = locus_parser.LocusParser()
        res = lp.parse(s, eNr)
        self.verifyResult(res, eNr, 7, 347837, 351421, -1)

    def test_non_local_coordinates(self):
        s, eNr = (
            "join(complement(FromElsewhere('AOTI010129901.1',540..627)),complement(406..534))",
            88,
        )
        lp = locus_parser.LocusParser()
        res = lp.parse(s, eNr)
        self.verifyResult(res, eNr, 2, 406, 627, -1)
