"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KongControlPlaneEKS = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
class KongControlPlaneEKS extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const rds_secrets = 'rds-secrets';
        const rds_secrets_provider_class = 'rds-secrets-provider-class';
        // const route_53_zone = aws_route53.HostedZone.fromLookup(this, 'MyZone', {
        //   domainName: props.endpoints.hostedZoneName,
        // });
        // const admin_cert = new aws_certificatemanager.Certificate(this, 'adminCert', {
        //   domainName: props.endpoints.adminDns,
        //   validation: aws_certificatemanager.CertificateValidation.fromDns(route_53_zone),
        // });
        // const cluster_cert = new aws_certificatemanager.Certificate(this, 'clusterCert', {
        //   domainName: props.endpoints.clusterDns,
        //   validation: aws_certificatemanager.CertificateValidation.fromDns(route_53_zone),
        // });
        // const telemetry_cert = new aws_certificatemanager.Certificate(this, 'telemetryCert', {
        //   domainName: props.endpoints.telemetryDns,
        //   validation: aws_certificatemanager.CertificateValidation.fromDns(route_53_zone),
        // });
        // const manager_cert = new aws_certificatemanager.Certificate(this, 'managerCert', {
        //   domainName: props.endpoints.managerDns,
        //   validation: aws_certificatemanager.CertificateValidation.fromDns(route_53_zone),
        // });
        const kong_namespace = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongNamespace', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'Namespace',
                    metadata: { name: props.namespace },
                },
            ],
        });
        const kong_cp_service_account = new aws_cdk_lib_1.aws_eks.ServiceAccount(this, 'KongCpSA', {
            cluster: props.cluster,
            name: 'kong-cp-service-account',
            namespace: props.namespace,
        });
        const aws_region = aws_cdk_lib_1.Stack.of(this).region;
        const aws_account = aws_cdk_lib_1.Stack.of(this).account;
        kong_cp_service_account.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            resources: [
                `arn:aws:secretsmanager:${aws_region}:${aws_account}:secret:${props.license_secret_name}-??????`,
                `${props.rds.secret?.secretFullArn}`,
            ],
            actions: [
                'secretsmanager:GetSecretValue',
                'secretsmanager:DescribeSecret',
            ],
        }));
        kong_cp_service_account.node.addDependency(kong_namespace);
        const kong_certificate = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongCert', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'cert-manager.io/v1',
                    kind: 'Certificate',
                    metadata: {
                        name: props.cacertname,
                        namespace: props.namespace,
                    },
                    spec: {
                        commonName: props.endpoints.hostedZoneName,
                        dnsNames: [
                            props.endpoints.adminDns,
                            props.endpoints.clusterDns,
                            props.endpoints.managerDns,
                            props.endpoints.telemetryDns,
                        ],
                        duration: '2160h0m0s',
                        issuerRef: {
                            group: 'awspca.cert-manager.io',
                            kind: 'AWSPCAClusterIssuer',
                            name: props.clusterIssuerName,
                        },
                        renewBefore: '360h0m0s',
                        secretName: `${props.clusterIssuerName}-secret`,
                        usages: [
                            'server auth',
                            'client auth',
                        ],
                        privateKey: {
                            algorithm: 'RSA',
                            size: 2048,
                        },
                    },
                },
            ],
        });
        kong_certificate.node.addDependency(kong_namespace);
        // const secretNameParts = Fn.split('-', props.rds.secret?.secretName!);
        // const secretNameWithoutSuffix = Fn.join('-', [Fn.select(0, secretNameParts), Fn.select(1, secretNameParts)]);
        const secrets_crd_mount = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'SecretsMount', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'secrets-store.csi.x-k8s.io/v1',
                    kind: 'SecretProviderClass',
                    metadata: {
                        name: rds_secrets_provider_class,
                        namespace: props.namespace,
                    },
                    spec: {
                        provider: 'aws',
                        parameters: {
                            objects: `- objectName: "${props.rds.secret?.secretFullArn}"\n  jmesPath:\n    - path: username\n      objectAlias: dbusername\n    - path: password\n      objectAlias: dbpassword\n    - path: dbname\n      objectAlias: dbname\n    - path: host\n      objectAlias: dbhost\n`,
                        },
                        secretObjects: [
                            {
                                secretName: rds_secrets,
                                type: 'Opaque',
                                data: [
                                    {
                                        objectName: 'dbusername',
                                        key: 'dbusername',
                                    },
                                    {
                                        objectName: 'dbpassword',
                                        key: 'dbpassword',
                                    },
                                    {
                                        objectName: 'dbname',
                                        key: 'dbname',
                                    },
                                    {
                                        objectName: 'dbhost',
                                        key: 'dbhost',
                                    },
                                ],
                            },
                        ],
                    },
                },
                {
                    apiVersion: 'secrets-store.csi.x-k8s.io/v1',
                    kind: 'SecretProviderClass',
                    metadata: {
                        name: props.license_secret_name,
                        namespace: props.namespace,
                    },
                    spec: {
                        provider: 'aws',
                        secretObjects: [
                            {
                                secretName: props.license_secret_name,
                                type: 'opaque',
                                data: [
                                    {
                                        objectName: props.license_secret_name,
                                        key: 'license',
                                    },
                                ],
                            },
                        ],
                        parameters: {
                            objects: `- objectName: "${props.license_secret_name}"\n  objectType: "secretsmanager"\n`,
                        },
                    },
                },
            ],
        });
        secrets_crd_mount.node.addDependency(kong_namespace);
        const deploy_kong_cp_helm = new aws_cdk_lib_1.aws_eks.HelmChart(this, 'KongCpHelm', {
            ...props.HelmOptions,
            cluster: props.cluster,
            repository: 'https://charts.konghq.com',
            chart: 'kong',
            release: 'kong',
            namespace: props.namespace,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            wait: true,
            createNamespace: true,
            values: {
                ingressController: {
                    enabled: false,
                },
                // image: {
                //   repository: 'kong/kong-gateway',
                //   tag: '2.5.0.0-alpine',
                // },
                env: {
                    cluster_data_plane_purge_delay: 86400,
                    database: 'postgres',
                    role: 'control_plane',
                    cluster_mtls: 'pki',
                    cluster_cert: `/etc/secrets/${props.clusterIssuerName}-secret/tls.crt`,
                    cluster_cert_key: `/etc/secrets/${props.clusterIssuerName}-secret/tls.key`,
                    cluster_ca_cert: `/etc/secrets/${props.clusterIssuerName}-secret/ca.crt`,
                    admin_ssl_cert: `/etc/secrets/${props.clusterIssuerName}-secret/tls.crt`,
                    admin_ssl_cert_key: `/etc/secrets/${props.clusterIssuerName}-secret/tls.key`,
                    admin_gui_listen: '0.0.0.0:8002, 0.0.0.0:8445 ssl',
                    admin_gui_url: `https://${props.endpoints.managerDns}`,
                    admin_gui_ssl_cert: `/etc/secrets/${props.clusterIssuerName}-secret/tls.crt`,
                    admin_gui_ssl_cert_key: `/etc/secrets/${props.clusterIssuerName}-secret/tls.key`,
                    cluster_telemetry_listen: '0.0.0.0:8006',
                    cluster_telemetry_endpoint: `${props.endpoints.telemetryDns}:443`,
                    pg_user: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbusername',
                            },
                        },
                    },
                    pg_password: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbpassword',
                            },
                        },
                    },
                    pg_database: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbname',
                            },
                        },
                    },
                    pg_host: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbhost',
                            },
                        },
                    },
                },
                cluster: {
                    enabled: true,
                    type: 'LoadBalancer',
                    tls: {
                        enabled: true,
                        servicePort: 443,
                    },
                    annotations: {
                        // 'service.beta.kubernetes.io/aws-load-balancer-proxy-protocol': '*',
                        'service.beta.kubernetes.io/aws-load-balancer-nlb-target-type': 'ip',
                        // 'service.beta.kubernetes.io/aws-load-balancer-ssl-cert': cluster_cert.certificateArn,
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'external',
                        'service.beta.kubernetes.io/aws-load-balancer-scheme': 'internet-facing',
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-cluster`,
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.clusterDns,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                clustertelemetry: {
                    enabled: true,
                    type: 'LoadBalancer',
                    tls: {
                        enabled: true,
                        servicePort: 443,
                    },
                    annotations: {
                        // 'service.beta.kubernetes.io/aws-load-balancer-proxy-protocol': '*',
                        'service.beta.kubernetes.io/aws-load-balancer-nlb-target-type': 'ip',
                        // 'service.beta.kubernetes.io/aws-load-balancer-ssl-cert': telemetry_cert.certificateArn,
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'external',
                        'service.beta.kubernetes.io/aws-load-balancer-scheme': 'internet-facing',
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.telemetryDns,
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-telemetry`,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                proxy: {
                    enabled: false,
                },
                admin: {
                    enabled: true,
                    type: 'LoadBalancer',
                    labels: {
                        'enable-metrics': true,
                    },
                    tls: {
                        servicePort: 443,
                    },
                    annotations: {
                        // 'service.beta.kubernetes.io/aws-load-balancer-proxy-protocol': '*',
                        'service.beta.kubernetes.io/aws-load-balancer-nlb-target-type': 'ip',
                        // 'service.beta.kubernetes.io/aws-load-balancer-ssl-cert': admin_cert.certificateArn,
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'external',
                        'service.beta.kubernetes.io/aws-load-balancer-scheme': 'internal',
                        'service.beta.kubernetes.io/aws-load-balancer-additional-resource-tags': 'Type=admin',
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.adminDns,
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-admin`,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                deployment: {
                    serviceAccount: {
                        create: false,
                        name: kong_cp_service_account.serviceAccountName,
                    },
                    userDefinedVolumes: [
                        {
                            name: props.license_secret_name,
                            csi: {
                                driver: 'secrets-store.csi.k8s.io',
                                readOnly: true,
                                volumeAttributes: {
                                    secretProviderClass: props.license_secret_name,
                                },
                            },
                        },
                        {
                            name: rds_secrets,
                            csi: {
                                driver: 'secrets-store.csi.k8s.io',
                                readOnly: true,
                                volumeAttributes: {
                                    secretProviderClass: rds_secrets_provider_class,
                                },
                            },
                        },
                    ],
                    userDefinedVolumeMounts: [
                        {
                            name: props.license_secret_name,
                            mountPath: '/mnt/license_secrets',
                            readOnly: true,
                        },
                        {
                            name: rds_secrets,
                            mountPath: '/mnt/rds_secrets',
                            readOnly: true,
                        },
                    ],
                },
                enterprise: {
                    enabled: true,
                    license_secret: props.license_secret_name,
                },
                serviceMonitor: {
                    enabled: true,
                },
                manager: {
                    enabled: true,
                    type: 'LoadBalancer',
                    tls: {
                        enabled: true,
                        servicePort: 443,
                    },
                    // http: {
                    //   enabled: false,
                    // },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-nlb-target-type': 'ip',
                        // 'service.beta.kubernetes.io/aws-load-balancer-proxy-protocol': '*',
                        // 'service.beta.kubernetes.io/aws-load-balancer-ssl-cert': manager_cert.certificateArn,
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'external',
                        'service.beta.kubernetes.io/aws-load-balancer-scheme': 'internal',
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-manager`,
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.managerDns,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                postgresql: {
                    enabled: false,
                },
                secretVolumes: [
                    `${props.clusterIssuerName}-secret`,
                ],
            },
        });
        deploy_kong_cp_helm.node.addDependency(secrets_crd_mount, kong_namespace, kong_cp_service_account);
    }
}
exports.KongControlPlaneEKS = KongControlPlaneEKS;
//# sourceMappingURL=data:application/json;base64,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