"""
Module for functional methods of lists.
Individual sequence elements are all computed, so a list can provide better
performance than a sequence in situations in which all the elements are used.
"""
import builtins as __builtin__
import functools as _functools
from .seq import pipe


class List:
    """
    Contains operations for working with values of type sequences (any generator, list, etc.)
    """
    # list methods ------------------------------------

    @staticmethod
    @pipe
    def sum(items):
        """
        Returns the sum of the elements in the list.
        """
        return __builtin__.sum(items)

    @staticmethod
    @pipe
    def sumBy(items, function):
        """
        Returns the sum of the results generated by applying the function to each
        element of the list.
        """
        return __builtin__.sum(map(function, items))

    @staticmethod
    @pipe
    def filter(items, function):
        """
        Returns a new collection containing only the elements of the collection for which
        the given predicate returns "true".
        """
        return [x for x in items if function(x)]

    @staticmethod
    @pipe
    def map(items, function):
        """
        Builds a new collection whose elements are the results of applying the given function
        to each of the elements of the collection.
        """
        return [function(x) for x in items]

    @staticmethod
    @pipe
    def max(items):
        """
        Return the greatest of all elements of the list, compared via built-in max operator.
        """
        return __builtin__.max(items)

    @staticmethod
    @pipe
    def reduce(items, function):
        """
        Apply a function to each element of the collection, threading an accumulator argument
        through the computation. Apply the function to the first two elements of the list. Then
        feed this result into the function along with the third element and so on. Return the final
        result. If the input function is f and the elements are i0...iN then computes
        f (... (f i0 i1) i2 ...) iN.
        """
        return _functools.reduce(function, items)

    @staticmethod
    @pipe
    def collect(items, function):
        """
        Applies the given function to each element of the sequence and concatenates all the results.
        """
        def gen():
            for seq in map(function, items):
                for x in seq:
                    yield x

        return list(gen())
