"""
Basic tests of langid module
"""

import pytest

from stanza.models.common.doc import Document
from stanza.pipeline.core import Pipeline
from stanza.pipeline.multilingual import MultilingualPipeline
from stanza.tests import *

#pytestmark = pytest.mark.skip

def test_langid():
    """
    Basic test of language identification
    """
    english_text = "This is an English sentence."
    french_text = "C'est une phrase française."
    docs = [english_text, french_text]

    nlp = Pipeline(dir=TEST_MODELS_DIR, lang='multilingual', processors="langid")
    docs = [Document([], text=text) for text in docs]
    nlp(docs)
    predictions = [doc.lang for doc in docs]
    assert predictions == ["en", "fr"]

def test_langid_benchmark():
    """
    Run lang id model on 500 examples, confirm reasonable accuracy.
    """
    examples = [
    {"text": "contingentiam in naturalibus causis.", "label": "la"},
    {"text": "I jak opowiadał nieżyjący już pan Czesław", "label": "pl"},
    {"text": "Sonera gilt seit längerem als Übernahmekandidat", "label": "de"},
    {"text": "与银类似，汞也可以与空气中的硫化氢反应。", "label": "zh-hans"},
    {"text": "contradictionem implicat.", "label": "la"},
    {"text": "Bis zu Prozent gingen die Offerten etwa im", "label": "de"},
    {"text": "inneren Sicherheit vorgeschlagene Ausweitung der", "label": "de"},
    {"text": "Multimedia-PDA mit Mini-Tastatur", "label": "de"},
    {"text": "Ponášalo sa to na rovnicu o dvoch neznámych.", "label": "sk"},
    {"text": "이처럼 앞으로 심판의 그 날에 다시 올 메시아가 예수 그리스도이며 , 그는 모든 인류의", "label": "ko"},
    {"text": "Die Arbeitsgruppe bedauert , dass der weit über", "label": "de"},
    {"text": "И только раз довелось поговорить с ним не вполне", "label": "ru"},
    {"text": "de a-l lovi cu piciorul și conștiința că era", "label": "ro"},
    {"text": "relación coas pretensións do demandante e que, nos", "label": "gl"},
    {"text": "med petdeset in sedemdeset", "label": "sl"},
    {"text": "Catalunya; el Consell Comarcal del Vallès Oriental", "label": "ca"},
    {"text": "kunnen worden.", "label": "nl"},
    {"text": "Witkin je ve většině ohledů zcela jiný.", "label": "cs"},
    {"text": "lernen, so zu agieren, dass sie positive oder auch", "label": "de"},
    {"text": "olurmuş...", "label": "tr"},
    {"text": "sarcasmo de Altman, desde as «peruas» que discutem", "label": "pt"},
    {"text": "خلاف فوجداری مقدمہ درج کرے۔", "label": "ur"},
    {"text": "Norddal kommune :", "label": "no"},
    {"text": "dem Windows-.-Zeitalter , soll in diesem Jahr", "label": "de"},
    {"text": "przeklętych ucieleśniają mit poety-cygana,", "label": "pl"},
    {"text": "We do not believe the suspect has ties to this", "label": "en"},
    {"text": "groziņu pīšanu.", "label": "lv"},
    {"text": "Senior Vice-President David M. Thomas möchte", "label": "de"},
    {"text": "neomylně vybral nějakou knihu a začetl se.", "label": "cs"},
    {"text": "Statt dessen darf beispielsweise der Browser des", "label": "de"},
    {"text": "outubro, alcançando R $ bilhões em .", "label": "pt"},
    {"text": "(Porte, ), as it does other disciplines", "label": "en"},
    {"text": "uskupení se mylně domnívaly, že podporu", "label": "cs"},
    {"text": "Übernahme von Next Ende an dem System herum , das", "label": "de"},
    {"text": "No podemos decir a la Hacienda que los alemanes", "label": "es"},
    {"text": "и рѣста еи братья", "label": "orv"},
    {"text": "الذي اتخذ قرارا بتجميد اعلان الدولة الفلسطينية", "label": "ar"},
    {"text": "uurides Rootsi sõjaarhiivist toodud . sajandi", "label": "et"},
    {"text": "selskapets penger til å pusse opp sin enebolig på", "label": "no"},
    {"text": "средней полосе и севернее в Ярославской,", "label": "ru"},
    {"text": "il-massa żejda fil-ġemgħat u superġemgħat ta'", "label": "mt"},
    {"text": "The Global Beauties on internetilehekülg, mida", "label": "et"},
    {"text": "이스라엘 인들은 하나님이 그 큰 팔을 펴 이집트 인들을 치는 것을 보고 하나님을 두려워하며", "label": "ko"},
    {"text": "Snad ještě dodejme jeden ekonomický argument.", "label": "cs"},
    {"text": "Spalio d. vykusiame pirmajame rinkimų ture", "label": "lt"},
    {"text": "und schlechter Journalismus ein gutes Geschäft .", "label": "de"},
    {"text": "Du sodiečiai sėdi ant potvynio apsemtų namų stogo.", "label": "lt"},
    {"text": "цей є автентичним.", "label": "uk"},
    {"text": "Și îndegrabă fu cu îngerul mulțime de șireaguri", "label": "ro"},
    {"text": "sobra personal cualificado.", "label": "es"},
    {"text": "Tako se u Njemačkoj dvije trećine liječnika služe", "label": "hr"},
    {"text": "Dual-Athlon-Chipsatz noch in diesem Jahr", "label": "de"},
    {"text": "यहां तक कि चीन के चीफ ऑफ जनरल स्टाफ भी भारत का", "label": "hi"},
    {"text": "Li forestier du mont avale", "label": "fro"},
    {"text": "Netzwerken für Privatanwender zu bewundern .", "label": "de"},
    {"text": "만해는 승적을 가진 중이 결혼할 수 없다는 불교의 계율을 시대에 맞지 않는 것으로 보았다", "label": "ko"},
    {"text": "balance and weight distribution but not really for", "label": "en"},
    {"text": "og så e # tente vi opp den om morgonen å sfyrte", "label": "nn"},
    {"text": "변화는 의심의 여지가 없는 것이지만 반면에 진화는 논쟁의 씨앗이다 .", "label": "ko"},
    {"text": "puteare fac aceastea.", "label": "ro"},
    {"text": "Waitt seine Führungsmannschaft nicht dem", "label": "de"},
    {"text": "juhtimisega, tulid sealt.", "label": "et"},
    {"text": "Veränderungen .", "label": "de"},
    {"text": "banda en el Bayer Leverkusen de la Bundesliga de", "label": "es"},
    {"text": "В туже зиму посла всеволодъ сн҃а своѥго ст҃ослава", "label": "orv"},
    {"text": "пославъ приведе я мастеры ѿ грекъ", "label": "orv"},
    {"text": "En un nou escenari difícil d'imaginar fa poques", "label": "ca"},
    {"text": "καὶ γὰρ τινὲς αὐτοὺς εὐεργεσίαι εἶχον ἐκ Κροίσου", "label": "grc"},
    {"text": "직접적인 관련이 있다 .", "label": "ko"},
    {"text": "가까운 듯하면서도 멀다 .", "label": "ko"},
    {"text": "Er bietet ein ähnliches Leistungsniveau und", "label": "de"},
    {"text": "民都洛水牛是獨居的，並不會以群族聚居。", "label": "zh-hant"},
    {"text": "την τρομοκρατία.", "label": "el"},
    {"text": "hurbiltzen diren neurrian.", "label": "eu"},
    {"text": "Ah dimenticavo, ma tutta sta caciara per fare un", "label": "it"},
    {"text": "На первом этапе (-) прошла так называемая", "label": "ru"},
    {"text": "of games are on the market.", "label": "en"},
    {"text": "находится Мост дружбы, соединяющий узбекский и", "label": "ru"},
    {"text": "lessié je voldroie que li saint fussent aporté", "label": "fro"},
    {"text": "Дошла очередь и до Гималаев.", "label": "ru"},
    {"text": "vzácným suknem táhly pouští, si jednou chtěl do", "label": "cs"},
    {"text": "E no terceiro tipo sitúa a familias (%), nos que a", "label": "gl"},
    {"text": "وجابت دوريات امريكية وعراقية شوارع المدينة، فيما", "label": "ar"},
    {"text": "Jeg har bodd her i år .", "label": "no"},
    {"text": "Pohrozil, že odbory zostří postoj, pokud se", "label": "cs"},
    {"text": "tinham conseguido.", "label": "pt"},
    {"text": "Nicht-Erkrankten einen Anfangsverdacht für einen", "label": "de"},
    {"text": "permanece em aberto.", "label": "pt"},
    {"text": "questi possono promettere rendimenti fino a un", "label": "it"},
    {"text": "Tema juurutatud kahevedurisüsteemita oleksid", "label": "et"},
    {"text": "Поведение внешне простой игрушки оказалось", "label": "ru"},
    {"text": "Bundesländern war vom Börsenverein des Deutschen", "label": "de"},
    {"text": "acció, 'a mesura que avanci l'estiu, amb l'augment", "label": "ca"},
    {"text": "Dove trovare queste risorse? Jay Naidoo, ministro", "label": "it"},
    {"text": "essas gordurinhas.", "label": "pt"},
    {"text": "Im zweiten Schritt sollen im übernächsten Jahr", "label": "de"},
    {"text": "allveelaeva pole enam vaja, kuna külm sõda on läbi", "label": "et"},
    {"text": "उपद्रवी दुकानों को लूटने के साथ ही उनमें आग लगा", "label": "hi"},
    {"text": "@user nella sfortuna sei fortunata ..", "label": "it"},
    {"text": "математических школ в виде грозовых туч.", "label": "ru"},
    {"text": "No cambiaremos nunca nuestra forma de jugar por un", "label": "es"},
    {"text": "dla tej klasy ani wymogów minimalnych, z wyjątkiem", "label": "pl"},
    {"text": "en todo el mundo, mientras que en España consiguió", "label": "es"},
    {"text": "политики считать надежное обеспечение военной", "label": "ru"},
    {"text": "gogoratzen du, genio alemana delakoaren", "label": "eu"},
    {"text": "Бычий глаз.", "label": "ru"},
    {"text": "Opeření se v pravidelných obdobích obnovuje", "label": "cs"},
    {"text": "I no és només la seva, es tracta d'una resposta", "label": "ca"},
    {"text": "오경을 가르쳤다 .", "label": "ko"},
    {"text": "Nach der so genannten Start-up-Periode vergibt die", "label": "de"},
    {"text": "Saulista huomasi jo lapsena , että hänellä on", "label": "fi"},
    {"text": "Министерство культуры сочло нецелесообразным, и", "label": "ru"},
    {"text": "znepřátelené tábory v Tádžikistánu předseda", "label": "cs"},
    {"text": "καὶ ἦν ὁ λαὸς προσδοκῶν τὸν Ζαχαρίαν καὶ ἐθαύμαζον", "label": "grc"},
    {"text": "Вечером, в продукте, этот же человек говорил о", "label": "ru"},
    {"text": "lugar á formación de xuizos máis complexos.", "label": "gl"},
    {"text": "cheaper, in the end?", "label": "en"},
    {"text": "الوزارة في شأن صفقات بيع الشركات العامة التي تم", "label": "ar"},
    {"text": "tärkeintä elämässäni .", "label": "fi"},
    {"text": "Виконання Мінських угод було заблоковано Росією та", "label": "uk"},
    {"text": "Aby szybko rozpoznać żołnierzy desantu, należy", "label": "pl"},
    {"text": "Bankengeschäfte liegen vorn , sagte Strothmann .", "label": "de"},
    {"text": "продолжение работы.", "label": "ru"},
    {"text": "Metro AG plant Online-Offensive", "label": "de"},
    {"text": "nu vor veni, și să vor osîndi, aceia nu pot porni", "label": "ro"},
    {"text": "Ich denke , es geht in Wirklichkeit darum , NT bei", "label": "de"},
    {"text": "de turism care încasează contravaloarea", "label": "ro"},
    {"text": "Aurkaria itotzea da helburua, baloia lapurtu eta", "label": "eu"},
    {"text": "com a centre de formació en Tecnologies de la", "label": "ca"},
    {"text": "oportet igitur quod omne agens in agendo intendat", "label": "la"},
    {"text": "Jerzego Andrzejewskiego, oparty na chińskich", "label": "pl"},
    {"text": "sau một vài câu chuyện xã giao không dính dáng tới", "label": "vi"},
    {"text": "что экономическому прорыву жесткий авторитарный", "label": "ru"},
    {"text": "DRAM-Preisen scheinen DSPs ein", "label": "de"},
    {"text": "Jos dajan nubbái: Mana!", "label": "sme"},
    {"text": "toți carii ascultară de el să răsipiră.", "label": "ro"},
    {"text": "odpowiedzialności, które w systemie własności", "label": "pl"},
    {"text": "Dvomesečno potovanje do Mollenda v Peruju je", "label": "sl"},
    {"text": "d'entre les agències internacionals.", "label": "ca"},
    {"text": "Fahrzeugzugangssysteme gefertigt und an viele", "label": "de"},
    {"text": "in an answer to the sharers' petition in Cuthbert", "label": "en"},
    {"text": "Europa-Domain per Verordnung zu regeln .", "label": "de"},
    {"text": "#Balotelli. Su ebay prezzi stracciati per Silvio", "label": "it"},
    {"text": "Ne na košickém trávníku, ale už včera v letadle se", "label": "cs"},
    {"text": "zaměstnanosti a investičních strategií.", "label": "cs"},
    {"text": "Tatínku, udělej den", "label": "cs"},
    {"text": "frecuencia con Mary.", "label": "es"},
    {"text": "Свеаборге.", "label": "ru"},
    {"text": "opatření slovenské strany o certifikaci nejvíce", "label": "cs"},
    {"text": "En todas me decían: 'Espera que hagamos un estudio", "label": "es"},
    {"text": "Die Demonstration sollte nach Darstellung der", "label": "de"},
    {"text": "Ci vorrà un assoluto rigore se dietro i disavanzi", "label": "it"},
    {"text": "Tatínku, víš, že Honzovi odešla maminka?", "label": "cs"},
    {"text": "Die Anzahl der Rechner wuchs um % auf und die", "label": "de"},
    {"text": "האמריקאית על אדמת סעודיה עלולה לסבך את ישראל, אין", "label": "he"},
    {"text": "Volán Egyesülés, a Közlekedési Főfelügyelet is.", "label": "hu"},
    {"text": "Schejbala, který stejnou hru s velkým úspěchem", "label": "cs"},
    {"text": "depends on the data type of the field.", "label": "en"},
    {"text": "Umsatzwarnung zu Wochenbeginn zeitweise auf ein", "label": "de"},
    {"text": "niin heti nukun .", "label": "fi"},
    {"text": "Mobilfunkunternehmen gegen die Anwendung der so", "label": "de"},
    {"text": "sapessi le intenzioni del governo Monti e dell'UE", "label": "it"},
    {"text": "Di chi è figlia Martine Aubry?", "label": "it"},
    {"text": "avec le reste du monde.", "label": "fr"},
    {"text": "Այդ մաքոքը ինքնին նոր չէ, աշխարհը արդեն մի քանի", "label": "hy"},
    {"text": "și în cazul destrămării cenaclului.", "label": "ro"},
    {"text": "befriedigen kann , und ohne die auftretenden", "label": "de"},
    {"text": "Κύκνον τ̓ ἐξεναρεῖν καὶ ἀπὸ κλυτὰ τεύχεα δῦσαι.", "label": "grc"},
    {"text": "færdiguddannede.", "label": "da"},
    {"text": "Schmidt war Sohn eines Rittergutsbesitzers.", "label": "de"},
    {"text": "и вдаша попадь ѡпрати", "label": "orv"},
    {"text": "cine nu știe învățătură”.", "label": "ro"},
    {"text": "détacha et cette dernière tenta de tuer le jeune", "label": "fr"},
    {"text": "Der har saka også ei lengre forhistorie.", "label": "nn"},
    {"text": "Pieprz roztłuc w moździerzu, dodać do pasty,", "label": "pl"},
    {"text": "Лежа за гребнем оврага, как за бруствером, Ушаков", "label": "ru"},
    {"text": "gesucht habe, vielen Dank nochmals!", "label": "de"},
    {"text": "инструментальных сталей, повышения", "label": "ru"},
    {"text": "im Halbfinale Patrick Smith und im Finale dann", "label": "de"},
    {"text": "البنوك التريث في منح تسهيلات جديدة لمنتجي حديد", "label": "ar"},
    {"text": "una bolsa ventral, la cual se encuentra debajo de", "label": "es"},
    {"text": "za SETimes.", "label": "sr"},
    {"text": "de Irak, a un piloto italiano que había violado el", "label": "es"},
    {"text": "Er könne sich nicht erklären , wie die Zeitung auf", "label": "de"},
    {"text": "Прохорова.", "label": "ru"},
    {"text": "la democrazia perde sulla tecnocrazia? #", "label": "it"},
    {"text": "entre ambas instituciones, confirmó al medio que", "label": "es"},
    {"text": "Austlandet, vart det funne om lag førti", "label": "nn"},
    {"text": "уровнями власти.", "label": "ru"},
    {"text": "Dá tedy primáři úplatek, a často ne malý.", "label": "cs"},
    {"text": "brillantes del acto, al llevar a cabo en el", "label": "es"},
    {"text": "eee druga zadeva je majhen priročen gre kamorkoli", "label": "sl"},
    {"text": "Das ATX-Board paßt in herkömmliche PC-ATX-Gehäuse", "label": "de"},
    {"text": "Za vodné bylo v prvním pololetí zaplaceno v ČR", "label": "cs"},
    {"text": "Даже на полсантиметра.", "label": "ru"},
    {"text": "com la del primer tinent d'alcalde en funcions,", "label": "ca"},
    {"text": "кількох оповідань в цілості — щось на зразок того", "label": "uk"},
    {"text": "sed ad divitias congregandas, vel superfluum", "label": "la"},
    {"text": "Norma Talmadge, spela mot Valentino i en version", "label": "sv"},
    {"text": "Dlatego chciał się jej oświadczyć w niezwykłym", "label": "pl"},
    {"text": "будут выступать на одинаковых снарядах.", "label": "ru"},
    {"text": "Orang-orang terbunuh di sana.", "label": "id"},
    {"text": "لدى رايت شقيق اسمه أوسكار, وهو يعمل كرسام للكتب", "label": "ar"},
    {"text": "Wirklichkeit verlagerten und kaum noch", "label": "de"},
    {"text": "как перемешивают костяшки перед игрой в домино, и", "label": "ru"},
    {"text": "В средине дня, когда солнце светило в нашу", "label": "ru"},
    {"text": "d'aventure aux rôles de jeune romantique avec une", "label": "fr"},
    {"text": "My teď hledáme organizace, jež by s námi chtěly", "label": "cs"},
    {"text": "Urteilsfähigkeit einbüßen , wenn ich eigene", "label": "de"},
    {"text": "sua appartenenza anche a voci diverse da quella in", "label": "it"},
    {"text": "Aufträge dieses Jahr verdoppeln werden .", "label": "de"},
    {"text": "M.E.: Miała szanse mnie odnaleźć, gdyby naprawdę", "label": "pl"},
    {"text": "secundum contactum virtutis, cum careat dimensiva", "label": "la"},
    {"text": "ezinbestekoa dela esan zuen.", "label": "eu"},
    {"text": "Anek hurbiltzeko eskatzen zion besaulkitik, eta", "label": "eu"},
    {"text": "perfectius alio videat, quamvis uterque videat", "label": "la"},
    {"text": "Die Strecke war anspruchsvoll und führte unter", "label": "de"},
    {"text": "саморазоблачительным уроком, западные СМИ не", "label": "ru"},
    {"text": "han representerer radikal islamisme .", "label": "no"},
    {"text": "Què s'hi respira pel que fa a la reforma del", "label": "ca"},
    {"text": "previsto para também ser desconstruido.", "label": "pt"},
    {"text": "Ὠκεανοῦ βαθυκόλποις ἄνθεά τ̓ αἰνυμένην, ῥόδα καὶ", "label": "grc"},
    {"text": "para jovens de a anos nos Cieps.", "label": "pt"},
    {"text": "संघर्ष को अंजाम तक पहुंचाने का ऐलान किया है ।", "label": "hi"},
    {"text": "objeví i u nás.", "label": "cs"},
    {"text": "kvitteringer.", "label": "da"},
    {"text": "This report is no exception.", "label": "en"},
    {"text": "Разлепват доносниците до избирателните списъци", "label": "bg"},
    {"text": "anderem ihre Bewegungsfreiheit in den USA", "label": "de"},
    {"text": "Ñu tegoon ca kaw gor ña ay njotti bopp yu kenn", "label": "wo"},
    {"text": "Struktur kann beispielsweise der Schwerpunkt mehr", "label": "de"},
    {"text": "% la velocidad permitida, la sanción es muy grave.", "label": "es"},
    {"text": "Teles-Einstieg in ADSL-Markt", "label": "de"},
    {"text": "ettekäändeks liiga suure osamaksu.", "label": "et"},
    {"text": "als Indiz für die geänderte Marktpolitik des", "label": "de"},
    {"text": "quod quidem aperte consequitur ponentes", "label": "la"},
    {"text": "de negociación para el próximo de junio.", "label": "es"},
    {"text": "Tyto důmyslné dekorace doznaly v poslední době", "label": "cs"},
    {"text": "največjega uspeha doslej.", "label": "sl"},
    {"text": "Paul Allen je jedan od suosnivača Interval", "label": "hr"},
    {"text": "Federal (Seac / DF) eo Sindicato das Empresas de", "label": "pt"},
    {"text": "Quartal mit . Mark gegenüber dem gleichen Quartal", "label": "de"},
    {"text": "otros clubes y del Barça B saldrán varios", "label": "es"},
    {"text": "Jaskula (Pol.) -", "label": "cs"},
    {"text": "umožnily říci, že je možné přejít k mnohem", "label": "cs"},
    {"text": "اعلن الجنرال تومي فرانكس قائد القوات الامريكية", "label": "ar"},
    {"text": "Telekom-Chef Ron Sommer und der Vorstandssprecher", "label": "de"},
    {"text": "My, jako průmyslový a finanční holding, můžeme", "label": "cs"},
    {"text": "voorlichting onder andere betrekking kan hebben:", "label": "nl"},
    {"text": "Hinrichtung geistig Behinderter applaudiert oder", "label": "de"},
    {"text": "wie beispielsweise Anzahl erzielte Klicks ,", "label": "de"},
    {"text": "Intel-PC-SDRAM-Spezifikation in der Version . (", "label": "de"},
    {"text": "plângere în termen de zile de la comunicarea", "label": "ro"},
    {"text": "и Испания ще изгубят втория си комисар в ЕК.", "label": "bg"},
    {"text": "इसके चलते इस आदिवासी जनजाति का क्षरण हो रहा है ।", "label": "hi"},
    {"text": "aunque se mostró contrario a establecer un", "label": "es"},
    {"text": "des letzten Jahres von auf Millionen Euro .", "label": "de"},
    {"text": "Ankara se također poziva da u cijelosti ratificira", "label": "hr"},
    {"text": "herunterlädt .", "label": "de"},
    {"text": "стрессовую ситуацию для организма, каковой", "label": "ru"},
    {"text": "Státního shromáždění (parlamentu).", "label": "cs"},
    {"text": "diskutieren , ob und wie dieser Dienst weiterhin", "label": "de"},
    {"text": "Verbindungen zu FPÖ-nahen Polizisten gepflegt und", "label": "de"},
    {"text": "Pražského volebního lídra ovšem nevybírá Miloš", "label": "cs"},
    {"text": "Nach einem Bericht der Washington Post bleibt das", "label": "de"},
    {"text": "للوضع آنذاك، لكني في قرارة نفسي كنت سعيداً لما", "label": "ar"},
    {"text": "не желаят запазването на статуквото.", "label": "bg"},
    {"text": "Offenburg gewesen .", "label": "de"},
    {"text": "ἐὰν ὑμῖν εἴπω οὐ μὴ πιστεύσητε", "label": "grc"},
    {"text": "all'odiato compagno di squadra Prost, il quale", "label": "it"},
    {"text": "historischen Gänselieselbrunnens.", "label": "de"},
    {"text": "למידע מלווייני הריגול האמריקאיים העוקבים אחר", "label": "he"},
    {"text": "οὐδὲν ἄρα διαφέρεις Ἀμάσιος τοῦ Ἠλείου, ὃν", "label": "grc"},
    {"text": "movementos migratorios.", "label": "gl"},
    {"text": "Handy und ein Spracherkennungsprogramm sämtliche", "label": "de"},
    {"text": "Kümne aasta jooksul on Eestisse ohjeldamatult", "label": "et"},
    {"text": "H.G. Bücknera.", "label": "pl"},
    {"text": "protiv krijumčarenja, ili pak traženju ukidanja", "label": "hr"},
    {"text": "Topware-Anteile mehrere Millionen Mark gefordert", "label": "de"},
    {"text": "Maar de mensen die nu over Van Dijk bij FC Twente", "label": "nl"},
    {"text": "poidan experimentar as percepcións do interesado,", "label": "gl"},
    {"text": "Miał przecież w kieszeni nóż.", "label": "pl"},
    {"text": "Avšak žádná z nich nepronikla za hranice přímé", "label": "cs"},
    {"text": "esim. helpottamalla luottoja muiden", "label": "fi"},
    {"text": "Podle předběžných výsledků zvítězila v", "label": "cs"},
    {"text": "Nicht nur das Web-Frontend , auch die", "label": "de"},
    {"text": "Regierungsinstitutionen oder Universitäten bei", "label": "de"},
    {"text": "Խուլեն Լոպետեգիին, պատճառաբանելով, որ վերջինս", "label": "hy"},
    {"text": "Афганистана, где в последние дни идут ожесточенные", "label": "ru"},
    {"text": "лѧхове же не идоша", "label": "orv"},
    {"text": "Mit Hilfe von IBMs Chip-Management-Systemen sollen", "label": "de"},
    {"text": ", als Manager zu Telefonica zu wechseln .", "label": "de"},
    {"text": "którym zajmuje się człowiek, zmienia go i pozwala", "label": "pl"},
    {"text": "činí kyperských liber, to je asi USD.", "label": "cs"},
    {"text": "Studienplätze getauscht werden .", "label": "de"},
    {"text": "учёных, орнитологов признают вид.", "label": "ru"},
    {"text": "acordare a concediilor prevăzute de legislațiile", "label": "ro"},
    {"text": "at større innsats for fornybar, berekraftig energi", "label": "nn"},
    {"text": "Politiet veit ikkje kor mange personar som deltok", "label": "nn"},
    {"text": "offentligheten av unge , sinte menn som har", "label": "no"},
    {"text": "însuși în jurul lapunei, care încet DISPARE în", "label": "ro"},
    {"text": "O motivo da decisão é evitar uma sobrecarga ainda", "label": "pt"},
    {"text": "El Apostolado de la prensa contribuye en modo", "label": "es"},
    {"text": "Teltow ( Kreis Teltow-Fläming ) ist Schmitt einer", "label": "de"},
    {"text": "grozījumus un iesniegt tos Apvienoto Nāciju", "label": "lv"},
    {"text": "Gestalt einer deutschen Nationalmannschaft als", "label": "de"},
    {"text": "D überholt zu haben , konterte am heutigen Montag", "label": "de"},
    {"text": "Softwarehersteller Oracle hat im dritten Quartal", "label": "de"},
    {"text": "Během nich se ekonomické podmínky mohou radikálně", "label": "cs"},
    {"text": "Dziki kot w górach zeskakuje z kamienia.", "label": "pl"},
    {"text": "Ačkoliv ligový nováček prohrál, opět potvrdil, že", "label": "cs"},
    {"text": "des Tages , Portraits internationaler Stars sowie", "label": "de"},
    {"text": "Communicator bekannt wurde .", "label": "de"},
    {"text": "τῷ δ’ ἄρα καὶ αὐτῷ ἡ γυνή ἐπίτεξ ἐοῦσα πᾶσαν", "label": "grc"},
    {"text": "Triadú tenia, mentre redactava 'Dies de memòria',", "label": "ca"},
    {"text": "دسته‌جمعی در درخشندگی ماه سیم‌گون زمزمه ستاینده و", "label": "fa"},
    {"text": "Книгу, наполненную мелочной заботой об одежде,", "label": "ru"},
    {"text": "putares canem leporem persequi.", "label": "la"},
    {"text": "В дальнейшем эта яркость слегка померкла, но в", "label": "ru"},
    {"text": "offizielles Verfahren gegen die Telekom", "label": "de"},
    {"text": "podrían haber sido habitantes de la Península", "label": "es"},
    {"text": "Grundlage für dieses Verfahren sind spezielle", "label": "de"},
    {"text": "Rechtsausschuß vorgelegten Entwurf der Richtlinie", "label": "de"},
    {"text": "Im so genannten Portalgeschäft sei das Unternehmen", "label": "de"},
    {"text": "ⲏ ⲉⲓϣⲁⲛϥⲓ ⲛⲉⲓⲇⲱⲗⲟⲛ ⲉⲧϩⲙⲡⲉⲕⲏⲓ ⲙⲏ ⲉⲓⲛⲁϣϩⲱⲡ ⲟⲛ ⲙⲡⲣⲏ", "label": "cop"},
    {"text": "juego podían matar a cualquier herbívoro, pero", "label": "es"},
    {"text": "Nach Angaben von Axent nutzen Unternehmen aus der", "label": "de"},
    {"text": "hrdiny Havlovy Zahradní slavnosti (premiéra ) se", "label": "cs"},
    {"text": "Een zin van heb ik jou daar", "label": "nl"},
    {"text": "hat sein Hirn an der CeBIT-Kasse vergessen .", "label": "de"},
    {"text": "καὶ τοὺς ἐκπλαγέντας οὐκ ἔχειν ἔτι ἐλεγχομένους", "label": "grc"},
    {"text": "nachgewiesenen langfristigen Kosten , sowie den im", "label": "de"},
    {"text": "jučer nakon četiri dana putovanja u Helsinki.", "label": "hr"},
    {"text": "pašto paslaugos teikėjas gali susitarti su", "label": "lt"},
    {"text": "В результате, эти золотые кадры переходят из одной", "label": "ru"},
    {"text": "द फाइव-ईयर एंगेजमेंट में अभिनय किया जिसमें जैसन", "label": "hi"},
    {"text": "výpis o počtu akcií.", "label": "cs"},
    {"text": "Enfin, elles arrivent à un pavillon chinois", "label": "fr"},
    {"text": "Tentu saja, tren yang berhubungandengan", "label": "id"},
    {"text": "Arbeidarpartiet og SV har sikra seg fleirtal mot", "label": "nn"},
    {"text": "eles: 'Tudo isso está errado' , disse um", "label": "pt"},
    {"text": "The islands are in their own time zone, minutes", "label": "en"},
    {"text": "Auswahl debütierte er am .", "label": "de"},
    {"text": "Bu komisyonlar, arazilerini satın almak için", "label": "tr"},
    {"text": "Geschütze gegen Redmond aufgefahren .", "label": "de"},
    {"text": "Time scything the hours, but at the top, over the", "label": "en"},
    {"text": "Di musim semi , berharap mengadaptasi Tintin untuk", "label": "id"},
    {"text": "крупнейшей геополитической катастрофой XX века.", "label": "ru"},
    {"text": "Rajojen avaaminen ei suju ongelmitta .", "label": "fi"},
    {"text": "непроницаемым, как для СССР.", "label": "ru"},
    {"text": "Ma non mancano le polemiche.", "label": "it"},
    {"text": "Internet als Ort politischer Diskussion und auch", "label": "de"},
    {"text": "incomplets.", "label": "ca"},
    {"text": "Su padre luchó al lado de Luis Moya, primer Jefe", "label": "es"},
    {"text": "informazione.", "label": "it"},
    {"text": "Primacom bietet für Telekom-Kabelnetz", "label": "de"},
    {"text": "Oświadczenie prezydencji w imieniu Unii", "label": "pl"},
    {"text": "foran rattet i familiens gamle Baleno hvis døra på", "label": "no"},
    {"text": "[speaker:laughter]", "label": "sl"},
    {"text": "Dog med langt mindre utstyr med seg.", "label": "nn"},
    {"text": "dass es nicht schon mit der anfänglichen", "label": "de"},
    {"text": "इस पर दोनों पक्षों में नोकझोंक शुरू हो गई ।", "label": "hi"},
    {"text": "کے ترجمان منیش تیواری اور دگ وجئے سنگھ نے بھی یہ", "label": "ur"},
    {"text": "dell'Assemblea Costituente che posseggono i", "label": "it"},
    {"text": "и аште вьси съблазнѧтъ сѧ нъ не азъ", "label": "cu"},
    {"text": "In Irvine hat auch das Logistikunternehmen Atlas", "label": "de"},
    {"text": "законодательных норм, принимаемых существующей", "label": "ru"},
    {"text": "Κροίσῳ προτείνων τὰς χεῖρας ἐπικατασφάξαι μιν", "label": "grc"},
    {"text": "МИНУСЫ: ИНФЛЯЦИЯ И КРИЗИС В ЖИВОТНОВОДСТВЕ.", "label": "ru"},
    {"text": "unterschiedlicher Meinung .", "label": "de"},
    {"text": "Jospa joku ystävällinen sielu auttaisi kassieni", "label": "fi"},
    {"text": "Añadió que, en el futuro se harán otros", "label": "es"},
    {"text": "Sessiz tonlama hem Fince, hem de Kuzey Sami", "label": "tr"},
    {"text": "nicht ihnen gehört und sie nicht alles , was sie", "label": "de"},
    {"text": "Etelästä Kuivajärveen laskee Tammelan Liesjärvestä", "label": "fi"},
    {"text": "ICANNs Vorsitzender Vint Cerf warb mit dem Hinweis", "label": "de"},
    {"text": "Norsk politikk frå til kan dermed, i", "label": "nn"},
    {"text": "Głosowało posłów.", "label": "pl"},
    {"text": "Danny Jones -- smithjones@ev.net", "label": "en"},
    {"text": "sebeuvědomění moderní civilizace sehrála lučavka", "label": "cs"},
    {"text": "относительно спокойный сон: тому гарантия", "label": "ru"},
    {"text": "A halte voiz prist li pedra a crïer", "label": "fro"},
    {"text": "آن‌ها امیدوارند این واکسن به‌زودی در دسترس بیماران", "label": "fa"},
    {"text": "vlastní důstojnou vousatou tváří.", "label": "cs"},
    {"text": "ora aprire la strada a nuove cause e alimentare il", "label": "it"},
    {"text": "Die Zahl der Vielleser nahm von auf Prozent zu ,", "label": "de"},
    {"text": "Finanzvorstand von Hotline-Dienstleister InfoGenie", "label": "de"},
    {"text": "entwickeln .", "label": "de"},
    {"text": "incolumità pubblica.", "label": "it"},
    {"text": "lehtija televisiomainonta", "label": "fi"},
    {"text": "joistakin kohdista eri mieltä.", "label": "fi"},
    {"text": "Hlavně anglická nezávislá scéna, Dead Can Dance,", "label": "cs"},
    {"text": "pásmech od do bodů bodové stupnice.", "label": "cs"},
    {"text": "Zu Beginn des Ersten Weltkrieges zählte das", "label": "de"},
    {"text": "Així van sorgir, damunt els antics cementiris,", "label": "ca"},
    {"text": "In manchem Gedicht der spätern Alten, wie zum", "label": "de"},
    {"text": "gaweihaida jah insandida in þana fairƕu jus qiþiþ", "label": "got"},
    {"text": "Beides sollte gelöscht werden!", "label": "de"},
    {"text": "modifiqués la seva petició inicial de anys de", "label": "ca"},
    {"text": "В день открытия симпозиума состоялась закладка", "label": "ru"},
    {"text": "tõestatud.", "label": "et"},
    {"text": "ἵππῳ πίπτει αὐτοῦ ταύτῃ", "label": "grc"},
    {"text": "bisher nie enttäuscht!", "label": "de"},
    {"text": "De bohte ollu tuollárat ja suttolaččat ja", "label": "sme"},
    {"text": "Klarsignal från röstlängdsläsaren, tre tryck i", "label": "sv"},
    {"text": "Tvůrcem nového termínu je Joseph Fisher.", "label": "cs"},
    {"text": "Nie miałem czasu na reakcję twierdzi Norbert,", "label": "pl"},
    {"text": "potentia Schöpfer.", "label": "de"},
    {"text": "Un poquito caro, pero vale mucho la pena;", "label": "es"},
    {"text": "οὔ τε γὰρ ἴφθιμοι Λύκιοι Δαναῶν ἐδύναντο τεῖχος", "label": "grc"},
    {"text": "vajec, sladového výtažku a některých vitamínových", "label": "cs"},
    {"text": "Настоящие герои, те, чьи истории потом", "label": "ru"},
    {"text": "praesumptio:", "label": "la"},
    {"text": "Olin justkui nende vastutusel.", "label": "et"},
    {"text": "Jokainen keinahdus tuo lähemmäksi hetkeä jolloin", "label": "fi"},
    {"text": "ekonomicky výhodných způsobů odvodnění těžkých,", "label": "cs"},
    {"text": "Poprvé ve své historii dokázala v kvalifikaci pro", "label": "cs"},
    {"text": "zpracovatelského a spotřebního průmyslu bude nutné", "label": "cs"},
    {"text": "Windows CE zu integrieren .", "label": "de"},
    {"text": "Armangué, a través d'un decret, ordenés l'aturada", "label": "ca"},
    {"text": "to, co nás Evropany spojuje, než to, co nás od", "label": "cs"},
    {"text": "ergänzt durch einen gesetzlich verankertes", "label": "de"},
    {"text": "Насчитал, что с начала года всего три дня были", "label": "ru"},
    {"text": "Borisovu tražeći od njega da prihvati njenu", "label": "sr"},
    {"text": "la presenza di ben veleni diversi: . chili di", "label": "it"},
    {"text": "καὶ τῶν ἐκλεκτῶν ἀγγέλων ἵνα ταῦτα φυλάξῃς χωρὶς", "label": "grc"},
    {"text": "pretraživale obližnju bolnicu i stambene zgrade u", "label": "hr"},
    {"text": "An rund Katzen habe Wolf seine Spiele getestet ,", "label": "de"},
    {"text": "investigating since March.", "label": "en"},
    {"text": "Tonböden (Mullböden).", "label": "de"},
    {"text": "Stálý dopisovatel LN v SRN Bedřich Utitz", "label": "cs"},
    {"text": "červnu předložené smlouvy.", "label": "cs"},
    {"text": "πνεύματι ᾧ ἐλάλει", "label": "grc"},
    {"text": ".%의 신장세를 보였다.", "label": "ko"},
    {"text": "Foae verde, foi de nuc, Prin pădure, prin colnic,", "label": "ro"},
    {"text": "διαπέμψας ἄλλους ἄλλῃ τοὺς μὲν ἐς Δελφοὺς ἰέναι", "label": "grc"},
    {"text": "المسلمين أو أي تيار سياسي طالما عمل ذلك التيار في", "label": "ar"},
    {"text": "As informações são da Dow Jones.", "label": "pt"},
    {"text": "Milliarde DM ausgestattet sein .", "label": "de"},
    {"text": "De utgår fortfarande från att kvinnans jämlikhet", "label": "sv"},
    {"text": "Sneeuw maakte in Davos bij de voorbereiding een", "label": "nl"},
    {"text": "De ahí que en este mercado puedan negociarse", "label": "es"},
    {"text": "intenzívnějšímu sbírání a studiu.", "label": "cs"},
    {"text": "और औसकर ४.० पैकेज का प्रयोग किया गया है ।", "label": "hi"},
    {"text": "Adipati Kuningan karena Kuningan menjadi bagian", "label": "id"},
    {"text": "Svako je bar jednom poželeo da mašine prosto umeju", "label": "sr"},
    {"text": "Im vergangenen Jahr haben die Regierungen einen", "label": "de"},
    {"text": "durat motus, aliquid fit et non est;", "label": "la"},
    {"text": "Dominować będą piosenki do tekstów Edwarda", "label": "pl"},
    {"text": "beantwortet .", "label": "de"},
    {"text": "О гуманитариях было кому рассказывать, а вот за", "label": "ru"},
    {"text": "Helsingin kaupunki riitautti vuokrasopimuksen", "label": "fi"},
    {"text": "chợt tan biến.", "label": "vi"},
    {"text": "avtomobil ločuje od drugih.", "label": "sl"},
    {"text": "Congress has proven itself ineffective as a body.", "label": "en"},
    {"text": "मैक्सिको ने इस तरह का शो इस समय आयोजित करने का", "label": "hi"},
    {"text": "No minimum order amount.", "label": "en"},
    {"text": "Convertassa .", "label": "fi"},
    {"text": "Как это можно сделать?", "label": "ru"},
    {"text": "tha mi creidsinn gu robh iad ceart cho saor shuas", "label": "gd"},
    {"text": "실제 일제는 이런 만해의 논리를 묵살하고 한반도를 침략한 다음 , 이어 만주를 침략하고", "label": "ko"},
    {"text": "Da un semplice richiamo all'ordine fino a grandi", "label": "it"},
    {"text": "pozoruhodný nejen po umělecké stránce, jež", "label": "cs"},
    {"text": "La comida y el servicio aprueban.", "label": "es"},
    {"text": "again, connected not with each other but to the", "label": "en"},
    {"text": "Protokol výslovně stanoví, že nikdo nemůže být", "label": "cs"},
    {"text": "ఒక విషయం అడగాలని ఉంది .", "label": "te"},
    {"text": "Безгранично почитая дирекцию, ловя на лету каждое", "label": "ru"},
    {"text": "rovnoběžných růstových vrstev, zůstávají krychlové", "label": "cs"},
    {"text": "प्रवेश और पूर्व प्रधानमंत्री लाल बहादुर शास्त्री", "label": "hi"},
    {"text": "Bronzen medaille in de Europese marathon.", "label": "nl"},
    {"text": "- gadu vecumā viņi to nesaprot.", "label": "lv"},
    {"text": "Realizó sus estudios primarios en la Escuela Julia", "label": "es"},
    {"text": "cuartos de final, su clasificación para la final a", "label": "es"},
    {"text": "Sem si pro něho přiletí americký raketoplán, na", "label": "cs"},
    {"text": "Way to go!", "label": "en"},
    {"text": "gehört der neuen SPD-Führung unter Parteichef", "label": "de"},
    {"text": "Somit simuliert der Player mit einer GByte-Platte", "label": "de"},
    {"text": "Berufung auf kommissionsnahe Kreise , die bereits", "label": "de"},
    {"text": "Dist Clarïen", "label": "fro"},
    {"text": "Schon nach den Gerüchten , die Telekom wolle den", "label": "de"},
    {"text": "Software von NetObjects ist nach Angaben des", "label": "de"},
    {"text": "si enim per legem iustitia ergo Christus gratis", "label": "la"},
    {"text": "ducerent in ipsam magis quam in corpus christi,", "label": "la"},
    {"text": "Neustar-Melbourne-IT-Partnerschaft NeuLevel .", "label": "de"},
    {"text": "forderte dagegen seine drastische Verschärfung.", "label": "de"},
    {"text": "pemmican på hundrede forskellige måder.", "label": "da"},
    {"text": "Lehån, själv matematiklärare, visar hur den nya", "label": "sv"},
    {"text": "I highly recommend his shop.", "label": "en"},
    {"text": "verità, giovani fedeli prostratevi #amen", "label": "it"},
    {"text": "उत्तर प्रदेश के अध्यक्ष पद से हटाए गए विनय कटियार", "label": "hi"},
    {"text": "() روزی مےں کشادگی ہوتی ہے۔", "label": "ur"},
    {"text": "Prozessorgeschäft profitieren kann , stellen", "label": "de"},
    {"text": "školy začalo počítat pytle s moukou a zjistilo, že", "label": "cs"},
    {"text": "प्रभावशाली पर गैर सरकारी लोगों के घरों में भी", "label": "hi"},
    {"text": "geschichtslos , oder eine Farce , wie sich", "label": "de"},
    {"text": "Ústrednými mocnosťami v marci však spôsobilo, že", "label": "sk"},
    {"text": "التسليح بدون مبرر، واستمرار الأضرار الناجمة عن فرض", "label": "ar"},
    {"text": "Například Pedagogická fakulta Univerzity Karlovy", "label": "cs"},
    {"text": "nostris ut eriperet nos de praesenti saeculo", "label": "la"}]
    
    nlp = Pipeline(dir=TEST_MODELS_DIR, lang="multilingual", processors="langid")
    docs = [Document([], text=example["text"]) for example in examples]
    gold_labels = [example["label"] for example in examples]
    nlp(docs)
    accuracy = sum([(doc.lang == label) for doc,label in zip(docs,gold_labels)])/len(docs)
    assert accuracy >= 0.98


def test_text_cleaning():
    """
    Basic test of cleaning text
    """
    docs = ["Bonjour le monde! #thisisfrench #ilovefrance",
            "Bonjour le monde! https://t.co/U0Zjp3tusD"]
    docs = [Document([], text=text) for text in docs]
    
    nlp = Pipeline(dir=TEST_MODELS_DIR, lang="multilingual", processors="langid")
    nlp(docs)
    assert [doc.lang for doc in docs] == ["it", "it"]
    
    nlp = Pipeline(dir=TEST_MODELS_DIR, lang="multilingual", processors="langid", langid_clean_text=True)
    assert nlp.processors["langid"]._clean_text
    nlp(docs)
    assert [doc.lang for doc in docs] == ["fr", "fr"]

def test_lang_subset():
    """
    Basic test of restricting output to subset of languages
    """
    docs = ["Bonjour le monde! #thisisfrench #ilovefrance",
            "Bonjour le monde! https://t.co/U0Zjp3tusD"]
    docs = [Document([], text=text) for text in docs]
    
    nlp = Pipeline(dir=TEST_MODELS_DIR, lang="multilingual", processors="langid")
    nlp(docs)
    assert [doc.lang for doc in docs] == ["it", "it"]
    
    nlp = Pipeline(dir=TEST_MODELS_DIR, lang="multilingual", processors="langid", langid_lang_subset=["en","fr"])
    assert nlp.processors["langid"]._model.lang_subset == ["en", "fr"]
    nlp(docs)
    assert [doc.lang for doc in docs] == ["fr", "fr"]
    
    nlp = Pipeline(dir=TEST_MODELS_DIR, lang="multilingual", processors="langid", langid_lang_subset=["en"])
    assert nlp.processors["langid"]._model.lang_subset == ["en"]
    nlp(docs)
    assert [doc.lang for doc in docs] == ["en", "en"]

def test_multilingual_pipeline():
    """
    Basic test of multilingual pipeline
    """
    english_text = "This is an English sentence."
    english_deps_gold = "\n".join((
        "('This', 5, 'nsubj')",
        "('is', 5, 'cop')",
        "('an', 5, 'det')",
        "('English', 5, 'amod')",
        "('sentence', 0, 'root')",
        "('.', 5, 'punct')"
    ))

    french_text = "C'est une phrase française."
    french_deps_gold = "\n".join((
        "(\"C'\", 4, 'nsubj')",
        "('est', 4, 'cop')",
        "('une', 4, 'det')",
        "('phrase', 0, 'root')",
        "('française', 4, 'amod')",
        "('.', 4, 'punct')"
    ))

    nlp = MultilingualPipeline(model_dir=TEST_MODELS_DIR)
    docs = [english_text, french_text]
    docs = nlp(docs)

    assert docs[0].lang == "en"
    assert docs[0].sentences[0].dependencies_string() == english_deps_gold
    assert docs[1].lang == "fr"
    assert docs[1].sentences[0].dependencies_string() == french_deps_gold

