# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['cbor_json']

package_data = \
{'': ['*']}

install_requires = \
['cbor2>=5.4.0,<6.0.0']

setup_kwargs = {
    'name': 'cbor-json',
    'version': '0.0.1',
    'description': "Three-way conversion between 'native' Python data representation, CBOR, and JSON",
    'long_description': '[![PyPI pyversions](https://img.shields.io/pypi/pyversions/ansicolortags.svg)](https://pypi.python.org/pypi/cbor-json/)\n[![PyPI status](https://img.shields.io/pypi/status/ansicolortags.svg)](https://pypi.python.org/pypi/cbor-json/)\n[![GitHub license](https://img.shields.io/github/license/Naereen/StrapDown.js.svg)](https://github.com/amaslyaev/cbor-json/blob/master/LICENSE)\n\n# cbor-json\n\nThis library provides three-way conversion between "native" Python data representation, [CBOR](https://cbor.io/) (Concise Binary Object Representation), and human- and machine-readable JSON (and also YAML, TOML, etc.) form.\n\n## Installation and import\nYou need at least Python 3.6 to use this library\n```shell\npip install cbor-json\n```\nImport:\n```python\nimport cbor_json\n```\n\n## Objective\nBinary data form is better than text because it consumes less space, and what is more important, better keeps its own identity. Sequence of zeros and ones equals itself and nothing else. Working with text data we always have to take into account that some "small" and "invisible" changes can occur at any stage of transmission/processing - changes in line ends, encoding adjustments, "illegal" symbols replacements, trailing line breaks removals, etc.\n\nText data representation is better because we can read it, understand and edit.\n\nThis library provides a simple way to enjoy benefits of both approaches: convert any valid CBOR to JSON (or YAML or TOML if you like), inspect it, make necessary changes, and convert to CBOR back.\n\n## Data forms\n1. **CBOR** - binary data format with excellent support in wide variety of languages including Python, C, C++, C#, Java, JavaScript, Go, Rust, PHP, Scala and many others. This library uses [cbor2](https://pypi.org/project/cbor2/) library as CBOR codec.\n2. **Native** Python representation - data as we use it in our scripts - strings, numbers, booleans, lists, dictionaries, sets, "bytes", datetimes. Also this library provides a possibility to implement three-way serialization (to/from CBOR and JSON) for custom classes.\n3. **Jsonable** representation - form that can be dumped to JSON without additional tricks.\n\n## Usage\nConversions:\n- **native_from_cbor** - decodes CBOR to the "native" Python representation\n- **cbor_from_native** - encodes Python data to CBOR\n- **jsonable_from_native** - transforms Python data to the form that can be passed to <code>json.dump</code> function without exceptions\n- **native_from_jsonable** - transformation back from jsonable form\n- **jsonable_from_cbor** and **cbor_from_jsonable** - decoding/encoding CBOR to/from jsonable representation\n\nLet\'s play with it\n```python\n>>> import cbor_json\n>>> c1 = cbor_json.cbor_from_native(\'Hello\')\n>>> c1\nb\'eHello\'\n>>> print(cbor_json.native_from_cbor(c1))\nHello\n>>> import json\n>>> json.dumps(cbor_json.jsonable_from_native(\'hello\'))\n\'"hello"\'\n>>> json.dumps(cbor_json.jsonable_from_cbor(c1))\n\'"Hello"\'\n```\nDump the current datetime to cbor and json:\n```python\n>>> import datetime\n>>> now = datetime.datetime.utcnow()\n>>> now\ndatetime.datetime(2021, 7, 21, 22, 44, 16, 381609)\n>>> cbor_json.cbor_from_native(now)\nb\'\\xc1\\xfbA\\xd8>(\\xd0\\x18lH\'\n>>> cbor_json.cbor_from_native(now).hex()\n\'c1fb41d83e28d0186c48\'\n>>> cbor_json.jsonable_from_native(now)\n{\'$type\': \'datetime\', \'$value\': \'2021-07-21T22:44:16.381609\'}\n>>> print(json.dumps(cbor_json.jsonable_from_native(now), indent=2))\n{\n  "$type": "datetime",\n  "$value": "2021-07-21T22:44:16.381609"\n}\n>>> # Roundtrip: native -> cbor -> json -> native\n>>> c2 = cbor_json.cbor_from_native(now)\n>>> jsoned = json.dumps(cbor_json.jsonable_from_cbor(c2))\n>>> cbor_json.native_from_jsonable(json.loads(jsoned))\ndatetime.datetime(2021, 7, 21, 22, 44, 16, 381609, tzinfo=datetime.timezone.utc)\n```\nNotice <code>tzinfo=datetime.timezone.utc</code> acquired after conversion to/from CBOR.\n\nMore encoding examples:\n```python\n>>> import math\n>>> import decimal\n>>> import fractions\n>>> import uuid\n>>> example = {\'simple values\': [None, True, False], \'numbers\': [0, 123, 0.123, math.nan, fractions.Fraction(1, 3), decimal.Decimal(\'123.45\')], \'id\': uuid.uuid4()}\n>>> cbor_json.cbor_from_native(example).hex()\n\'a3626964d82550962529e447f94237808137d11b1bb033676e756d626572738600187bfb3fbf7ced916872b0f97e00d81e820103c482211930396d73696d706c652076616c75657383f6f5f4\'\n>>> print(json.dumps(cbor_json.jsonable_from_native(example), indent=2))\n{\n  "simple values": [\n    null,\n    true,\n    false\n  ],\n  "numbers": [\n    0,\n    123,\n    0.123,\n    NaN,\n    {\n      "$type": "fraction",\n      "$value": "1/3"\n    },\n    {\n      "$type": "decimal",\n      "$value": "123.45"\n    }\n  ],\n  "id": {\n    "$type": "uuid",\n    "$value": "962529e4-47f9-4237-8081-37d11b1bb033"\n  }\n}\n```\n\nLet\'s try to encode something recursive\n```python\n>>> a1 = [1, ]\n>>> a2 = [2, a1]\n>>> a1.append(a2)\n>>> a1\n[1, [2, [...]]]\n>>> a2\n[2, [1, [...]]]\n>>> cbor_json.cbor_from_native(a1)\nTraceback (most recent call last):\n  File "<stdin>", line 1, in <module>\n  ... cut ...\n  File "***/cbor_json/_cbor_json_codecs.py", line 60, in _cborable_from_native\n    raise ValueError(\'Cannot encode a recursively linked structure\')\nValueError: Cannot encode a recursively linked structure\n```\nIt is not possible. Get rid of recursive links before encoding.\n\nLet\'s encode a pandas dataframe\n```python\n>>> import pandas as pd  # assume it is pip-installed\n>>> dframe = pd.DataFrame({\'name\': [\'John\', \'Jane\'], \'age\': [23, 22]})\n>>> dframe\n   name  age\n0  John   23\n1  Jane   22\n>>> from cbor_json.custom_objects import DataFrameSerialized\n>>> c3 = cbor_json.cbor_from_native(DataFrameSerialized(dframe))\n>>> c3.hex()\n\'d81b8362646682646e616d65636167658282644a6f686e1782644a616e6516\'\n>>> dframe_decoded = pd.DataFrame(cbor_json.native_from_cbor(c3).columns_data())\n>>> dframe_decoded\n   name  age\n0  John   23\n1  Jane   22\n>>> print(json.dumps(cbor_json.jsonable_from_cbor(c3), indent=2))\n{\n  "$type": "custom-object",\n  "$class": "DataFrame (columns, data)",\n  "$class_tag": "df",\n  "$value": [\n    [\n      "name",\n      "age"\n    ],\n    [\n      [\n        "John",\n        23\n      ],\n      [\n        "Jane",\n        22\n      ]\n    ]\n  ]\n}\n```\n\n### Defining your own custom class serialization\n1. Inherit from <code>cbor_json.SerializableToCbor</code>.\n2. In the implementation define cbor_cc_classtag and cbor_cc_descr class variables.\n3. Implement get_cbor_cc_values and put_cbor_cc_values methods.\n4. Call a cbor_json.register_custom_class to register your class for codec.\n\n```python\n>>> class Point(cbor_json.SerializableToCbor):\n...     cbor_cc_classtag = \'p\'\n...     cbor_cc_descr = \'Point (x, y)\'\n...     def __init__(self, x=None, y=None):\n...         self.x, self.y = x, y\n...     def get_cbor_cc_values(self):\n...         return [self.x, self.y]\n...     def put_cbor_cc_values(self, x, y):\n...         self.x, self.y = x, y\n... \n>>> cbor_json.register_custom_class(Point)\n>>> c4 = cbor_json.cbor_from_native(Point(1.23, 4.56))\n>>> c4.hex()\n\'d81b836170fb3ff3ae147ae147aefb40123d70a3d70a3d\'\n>>> cbor_json.native_from_cbor(c4).x\n1.23\n>>> print(json.dumps(cbor_json.jsonable_from_cbor(c4), indent=2))\n{\n  "$type": "custom-object",\n  "$class": "Point (x, y)",\n  "$class_tag": "p",\n  "$value": [\n    1.23,\n    4.56\n  ]\n}\n```\n\n### Guidelines for assigning class tags\n\n1. At the moment these class tags are in use:\n\n| class tag | class | Description |\n| :---:     | :---  |  :---       |\n| #   | cbor_json.custom_objects.HashSha3_224 | sha3_224 hash |\n| #1  | cbor_json.custom_objects.HashSha1     | sha1 hash |\n| #2  | cbor_json.custom_objects.HashSha256   | sha256 hash |\n| #3  | cbor_json.custom_objects.HashSha3_256 | sha3_256 hash |\n| #5  | cbor_json.custom_objects.HashMd5      | md5 hash |\n| #0  | cbor_json.custom_objects.HashCrc32    | crc32 checksum |\n| df  | cbor_json.custom_objects.DataFrameSerialized | pandas dataframe |\n\n2. Keep class tags short.\n3. Tags of 1 and 2 characters long meant to be a subject of general consent. If you have an idea to add something undoubtedly useful, create an issue and/or a PR.\n4. Registering class tags that start with "~" sign is prohibited. Data marked this way is meant to be interpreted as <code>cbor_json.UnrecognizedCustomObject</code>, and it might be useful sometimes.\n\n## Additional notes\n- Roundtrip "CBOR -> decode -> encode -> CBOR" usually produces exactly the same result, but with some exceptions:\n  - Encoding always produces so called "canonical" format, so if decoded cbor was not canonical, the result will be different.\n  - Here we encode datetimes as timestamps (cbor tag 1), so if they were encoded as datetime strings (cbor tag 0), the result will change.\n  - Floats... No guarantees for them, as usual.\n- Roundtrip "Native -> CBOR -> native" logically produces the same result except dicts keys order.\n- Roundtrip "JSON -> native or CBOR -> JSON" sometimes produces the same result, but no guarantees at all.\n- Not every imaginable json can be processed by this tool. For instance, \'{"$type": "Hahaha"}\' will fail.\n- If some valid CBOR cannot be processed by this tool, please create an issue.\n- Please take care of backward compatibility. Do not redefine class tags.\n',
    'author': 'Alexander Maslyeav',
    'author_email': 'maslyaev@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/amaslyaev/cbor-json/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7.1,<4.0.0',
}


setup(**setup_kwargs)
