# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['mango',
 'mango.hedging',
 'mango.layouts',
 'mango.marketmaking',
 'mango.marketmaking.orderchain',
 'mango.oracles',
 'mango.oracles.ftx',
 'mango.oracles.market',
 'mango.oracles.pythnetwork',
 'mango.oracles.stub',
 'mango.simplemarketmaking']

package_data = \
{'': ['*']}

install_requires = \
['Rx>=3.2.0,<4.0.0',
 'base58>=2.1.1,<3.0.0',
 'jsons>=1.6.1,<2.0.0',
 'numpy>=1.22.1,<2.0.0',
 'pandas>=1.4.1,<2.0.0',
 'pyserum>=0.5.0a0,<0.6.0',
 'python-dateutil>=2.8.2,<3.0.0',
 'requests>=2.27.1,<3.0.0',
 'rxpy-backpressure>=1.0.0,<2.0.0',
 'solana>=0.25.0,<0.26.0',
 'websocket-client>=1.2.1,<2.0.0',
 'zstandard>=0.17.0,<0.18.0']

setup_kwargs = {
    'name': 'tmp-patch-mango-explorer',
    'version': '3.4.9',
    'description': 'Python integration for https://mango.markets',
    'long_description': '# 🥭 Mango Explorer\n\n## ⚠ Warning\n\nTHE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.\n\n\n## 🥭 Introduction\n\n`mango-explorer` provides Python code to integrate with [Mango Markets](https://mango.markets). The canonical client is written in typescript and by far better maintaned, hence we recommend to use it instead. You can find it [here](https://github.com/blockworks-foundation/mango-client-v3).\n\nIt includes some useful command-line programs, such as:\n* `deposit`\n* `withdraw`\n* `place-order`\n* `market-buy`\n* `market-sell`\n* `send-tokens`\n\nThere are also more complex programs such as a runnable [marketmaker](docs/MarketmakingQuickstart.md) for Serum, Spot and Perp markets.\n\n\n## 📦 Installation\n\n![PyPI](https://img.shields.io/pypi/v/mango-explorer)\n\n`mango-explorer` is available as a [Python package on Pypi](https://pypi.org/project/mango-explorer) and can be installed as:\n\n```\npip install mango-explorer\n```\n\nA simple [installation walkthrough](docs/Installation.md) is available, and of course other ways of installing it or adding it as a dependency are possible and will depend on the particular tools you are using.\n\n\n## 🌳 Branches\n\nThe latest version of the code is in the [main branch on Github](https://github.com/blockworks-foundation/mango-explorer).\n\nCode to integrate with Version 2 of Mango is in the [v2 branch](https://github.com/blockworks-foundation/mango-explorer/tree/v2).\n\n\n## 📓 Examples\n\nHere are some example snippets to get you started.\n\nMany more examples are provided in a separate [Github repo](https://github.com/blockworks-foundation/mango-explorer-examples) and can be [run in your browser (no installation required!) at Binder](https://mybinder.org/v2/gh/blockworks-foundation/mango-explorer-examples/HEAD).\n\n\n### 🏃 Show OrderBook\n> [![Binder](https://mybinder.org/badge_logo.svg)](https://mybinder.org/v2/gh/blockworks-foundation/mango-explorer-examples/HEAD?labpath=ShowOrderBook.ipynb) [Full runnable code in `mango-explorer-examples` repo](https://github.com/blockworks-foundation/mango-explorer-examples/blob/main/ShowOrderBook.ipynb)\n\nThis code will connect to the _devnet_ cluster, fetch the orderbook for BTC-PERP, and print out a summary of it:\n```\nimport mango\n\nwith mango.ContextBuilder.build(cluster_name="devnet") as context:\n    market = mango.market(context, "BTC-PERP")\n    print(market.fetch_orderbook(context))\n```\n\n\n### 🏃 Subscribe to OrderBook\n> [![Binder](https://mybinder.org/badge_logo.svg)](https://mybinder.org/v2/gh/blockworks-foundation/mango-explorer-examples/HEAD?labpath=SubscribeOrderBook.ipynb) [Full runnable code in `mango-explorer-examples` repo](https://github.com/blockworks-foundation/mango-explorer-examples/blob/main/SubscribeOrderBook.ipynb)\n\nThis code will connect to the _devnet_ cluster and will print out the latest orderbook every time the orderbook changes, and will exit after 60 seconds.\n```\nimport datetime\nimport mango\nimport time\n\nwith mango.ContextBuilder.build(cluster_name="devnet") as context:\n    market = mango.market(context, "BTC-PERP")\n    subscription = market.on_orderbook_change(context, lambda ob: print("\\n", datetime.datetime.now(), "\\n", ob))\n\n    time.sleep(60)\n\n    subscription.dispose()\n```\n\n\n### 🏃 Show Perp Fills\n> [![Binder](https://mybinder.org/badge_logo.svg)](https://mybinder.org/v2/gh/blockworks-foundation/mango-explorer-examples/HEAD?labpath=ShowPerpFills.ipynb) [Full runnable code in `mango-explorer-examples` repo](https://github.com/blockworks-foundation/mango-explorer-examples/blob/main/ShowPerpFills.ipynb)\n\nA \'fill\' is when a maker order from the orderbook is matched with an incoming taker order. It can be useful to see these.\n\nThis code will connect to the _devnet_ cluster and fetch all recent events. It will then show all the fill events.\n```\nimport mango\n\nwith mango.ContextBuilder.build(cluster_name="devnet") as context:\n    market = mango.market(context, "BTC-PERP")\n    event_queue = mango.PerpEventQueue.load(context, market.event_queue_address, market.lot_size_converter)\n    print(event_queue.fills)\n```\n\n\n### 🏃 Subscribe to Fills\n> [![Binder](https://mybinder.org/badge_logo.svg)](https://mybinder.org/v2/gh/blockworks-foundation/mango-explorer-examples/HEAD?labpath=SubscribeFills.ipynb) [Full runnable code in `mango-explorer-examples` repo](https://github.com/blockworks-foundation/mango-explorer-examples/blob/main/SubscribeFills.ipynb)\n\nThis code will connect to the _devnet_ cluster and print out every fill that happens. It will exit after 60 seconds.\n```\nimport datetime\nimport mango\nimport time\n\nwith mango.ContextBuilder.build(cluster_name="devnet") as context:\n    market = mango.market(context, "BTC-PERP")\n    subscription = market.on_fill(context, lambda ob: print("\\n", datetime.datetime.now(), "\\n", ob))\n\n    time.sleep(60)\n\n    subscription.dispose()\n```\n\n### 🏃 Place Order\n> [![Binder](https://mybinder.org/badge_logo.svg)](https://mybinder.org/v2/gh/blockworks-foundation/mango-explorer-examples/HEAD?labpath=PlaceOrder.ipynb) [Full runnable code in `mango-explorer-examples` repo](https://github.com/blockworks-foundation/mango-explorer-examples/blob/main/PlaceOrder.ipynb)\n\nThis code will load the \'example\' wallet, connect to the _devnet_ cluster, place an order and wait for the order\'s transaction signature to be confirmed.\n\nThe order is an IOC (\'Immediate Or Cancel\') order to buy 2 SOL-PERP at $1. This order is unlikely to be filled (SOL currently costs substantially more than $1), but that saves having to show order cancellation code here.\n\nPossible `order_type` values are:\n* mango.OrderType.LIMIT\n* mango.OrderType.IOC\n* mango.OrderType.POST_ONLY\n* mango.OrderType.MARKET (it\'s nearly always better to use IOC and a price with acceptable slippage)\n* mango.OrderType.POST_ONLY_SLIDE (only available on perp markets)\n```\nimport decimal\nimport mango\n\nfrom solana.publickey import PublicKey\n\n# Use our hard-coded devnet wallet for DeekipCw5jz7UgQbtUbHQckTYGKXWaPQV4xY93DaiM6h.\n# For real-world use you\'d load the bytes from the environment or a file. Later we use\n# its Mango Account at HhepjyhSzvVP7kivdgJH9bj32tZFncqKUwWidS1ja4xL.\nwallet = mango.Wallet(bytes([67,218,68,118,140,171,228,222,8,29,48,61,255,114,49,226,239,89,151,110,29,136,149,118,97,189,163,8,23,88,246,35,187,241,107,226,47,155,40,162,3,222,98,203,176,230,34,49,45,8,253,77,136,241,34,4,80,227,234,174,103,11,124,146]))\n\nwith mango.ContextBuilder.build(cluster_name="devnet") as context:\n    group = mango.Group.load(context)\n    account = mango.Account.load(context, PublicKey("HhepjyhSzvVP7kivdgJH9bj32tZFncqKUwWidS1ja4xL"), group)\n    market_operations = mango.operations(context, wallet, account, "SOL-PERP", dry_run=False)\n\n    # Try to buy 2 SOL for $1.\n    order = mango.Order.from_values(side=mango.Side.BUY,\n                                    price=decimal.Decimal(1),\n                                    quantity=decimal.Decimal(2),\n                                    order_type=mango.OrderType.IOC)\n    print("Placing order:", order)\n    placed_order_signatures = market_operations.place_order(order)\n\n    print("Waiting for place order transaction to confirm...\\n", placed_order_signatures)\n    mango.WebSocketTransactionMonitor.wait_for_all(\n            context.client.cluster_ws_url, placed_order_signatures, commitment="processed"\n        )\n```\n\n### 🏃 Place and Cancel Order\n> [![Binder](https://mybinder.org/badge_logo.svg)](https://mybinder.org/v2/gh/blockworks-foundation/mango-explorer-examples/HEAD?labpath=PlaceAndCancelOrders.ipynb) [Full runnable code in `mango-explorer-examples` repo](https://github.com/blockworks-foundation/mango-explorer-examples/blob/main/PlaceAndCancelOrders.ipynb)\n\nThis code will load the \'example\' wallet, connect to the _devnet_ cluster, place an order and then cancel it.\n\nThis is a bit longer than the Place Order example but only because it performs most of the Place Order code as a setup to create the order so there\'s an order to cancel.\n\nThe key point is that `cancel_order()` takes an `Order` as a parameter, and that `Order` needs either the `id` or `client_id` property to be set so the Mango program can find the equivalent order data on-chain. The sample code specified 1001 as the `client_id` when the order was instantiated so it\'s fine for use here.\n```\nimport decimal\nimport mango\n\nfrom solana.publickey import PublicKey\n\n# Use our hard-coded devnet wallet for DeekipCw5jz7UgQbtUbHQckTYGKXWaPQV4xY93DaiM6h.\n# For real-world use you\'d load the bytes from the environment or a file. Later we use\n# its Mango Account at HhepjyhSzvVP7kivdgJH9bj32tZFncqKUwWidS1ja4xL.\nwallet = mango.Wallet(bytes([67,218,68,118,140,171,228,222,8,29,48,61,255,114,49,226,239,89,151,110,29,136,149,118,97,189,163,8,23,88,246,35,187,241,107,226,47,155,40,162,3,222,98,203,176,230,34,49,45,8,253,77,136,241,34,4,80,227,234,174,103,11,124,146]))\n\nwith mango.ContextBuilder.build(cluster_name="devnet") as context:\n    group = mango.Group.load(context)\n    account = mango.Account.load(context, PublicKey("HhepjyhSzvVP7kivdgJH9bj32tZFncqKUwWidS1ja4xL"), group)\n    market_operations = mango.operations(context, wallet, account, "SOL-PERP", dry_run=False)\n\n    print("Orders (initial):")\n    print(market_operations.load_orderbook())\n\n    order = mango.Order.from_values(side=mango.Side.BUY,\n                                    price=decimal.Decimal(10),\n                                    quantity=decimal.Decimal(1),\n                                    order_type=mango.OrderType.POST_ONLY)\n    print("Placing order:", order)\n    placed_order_signatures = market_operations.place_order(order)\n\n    print("Waiting for place order transaction to confirm...\\n", placed_order_signatures)\n    mango.WebSocketTransactionMonitor.wait_for_all(\n            context.client.cluster_ws_url, placed_order_signatures, commitment="processed"\n        )\n\n    print("\\n\\nOrders (including our new order):")\n    orderbook = market_operations.load_orderbook()\n    print(orderbook)\n\n    # Order has the client ID 1001 so we can use that Order object as a parameter here.\n    cancellaton_signatures = market_operations.cancel_order(order)\n\n    print("Waiting for cancel order transaction to confirm...\\n", cancellaton_signatures)\n    mango.WebSocketTransactionMonitor.wait_for_all(\n            context.client.cluster_ws_url, cancellaton_signatures, commitment="processed"\n        )\n\n    print("\\n\\nOrders (without our order):")\n    print(market_operations.load_orderbook())\n```\n\n\n### 🏃 Refresh Orders\n\nSolana\'s transaction mechanism allows for atomic cancel-and-replace of orders - either the entire transaction succeeds (old orders are cancelled, new orders are placed), or the entire transaction fails (no orders are cancelled, no orders are placed).\n\nNeither Serum nor Mango supports \'editing\' or changing an order - to change the price or quantity for an order you must cancel it and replace it with an order with updated values.\n\nThis code will loop 3 times around:\n* in one transaction: cancelling all perp orders and placing bid and ask perp orders on SOL-PERP\n* wait for that transaction to confirm\n* pause for 5 seconds\n\nYou can verify the transaction signatures in [Solana Explorer](https://explorer.solana.com/?cluster=devnet) to see there is a single transaction containing a `CancelAllPerpOrders` instruction followed by two `PlacePerpOrder2` instructions. Since they\'re all in the same transaction, they will all succeed or all fail - if any instruction fails, the previous instructions are not committed to the chain, as if they never happened.\n```\nimport decimal\nimport mango\nimport time\n\nfrom solana.publickey import PublicKey\n\n# Use our hard-coded devnet wallet for DeekipCw5jz7UgQbtUbHQckTYGKXWaPQV4xY93DaiM6h.\n# For real-world use you\'d load the bytes from the environment or a file. Later we use\n# its Mango Account at HhepjyhSzvVP7kivdgJH9bj32tZFncqKUwWidS1ja4xL.\nwallet = mango.Wallet(bytes([67,218,68,118,140,171,228,222,8,29,48,61,255,114,49,226,239,89,151,110,29,136,149,118,97,189,163,8,23,88,246,35,187,241,107,226,47,155,40,162,3,222,98,203,176,230,34,49,45,8,253,77,136,241,34,4,80,227,234,174,103,11,124,146]))\n\nwith mango.ContextBuilder.build(cluster_name="devnet") as context:\n    group = mango.Group.load(context)\n    account = mango.Account.load(context, PublicKey("HhepjyhSzvVP7kivdgJH9bj32tZFncqKUwWidS1ja4xL"), group)\n    market_operations = mango.operations(context, wallet, account, "SOL-PERP", dry_run=False)\n    market_instructions: mango.PerpMarketInstructionBuilder = mango.instruction_builder(context, wallet, account, "SOL-PERP", dry_run=False)\n\n    signers: mango.CombinableInstructions = mango.CombinableInstructions.from_wallet(wallet)\n\n    for counter in range(3):\n        print("\\n\\nOrders:")\n        orderbook = market_operations.load_orderbook()\n        print(orderbook)\n\n        instructions = signers\n        instructions += market_instructions.build_cancel_all_orders_instructions()\n        buy = mango.Order.from_values(side=mango.Side.BUY,\n                                      price=orderbook.top_bid.price,\n                                      quantity=decimal.Decimal(1),\n                                      order_type=mango.OrderType.POST_ONLY,\n                                      client_id=counter+10)\n        print(buy)\n        instructions += market_instructions.build_place_order_instructions(buy)\n        sell = mango.Order.from_values(side=mango.Side.SELL,\n                                       price=orderbook.top_ask.price,\n                                       quantity=decimal.Decimal(1),\n                                       order_type=mango.OrderType.POST_ONLY,\n                                       client_id=counter+20)\n        print(sell)\n        instructions += market_instructions.build_place_order_instructions(sell)\n\n        signatures = instructions.execute(context)\n\n        print("Waiting for refresh order transaction to confirm...\\n", signatures)\n        mango.WebSocketTransactionMonitor.wait_for_all(\n                context.client.cluster_ws_url, signatures, commitment="processed"\n            )\n\n        print("Sleeping for 5 seconds...")\n        time.sleep(5)\n\nprint("Cleaning up...")\ninstructions = signers\ninstructions += market_instructions.build_cancel_all_orders_instructions()\n\ncleanup_signatures = instructions.execute(context)\n\nprint("Waiting for cleanup transaction to confirm...\\n", cleanup_signatures)\nmango.WebSocketTransactionMonitor.wait_for_all(\n        context.client.cluster_ws_url, cleanup_signatures, commitment="processed"\n    )\n\nprint("Example complete.")\n```\n\n### 🏃 Show Account Data\n> [![Binder](https://mybinder.org/badge_logo.svg)](https://mybinder.org/v2/gh/blockworks-foundation/mango-explorer-examples/HEAD?labpath=ShowAccountDataFrame.ipynb) [Full runnable code in `mango-explorer-examples` repo](https://github.com/blockworks-foundation/mango-explorer-examples/blob/main/ShowAccountDataFrame.ipynb)\n\nThis code will connect to the _devnet_ cluster and show important data from a Mango Account.\n\nData on deposits, borrows and perp positions are all available in a `pandas` `DataFrame` for you to perform your own calculations upon. The `Account` class also has some methods to take this `DataFrame` and run common calculations on it, such as calculating the total value of the `Account` (using `total_value()`), the health of the account (using `init_health()` and `maint_health()`), or the account\'s leverage (using `leverage()`).\n```\nimport mango\n\nfrom solana.publickey import PublicKey\n\nwith mango.ContextBuilder.build(cluster_name="devnet") as context:\n    group = mango.Group.load(context)\n    cache: mango.Cache = mango.Cache.load(context, group.cache)\n\n    account = mango.Account.load(context, PublicKey("HhepjyhSzvVP7kivdgJH9bj32tZFncqKUwWidS1ja4xL"), group)\n    open_orders = account.load_all_spot_open_orders(context)\n    frame = account.to_dataframe(group, open_orders, cache)\n    print(frame)\n    print(f"Init Health: {account.init_health(frame)}")\n    print(f"Maint Health: {account.maint_health(frame)}")\n    print(f"Total Value: {account.total_value(frame)}")\n    print(f"Leverage: {account.leverage(frame):,.2f}x")\n```\n\n### 🏃 Subscribe to Account changes\n> [![Binder](https://mybinder.org/badge_logo.svg)](https://mybinder.org/v2/gh/blockworks-foundation/mango-explorer-examples/HEAD?labpath=SubscribeAccount.ipynb) [Full runnable code in `mango-explorer-examples` repo](https://github.com/blockworks-foundation/mango-explorer-examples/blob/main/SubscribeAccount.ipynb)\n\nThis code will connect to the _devnet_ cluster and print out the Mango `Account` every time it changes. It will exit after 60 seconds.\n\nThe `subscribe()` method takes a lambda callback as a parameter - this lambda will be called with a new `Account` object when an update is received.\n\nThis code will exit after 60 seconds.\n\nThis pattern can also be used with `Group`, `Cache` and `EventQueue` objects.\n\n**NOTE:** This will send an updated `Account` object, including whatever updated values triggered the update. However a proper valuation of an `Account` involves several more Mango data accounts so it\'s important to fetch all data fresh (including `Group`, `Cache` and all `OpenOrders`) to calculate values.\n```\nimport datetime\nimport mango\nimport time\n\nfrom solana.publickey import PublicKey\n\nwith mango.ContextBuilder.build(cluster_name="devnet") as context:\n    # Load our devnet Random Taker Mango Account - it should have some activity\n    group = mango.Group.load(context)\n    account = mango.Account.load(context, PublicKey("5JDWiJGmnvs1DZomMV3s9Ev6DAgCQ9Svxd81EHCapjnD"), group)\n\n    manager = mango.IndividualWebSocketSubscriptionManager(context)\n    subscription = account.subscribe(context, manager, lambda acc: print(datetime.datetime.now(), "\\n", acc))\n    manager.open()\n\n    time.sleep(60)\n\n    manager.dispose()\n```\n\n\n### 🏃 Deposit\n> [![Binder](https://mybinder.org/badge_logo.svg)](https://mybinder.org/v2/gh/blockworks-foundation/mango-explorer-examples/HEAD?labpath=Deposit.ipynb) [Full runnable code in `mango-explorer-examples` repo](https://github.com/blockworks-foundation/mango-explorer-examples/blob/main/Deposit.ipynb)\n\nThis code will connect to the _devnet_ cluster and deposit 0.1 SOL into a Mango `Account`.\n```\nimport decimal\nimport mango\n\nfrom solana.publickey import PublicKey\n\n\n# Use our hard-coded devnet wallet for DeekipCw5jz7UgQbtUbHQckTYGKXWaPQV4xY93DaiM6h.\n# For real-world use you\'d load the bytes from the environment or a file. This wallet\n# has a Mango Account at HhepjyhSzvVP7kivdgJH9bj32tZFncqKUwWidS1ja4xL.\nwallet = mango.Wallet(bytes([67,218,68,118,140,171,228,222,8,29,48,61,255,114,49,226,239,89,151,110,29,136,149,118,97,189,163,8,23,88,246,35,187,241,107,226,47,155,40,162,3,222,98,203,176,230,34,49,45,8,253,77,136,241,34,4,80,227,234,174,103,11,124,146]))\n\nwith mango.ContextBuilder.build(cluster_name="devnet") as context:\n    group = mango.Group.load(context)\n    account = mango.Account.load(context, PublicKey("HhepjyhSzvVP7kivdgJH9bj32tZFncqKUwWidS1ja4xL"), group)\n    sol_token = mango.token(context, "SOL")\n\n    print("Wrapped SOL in account", account.slot_by_instrument(sol_token).net_value)\n\n    deposit_value = mango.InstrumentValue(sol_token, decimal.Decimal("0.1"))\n    deposit_signatures = account.deposit(context, wallet, deposit_value)\n    print("Waiting for deposit transaction to confirm...", deposit_signatures)\n    mango.WebSocketTransactionMonitor.wait_for_all(\n            context.client.cluster_ws_url, deposit_signatures, commitment="processed"\n        )\n\n    account = mango.Account.load(context, account.address, group)\n    print("Wrapped SOL after deposit", account.slot_by_instrument(sol_token).net_value)\n```\n\n\n### 🏃 Withdraw\n> [![Binder](https://mybinder.org/badge_logo.svg)](https://mybinder.org/v2/gh/blockworks-foundation/mango-explorer-examples/HEAD?labpath=Withdraw.ipynb) [Full runnable code in `mango-explorer-examples` repo](https://github.com/blockworks-foundation/mango-explorer-examples/blob/main/Withdraw.ipynb)\n\nThis code will connect to the _devnet_ cluster and withdraw 0.1 SOL from a Mango `Account`.\n\nUnlike `deposit()`, you specify an address as the destination. You can withdraw from your Mango `Account` to anyone\'s wallet (be careful!). Also, you can withdraw tokens even if you don\'t own them (as long as you have sufficient collateral) by borrowing them - `withdraw()`\'s `allow_borrow` boolean allows or prohibits borrowing tokens the `Account` doesn\'t own.\n```\nimport decimal\nimport mango\n\nfrom solana.publickey import PublicKey\n\n\n# Use our hard-coded devnet wallet for DeekipCw5jz7UgQbtUbHQckTYGKXWaPQV4xY93DaiM6h.\n# For real-world use you\'d load the bytes from the environment or a file. This wallet\n# has a Mango Account at HhepjyhSzvVP7kivdgJH9bj32tZFncqKUwWidS1ja4xL.\nwallet = mango.Wallet(bytes([67,218,68,118,140,171,228,222,8,29,48,61,255,114,49,226,239,89,151,110,29,136,149,118,97,189,163,8,23,88,246,35,187,241,107,226,47,155,40,162,3,222,98,203,176,230,34,49,45,8,253,77,136,241,34,4,80,227,234,174,103,11,124,146]))\n\nwith mango.ContextBuilder.build(cluster_name="devnet") as context:\n    group = mango.Group.load(context)\n    account = mango.Account.load(context, PublicKey("HhepjyhSzvVP7kivdgJH9bj32tZFncqKUwWidS1ja4xL"), group)\n    sol_token = mango.token(context, "SOL")\n\n    print("Wrapped SOL in account", account.slot_by_instrument(sol_token).net_value)\n\n    withdraw_value = mango.InstrumentValue(sol_token, decimal.Decimal("0.1"))\n    withdrawal_signatures = account.withdraw(context, wallet, wallet.address, withdraw_value, False)\n    print("Waiting for withdraw transaction to confirm...", withdrawal_signatures)\n    mango.WebSocketTransactionMonitor.wait_for_all(\n            context.client.cluster_ws_url, withdrawal_signatures, commitment="processed"\n        )\n\n    account = mango.Account.load(context, account.address, group)\n    print("Wrapped SOL after withdrawal", account.slot_by_instrument(sol_token).net_value)\n```\n\n\n## 🏛️ Running the marketmaker\n\nThere is a [Marketmaking Quickstart](docs/MarketmakingQuickstart.md) - a walkthrough of setting up and running the marketmaker on devnet, from setting up the account, depositing tokens, to running the marketmaker itself.\n\nIt can take around 30 minutes to run through.\n\nRequirements:\n* A server with docker installed\n\n**Note** This walkthrough is devnet-only so no actual funds are used or at-risk.\n\n## 🔖 References\n\n* [SolanaPy](https://github.com/michaelhly/solana-py/)\n* [PySerum](https://github.com/serum-community/pyserum/)\n* [Python Decimal Class](https://docs.python.org/3/library/decimal.html)\n* [Python Construct Library](https://construct.readthedocs.io/en/latest/)\n* [Python Observables](https://rxpy.readthedocs.io/en/latest/)\n* [RxPy Backpressure](https://github.com/daliclass/rxpy-backpressure)\n\n\n## 🥭 Support\n\n[🥭 Mango Markets](https://mango.markets/) support is available at: [Docs](https://docs.mango.markets/) | [Discord](https://discord.gg/67jySBhxrg) | [Twitter](https://twitter.com/mangomarkets) | [Github](https://github.com/blockworks-foundation) | [Email](mailto:hello@blockworks.foundation)\n',
    'author': 'Geoff Taylor',
    'author_email': 'geoff@mango.markets',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://mango.markets',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<3.11',
}


setup(**setup_kwargs)
