# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['typingx']

package_data = \
{'': ['*']}

extras_require = \
{':python_version < "3.9"': ['typing_extensions>=3.7.4.3,<4.0.0.0']}

setup_kwargs = {
    'name': 'typingx',
    'version': '0.3.0',
    'description': 'Extend typing package functionalities',
    'long_description': '# typingx\n[![Tests](https://github.com/PrettyWood/typingx/workflows/Tests/badge.svg)](https://github.com/PrettyWood/typingx/actions)\n[![codecov](https://codecov.io/gh/PrettyWood/typingx/branch/main/graph/badge.svg)](https://codecov.io/gh/PrettyWood/typingx)\n[![pypi](https://img.shields.io/pypi/v/typingx.svg)](https://pypi.python.org/pypi/typingx)\n[![versions](https://img.shields.io/pypi/pyversions/typingx.svg)](https://github.com/PrettyWood/typingx)\n[![license](https://img.shields.io/github/license/PrettyWood/typingx.svg)](https://github.com/PrettyWood/typingx/blob/master/LICENSE)\n\n\nHow many times have you wanted to check the shape of your data either in application code or while testing?\nWith this library, you can leverage `typing` types at runtime to do that!\n\n```python\n# Check if `my_list` is a list of integers\nisinstancex(my_list, [int])  # shortcut for typing.List[int]\n\n# Check if `my_list` has only numbers (3.10 syntax)\nisinstancex([3, 4, 3.14], list[int | float])\n\n# Check if `my_list` starts with 2 integers and then has only strings\nisinstancex(my_list, [int, int, str, ...])  # shortcut for Listx[int, int, str, ...] (see extra types)\n\n# Check if `my_dict` is a mapping between integers and strings\nisinstancex(my_dict, {int: str})  # shortcut for `typing.Dict[int, str]`\n\n# Check deeper the shape of `my_dict`\nisinstancex(my_dict, {\'a\': int, \'b\': bool, ...: str})  # shortcut for `typing.TypedDict`\n```\n\nSince `typing` changed a lot since python `3.6`, this library also makes sure the whole behaviour\nis consistent with `3.10` for all python versions.\n\nIt hence provides:\n- `isinstancex`: like `isinstance` but with `typing` types and extra types provided by this library\n- `get_args` and `get_origin` that have the exact same behaviour as the `3.10` python version ones, no matter which python version is used (the only exception being `get_args` with `Generic` on python 3.6)\n- `is_literal`, `is_newtype`, `is_typeddict` helpers\n- most `typing` types but with homogeneous behaviour (e.g. with `3.8`, this libray will choose `typing_extensions.TypedDict` instead of `typing.TypedDict` since the latter doesn\'t store information to distinguish optional and required keys)\n- extanded types:\n  * `TypedDict` with `...` field to allow type checking on optional fields\n- extra types:\n  * `Listx` and `Tuplex`: more sophisticated versions of `List` and `Tuple` to add `...` anywhere in the parameters\n\n## Installation\n\n``` bash\n    pip install typingx\n```\n\n## Usage\n```python\nfrom collections import ChainMap, Counter\n\nfrom typingx import *\n\n# Dict\nassert isinstancex({"a": 1, "b": 2}, Dict[str, int]) is True\nassert isinstancex({"a": 1, "b": 2}, Dict[str, str]) is False\nassert isinstancex({"a": 1, "b": 2}, Dict[int, str]) is False\nassert isinstancex({"a": 1, "b": 2}, Dict[str, Any]) is True\n\n# Dict (shortcut)\nassert isinstancex({"a": 1, "b": 2}, {str: int}) is True\n\n# List\nassert isinstancex([1, 2, 3], List[int]) is True\nassert isinstancex([1, 2, "q"], List[int]) is False\nassert isinstancex([1, 2, "q"], List[Union[str, int]]) is True\n\n# Listx\nassert isinstancex([1, 2, 3, 4], Listx[int]) is True\nassert isinstancex([1, 2, "q"], Listx[int, ..., str]) is True\nassert isinstancex([1, 2, "q", "w", "e"], Listx[int, ..., str]) is False\nassert isinstancex([1, 2, "q", "w", "e"], Listx[int, ..., str, ...]) is True\nassert isinstancex([1, 2, "q", "w", b"xyz", "e"], Listx[int, ..., str, ...]) is False\nassert isinstancex([1, 2, "q", "w", b"xyz", "e"], Listx[int, ..., Union[str, bytes], ...]) is True\n\n# Listx (shortcut)\nassert isinstancex([1, 2, 3, 4, "q"], [int, ..., str]) is True\n\n# Literal\nassert isinstancex("a", Literal["a"]) is True\nassert isinstancex(Literal["a"], Literal["a"]) is True\nassert isinstancex("b", Literal["a"]) is False\nassert isinstancex("b", Literal["a", Literal[Literal["b"]]]) is True\nassert isinstancex(Literal["a", "b"], Literal["b", "a", "c"]) is True\n\n# Mapping\nassert isinstancex(Counter({"red": 4, "blue": 2}), Mapping[str, int]) is True\nassert isinstancex(ChainMap({"art": "van gogh"}, {"music": "bach"}), Mapping[str, str]) is True\n\n# NewType\nUserId = NewType("UserId", int)\nProUserId = NewType("ProUserId", UserId)\n\nassert isinstancex(1, UserId) is True\nassert isinstancex(1, ProUserId) is True\nassert isinstancex(UserId(1), UserId) is True\nassert isinstancex("3", UserId) is False\n\n# None\nassert isinstancex([None, None], List[None]) is True\nassert isinstancex([None, None], List[NoneType]) is True\nassert isinstancex([None, None], List[type(None)]) is True\nassert isinstancex([None, None], List[Literal[None]]) is True\n\n# Sequence\nassert isinstancex("abc", Sequence[Any]) is True\nassert isinstancex("abc", Sequence[int]) is False\nassert isinstancex((1, 3, 5), Sequence[int]) is True\n\n# Set\nassert isinstancex({"a", "b"}, Set[str]) is True\nassert isinstancex({"a", "b"}, Set[int]) is False\n# Can be written with the shortcut!\nassert isinstancex({"a", "b"}, {str}) is True\n\n# Tuple\nassert isinstancex((1, 2), Tuple[int, ...]) is True\nassert isinstancex((1, 2), Tuple[int, int]) is True\nassert isinstancex((1, 2), Tuple[int, int, int]) is False\n\n# Tuplex\nassert isinstancex((3, "a", "b"), Tuplex[int, str, ...]) is True\nassert isinstancex((3, "a", "b", "c"), Tuplex[int, str, ...]) is True\nassert isinstancex((3, "a", "b", "c"), Tuplex[int, str, ..., bool]) is False\nassert isinstancex((3, "a", "b", "c", True), Tuplex[int, str, ..., bool]) is True\nassert isinstancex((3, "a", "b", "c", 3), Tuplex[int, str, ..., bool]) is False\nassert isinstancex((3, "a", "b", "c", True, False), Tuplex[int, str, ..., bool, ...]) is True\n\n# Type\nclass User: ...\nclass BaseUser(User): ...\n\nassert isinstancex(BaseUser, Type[BaseUser]) is True\nassert isinstancex(BaseUser, Type[User]) is True\nassert isinstancex(User, Type[User]) is True\nassert isinstancex(User, Type[BaseUser]) is False\n\n# TypedDict\nFullMovie = TypedDict("FullMovie", {"name": str, "year": int})\n\nclass PartialMovie(TypedDict, total=False):\n    name: str\n    year: int\n\nclass ExtraMovie(TypedDict):\n    name: str\n    year: int\n    __extra__: str\n\nassert isinstancex({"name": "The Matrix", "year": 1999}, FullMovie) is True\nassert isinstancex({"name": "The Matrix", "year": "1999"}, FullMovie) is False\nassert isinstancex({"name": "The Matrix"}, FullMovie) is False\nassert isinstancex({"name": "The Matrix", "year": 1999, "extra": "qwe"}, FullMovie) is False\n\nassert isinstancex({"name": "The Matrix", "year": 1999}, PartialMovie) is True\nassert isinstancex({"name": "The Matrix"}, PartialMovie) is True\nassert isinstancex({"name": "The Matrix", "year": 1999, "extra": "qwe"}, PartialMovie) is False\n\nassert isinstancex({"name": "The Matrix", "year": 1999}, ExtraMovie) is True\nassert isinstancex({"name": "The Matrix", "year": 1999, "q": "w", "e": "r"}, ExtraMovie) is True\nassert isinstancex({"name": "The Matrix", "year": 1999, "q": "w", "e": 1}, ExtraMovie) is False\n\n# TypedDict (shortcut)\nassert isinstancex({"name": "The Matrix", "year": 1999, "q": "w", "e": "r"}, {"name": str, "year": int, ...: str}) is True\n```\n',
    'author': 'Eric Jolibois',
    'author_email': 'em.jolibois@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/PrettyWood/typingx',
    'packages': packages,
    'package_data': package_data,
    'extras_require': extras_require,
    'python_requires': '>=3.6.1,<4.0.0',
}


setup(**setup_kwargs)
