# Contributing to buildzr

Thank you for considering contributing to buildzr! Your help is greatly appreciated. Below are some guidelines to help you get started.

_Happy Hacking 👩‍💻!_

## Getting Started

### Codespaces

I recommend using GitHub Codespaces for development. Codespaces provides a consistent development environment and simplifies the setup process.

You can quickly set up and start contributing to our project using GitHub Codespaces via GitHub CLI. Follow these steps:

1. Install the GitHub CLI if you haven't already. Visit https://cli.github.com/.

2. Authenticate with your GitHub account by running:

    ```bash
    gh auth login
    ```

3. Create a new codespace for this repository using the following command:

    ```bash
    gh codespace create -r amirulmenjeni/buildzr
    ```

4. Once the codespace is created, you can [connect to it via VS Code](https://docs.github.com/en/codespaces/developing-in-a-codespace/using-github-codespaces-in-visual-studio-code#opening-a-codespace-in-vs-code) or `ssh`
   into it:

    ```bash
    gh codespace ssh
    ```

5. Inside the workspace, setup and activate the Python environment:

    ```bash
    conda env create -f environment.yml
    ```

    To activate:

    ```bash
    conda activate buildzr-dev
    ```

The codespace environment also includes useful tools developed by the Structurizr team such as [Structurizr CLI](https://docs.structurizr.com/cli) and [Structurizr Lite](https://docs.structurizr.com/lite) to view diagrams defined in [Structurizr DSL](https://docs.structurizr.com/dsl) or [JSON](https://structurizr.com/json). They're mostly helpful during development to validate the [Structurizr JSON schema](https://github.com/structurizr/json) generated by `buildzr`.

### Running Tests

All tests are located in the `tests` folder. To run the tests, use the following command:
```sh
pytest --mypy tests
```

### Testing `buildzr` Samples

Sometimes it is valuable to manually check the output of the Structurizr workspace generated by `buildzr` visually. For this reason, you can create `buildzr` samples in the [`tests/samples`](https://github.com/amirulmenjeni/buildzr/tree/master/tests/samples) directory.

Here's an example of a simple workspace to showcase the [group](https://docs.structurizr.com/dsl/cookbook/groups/) feature in Structurizr DSL used for grouping elements together:

```python
from buildzr.dsl import (
    Workspace,
    SoftwareSystem,
    Container,
    Component,
    Group,
    SystemLandscapeView,
    SystemContextView,
    ContainerView,
)

with Workspace("w") as w:
    with Group("Company 1"):
        system_a = SoftwareSystem("A")
        with system_a:
            a1 = Container("a1")
            a2 = Container("a2")

    with Group("Company 2"):
        system_b = SoftwareSystem("B")
        with system_b:
            b1 = Container("b1")
            b2 = Container("b2")
            with b2:
                c1 = Component("c1")

    system_c = SoftwareSystem("C")

    # Define relationships
    system_a >> "Uses" >> system_b
    a1 >> "Uses" >> b1
    system_a >> "Uses" >> system_c

    # Create views
    SystemLandscapeView(
        key='groups-sample',
        description="Groups Sample"
    )

    SystemContextView(
        software_system_selector=system_a,
        key='groups-sample-a',
        description="Groups Sample - Software System A"
    )

    SystemContextView(
        software_system_selector=system_b,
        key='groups-sample-b',
        description="Groups Sample - Software System B"
    )

    ContainerView(
        software_system_selector=system_b,
        key='groups-sample-b2',
        description="Groups Sample - Container B2"
    )

    # Export to JSON
    w.to_json('groups-sample.json')
```

The example above demonstrates the current `buildzr` API using context managers (`with` statements) for a cleaner, more Pythonic syntax. Key features shown:

- Use `with Workspace("name") as w:` to create a workspace context
- Use `with Group("name"):` to group elements together
- Use `with software_system:` or `with container:` to nest containers and components
- Define relationships using the `>>` operator
- Views and styles are created within the workspace context
- Export using `w.to_json('filename.json')`

You can run this code directly or create sample files in the `tests/samples` directory for testing purposes.

### Working with the `.json` files

What can you do with the generated `.json` files?

As mentioned, the `.json` file, if successfully and correctly generated using
`buildzr`, should follow the Structurizr JSON schema. Thus you can make use of existing and mature Structurizr tools such as the [Structurizr CLI](https://docs.structurizr.com/cli) and [Structurizr Lite](https://docs.structurizr.com/lite).

If you use codespaces (_highly recommended_), these tools are already pre-installed in the devcontainer image.

For example, to export the `tests/samples/.tests.samples.groups.json` to other format (say, [PlantUML](https://plantuml.com/)):

```bash
structurizr.sh \
    export \
    --format plantuml \
    --workspace tests/samples/.tests.samples.groups.json \
    --output tests/samples/.tests.samples.groups.puml
```

To view the rendering of `tests/samples/.tests.samples.groups.json` in the
browser at http://localhost:8080:

```bash
structurizr-lite.sh tests/samples/.tests.samples.groups.json
```

## Submitting Changes

As usual, just fork the repo, make changes, and submit PRs!