# coding :utf-8
#
# The MIT License (MIT)
#
# Copyright (c) 2016-2019 XuHaiJiang/QFF
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
import os
import sys

sys.path.append(os.path.abspath(os.path.join(os.path.join(os.path.dirname(os.path.abspath(__file__)), os.path.pardir),
                                             os.path.pardir)))
"""
获取股票相关数据，并保存至数据库中。
根据操作系统设置定时任务，执行本文件。
"""
import pandas as pd
import datetime
import time
from qff.price.fetch import fetch_stock_list, fetch_price, fetch_stock_xdxr, fetch_stock_info, fetch_stock_block
from qff.tools.date import run_time, get_real_trade_date, get_next_trade_day, get_trade_days
from qff.tools.config import DATABASE
from qff.tools.utils import util_to_json_from_pandas
from pymongo.errors import PyMongoError


@run_time
def save_security_list(market='stock'):
    """
    从通达信获取股票/指数/ETF列表，并保存到数据库中
    :param market: 市场类型，目前支持“stock/index/etf", 默认“stock".
    :return: None
    """
    try:
        table_name = market + '_list'
        DATABASE.drop_collection(table_name)
        coll = DATABASE.get_collection(table_name)
        coll.create_index('code')
        print(f'##JOB01 Now Saving {table_name.upper()} ====')
        df = fetch_stock_list(market).reset_index()
        pandas_data = util_to_json_from_pandas(df)
        coll.insert_many(pandas_data)
        print(f'##JOB01 Save {table_name.upper()} Done!')
    except Exception as e:
        print(" Error save_security_list exception!")
        print(e)


@run_time
def save_security_day(market='stock'):
    """
    从通达信获取交易日数据，并保存到数据库中
    :param market: 市场类型，目前支持“stock/index/etf", 默认“stock".
    """
    try:

        end_date = _now_time()[:10]
        stock_list = fetch_stock_list(market).index.to_list()
        table_name = market + '_day'
        print(f'##JOB02 Now Saving {table_name.upper()} ====')
        coll = DATABASE.get_collection(table_name)
        coll.create_index([("code", 1), ("date", 1)], unique=True)
        coll.create_index("date")

        data_num = 0
        data_list = []
        for item in range(len(stock_list)):
            print(f'SAVE_{table_name.upper()} : The {item} of Total {len(stock_list)}')
            code = stock_list[item]
            try:
                start_date = coll.find_one({'code': code}, sort=[('date', -1)])['date']
            except TypeError or PyMongoError:
                start_date = '1990-01-01'

            if start_date != end_date:
                try:
                    start_date = get_next_trade_day(start_date)
                    print('Trying updating {} from {}'.format(code, start_date))
                    data = fetch_price(code, freq='day', market=market, start=start_date)
                    if data is None or len(data) == 0:
                        continue
                    data = data.loc[:end_date]
                    data.reset_index(inplace=True)
                    data_num += len(data)
                    data_list.append(data)
                    if data_num > 2000:
                        data = pd.concat(data_list)
                        data_num = 0
                        data_list.clear()
                        coll.insert_many(util_to_json_from_pandas(data))

                except Exception as e:
                    print(f'updating {code} data error!')
                    print('Exception:' + str(e))

        if data_num > 0:
            data = pd.concat(data_list)
            coll.insert_many(util_to_json_from_pandas(data))

        print(f'SUCCESS SAVE {table_name.upper()} ^_^')
    except EOFError:
        time.sleep(1)
    except Exception as e:
        print(" Error save_security_day exception!")
        print(str(e))


@run_time
def save_security_min(market='stock', freq='1min'):
    """
    从通达信获取交易日数据，并保存到数据库中
    :param market: 市场类型，目前支持“stock/index/etf", 默认“stock".
     :param freq: 分钟频率，支持1min/5min/15min/30min/60min.
    """
    if freq not in ["1min", "5min", "15min", "30min", "60min"] or\
       market not in ["stock", "index", "etf"]:
        print("save_security_min: 输入参数错误！")
        return

    print(f"START SAVE {market.upper()} {freq.upper()} DATA ======")

    try:
        end_date = _now_time()
        stock_list = fetch_stock_list(market).index.to_list()
        table_name = market + '_min'
        print(f'##JOB03 Now Saving {table_name.upper()} ====')
        coll = DATABASE.get_collection(table_name)
        coll.create_index([("type", 1), ("code", 1), ("datetime", 1)], unique=True)

        data_num = 0
        data_list = []
        for item in range(len(stock_list)):
            print(f'SAVE_{table_name.upper()}  {freq}: The {item} of Total {len(stock_list)}')
            code = stock_list[item]

            try:
                start_date = coll.find_one({'type': freq, 'code': code}, sort=[('datetime', -1)])['datetime']
            except TypeError or PyMongoError:
                start_date = '1990-01-01'

            if start_date != end_date:
                try:
                    start_date = get_next_trade_day(start_date)
                    print('Trying updating {} {} data from {}'.format(code, freq, start_date))
                    data = fetch_price(code, freq=freq, market=market, start=start_date)
                    if data is None or len(data) == 0:
                        continue
                    data = data.loc[:end_date]
                    data.reset_index(inplace=True)
                    data['type'] = freq

                    data_num += len(data)
                    data_list.append(data)
                    if data_num > 2000:
                        data = pd.concat(data_list)
                        data_num = 0
                        data_list.clear()
                        coll.insert_many(util_to_json_from_pandas(data))

                except Exception as e:
                    print(f'updating {code} {freq} data error!')
                    print('Exception:' + str(e))
        if data_num > 0:
            data = pd.concat(data_list)
            coll.insert_many(util_to_json_from_pandas(data))

        print(f'SUCCESS SAVE {table_name.upper()} {freq} DATA! ^_^')
    except EOFError:
        time.sleep(1)
    except Exception as e:
        print(f" Error save_security_min exception!:{market.upper()} {freq.upper()} DATA")
        print(e)


@run_time
def save_stock_xdxr():
    """
    从通达信获取除权除息分红数据，并保存到数据库中
    """
    try:
        coll = DATABASE.get_collection('stock_xdxr')
        coll.create_index([('code', 1), ('date', 1)], unique=True)

        coll_adj = DATABASE.get_collection('stock_adj')
        coll_adj.create_index([('code', 1), ('date', 1)], unique=True)
    except Exception as e:
        print(e)
        DATABASE.drop_collection('stock_xdxr')
        coll = DATABASE.get_collection('stock_xdxr')
        coll.create_index([('code', 1), ('date', 1)], unique=True)

        DATABASE.drop_collection('stock_adj')
        coll_adj = DATABASE.get_collection('stock_adj')
        coll_adj.create_index([('code', 1), ('date', 1)], unique=True)

    try:
        print(f'##JOB04 Now Saving stock_xdxr ====')
        stock_list = fetch_stock_list('stock').index.to_list()
        for item in range(len(stock_list)):
            code = stock_list[item]
            print(f'SAVE_stock_xdxr : The {item} of Total {len(stock_list)}，code={code}')
            xdxr = fetch_stock_xdxr(str(code))
            if xdxr is None or len(xdxr) == 0:
                continue
            try:
                coll.insert_many(util_to_json_from_pandas(xdxr))
            except PyMongoError:
                pass

            try:
                data = fetch_price(code, freq='day', market='stock', start='1990-01-01')
                qfq = _calc_stock_to_fq(data, xdxr, 'qfq').reset_index()
                qfq = qfq.assign(date=qfq.date.apply(lambda x: str(x)[0:10]))
                adjdata = util_to_json_from_pandas(qfq.loc[:, ['date', 'code', 'adj']])
                coll_adj.delete_many({'code': code})
                coll_adj.insert_many(adjdata)
            except Exception as e:
                print(e)

        print(f'SUCCESS save_stock_xdxr! ^_^')
    except EOFError:
        time.sleep(1)
    except Exception as e:
        print(" Error save_stock_xdxr exception!")
        print(e)


@run_time
def save_security_info():
    """
    从通达信获取股票基本信息，并保存到数据库中
    :return: None
    """
    try:
        table_name = 'stock_info'
        DATABASE.drop_collection(table_name)
        coll = DATABASE.get_collection(table_name)
        coll.create_index('code')
        print(f'##JOB05 Now Saving {table_name.upper()} ====')
        stock_list = fetch_stock_list().index.to_list()
        info = []
        for code in stock_list:
            data = fetch_stock_info(code)
            if data is not None and len(data) > 0:
                info.append(data)

        coll.insert_many(util_to_json_from_pandas(pd.concat(info)))
        print(f'SUCCESS SAVE {table_name.upper()} ^_^')

    except Exception as e:
        print(" Error save_security_info exception!")
        print(e)


@run_time
def save_security_block():
    """
    从通达信获取股票板块信息，并保存到数据库中
    :return: None
    """
    try:
        table_name = 'stock_block'
        DATABASE.drop_collection(table_name)
        coll = DATABASE.get_collection(table_name)
        coll.create_index('code')
        print(f'##JOB06 Now Saving {table_name.upper()} ====')
        data = fetch_stock_block()
        if data is not None:
            coll.insert_many(util_to_json_from_pandas(data))
        print(f'SUCCESS SAVE {table_name.upper()} ^_^')

    except Exception as e:
        print(" Error save_security_info exception!")
        print(e)


@run_time
def save_stock_name():

    try:
        coll = DATABASE.stock_name
        coll.create_index([("date", 1), ("code", 1)], unique=True)

        cursor = DATABASE.stock_list.find(
            {},
            {"_id": 0, "code": 1, "name": 1},
        )
        df1 = pd.DataFrame([item for item in cursor])
        df1 = df1.set_index('code')

        cursor = DATABASE.stock_info.find(
            {},
            {"_id": 0, "code": 1, "ipo_date": 1},
        )
        df2 = pd.DataFrame([item for item in cursor])
        df2 = df2.set_index('code')

        df3 = df1.join(df2, how="left")
        df4: pd.DataFrame = df3[df3['ipo_date'] > 0]
        df4 = df4.reset_index()
        df4.drop('ipo_date', axis=1, inplace=True)

        end_date = _now_time()[:10]
        try:
            last_date = coll.find_one({'code': df1.index[0]}, sort=[('date', -1)])['date']
            start = get_next_trade_day(last_date)
        except Exception as error0:
            print(error0)
            start = end_date

        if start > end_date:
            print("stock_name 数据已经最新!")
            return False

        for today in get_trade_days(start, end_date):
            df4.insert(0, 'date', value=today)
            coll.insert_many(util_to_json_from_pandas(df4))

        print("SAVE stock_name DONE!")
        return True
    except Exception as error0:
        print(error0)
        return False


##########################################################################################################
def _now_time():
    return str(get_real_trade_date(str(datetime.date.today() - datetime.timedelta(days=1)))) + \
           ' 15:00:00' if datetime.datetime.now().hour < 15 \
           else str(get_real_trade_date(str(datetime.date.today()))) + ' 15:00:00'


def _calc_stock_to_fq(bfq_data, xdxr_data, fq_type='qfq'):
    """
    计算复权系数
    :param bfq_data: 原始股票数据
    :param xdxr_data: 除权数据表
    :param fq_type: 复权类型,目前仅支持前复权,即qfq
    :return:
    """

    info = xdxr_data.query('category==1')
    bfq_data = bfq_data.assign(if_trade=1)

    if len(info) > 0:
        data = pd.concat(
            [
                bfq_data,
                info.loc[bfq_data.index[0]:bfq_data.index[-1],
                         ['category']]
            ],
            axis=1
        )

        data['if_trade'].fillna(value=0, inplace=True)
        data = data.fillna(method='ffill')

        data = pd.concat(
            [
                data,
                info.loc[bfq_data.index[0]:bfq_data.index[-1],
                         ['fenhong',
                          'peigu',
                          'peigujia',
                          'songzhuangu']]
            ],
            axis=1
        )
    else:
        data = pd.concat(
            [
                bfq_data,
                info.
                loc[:,
                    ['category',
                     'fenhong',
                     'peigu',
                     'peigujia',
                     'songzhuangu']]
            ],
            axis=1
        )
    data = data.fillna(0)
    data['preclose'] = (
        data['close'].shift(1) * 10 - data['fenhong'] +
        data['peigu'] * data['peigujia']
    ) / (10 + data['peigu'] + data['songzhuangu'])

    if fq_type in ['01', 'qfq']:
        data['adj'] = (data['preclose'].shift(-1) /
                       data['close']).fillna(1)[::-1].cumprod()
    else:
        data['adj'] = (data['close'] /
                       data['preclose'].shift(-1)).cumprod().shift(1).fillna(1)

    for col in ['open', 'high', 'low', 'close', 'preclose']:
        data[col] = data[col] * data['adj']
    data['volume'] = data['volume'] / \
        data['adj'] if 'volume' in data.columns else data['vol']/data['adj']
    try:
        data['high_limit'] = data['high_limit'] * data['adj']
        data['low_limit'] = data['high_limit'] * data['adj']
    except:
        pass
    return data.query('if_trade==1 and open != 0').drop(
        ['fenhong',
         'peigu',
         'peigujia',
         'songzhuangu',
         'if_trade',
         'category'],
        axis=1,
        errors='ignore'
    )


if __name__ == '__main__':

    for market_ in ['stock', 'index', 'etf']:
        save_security_list(market_)
        save_security_day(market_)
        for freq_ in ["1min", "5min", "15min", "30min", "60min"]:
            save_security_min(market=market_, freq=freq_)

    save_security_info()
    save_stock_name()
    save_security_block()
    save_stock_xdxr()
    # save_security_min('stock', "1min")
    # save_security_day('stock')
    # save_security_min(market='etf', freq="5min")