"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodePipelineMergeAction = void 0;
const core_1 = require("@aws-cdk/core");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_codepipeline_1 = require("@aws-cdk/aws-codepipeline");
const aws_codepipeline_actions_1 = require("@aws-cdk/aws-codepipeline-actions");
const merge_branches_function_1 = require("./merge-branches-function");
/**
 * Represents a reference to a CodePipelineMergeAction.
 */
class CodePipelineMergeAction extends aws_codepipeline_actions_1.Action {
    constructor(props) {
        super({
            ...props,
            category: aws_codepipeline_1.ActionCategory.INVOKE,
            provider: 'Lambda',
            artifactBounds: {
                minInputs: 0,
                maxInputs: 0,
                minOutputs: 0,
                maxOutputs: 0,
            },
        });
        this.props = props;
    }
    bound(scope, _stage, options) {
        const { repository, sourceCommitSpecifier, destinationCommitSpecifier, crossAccountRole, } = this.props;
        const mergeBranchesFunction = new merge_branches_function_1.MergeBranchesFunction(scope, 'MergeBranchesFunction', {
            repository,
            crossAccountRole,
        });
        // allow pipeline to list functions
        options.role.addToPolicy(new aws_iam_1.PolicyStatement({
            actions: ['lambda:ListFunctions'],
            resources: ['*'],
        }));
        // allow pipeline to invoke this lambda functionn
        options.role.addToPolicy(new aws_iam_1.PolicyStatement({
            actions: ['lambda:InvokeFunction'],
            resources: [mergeBranchesFunction.functionArn],
        }));
        // allow lambda to put job results for this pipeline
        // CodePipeline requires this to be granted to '*'
        // (the Pipeline ARN will not be enough)
        mergeBranchesFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            resources: ['*'],
            actions: [
                'codepipeline:PutJobSuccessResult',
                'codepipeline:PutJobFailureResult',
            ],
        }));
        return {
            configuration: {
                FunctionName: mergeBranchesFunction.functionName,
                UserParameters: core_1.Stack.of(scope).toJsonString({
                    repositoryName: repository.repositoryName,
                    sourceCommitSpecifier,
                    destinationCommitSpecifier,
                }),
            },
        };
    }
}
exports.CodePipelineMergeAction = CodePipelineMergeAction;
//# sourceMappingURL=data:application/json;base64,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