# -*- coding: utf-8 -*-


import requests


class VK:
    """ VK api wrapper """

    def __init__(self, access_token: str, v: int = 5.101):
        """
        :param str access_token: vk access token
        :param float v: vk api version
        """

        self.access_token = access_token
        self.v = v
        self.session = requests.session()

    def _call(self, method: str, params: dict = {}) -> dict:
        """

        :param str method: vk api method name
        :param dict params: method parameters
        :return: dict of api response
        """

        url = 'https://api.vk.com/method/{}'.format(method)
        params['access_token'] = self.access_token
        params['v'] = self.v
        try:
            response = self.session.post(url=url, params=params)
            response = response.json()
            response = response['response']
            return response
        except KeyError as e:
            print(response['error'])
            raise e

        except Exception as e:
            # TODO: ccreate exceptions
            raise e

    def account_ban(self, owner_id=None):
        """
        Добавляет пользователя или группу в черный список.

        :param owner_id: идентификатор пользователя или сообщества, которое будет добавлено в черный список. целое число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('account.ban', params=params)

    def account_changePassword(self, restore_sid=None, change_password_hash=None, old_password=None, new_password=None):
        """
        Позволяет сменить пароль пользователя после успешного восстановления доступа к аккаунту через СМС, используя метод auth.restore.

        :param restore_sid: идентификатор сессии, полученный при восстановлении доступа используя метод auth.restore. (В случае если пароль меняется сразу после восстановления доступа) строка
        :param change_password_hash: хэш, полученный при успешной OAuth авторизации по коду полученному по СМС (В случае если пароль меняется сразу после восстановления доступа) строка
        :param old_password: текущий пароль пользователя. строка
        :param new_password: новый пароль, который будет установлен в качестве текущего. строка, минимальная длина 6, обязательный параметр
        :return: Возвращает объект с единственным полем token, содержащим новый токен.
        """

        params = locals()
        params.pop('self')
        return self._call('account.changePassword', params=params)

    def account_getActiveOffers(self, offset=None, count=None):
        """
        Возвращает список активных рекламных предложений (офферов), выполнив которые пользователь сможет получить соответствующее количество голосов на свой счёт внутри приложения. Для отображения пользователю одного или всех рекламных предложений можно использовать метод JS API showOrderBox. После выполнения пользователем одной из предложенных ему акций, голоса будут зачислены на его счёт внутри приложения автоматически. В этом случае приложение может сразу списать необходимое количество голосов, сконвертировав голоса во внутреннюю валюту приложения.

        :param offset: смещение, необходимое для выборки определенного подмножества офферов. положительное число, по умолчанию 0
        :param count: количество офферов, которое необходимо получить положительное число, по умолчанию 100, максимальное значение 100
        :return: Возвращает массив, состоящий из общего количества старгетированных на текущего пользователя специальных предложений (первый элемент), и списка объектов с информацией о предложениях.  В случае, если на пользователя не старгетировано ни одного специального предложения, массив будет содержать элемент 0 (количество специальных предложений).
        """

        params = locals()
        params.pop('self')
        return self._call('account.getActiveOffers', params=params)

    def account_getAppPermissions(self, user_id=None):
        """
        Получает настройки текущего пользователя в данном приложении.

        :param user_id: идентификатор пользователя, информацию о настройках которого необходимо получить. По умолчанию — текущий пользователь. положительное число, по умолчанию идентификатор текущего пользователя, обязательный параметр
        :return: После успешного выполнения возвращает битовую маску настроек текущего пользователя в данном приложении.   Пример Если Вы хотите получить права на Доступ к друзьям и Доступ к статусам пользователя, то Ваша битовая маска будет равна: 2 + 1024 = 1026.  Если, имея битовую маску 1026, Вы хотите проверить, имеет ли она доступ к друзьям — Вы можете сделать 1026 & 2. Например alert(1026 & 2);  см. Список возможных настроек прав доступа
        """

        params = locals()
        params.pop('self')
        return self._call('account.getAppPermissions', params=params)

    def account_getBanned(self, offset=None, count=None):
        """
        Возвращает список пользователей, находящихся в черном списке.

        :param offset: смещение, необходимое для выборки определенного подмножества черного списка. положительное число
        :param count: количество объектов, информацию о которых необходимо вернуть. положительное число, по умолчанию 20, максимальное значение 200
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов, описывающих пользователей в поле items.
        """

        params = locals()
        params.pop('self')
        return self._call('account.getBanned', params=params)

    def account_getCounters(self, filter=None):
        """
        Возвращает ненулевые значения счетчиков пользователя.

        :param filter: счетчики, информацию о которых нужно вернуть. Может включать следующие значения:   friends — новые заявки в друзья;  friends_suggestions — предлагаемые друзья;  messages — новые сообщения;  photos — новые отметки на фотографиях;  videos — новые отметки на видеозаписях;  gifts — подарки;  events — события;  groups — сообщества;  notifications — ответы;  sdk — запросы в мобильных играх;  app_requests — уведомления от приложений.  friends_recommendations — рекомендации друзей.  список слов, разделенных через запятую
        :return: Возвращает объект, который может содержать поля friends, messages, photos, videos, notes, gifts, events, groups, notifications, sdk, app_requests, friends_recommendations со значением счетчиков (integer).
        """

        params = locals()
        params.pop('self')
        return self._call('account.getCounters', params=params)

    def account_getInfo(self, fields=None):
        """
        Возвращает информацию о текущем аккаунте.

        :param fields: список полей, которые необходимо вернуть. Возможные значения:   country  https_required  own_posts_default  no_wall_replies  intro  lang  По умолчанию будут возвращены все поля. список слов, разделенных через запятую
        :return: Метод возвращает объект, содержащий следующие поля:  country  stringстроковой код страны, определенный по IP адресу, с которого сделан запрос. https_required  integer, [0,1]информация о том, включено ли безопасное соединение для аккаунта. 1 — включено, 0 — не включено. 2fa_required  integer, [0,1]информация о том, включена ли двухфакторная аутентификация для аккаунта. 1 — включена, 0 — не включена. own_posts_default  integer, [0,1]информация о том, показываются ли по умолчанию на стене только записи пользователя. 1 — да, 0 — нет. no_wall_replies  integer, [0,1]информация о том, отключено ли комментирование записей на стене. 1 — да, 0 — нет. intro  integer, [0,1]информация о том, прошел ли пользователь обучение по использованию приложения. lang  integerидентификатор текущего языка пользователя.
        """

        params = locals()
        params.pop('self')
        return self._call('account.getInfo', params=params)

    def account_getProfileInfo(self):
        """
        Возвращает информацию о текущем профиле.

        :return:    В ходе выполнения могут произойти общие ошибки. Их описание находится на отдельной странице.
        """

        params = locals()
        params.pop('self')
        return self._call('account.getProfileInfo', params=params)

    def account_getPushSettings(self, device_id=None, token=None):
        """
        Позволяет получать настройки Push-уведомлений.

        :param device_id: уникальный идентификатор устройства. строка, доступен начиная с версии 5.31
        :param token: Идентификатор устройства, используемый для отправки уведомлений. (для mpns идентификатор должен представлять из себя URL для отправки уведомлений) строка, устаревший параметр, доступен только для версий меньше 5.31
        :return: Возвращает объект, содержащий поля:   disabled — отключены ли уведомления.  disabled_until — unixtime-значение времени, до которого временно отключены уведомления.  conversations — список, содержащий настройки конкретных диалогов, и их количество первым элементом.  settings — объект с настройками Push-уведомлений в специальном формате.
        """

        params = locals()
        params.pop('self')
        return self._call('account.getPushSettings', params=params)

    def account_registerDevice(self, token=None, device_model=None, device_year=None, device_id=None,
                               system_version=None, settings=None, sandbox=None, no_text=None, subscribe=None):
        """
        Подписывает устройство на базе iOS, Android, Windows Phone или Mac на получение Push-уведомлений. Обратите внимание, для корректной работы Push-уведомлений необходимо заполнить поле bundle_id в настройках приложения.

        :param token: идентификатор устройства, используемый для отправки уведомлений. (для mpns идентификатор должен представлять из себя URL для отправки уведомлений). строка, обязательный параметр
        :param device_model: cтроковое название модели устройства. строка
        :param device_year: год устройства. целое число
        :param device_id: уникальный идентификатор устройства. строка, обязательный параметр
        :param system_version: строковая версия операционной системы устройства. строка
        :param settings: сериализованный JSON-объект, описывающий настройки уведомлений в специальном формате. данные в формате JSON, доступен начиная с версии 5.31
        :param sandbox: флаг предназначен для iOS устройств. 1 — использовать sandbox сервер для отправки push-уведомлений, 0 — не использовать. флаг, может принимать значения 1 или 0, по умолчанию 0
        :param no_text: 1 — не передавать текст сообщения в push уведомлении. 0 — (по умолчанию) текст сообщения передаётся. целое число, устаревший параметр, доступен только для версий меньше 5.31
        :param subscribe: Список типов уведомлений, которые следует присылать. Могут быть переданы следующие типы: msg, friend, call, reply, mention, group, like. По умолчанию присылаются: msg, friend. строка, устаревший параметр, доступен только для версий меньше 5.31
        :return: После успешного выполнения возвращает 1.  На iOS и Windows Phone push-уведомления будут отображены без какой либо обработки.  На Android будут приходить события в следующем формате.
        """

        params = locals()
        params.pop('self')
        return self._call('account.registerDevice', params=params)

    def account_saveProfileInfo(self, first_name=None, last_name=None, maiden_name=None, screen_name=None,
                                cancel_request_id=None, sex=None, relation=None, relation_partner_id=None, bdate=None,
                                bdate_visibility=None, home_town=None, country_id=None, city_id=None, status=None):
        """
        Редактирует информацию текущего профиля.

        :param first_name: имя пользователя. Обязательно с большой буквы. строка
        :param last_name: фамилия пользователя. Обязательно с большой буквы. строка
        :param maiden_name: девичья фамилия пользователя (только для женского пола). строка
        :param screen_name: короткое имя страницы. строка
        :param cancel_request_id: идентификатор заявки на смену имени, которую необходимо отменить.  Если передан этот параметр, все остальные параметры игнорируются. положительное число
        :param sex: пол пользователя. Возможные значения:   1 — женский;  2 — мужской.  положительное число
        :param relation: семейное положение пользователя. Возможные значения:   1 — не женат/не замужем;  2 — есть друг/есть подруга;  3 — помолвлен/помолвлена;  4 — женат/замужем;  5 — всё сложно;  6 — в активном поиске;  7 — влюблён/влюблена;  8 — в гражданском браке;  0 — не указано.  положительное число
        :param relation_partner_id: идентификатор пользователя, с которым связано семейное положение. положительное число
        :param bdate: дата рождения пользователя в формате DD.MM.YYYY, например "15.11.1984". строка
        :param bdate_visibility: видимость даты рождения. Возможные значения:   1 — показывать дату рождения;  2 — показывать только месяц и день;  0 — не показывать дату рождения.  положительное число
        :param home_town: родной город пользователя. строка
        :param country_id: идентификатор страны пользователя. положительное число
        :param city_id: идентификатор города пользователя. положительное число
        :param status: статус пользователя, который также может быть изменен методом status.set строка
        :return: Метод возвращает объект, содержащий следующие поля:   changed – возвращает 1 — если информация была сохранена, 0 — если ни одно из полей не было сохранено.  Если в параметрах передавалось имя или фамилия пользователя, в объекте также будет возвращено поле name_request – объект, содержащий информацию о заявке на смену имени, со следующими полями:   id – идентификатор заявки, необходимый для её отмены (только если status равен processing). Для отмены заявки необходимо вызвать метод account.saveProfileInfo с параметром cancel_request_id;  status – статус заявки, возвращаемые значения:   success – имя было успешно изменено;  processing – заявка рассматривается;  declined – заявка отклонена;  was_accepted – недавно уже была одобрена заявка, повторную заявку можно подать не раньше даты, указанной в поле repeat_date;  was_declined – предыдущая заявка была отклонена, повторную заявку можно подать не раньше даты, указанной в поле repeat_date;   first_name – имя пользователя, указанное в заявке;  last_name – фамилия пользователя, указанная в заявке.
        """

        params = locals()
        params.pop('self')
        return self._call('account.saveProfileInfo', params=params)

    def account_setInfo(self, name=None, value=None, intro=None, own_posts_default=None, no_wall_replies=None):
        """
        Позволяет редактировать информацию о текущем аккаунте.

        :param name: имя настройки строка, доступен начиная с версии 5.49
        :param value: значение настройки строка, доступен начиная с версии 5.49
        :param intro: битовая маска, отвечающая за прохождение обучения в мобильных клиентах. положительное число, устаревший параметр, доступен только для версий меньше 5.49
        :param own_posts_default: 1 – на стене пользователя по-умолчанию должны отображаться только собственные записи;  0 – на стене пользователя должны отображаться все записи. флаг, может принимать значения 1 или 0, устаревший параметр, доступен только для версий меньше 5.49
        :param no_wall_replies: 1 – отключить комментирование записей на стене;  0 – разрешить комментирование. флаг, может принимать значения 1 или 0, устаревший параметр, доступен только для версий меньше 5.49
        :return: В результате успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('account.setInfo', params=params)

    def account_setNameInMenu(self, user_id=None, name=None):
        """
        Устанавливает короткое название приложения (до 17 символов), которое выводится пользователю в левом меню. Это происходит только в том случае, если пользователь добавил приложение в левое меню со страницы приложения, списка приложений или настроек.

        :param user_id: идентификатор пользователя. положительное число, по умолчанию идентификатор текущего пользователя, обязательный параметр
        :param name: короткое название приложения. строка
        :return: Возвращает 1 в случае успешной установки короткого названия.  Если пользователь не установил приложение в левое меню, метод вернет ошибку 148 (Access to the menu of the user denied). Избежать этой ошибки можно с помощью метода account.getAppPermissions.
        """

        params = locals()
        params.pop('self')
        return self._call('account.setNameInMenu', params=params)

    def account_setOffline(self):
        """
        Помечает текущего пользователя как offline (только в текущем приложении).

        :return:    В ходе выполнения могут произойти общие ошибки. Их описание находится на отдельной странице.
        """

        params = locals()
        params.pop('self')
        return self._call('account.setOffline', params=params)

    def account_setOnline(self, voip=None):
        """
        Помечает текущего пользователя как online на 5 минут.

        :param voip: возможны ли видеозвонки для данного устройства флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('account.setOnline', params=params)

    def account_setPushSettings(self, device_id=None, settings=None, key=None, value=None):
        """
        Изменяет настройку Push-уведомлений.

        :param device_id: уникальный идентификатор устройства. строка, обязательный параметр
        :param settings: сериализованный JSON-объект, описывающий настройки уведомлений в специальном формате. данные в формате JSON
        :param key: ключ уведомления. строка
        :param value: новое значение уведомления в специальном формате. список слов, разделенных через запятую
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('account.setPushSettings', params=params)

    def account_setSilenceMode(self, device_id=None, time=None, peer_id=None, sound=None, token=None, chat_id=None,
                               user_id=None):
        """
        Отключает push-уведомления на заданный промежуток времени.

        :param device_id: уникальный идентификатор устройства. строка, доступен начиная с версии 5.31
        :param time: время в секундах на которое требуется отключить уведомления, -1 отключить навсегда целое число
        :param peer_id: идентификатор назначения. Для пользователя:  id  пользователя.   Для групповой беседы:  2000000000 + id беседы.   Для сообщества:  -id сообщества.   целое число, доступен начиная с версии 5.46
        :param sound: 1 — включить звук в этом диалоге, 0 — отключить звук (параметр работает, только если в peer_id передан идентификатор групповой беседы или пользователя). целое число
        :param token: идентификатор устройства для сервиса push уведомлений. строка, устаревший параметр, доступен только для версий меньше 5.31
        :param chat_id: идентификатор чата, для которого следует отключить уведомления целое число, устаревший параметр, доступен только для версий меньше 5.46
        :param user_id: идентификатор пользователя, для которого следует отключить уведомления целое число, устаревший параметр, доступен только для версий меньше 5.46
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('account.setSilenceMode', params=params)

    def account_unban(self, owner_id=None):
        """
        Удаляет пользователя или группу из черного списка.

        :param owner_id: идентификатор пользователя или группы, которого нужно удалить из черного списка. целое число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('account.unban', params=params)

    def account_unregisterDevice(self, device_id=None, sandbox=None, token=None):
        """
        Отписывает устройство от Push уведомлений.

        :param device_id: уникальный идентификатор устройства. строка, доступен начиная с версии 5.31
        :param sandbox: флаг предназначен для iOS устройств. Возможные значения:   1 — отписать устройство, использующего sandbox сервер для отправки push-уведомлений;  0 — отписать устройство, не использующее sandbox сервер.  флаг, может принимать значения 1 или 0, по умолчанию 0
        :param token: идентификатор устройства, используемый для отправки уведомлений. строка, устаревший параметр, доступен только для версий меньше 5.31
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('account.unregisterDevice', params=params)

    def appWidgets_getAppImageUploadServer(self, image_type=None):
        """
        Позволяет получить адрес для загрузки фотографии в коллекцию приложения для виджетов приложений сообществ.

        :param image_type: тип изображения. Возможные значения:   24x24;  50x50;  160x160;  160x240;  510x128.  обязательный параметр
        :return: Возвращает объект с единственным полем upload_url, содержащим URL для загрузки изображения.  Для загрузки изображения сгенерируйте POST-запрос с файлом в поле image на полученный адрес, а затем вызовите метод appWidgets.saveAppImage.
        """

        params = locals()
        params.pop('self')
        return self._call('appWidgets.getAppImageUploadServer', params=params)

    def appWidgets_getAppImages(self):
        """
        Позволяет получить коллекцию изображений, загруженных для приложения, в виджетах приложений сообществ.

        :return: Возвращает общее число результатов в поле count (integer) и массив объектов, описывающих изображения, в поле items (array).  Каждый объект массива  items содержит следующие поля:   id (string) — идентификатор изображения. Обратите внимание, идентификаторы изображений для виджетов отличаются от обычных фотографий, и НЕ представляют собой сочетание owner_id+"_"+photo_id. Полученный идентификатор нужно использовать в коде виджета «как есть».  type (string) — тип изображения. Возможные значения:   160x160,  160x240,  24x24,  510x128,  50x50.   images  (array) — массив копий изображения. Каждый объект в массиве содержит следующие поля:   url (string) — URL копии;  width (integer) — ширина в px;  height (integer) — высота в px.
        """

        params = locals()
        params.pop('self')
        return self._call('appWidgets.getAppImages', params=params)

    def appWidgets_getGroupImageUploadServer(self):
        """
        Позволяет получить адрес для загрузки фотографии в коллекцию сообщества для виджетов приложений сообществ.

        :return: Возвращает объект с единственным полем upload_url, содержащим URL для загрузки изображения.  Для загрузки изображения сгенерируйте POST-запрос с файлом в поле image на полученный адрес, а затем вызовите метод appWidgets.saveGroupImage.
        """

        params = locals()
        params.pop('self')
        return self._call('appWidgets.getGroupImageUploadServer', params=params)

    def appWidgets_getGroupImages(self):
        """
        Позволяет получить коллекцию изображений, загруженных для приложения, в виджетах приложений сообществ.

        :return: Возвращает общее число результатов в поле count (integer) и массив объектов, описывающих изображения, в поле items (array).  Каждый объект массива  items содержит следующие поля:   id (string) — идентификатор изображения. Обратите внимание, идентификаторы изображений для виджетов отличаются от обычных фотографий, и НЕ представляют собой сочетание owner_id+"_"+photo_id. Полученный идентификатор нужно использовать в коде виджета «как есть».  type (string) — тип изображения. Возможные значения:   160x160,  160x240,  24x24,  510x128,  50x50.   images  (array) — массив копий изображения. Каждый объект в массиве содержит следующие поля:   url (string) — URL копии;  width (integer) — ширина в px;  height (integer) — высота в px.
        """

        params = locals()
        params.pop('self')
        return self._call('appWidgets.getGroupImages', params=params)

    def appWidgets_getImagesById(self):
        """
        Позволяет получить изображение для виджетов приложений сообществ по его идентификатору.

        :return: Возвращает объект, который содержит следующие поля:   id (string) — идентификатор изображения. Обратите внимание, идентификаторы изображений для виджетов отличаются от обычных фотографий, и НЕ представляют собой сочетание owner_id+"_"+photo_id. Полученный идентификатор нужно использовать в коде виджета «как есть».  type (string) — тип изображения. Возможные значения:   160x160,  160x240,  24x24,  510x128,  50x50.   images  (array) — массив копий изображения. Каждый объект в массиве содержит следующие поля:   url (string) — URL копии;  width (integer) — ширина в px;  height (integer) — высота в px.
        """

        params = locals()
        params.pop('self')
        return self._call('appWidgets.getImagesById', params=params)

    def appWidgets_saveAppImage(self, hash=None, image=None):
        """
        Позволяет сохранить изображение в коллекцию приложения для виджетов приложений сообществ после загрузки на сервер.

        :param hash: параметр hash, полученный после загрузки на сервер. строка, обязательный параметр
        :param image: параметр image, полученный после загрузки на сервер. строка, обязательный параметр
        :return: Возвращает объект, который содержит следующие поля:   id (string) — идентификатор изображения.  type (string) — тип изображения. Возможные значения:   160x160,  160x240,  24x24,  510x128,  50x50.   images  (array) — массив копий изображения. Каждый объект в массиве содержит следующие поля:   url (string) — URL копии;  width (integer) — ширина в px;  height (integer) — высота в px.
        """

        params = locals()
        params.pop('self')
        return self._call('appWidgets.saveAppImage', params=params)

    def appWidgets_saveGroupImage(self):
        """
        Позволяет сохранить изображение в коллекцию сообщества для виджетов приложений сообществ. после загрузки на сервер.

        :return: Возвращает объект, который содержит следующие поля:   id (string) — идентификатор изображения.  type (string) — тип изображения. Возможные значения:   160x160,  160x240,  24x24,  510x128,  50x50.   images  (array) — массив копий изображения. Каждый объект в массиве содержит следующие поля:   url (string) — URL копии;  width (integer) — ширина в px;  height (integer) — высота в px.
        """

        params = locals()
        params.pop('self')
        return self._call('appWidgets.saveGroupImage', params=params)

    def appWidgets_update(self):
        """
        Позволяет обновить виджет приложения сообщества. Этот метод можно вызывать не чаще 1 раза в 10 секунд.

        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('appWidgets.update', params=params)

    def apps_deleteAppRequests(self):
        """
        Удаляет все уведомления о запросах, отправленных из текущего приложения

        :return:    В ходе выполнения могут произойти общие ошибки. Их описание находится на отдельной странице.
        """

        params = locals()
        params.pop('self')
        return self._call('apps.deleteAppRequests', params=params)

    def apps_get(self, app_id=None, app_ids=None, platform=None, extended=None, return_friends=None, fields=None,
                 name_case=None):
        """
        Возвращает данные о запрошенном приложении.

        :param app_id: идентификатор приложения, данные которого необходимо получить. положительное число
        :param app_ids: список идентификаторов приложений, данные которых необходимо получить. список слов, разделенных через запятую, количество элементов должно составлять не более 100
        :param platform: платформа, для которой необходимо вернуть данные. Возможные значения:   ios — iOS;  android — Android;  winphone — Windows Phone;  web — приложения на vk.com.  По умолчанию: web. строка, по умолчанию web
        :param extended: 1 — возвращать дополнительные поля. По умолчанию возвращает только основные поля приложений. флаг, может принимать значения 1 или 0, по умолчанию 0
        :param return_friends: 1 – возвращать список друзей, установивших это приложение. По умолчанию: 0.  Параметр учитывается только при передаче access_token. флаг, может принимать значения 1 или 0, по умолчанию 0
        :param fields: список дополнительных полей, которые необходимо вернуть для профилей пользователей.  Доступные значения: sex, bdate, city, country, photo_50, photo_100, photo_200_orig, photo_200, photo_400_orig, photo_max, photo_max_orig, online, online_mobile, lists, domain, has_mobile, contacts, connections, site, education, universities, schools, can_post, can_see_all_posts, can_see_audio, can_write_private_message, status, last_seen, common_count, relation, relatives, counters,screen_name,timezone  Параметр учитывается только при return_friends = 1. список слов, разделенных через запятую
        :param name_case: падеж для склонения имени и фамилии пользователей. Возможные значения: * nom — именительный (по умолчанию);   gen — родительный;  dat — дательный;  acc — винительный;  ins — творительный;  abl — предложный;   Параметр учитывается только при return_friends = 1. строка
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов, описывающих приложения, в поле items.
        """

        params = locals()
        params.pop('self')
        return self._call('apps.get', params=params)

    def apps_getCatalog(self, sort=None, offset=None, count=None, platform=None, extended=None, return_friends=None,
                        fields=None, name_case=None, q=None, genre_id=None, filter=None):
        """
        Возвращает список приложений, доступных для пользователей сайта через каталог приложений.

        :param sort: способ сортировки приложений. Возможные значения:   popular_today — популярные за день;  visitors — по посещаемости;  create_date — по дате создания приложения;  growth_rate — по скорости роста;  popular_week — популярные за неделю.  По умолчанию: popular_today. строка
        :param offset: смещение, необходимое для выборки определенного подмножества приложений. положительное число
        :param count: количество приложений, информацию о которых необходимо вернуть. положительное число, по умолчанию 100, обязательный параметр
        :param platform: платформа, для которой необходимо вернуть приложения. Возможные значения:   ios — iOS;  android — Android;  winphone — Windows Phone;  web — приложения на vk.com;  html5 — Direct Games.  По умолчанию: web. строка
        :param extended: 1 — возвращать дополнительные поля приложений. Если указан extended – count не должен быть больше 100. флаг, может принимать значения 1 или 0
        :param return_friends: 1 – возвращать список друзей, установивших это приложение. По умолчанию: 0.  Параметр учитывается только при передаче access_token. флаг, может принимать значения 1 или 0
        :param fields: список дополнительных полей, которые необходимо вернуть для профилей пользователей.  Доступные значения: sex, bdate, city, country, photo_50, photo_100, photo_200_orig, photo_200, photo_400_orig, photo_max, photo_max_orig, online, online_mobile, lists, domain, has_mobile, contacts, connections, site, education, universities, schools, can_post, can_see_all_posts, can_see_audio, can_write_private_message, status, last_seen, common_count, relation, relatives, counters,screen_name,timezone  Параметр учитывается только при return_friends = 1. список слов, разделенных через запятую
        :param name_case: падеж для склонения имени и фамилии пользователей. Возможные значения: именительный – nom, родительный – gen, дательный – dat, винительный – acc, творительный – ins, предложный – abl. По умолчанию: nom.  Параметр учитывается только при return_friends = 1. строка
        :param q: поисковая строка для поиска по каталогу приложений. строка
        :param genre_id: идентификатор жанра положительное число
        :param filter: installed — возвращает список установленных приложений (только для мобильных приложений),  featured — возвращает список приложений, установленных в "Выбор редакции" (только для мобильных приложений). строка
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов, описывающих приложения, в поле items.  К методу можно делать не более 60 запросов в минуту с одного IP или id.
        """

        params = locals()
        params.pop('self')
        return self._call('apps.getCatalog', params=params)

    def apps_getFriendsList(self, extended=None, count=None, offset=None, type=None, fields=None):
        """
        Создает список друзей, который будет использоваться при отправке пользователем приглашений в приложение и игровых запросов.

        :param extended: параметр, определяющий необходимость возвращать расширенную информацию о пользователях. Возможные значения:   0 — возвращаются только идентификаторы;  1 — будут дополнительно возвращены имя и фамилия.   По умолчанию: 0. флаг, может принимать значения 1 или 0, по умолчанию 0
        :param count: количество пользователей в создаваемом списке. положительное число, по умолчанию 20, максимальное значение 5000
        :param offset: смещение относительно первого пользователя для выборки определенного подмножества. положительное число, по умолчанию 0
        :param type: тип создаваемого списка друзей. Возможные значения:   invite — доступные для приглашения (не играют в игру);  request — доступные для отправки запроса (уже играют).   По умолчанию: invite. строка, по умолчанию invite
        :param fields: список дополнительных полей профилей, которые необходимо вернуть. См. подробное описание. список слов, разделенных через запятую
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов, описывающих пользователей, в поле items.
        """

        params = locals()
        params.pop('self')
        return self._call('apps.getFriendsList', params=params)

    def apps_getLeaderboard(self, type=None, global_=None, extended=None):
        """
        Возвращает рейтинг пользователей в игре.

        :param type: тип турнирной таблицы. Возможные значения:   level — по уровням;  points — по очкам, начисленным за выполнение миссий;  score — по очкам, начисленным напрямую (apps.getScore).  строка, обязательный параметр
        :param global: тип рейтинга. Возможные значения:   1 — глобальный рейтинг по всем игрокам (возвращается не более 20 результатов);  0 — рейтинг по друзьям пользователя.  флаг, может принимать значения 1 или 0, по умолчанию 1
        :param extended: 1 — возвращать дополнительную информацию о пользователе. флаг, может принимать значения 1 или 0, по умолчанию 0
        :return: После успешного выполнения возвращает список друзей с текущим уровнем и количеством очков в игре, отсортированный по убыванию текущего уровня или количества очков.
        """

        params = locals()
        params.pop('self')
        return self._call('apps.getLeaderboard', params=params)

    def apps_getScopes(self, type=None):
        """


        :param type: строка, по умолчанию user
        :return:    В ходе выполнения могут произойти общие ошибки. Их описание находится на отдельной странице.
        """

        params = locals()
        params.pop('self')
        return self._call('apps.getScopes', params=params)

    def apps_getScore(self, user_id=None):
        """
        Метод возвращает количество очков пользователя в этой игре.

        :param user_id: идентификатор пользователя. положительное число, по умолчанию идентификатор текущего пользователя, обязательный параметр
        :return: После успешного выполнения возвращает число очков для пользователя.
        """

        params = locals()
        params.pop('self')
        return self._call('apps.getScore', params=params)

    def apps_promoHasActiveGift(self, promo_id=None, user_id=None):
        """
        Проверить есть ли у пользователя подарок в игре. С помощью данного метода можно проверить есть ли у пользователя подарок в вашей игре, полученный в одной и промо-акций платформы

        :param promo_id: Идентификатор промо-акции положительное число, обязательный параметр
        :param user_id: Идентификатор пользователя. Используется только при запросе с сервисным токеном положительное число
        :return: JSON = {    "response": 0|1  }  Где в поле response хранятся данные о наличии подарка в игре  1 - у пользователя есть активный подарок, нужно выдать подарок пользователю в игре и затем погасить его используя метод apps.promoUseGift  0 - у пользователя нет активного подарка в вашей игре
        """

        params = locals()
        params.pop('self')
        return self._call('apps.promoHasActiveGift', params=params)

    def apps_promoUseGift(self, promo_id=None, user_id=None):
        """
        Использовать подарок, полученный пользователем в промо-акции. Данный метод необходимо вызвать после выдачи пользователю внутриигрового подарка, если при вызове метода apps.promoHasActiveGift был получен положительный результат. (т.е. в одной из механик пользователь получил подарок в вашей игре)

        :param promo_id: Идентификатор акции. положительное число, обязательный параметр
        :param user_id: Идентификатор пользователя. Используется только при запросе с сервисным токеном. положительное число
        :return: JSON = {    "response": 0|1  }  Где в поле response хранится результат выполнения операции.  1 -  успех операции  0 - неудача  Также в ходе выполнения метода могут вернуться стандартные ошибки апи
        """

        params = locals()
        params.pop('self')
        return self._call('apps.promoUseGift', params=params)

    def apps_sendRequest(self, user_id=None, text=None, type=None, name=None, key=None, separate=None):
        """
        Позволяет отправить запрос другому пользователю в приложении, использующем авторизацию ВКонтакте. Метод требует использования окна подтверждения. Отправка запросов в приложениях, не опубликованных в Мобильном каталоге, доступна только администраторам приложения.

        :param user_id: идентификатор пользователя, которому следует отправить запрос. положительное число, обязательный параметр
        :param text: текст запроса. строка
        :param type: тип запроса. Возможные значения:   invite – если запрос отправляется пользователю, не установившему приложение;  request – если пользователь уже установил приложение.   Обратите внимание! Для запросов с type = invite действует ограничение — одному и тому же пользователю нельзя отправить запрос чаще одного раза в неделю. строка, по умолчанию request
        :param name: уникальное в рамках приложения имя для каждого вида отправляемого запроса. строка, максимальная длина 128
        :param key: строка, которая будет возвращена назад при переходе пользователя по запросу в приложение. Может использоваться для подсчета конверсии. строка
        :param separate: запрет на группировку запроса с другими, имеющими тот же name. 1 — группировка запрещена, 0 — группировка разрешена. По умолчанию: 0. флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает идентификатор созданного запроса.
        """

        params = locals()
        params.pop('self')
        return self._call('apps.sendRequest', params=params)

    def auth_checkPhone(self, phone=None, client_id=None, client_secret=None, auth_by_phone=None):
        """
        Проверяет правильность введённого номера (возможность его использования для регистрации или авторизации).

        :param phone: номер телефона регистрируемого пользователя. строка, обязательный параметр
        :param client_id: идентификатор Вашего приложения. целое число
        :param client_secret: секретный ключ приложения, доступный в разделе редактирования приложения. строка
        :param auth_by_phone: 1 — проверить правильность номера для авторизации, а не для регистрации нового аккаунта. По умолчанию: 0. флаг, может принимать значения 1 или 0
        :return: В случае, если номер пользователя является корректным, будет возвращён 1.
        """

        params = locals()
        params.pop('self')
        return self._call('auth.checkPhone', params=params)

    def auth_restore(self, phone=None, last_name=None):
        """
        Позволяет восстановить доступ к аккаунту, используя код, полученный через SMS.  Данный метод доступен только приложениям, имеющим доступ к Прямой авторизации.

        :param phone: номер телефона пользователя. строка, обязательный параметр
        :param last_name: фамилия пользователя. строка, обязательный параметр
        :return: В случае успеха метод возвращает объект содержащий следующие поля:  success – 1;  sid – параметр необходимый для получения доступа по коду.  Для завершения восстановления доступа необходимо обратиться по адресу: https://oauth.vk.com/token?grant_type=restore_code&client_id={Идентификатор приложения}&client_secret={Секретный_ключ}&username={Номер телефона}&scope={Список прав доступа}&sid={Параметр, получаемый в данном методе}&code={Код полученный через SMS}  Список параметров:  grant_type – необходимо передать значение: restore_code;  client_id – Идентификатор приложения;  client_secret – Секретный ключ;  username – Номер телефона по которому был восстановлен пароль;  scope – список прав доступа, разделенных через запятую;  sid – идентификатор сессии, полученный в результате выполнения этого метода;  code – код, полученный через SMS.  В результате авторизации через restore_code OAuth вернет данные аналогичные обычной авторизации, с дополнительным параметром change_password_hash необходимым для метода account.changePassword.
        """

        params = locals()
        params.pop('self')
        return self._call('auth.restore', params=params)

    def board_addTopic(self, group_id=None, title=None, text=None, from_group=None, attachments=None):
        """
        Создает новую тему в списке обсуждений группы.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :param title: название обсуждения. строка, обязательный параметр
        :param text: текст первого сообщения в обсуждении. строка
        :param from_group: 1 — тема будет создана от имени группы, 0 — тема будет создана от имени пользователя (по умолчанию). флаг, может принимать значения 1 или 0
        :param attachments: список объектов, приложенных к записи, в формате: <type><owner_id>_<media_id>,<type><owner_id>_<media_id>   <type> — тип медиа-приложения:   photo — фотография;  video — видеозапись ;  audio — аудиозапись;  doc — документ;   <owner_id> — идентификатор владельца медиа-приложения  <media_id> — идентификатор медиа-приложения.  Например:photo100172_166443618,photo66748_265827614  Параметр является обязательным, если не задан параметр text. список слов, разделенных через запятую
        :return: В случае успеха возвращает идентификатор созданной темы.
        """

        params = locals()
        params.pop('self')
        return self._call('board.addTopic', params=params)

    def board_closeTopic(self, group_id=None, topic_id=None):
        """
        Закрывает тему в списке обсуждений группы (в такой теме невозможно оставлять новые сообщения).

        :param group_id: идентификатор сообщества, в котором размещено обсуждение. положительное число, обязательный параметр
        :param topic_id: идентификатор обсуждения. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('board.closeTopic', params=params)

    def board_createComment(self, group_id=None, topic_id=None, message=None, attachments=None, from_group=None,
                            sticker_id=None, guid=None):
        """
        Добавляет новый комментарий в обсуждении.

        :param group_id: идентификатор сообщества, в котором находится обсуждение. положительное число, обязательный параметр
        :param topic_id: идентификатор темы, в которой необходимо оставить комментарий. положительное число, обязательный параметр
        :param message: текст комментария. Обязательный параметр, если не передано значение attachments. строка, строка
        :param attachments: Список объектов, приложенных к комментарию и разделённых символом ",". Поле attachments представляется в формате: <type><owner_id>_<media_id>,<type><owner_id>_<media_id>  <type> — тип медиа-вложения: photo — фотография; video — видеозапись; audio — аудиозапись; doc — документ;  <owner_id> — идентификатор владельца медиа-вложения; <media_id> — идентификатор медиа-вложения.  Например: photo100172_166443618,photo66748_265827614 список слов, разделенных через запятую
        :param from_group: 1 — сообщение будет опубликовано от имени группы, 0 — сообщение будет опубликовано от имени пользователя (по умолчанию). флаг, может принимать значения 1 или 0
        :param sticker_id: идентификатор стикера. положительное число
        :param guid: уникальный идентификатор, предназначенный для предотвращения повторной отправки одинакового комментария. строка
        :return: В случае успеха возвращает идентификатор созданного комментария.
        """

        params = locals()
        params.pop('self')
        return self._call('board.createComment', params=params)

    def board_deleteComment(self, group_id=None, topic_id=None, comment_id=None):
        """
        Удаляет сообщение темы в обсуждениях сообщества.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :param topic_id: идентификатор обсуждения. положительное число, обязательный параметр
        :param comment_id: идентификатор комментария в обсуждении. положительное число, обязательный параметр
        :return: После успешного выполнения, а также в том случае, если комментарий уже удален, возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('board.deleteComment', params=params)

    def board_deleteTopic(self, group_id=None, topic_id=None):
        """
        Удаляет тему в обсуждениях группы.

        :param group_id: идентификатор сообщества, в котором размещено обсуждение. положительное число, обязательный параметр
        :param topic_id: идентификатор обсуждения. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('board.deleteTopic', params=params)

    def board_editComment(self, group_id=None, topic_id=None, comment_id=None, message=None, attachments=None):
        """
        Редактирует одно из сообщений в обсуждении сообщества.

        :param group_id: идентификатор сообщества, в котором размещено обсуждение. положительное число, обязательный параметр
        :param topic_id: идентификатор обсуждения. положительное число, обязательный параметр
        :param comment_id: идентификатор комментария в обсуждении. положительное число, обязательный параметр
        :param message: новый текст комментария (является обязательным, если не задан параметр attachments). строка, строка
        :param attachments: новый список объектов, приложенных к комментарию и разделённых символом ",". Поле attachments представляется в формате: <type><owner_id>_<media_id>,<type><owner_id>_<media_id> <type> — тип медиа-вложения: photo — фотография  video — видеозапись  audio — аудиозапись  doc — документ <owner_id> — идентификатор владельца медиа-вложения  <media_id> — идентификатор медиа-вложения.   Например: photo100172_166443618,photo66748_265827614 Параметр является обязательным, если не задан параметр message. список слов, разделенных через запятую
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('board.editComment', params=params)

    def board_editTopic(self, group_id=None, topic_id=None, title=None):
        """
        Изменяет заголовок темы в списке обсуждений группы.

        :param group_id: идентификатор сообщества, в котором размещено обсуждение. положительное число, обязательный параметр
        :param topic_id: идентификатор обсуждения. положительное число, обязательный параметр
        :param title: новое название обсуждения. строка, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('board.editTopic', params=params)

    def board_fixTopic(self, group_id=None, topic_id=None):
        """
        Закрепляет тему в списке обсуждений группы (такая тема при любой сортировке выводится выше остальных).

        :param group_id: идентификатор сообщества, в котором размещено обсуждение. положительное число, обязательный параметр
        :param topic_id: идентификатор обсуждения. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('board.fixTopic', params=params)

    def board_getComments(self, group_id=None, topic_id=None, need_likes=None, start_comment_id=None, offset=None,
                          count=None, extended=None, sort=None):
        """
        Возвращает список сообщений в указанной теме.

        :param group_id: идентификатор сообщества, информацию об обсуждениях которого нужно получить. положительное число, обязательный параметр
        :param topic_id: идентификатор обсуждения. положительное число, обязательный параметр
        :param need_likes: 1 — будет возвращено дополнительное поле likes. По умолчанию поле likes не возвращается. флаг, может принимать значения 1 или 0
        :param start_comment_id: идентификатор комментария, начиная с которого нужно вернуть список (подробности см. ниже). положительное число, доступен начиная с версии 5.33
        :param offset: смещение, необходимое для выборки определенного подмножества сообщений. целое число
        :param count: количество сообщений, которое необходимо получить (но не более 100). По умолчанию — 20. положительное число, по умолчанию 20, максимальное значение 100
        :param extended: Если указать в качестве этого параметра 1, то будет возвращена информация о пользователях, являющихся авторами сообщений. По умолчанию 0. флаг, может принимать значения 1 или 0
        :param sort: порядок сортировки комментариев: asc — хронологический desc — антихронологический строка
        :return: В случае успеха возвращает объект, содержащий поле items, поле poll, если к теме прикреплен опрос, а также поле profiles (если был указан параметр extended).  Поле items представляет собой массив, содержащий объекты, описывающие комментарии.  Если к теме был прикреплен опрос, возвращается поле poll следующего вида:   owner_id — идентификатор владельца опроса  poll_id — идентификатор опроса;  created — время создания опроса в формате unixtime;  is_closed — 1, если опрос закрыт (в нем больше нельзя принимать участие) или 0 в противном случае;  question — текст вопроса для опроса;  votes — общее количество ответивших пользователей;  answer_id — идентификатор ответа текущего пользователя (если текущий пользователь еще не отвечал в данном опросе, то содержит 0);  answers — массив, содержащий объекты с вариантами ответа на вопрос в опросе:   id — идентификатор ответа на вопрос;  text — текст ответа на вопрос;  votes — количество пользователей, проголосовавших за данный вариант ответа;  rate — рейтинг данного варианта ответа, выраженный в процентах.    В случае передачи параметра extended равным 1, поле profiles содержит массив объектов user  с информацией о данных пользователей, являющихся авторами сообщений.  Если был передан параметр start_comment_id, будет также возвращено поле real_offset – итоговое смещение данного подмножества комментариев (оно может быть отрицательным, если был указан отрицательный offset).
        """

        params = locals()
        params.pop('self')
        return self._call('board.getComments', params=params)

    def board_getTopics(self, group_id=None, topic_ids=None, order=None, offset=None, count=None, extended=None,
                        preview=None, preview_length=None):
        """
        Возвращает список тем в обсуждениях указанной группы.

        :param group_id: идентификатор сообщества, информацию об обсуждениях которого необходимо получить. Если сообщество закрытое или частное, для вызова метода потребуется право доступа groups. положительное число, обязательный параметр
        :param topic_ids: список идентификаторов тем, которые необходимо получить (не более 100). По умолчанию возвращаются все темы. Если указан данный параметр, игнорируются параметры order, offset и count (возвращаются все запрошенные темы в указанном порядке). список целых чисел, разделенных запятыми
        :param order: Порядок, в котором необходимо вернуть список тем. Возможные значения:   1 — по убыванию даты обновления;  2 — по убыванию даты создания;  -1 — по возрастанию даты обновления;  -2 — по возрастанию даты создания.  По умолчанию темы возвращаются в порядке, установленном администратором группы. "Прилепленные" темы при любой сортировке возвращаются первыми в списке. целое число
        :param offset: смещение, необходимое для выборки определенного подмножества тем. положительное число
        :param count: количество тем, которое необходимо получить (но не более 100). По умолчанию — 40. положительное число, по умолчанию 40, максимальное значение 100
        :param extended: Если указать в качестве этого параметра 1, то будет возвращена информация о пользователях, являющихся создателями тем или оставившими в них последнее сообщение. По умолчанию 0. флаг, может принимать значения 1 или 0
        :param preview: Набор флагов, определяющий, необходимо ли вернуть вместе с информацией о темах текст первых и последних сообщений в них. Является суммой флагов:   1 — вернуть первое сообщение в каждой теме (поле first_comment);  2 — вернуть последнее сообщение в каждой теме (поле last_comment). По умолчанию — 0 (не возвращать текст сообщений).  целое число
        :param preview_length: Количество символов, по которому нужно обрезать первое и последнее сообщение. Укажите 0, если Вы не хотите обрезать сообщение. (по умолчанию — 90). положительное число, по умолчанию 90
        :return: После успешного выполнения возвращает объект, содержащий следующие поля:  count integer число результатов. items array массив объектов, описывающих обсуждения. default_order integer тип сортировки обсуждений в сообществе. Возможные значения:   1 — по убыванию даты обновления;  2 — по убыванию даты создания;  -1 — по возрастанию даты обновления;  -2 — по возрастанию даты создания. can_add_topics integer, [0,1] информация о том, может ли текущий пользователь создать обсуждение в сообществе. Возможные значения:   0 — не может;  1 — может. profiles array (только для extended = 1). Массив объектов, описывающих пользователей. Каждый из объектов содержит базовые поля, а также поля sex, photo_50, photo_100, online и type (= profile).
        """

        params = locals()
        params.pop('self')
        return self._call('board.getTopics', params=params)

    def board_openTopic(self, group_id=None, topic_id=None):
        """
        Открывает ранее закрытую тему (в ней станет возможно оставлять новые сообщения).

        :param group_id: идентификатор сообщества, в котором размещено обсуждение. положительное число, обязательный параметр
        :param topic_id: идентификатор обсуждения. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('board.openTopic', params=params)

    def board_restoreComment(self, group_id=None, topic_id=None, comment_id=None):
        """
        Восстанавливает удаленное сообщение темы в обсуждениях группы.

        :param group_id: идентификатор сообщества, в котором размещено обсуждение. положительное число, обязательный параметр
        :param topic_id: идентификатор обсуждения. положительное число, обязательный параметр
        :param comment_id: идентификатор комментария. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('board.restoreComment', params=params)

    def board_unfixTopic(self, group_id=None, topic_id=None):
        """
        Отменяет прикрепление темы в списке обсуждений группы (тема будет выводиться согласно выбранной сортировке).

        :param group_id: идентификатор сообщества, в котором размещено обсуждение. положительное число, обязательный параметр
        :param topic_id: идентификатор обсуждения. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('board.unfixTopic', params=params)

    def database_getChairs(self, faculty_id=None, offset=None, count=None):
        """
        Возвращает список кафедр университета по указанному факультету.

        :param faculty_id: идентификатор факультета, кафедры которого необходимо получить. положительное число, обязательный параметр
        :param offset: отступ, необходимый для получения определенного подмножества кафедр. положительное число
        :param count: количество кафедр которое необходимо получить. положительное число, по умолчанию 100, максимальное значение 10000
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов, описывающих кафедры, в поле items.  Каждый объект содержит следующие поля:   id (integer) — идентификатор кафедры;  title (string) — название кафедры.
        """

        params = locals()
        params.pop('self')
        return self._call('database.getChairs', params=params)

    def database_getCities(self, country_id=None, region_id=None, q=None, need_all=None, offset=None, count=None):
        """
        Возвращает список городов.

        :param country_id: идентификатор страны, полученный  database.getCountries. положительное число, обязательный параметр
        :param region_id: идентификатор региона, города которого необходимо получить. положительное число
        :param q: строка поискового запроса. Например, Санкт.  Максимальная длина строки — 15 символов. строка
        :param need_all:   1 – возвращать все города.  0 – возвращать только основные города.  флаг, может принимать значения 1 или 0
        :param offset: отступ, необходимый для получения определенного подмножества городов. положительное число
        :param count: количество городов, которые необходимо вернуть. положительное число, по умолчанию 100, максимальное значение 1000
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов, описывающих города, в поле items.  Каждый объект содержит следующие поля:   id (integer) — идентификатор города;  title (string) — название города.   Объект может содержать дополнительные поля:   area (string) — область;  region (string) — регион.   Если не задан параметр q, то будет возвращен список самых крупных городов в заданной стране. Если задан параметр q, то будет возвращен список городов, которые релевантны поисковому запросу. Полем important отмечены ключевые города для текущего пользователя.
        """

        params = locals()
        params.pop('self')
        return self._call('database.getCities', params=params)

    def database_getCitiesById(self, city_ids=None):
        """
        Возвращает информацию о городах и регионах по их идентификаторам. Идентификаторы (id) могут быть получены с помощью методов users.get, places.getById, places.search, places.getCheckins.

        :param city_ids: идентификаторы городов и регионов. список положительных чисел, разделенных запятыми, количество элементов должно составлять не более 1000
        :return: Возвращает массив объектов, описывающих города.  Каждый объект содержит следующие поля:   id (integer) — идентификатор города или региона;  title (string) — название города или региона.
        """

        params = locals()
        params.pop('self')
        return self._call('database.getCitiesById', params=params)

    def database_getCountries(self, need_all=None, code=None, offset=None, count=None):
        """
        Возвращает список стран.

        :param need_all: 1 — вернуть список всех стран. флаг, может принимать значения 1 или 0
        :param code: перечисленные через запятую двухбуквенные коды стран в стандарте ISO 3166-1 alpha-2, для которых необходимо выдать информацию. Пример значения code: RU,UA,BY строка
        :param offset: отступ, необходимый для выбора определенного подмножества стран. положительное число
        :param count: количество стран, которое необходимо вернуть. положительное число, по умолчанию 100, максимальное значение 1000
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов, описывающих страны, в поле items.  Каждый объект содержит следующие поля:   id (integer) — идентификатор страны;  title (string) — название страны.   Если не заданы параметры need_all и code, то возвращается краткий список стран, расположенных наиболее близко к стране текущего пользователя. Если задан параметр need_all, то будет возвращен список всех стран. Если задан параметр code, то будут возвращены только страны с перечисленными ISO 3166-1 alpha-2 кодами.
        """

        params = locals()
        params.pop('self')
        return self._call('database.getCountries', params=params)

    def database_getCountriesById(self, country_ids=None):
        """
        Возвращает информацию о странах по их идентификаторам Идентификаторы (id) могут быть получены с помощью методов users.get, places.getById, places.search, places.getCheckins.

        :param country_ids: идентификаторы стран. список положительных чисел, разделенных запятыми, количество элементов должно составлять не более 1000
        :return: Возвращает массив объектов, описывающих страны.  Каждый объект содержит следующие поля:   id (integer) — идентификатор региона;  title (string) — название региона.
        """

        params = locals()
        params.pop('self')
        return self._call('database.getCountriesById', params=params)

    def database_getFaculties(self, university_id=None, offset=None, count=None):
        """
        Возвращает список факультетов.

        :param university_id: идентификатор университета, факультеты которого необходимо получить. положительное число, обязательный параметр
        :param offset: отступ, необходимый для получения определенного подмножества факультетов. положительное число
        :param count: количество факультетов которое необходимо получить. положительное число, по умолчанию 100, максимальное значение 10000
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов, описывающих факультеты, в поле items.  Каждый объект содержит следующие поля:   id (integer) — идентификатор факультета;  title (string) — название факультета.
        """

        params = locals()
        params.pop('self')
        return self._call('database.getFaculties', params=params)

    def database_getMetroStations(self, city_id=None, offset=None, count=None, extended=None):
        """
        Возвращает список станций метро

        :param city_id: идентификатор города, полученный методом database.getCities. положительное число, обязательный параметр
        :param offset: отступ, необходимый для выбора определенного подмножества станций. положительное число
        :param count: количество станций, которое необходимо вернуть. положительное число, по умолчанию 100, максимальное значение 500
        :param extended: флаг, может принимать значения 1 или 0, по умолчанию
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов, описывающих станции метро, в поле items.  Каждый объект содержит следующие поля:   id (integer) — идентификатор;  name (string) — название;  color (string) — цвет линии.
        """

        params = locals()
        params.pop('self')
        return self._call('database.getMetroStations', params=params)

    def database_getMetroStationsById(self, station_ids=None):
        """
        Возвращает информацию об одной или нескольких станциях метро по их идентификаторам.

        :param station_ids: список идентификаторов станций метро список положительных чисел, разделенных запятыми
        :return: После успешного выполнения возвращает массив объектов, описывающих станции метро.  Каждый объект содержит следующие поля:   id (integer) — идентификатор;  name (string) — название;  color (string) — цвет линии.
        """

        params = locals()
        params.pop('self')
        return self._call('database.getMetroStationsById', params=params)

    def database_getRegions(self, country_id=None, q=None, offset=None, count=None):
        """
        Возвращает список регионов.

        :param country_id: идентификатор страны, полученный в методе database.getCountries. положительное число, обязательный параметр
        :param q: строка поискового запроса. Например, Лен. строка
        :param offset: отступ, необходимый для выбора определенного подмножества регионов. положительное число
        :param count: количество регионов, которое необходимо вернуть. положительное число, по умолчанию 100, максимальное значение 1000
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов, описывающих регионы, в поле items.  Каждый объект содержит следующие поля:   id (integer) — идентификатор региона;  title (string) — название региона.   Если не задан параметр q, то будет возвращен список всех регионов в заданной стране. Если задан параметр q, то будет возвращен список регионов, которые релевантны поисковому запросу.
        """

        params = locals()
        params.pop('self')
        return self._call('database.getRegions', params=params)

    def database_getSchoolClasses(self, country_id=None):
        """
        Возвращает список классов, характерных для школ определенной страны.

        :param country_id: идентификатор страны, доступные классы в которой необходимо вернуть. положительное число
        :return: Возвращает массив, каждый элемент которого представляет собой пару: идентификатор и строковое обозначение класса.
        """

        params = locals()
        params.pop('self')
        return self._call('database.getSchoolClasses', params=params)

    def database_getSchools(self, q=None, city_id=None, offset=None, count=None):
        """
        Возвращает список школ.

        :param q: строка поискового запроса. Например, гимназия. строка
        :param city_id: идентификатор города, школы которого необходимо вернуть. положительное число, обязательный параметр
        :param offset: отступ, необходимый для получения определенного подмножества школ. положительное число
        :param count: количество школ, которое необходимо вернуть. положительное число, по умолчанию 100, максимальное значение 10000
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов, описывающих школы, в поле items.  Каждый объект содержит следующие поля:   id (integer) — идентификатор школы;  title (string) — название школы.
        """

        params = locals()
        params.pop('self')
        return self._call('database.getSchools', params=params)

    def database_getUniversities(self, q=None, country_id=None, city_id=None, offset=None, count=None):
        """
        Возвращает список высших учебных заведений.

        :param q: строка поискового запроса. Например, СПБ. строка
        :param country_id: идентификатор страны, учебные заведения которой необходимо вернуть. положительное число
        :param city_id: идентификатор города, учебные заведения которого необходимо вернуть. положительное число
        :param offset: отступ, необходимый для получения определенного подмножества учебных заведений. положительное число
        :param count: количество учебных заведений, которое необходимо вернуть. положительное число, по умолчанию 100, максимальное значение 10000
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов, описывающих университеты, в поле items.  Каждый объект содержит следующие поля:   id (integer) — идентификатор университета;  title (string) — название университета.
        """

        params = locals()
        params.pop('self')
        return self._call('database.getUniversities', params=params)

    def docs_add(self, owner_id=None, doc_id=None, access_key=None):
        """
        Копирует документ в документы текущего пользователя.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит документ.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param doc_id: идентификатор документа. положительное число, обязательный параметр
        :param access_key: ключ доступа документа. Этот параметр следует передать, если вместе с остальными данными о документе было возвращено поле access_key. строка
        :return: После успешного выполнения возвращает идентификатор созданного документа.
        """

        params = locals()
        params.pop('self')
        return self._call('docs.add', params=params)

    def docs_delete(self, owner_id=None, doc_id=None):
        """
        Удаляет документ пользователя или группы.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит документ.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param doc_id: идентификатор документа. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('docs.delete', params=params)

    def docs_edit(self, owner_id=None, doc_id=None, title=None, tags=None):
        """
        Редактирует документ пользователя или группы.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит документ.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя, обязательный параметр
        :param doc_id: идентификатор документа. положительное число, обязательный параметр
        :param title: название документа. строка, максимальная длина 128
        :param tags: метки для поиска. список слов, разделенных через запятую
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('docs.edit', params=params)

    def docs_get(self, count=None, offset=None, type=None, owner_id=None, return_tags=None):
        """
        Возвращает расширенную информацию о документах пользователя или сообщества.

        :param count: количество документов, информацию о которых нужно вернуть.  По умолчанию: все документы. Максимальное количество документов, которое можно получить: 2000. положительное число
        :param offset: смещение, необходимое для выборки определенного подмножества документов. Максимальное значение: 1999. положительное число
        :param type: фильтр по типу документа.  Возможные варианты:   1 — текстовые документы;  2 — архивы;  3 — gif;  4 — изображения;  5 — аудио;  6 — видео;  7 — электронные книги;  8 — неизвестно.  положительное число, по умолчанию 0
        :param owner_id: идентификатор пользователя или сообщества, которому принадлежат документы.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param return_tags: флаг, может принимать значения 1 или 0, по умолчанию
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов документов в поле items.
        """

        params = locals()
        params.pop('self')
        return self._call('docs.get', params=params)

    def docs_getById(self, docs=None, return_tags=None):
        """
        Возвращает информацию о документах по их идентификаторам.

        :param docs: идентификаторы документов, информацию о которых нужно вернуть. Пример значения docs:  66748_91488,66748_91455.  Метод поддерживает передачу access_key. список слов, разделенных через запятую, обязательный параметр
        :param return_tags: флаг, может принимать значения 1 или 0, по умолчанию
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов документов в поле items.
        """

        params = locals()
        params.pop('self')
        return self._call('docs.getById', params=params)

    def docs_getMessagesUploadServer(self, type=None, peer_id=None):
        """
        Получает адрес сервера для загрузки документа в личное сообщение.

        :param type: тип документа. Возможные значения:   doc — обычный документ;  audio_message — голосовое сообщение;  graffiti — граффити.  строка, по умолчанию doc
        :param peer_id: идентификатор назначения.  Для пользователя:  id  пользователя.   Для групповой беседы:  2000000000 + id беседы.   Для сообщества:  -id сообщества.   целое число
        :return: Возвращает объект с единственным полем upload_url (string), содержащим адрес сервера для загрузки документа.
        """

        params = locals()
        params.pop('self')
        return self._call('docs.getMessagesUploadServer', params=params)

    def docs_getTypes(self, owner_id=None):
        """
        Возвращает доступные для пользователя типы документов.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежат документы. целое число, по умолчанию идентификатор текущего пользователя, обязательный параметр
        :return: После успешного выполнения возвращает список объектов type с полями, описанными ниже:  id  integer идентификатор типа. name  string название типа. count   integer число документов;
        """

        params = locals()
        params.pop('self')
        return self._call('docs.getTypes', params=params)

    def docs_getUploadServer(self, group_id=None):
        """
        Возвращает адрес сервера для загрузки документов.

        :param group_id: идентификатор сообщества (если необходимо загрузить документ в список документов сообщества). Если документ нужно загрузить в список пользователя, метод вызывается без дополнительных параметров. положительное число
        :return: Возвращает объект с полем upload_url, содержащим адрес для загрузки файла. После успешной загрузки документа необходимо воспользоваться методом docs.save.
        """

        params = locals()
        params.pop('self')
        return self._call('docs.getUploadServer', params=params)

    def docs_getWallUploadServer(self, group_id=None):
        """
        Возвращает адрес сервера для загрузки документов в папку Отправленные, для последующей отправки документа на стену или личным сообщением.

        :param group_id: идентификатор сообщества, в которое нужно загрузить документ. положительное число
        :return: Возвращает объект с полем upload_url. После успешной загрузки документа необходимо воспользоваться методом docs.save.
        """

        params = locals()
        params.pop('self')
        return self._call('docs.getWallUploadServer', params=params)

    def docs_save(self, file=None, title=None, tags=None, return_tags=None):
        """
        Сохраняет документ после его успешной  загрузки на сервер.

        :param file: параметр, возвращаемый в результате  загрузки файла на сервер. строка, обязательный параметр
        :param title: название документа. строка
        :param tags: метки для поиска. строка
        :param return_tags: флаг, может принимать значения 1 или 0, по умолчанию
        :return: После успешного выполнения возвращает массив объектов, описывающих загруженные документы. Каждый объект массива содержит поле type (string) с возможными значениями graffiti, audio_message, doc, и одноименный объект, описывающий загруженный документ.  Для type = graffiti объект graffiti, описывающий граффити.  Для type = audio_message объект audio_message, описывающий голосовое сообщение.  Для type = doc объект doc, описывающий другие типы документов.
        """

        params = locals()
        params.pop('self')
        return self._call('docs.save', params=params)

    def docs_search(self, q=None, search_own=None, count=None, offset=None, return_tags=None):
        """
        Возвращает результаты поиска по документам.

        :param q: строка поискового запроса. Например, зеленые тапочки. строка, обязательный параметр, максимальная длина 512
        :param search_own: 1 — искать среди собственных документов пользователя. флаг, может принимать значения 1 или 0
        :param count: количество документов, информацию о которых нужно вернуть. Обратите внимание — даже при использовании параметра offset для получения информации доступны только первые 1000 результатов.   положительное число, по умолчанию 20
        :param offset: смещение, необходимое для выборки определенного подмножества документов. положительное число
        :param return_tags: флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает список объектов документов.  Если установлен флаг search_own, в поле local дополнительно вернется список документов из локального поиска.
        """

        params = locals()
        params.pop('self')
        return self._call('docs.search', params=params)

    def execute(self, code=None):
        """
        Универсальный метод, который позволяет запускать последовательность других методов, сохраняя и фильтруя промежуточные результаты.

        :param code: код алгоритма в VKScript - формате, похожем на JavaSсript или ActionScript (предполагается совместимость с ECMAScript). Алгоритм должен завершаться командой return %выражение%. Операторы должны быть разделены точкой с запятой. строка
        :return: Возвращает данные, запрошенные алгоритмом. При работе с методом execute структура ответа в XML ближе к JSON и может незначительно отличаться от стандартного представления других методов.  Обратите внимание: версия API, указанная при вызове execute, не применяется автоматически к методам, вызываемым внутри code.  Обратите внимание! Существует ограничение на объем ответа. В случае, если данные в ответе занимают больше 5 МБ, вернется ошибка 13 с комментарием "response size is too big".
        """

        params = locals()
        params.pop('self')
        return self._call('execute', params=params)

    def fave_addArticle(self, url=None):
        """
        Добавляет статью в закладки.

        :param url: ссылка на добавляемую статью. строка, обязательный параметр
        :return: В случае успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('fave.addArticle', params=params)

    def fave_addLink(self, link=None):
        """
        Добавляет ссылку в закладки.

        :param link: адрес добавляемой ссылки. Поддерживаются только внутренние ссылки на vk.com. строка, обязательный параметр
        :return: В случае успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('fave.addLink', params=params)

    def fave_addPage(self, user_id=None, group_id=None):
        """
        Добавляет сообщество или пользователя в закладки.

        :param user_id: идентификатор пользователя, которого нужно добавить в закладки. положительное число
        :param group_id: идентификатор сообщества, которое нужно добавить в закладки. положительное число
        :return: В случае успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('fave.addPage', params=params)

    def fave_addPost(self, owner_id=None, id=None, access_key=None, ref=None, track_code=None, source=None):
        """
        Добавляет запись со стены пользователя или сообщества в закладки.

        :param owner_id: идентификатор пользователя или сообщества, чью запись со стены требуется добавить в закладки. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param id: идентификатор записи, которую необходимо добавить в закладки. целое число, обязательный параметр
        :param access_key: специальный код доступа для приватных постов. строка
        :param ref: строка
        :param track_code: строка
        :param source: строка
        :return: В случае успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('fave.addPost', params=params)

    def fave_addProduct(self, owner_id=None, id=None, access_key=None):
        """
        Добавляет товар в закладки.

        :param owner_id: идентификатор пользователя или сообщества, чей товар требуется добавить в закладки. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param id: идентификатор товара. целое число, обязательный параметр
        :param access_key: специальный код доступа для приватных товаров. строка
        :return: В случае успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('fave.addProduct', params=params)

    def fave_addTag(self, name=None, position=None):
        """
        Создает метку закладок.

        :param name: название метки. строка, максимальная длина 50
        :param position: положение метки.  Возможные значения:   front - начало списка,  back - конец списка.  строка, по умолчанию back
        :return: В случае успешного выполнения возвращает объект метки с полями name - названием метки и id - идентификатором созданной метки.
        """

        params = locals()
        params.pop('self')
        return self._call('fave.addTag', params=params)

    def fave_addVideo(self, owner_id=None, id=None, access_key=None, ref=None):
        """
        Добавляет видеозапись в закладки.

        :param owner_id: идентификатор пользователя или сообщества, владельца видеозаписи, которую требуется добавить в закладки. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param id: идентификатор видеозаписи. целое число, обязательный параметр
        :param access_key: специальный код доступа для приватных видеозаписей. строка
        :param ref: строка
        :return: В случае успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('fave.addVideo', params=params)

    def fave_editTag(self, id=None, name=None):
        """
        Редактирует метку.

        :param id: идентификатор метки. целое число, обязательный параметр
        :param name: новое название метки. строка, обязательный параметр, максимальная длина 50
        :return: В случае успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('fave.editTag', params=params)

    def fave_get(self, extended=None, item_type=None, tag_id=None, offset=None, count=None, fields=None,
                 is_from_snackbar=None):
        """
        Возвращает объекты, добавленные в закладки пользователя.

        :param extended: 1, если необходимо получить информацию о пользователе. По умолчанию: 0. флаг, может принимать значения 1 или 0, по умолчанию
        :param item_type: Типы объектов, которые необходимо вернуть. Возможные значения:   post — записи;  video — видеозаписи;  product — товары;  article — статьи;  link — ссылки.  строка
        :param tag_id: идентификатор метки, закладки отмеченные которой требуется вернуть. целое число
        :param offset: смещение относительно первого объекта в закладках пользователя для выборки определенного подмножества. положительное число
        :param count: количество возвращаемых закладок. целое число, по умолчанию 50, минимальное значение 1, максимальное значение 100
        :param fields: список дополнительных полей профилей, которые необходимо вернуть. См. подробное описание. строка
        :param is_from_snackbar: флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов, добавленных в закладки, в поле items.  Каждый из объектов содержит следующие поля:  type  stringтип объекта, добавленного в закладки. Возможные значения:   post — запись на стене;  video — видеозапись;  product — товар;  article — статья;  link — ссылки. seen  booleanявляется ли закладка просмотренной. added_date  integerдата добавления объекта в закладки в формате Unixtime. tags  arrayмассив меток объекта в списке закладок. {object_type}  objectобъект, добавленный в закладки.   Если был задан параметр extended=1, возвращает число результатов в поле count, отдельно массив объектов пользователей в поле profiles и сообществ в поле groups.
        """

        params = locals()
        params.pop('self')
        return self._call('fave.get', params=params)

    def fave_getPages(self, offset=None, count=None, type=None, fields=None, tag_id=None):
        """
        Возвращает страницы пользователей и сообществ, добавленных в закладки.

        :param offset: смещение относительно первого объекта в закладках пользователя для выборки определенного подмножества. положительное число, максимальное значение 10000
        :param count: количество возвращаемых закладок. положительное число, по умолчанию 50, минимальное значение 1, максимальное значение 500
        :param type: Типы объектов, которые необходимо вернуть. Возможные значения:   users — вернуть только пользователей;  groups — вернуть только сообщества;  hints — топ сообществ и пользователей.   Если параметр не указан — вернутся объекты пользователей и сообществ, добавленных в закладки, в порядке добавления. строка
        :param fields: список дополнительных полей для объектов user и group, которые необходимо вернуть. список слов, разделенных через запятую
        :param tag_id: идентификатор метки, закладки отмеченные которой требуется вернуть. целое число
        :return:
        """

        params = locals()
        params.pop('self')
        return self._call('fave.getPages', params=params)

    def fave_getTags(self):
        """
        Возвращает список меток в закладках.

        :return:    В ходе выполнения могут произойти общие ошибки. Их описание находится на отдельной странице.
        """

        params = locals()
        params.pop('self')
        return self._call('fave.getTags', params=params)

    def fave_markSeen(self):
        """
        Отмечает закладки как просмотренные.

        :return:    В ходе выполнения могут произойти общие ошибки. Их описание находится на отдельной странице.
        """

        params = locals()
        params.pop('self')
        return self._call('fave.markSeen', params=params)

    def fave_removeArticle(self, owner_id=None, article_id=None, ref=None):
        """
        Удаляет статью из закладок.

        :param owner_id: идентификатор владельца статьи. целое число, обязательный параметр
        :param article_id: идентификатор статьи. положительное число, обязательный параметр
        :param ref: строка
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('fave.removeArticle', params=params)

    def fave_removeLink(self, link_id=None):
        """
        Удаляет ссылку из списка закладок пользователя.

        :param link_id: идентификатор ссылки. строка
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('fave.removeLink', params=params)

    def fave_removePage(self, user_id=None, group_id=None):
        """
        Удаляет из закладок сообщество или страницу пользователя.

        :param user_id: идентификатор пользователя, которого следует удалить из закладок. целое число
        :param group_id: идентификатор сообщества, которое следует удалить из закладок. целое число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('fave.removePage', params=params)

    def fave_removePost(self, owner_id=None, id=None):
        """
        Удаляет из закладок запись на стене пользователя или сообщества.

        :param owner_id: идентификатор владельца записи. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param id: идентификатор записи на стене. целое число, обязательный параметр
        :return: В случае успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('fave.removePost', params=params)

    def fave_removeProduct(self, owner_id=None, id=None):
        """
        Удаляет товар из закладок.

        :param owner_id: идентификатор пользователя или сообщества, чей товар требуется удалить из закладок. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param id: идентификатор товара. целое число, обязательный параметр
        :return: В случае успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('fave.removeProduct', params=params)

    def fave_removeTag(self, id=None):
        """
        Удаляет метку закладок.

        :param id: идентификатор метки. целое число, обязательный параметр
        :return: В случае успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('fave.removeTag', params=params)

    def fave_removeVideo(self, owner_id=None, id=None):
        """
        Удаляет видеозапись из списка закладок.

        :param owner_id: идентификатор пользователя или сообщества, владельца видеозаписи, которую требуется удалить из закладок. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param id: идентификатор видеозаписи. целое число, обязательный параметр
        :return: В случае успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('fave.removeVideo', params=params)

    def fave_reorderTags(self, ids=None):
        """
        Меняет порядок меток закладок в списке.

        :param ids: перечисленные через запятую все идентификаторы меток пользователя в том порядке, в котором их требуется отображать. список целых чисел, разделенных запятыми, обязательный параметр
        :return: В случае успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('fave.reorderTags', params=params)

    def fave_setPageTags(self, user_id=None, group_id=None, tag_ids=None):
        """
        Устанавливает метку странице пользователя или сообщества.

        :param user_id: идентификатор пользователя, которому требуется проставить метку в закладках. Обязательный параметр, если не задан параметр group_id. положительное число
        :param group_id: идентификатор сообщества, которому требуется проставить метку в закладках. Обязательный параметр, если не задан параметр user_id. положительное число
        :param tag_ids: перечисленные через запятую идентификаторы тегов, которые требуется присвоить странице. список целых чисел, разделенных запятыми
        :return: В случае успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('fave.setPageTags', params=params)

    def fave_setTags(self, item_type=None, item_owner_id=None, item_id=None, tag_ids=None, link_id=None, link_url=None):
        """
        Устанавливает метку выбранному объекту в списке закладок.

        :param item_type: Тип объекта, которому необходимо присвоить метку. Возможные значения:   post — запись на стене;  video — видеозапись;  product — товар;  article — статья;  link — ссылка.   Для работы с объектами пользователя или сообщества используйте метод fave.setPageTags строка
        :param item_owner_id: идентификатор владельца объекта, которому требуется присвоить метку. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число
        :param item_id: идентификатор объекта. целое число
        :param tag_ids: идентификатор метки, которую требуется присвоить объекту. список целых чисел, разделенных запятыми
        :param link_id: идентификатор ссылки, которой требуется присвоить метку. строка
        :param link_url: строка
        :return: В случае успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('fave.setTags', params=params)

    def fave_trackPageInteraction(self, user_id=None, group_id=None):
        """
        Устанавливает страницу пользователя или сообщества в топ закладок. Метод для работы с разделом "Недавние" на странице https://vk.com/bookmarks.  Вызовите этот метод при переходе из закладок на страницу пользователя или сообщества, чтобы поднять закладку на первое место в списке недавно посещенных.  Всего в списке может быть не более 20 объектов.

        :param user_id: идентификатор пользователя, которому требуется проставить метку в закладках. Обязательный параметр, если не задан параметр group_id. положительное число
        :param group_id: идентификатор сообщества, которому требуется проставить метку в закладках. Обязательный параметр, если не задан параметр user_id. положительное число
        :return: В случае успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('fave.trackPageInteraction', params=params)

    def friends_add(self, user_id=None, text=None, follow=None):
        """
        Одобряет или создает заявку на добавление в друзья. Если идентификатор выбранного пользователя присутствует в списке заявок на добавление в друзья, полученном методом friends.getRequests, то одобряет заявку на добавление и добавляет выбранного пользователя в друзья к текущему пользователю. В противном случае создает заявку на добавление в друзья текущего пользователя к выбранному пользователю.

        :param user_id: идентификатор пользователя, которому необходимо отправить заявку, либо заявку от которого необходимо одобрить. положительное число
        :param text: текст сопроводительного сообщения для заявки на добавление в друзья. Максимальная длина сообщения — 500 символов. строка
        :param follow: 1, если необходимо отклонить входящую заявку (оставить пользователя в подписчиках). флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает одно из следующих значений:   1 — заявка на добавление данного пользователя в друзья отправлена;  2 — заявка на добавление в друзья от данного пользователя одобрена;  4 — повторная отправка заявки.
        """

        params = locals()
        params.pop('self')
        return self._call('friends.add', params=params)

    def friends_addList(self, name=None, user_ids=None):
        """
        Создает новый список друзей у текущего пользователя.

        :param name: название создаваемого списка друзей. строка, обязательный параметр
        :param user_ids: идентификаторы пользователей, которых необходимо поместить в созданный список. список положительных чисел, разделенных запятыми
        :return: После успешного выполнения возвращает идентификатор (list_id) созданного списка друзей.
        """

        params = locals()
        params.pop('self')
        return self._call('friends.addList', params=params)

    def friends_areFriends(self, user_ids=None, need_sign=None):
        """
        Возвращает информацию о том, добавлен ли текущий пользователь в друзья у указанных пользователей. Также возвращает информацию о наличии исходящей или входящей заявки в друзья (подписки).

        :param user_ids: идентификаторы пользователей, статус дружбы с которыми нужно проверить. список целых чисел, разделенных запятыми, обязательный параметр
        :param need_sign: 1 – необходимо вернуть поле sign которое равно:  md5("{id}_{user_id}_{friends_status}_{application_secret}"), где id - это идентификатор пользователя, для которого выполняется запрос.  и позволяет на сервере убедиться что данные не были подделаны на клиенте.  0 – поле sign возвращать не нужно. флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает массив объектов, каждый из которых содержит следующие поля:   user_id — идентификатор пользователя (из числа переданных в параметре user_ids);  friend_status — статус дружбы с пользователем:   0 – пользователь не является другом,  1 – отправлена заявка/подписка пользователю,  2 – имеется входящая заявка/подписка от пользователя,  3 – пользователь является другом;   request_message — текст сообщения, прикрепленного к заявке в друзья (если есть).  read_state — статус заявки (0 — не просмотрена, 1 — просмотрена), возвращается только если friend_status = 2;
        """

        params = locals()
        params.pop('self')
        return self._call('friends.areFriends', params=params)

    def friends_delete(self, user_id=None):
        """
        Удаляет пользователя из списка друзей или отклоняет заявку в друзья. Если идентификатор выбранного пользователя присутствует в списке заявок на добавление в друзья, полученном методом friends.getRequests, то отклоняет заявку на добавление в друзья к текущему пользователю. В противном случае удаляет выбранного пользователя из списка друзей текущего пользователя, который может быть получен методом friends.get.

        :param user_id: идентификатор пользователя, которого необходимо удалить из списка друзей, либо заявку от которого необходимо отклонить. положительное число
        :return: После успешного выполнения начиная с версии 5.28 возвращается объект с полями:   success — удалось успешно удалить друга  friend_deleted — был удален друг  out_request_deleted  — отменена исходящая заявка  in_request_deleted  — отклонена входящая заявка  suggestion_deleted  — отклонена рекомендация друга   Для версии 5.27 и более старых возвращает одно из следующих значений:   1 — пользователь удален из списка друзей;  2 — заявка на добавление в друзья данного пользователя отклонена (входящая или исходящая);  3 — рекомендация добавить в друзья данного пользователя удалена.
        """

        params = locals()
        params.pop('self')
        return self._call('friends.delete', params=params)

    def friends_deleteAllRequests(self):
        """
        Отмечает все входящие заявки на добавление в друзья как просмотренные.

        :return:    В ходе выполнения могут произойти общие ошибки. Их описание находится на отдельной странице.
        """

        params = locals()
        params.pop('self')
        return self._call('friends.deleteAllRequests', params=params)

    def friends_deleteList(self, list_id=None):
        """
        Удаляет существующий список друзей текущего пользователя.

        :param list_id: идентификатор списка друзей, который необходимо удалить. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('friends.deleteList', params=params)

    def friends_edit(self, user_id=None, list_ids=None):
        """
        Редактирует списки друзей для выбранного друга.

        :param user_id: идентификатор пользователя (из числа друзей), для которого необходимо отредактировать списки друзей. положительное число, обязательный параметр
        :param list_ids: идентификаторы списков друзей, в которые нужно добавить пользователя. список положительных чисел, разделенных запятыми
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('friends.edit', params=params)

    def friends_editList(self, name=None, list_id=None, user_ids=None, add_user_ids=None, delete_user_ids=None):
        """
        Редактирует существующий список друзей текущего пользователя.

        :param name: название списка друзей. строка
        :param list_id: идентификатор списка друзей. положительное число, обязательный параметр
        :param user_ids: идентификаторы пользователей, включенных в список. список положительных чисел, разделенных запятыми
        :param add_user_ids: идентификаторы пользователей, которых необходимо добавить в список. (в случае если не передан user_ids) список положительных чисел, разделенных запятыми
        :param delete_user_ids: идентификаторы пользователей, которых необходимо изъять из списка. (в случае если не передан user_ids) список положительных чисел, разделенных запятыми
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('friends.editList', params=params)

    def friends_get(self, user_id=None, order=None, list_id=None, count=None, offset=None, fields=None, name_case=None,
                    ref=None):
        """
        Возвращает список идентификаторов друзей пользователя или расширенную информацию о друзьях пользователя (при использовании параметра fields). Если Вы используете социальный граф пользователя ВКонтакте в своем приложении, обратите внимание на п. 4.4. Правил платформы.

        :param user_id: идентификатор пользователя, для которого необходимо получить список друзей. Если параметр не задан, то считается, что он равен идентификатору текущего пользователя (справедливо для вызова с передачей access_token). целое число
        :param order: порядок, в котором нужно вернуть список друзей. Допустимые значения:   hints — сортировать по рейтингу, аналогично тому, как друзья сортируются в разделе Мои друзья (Это значение доступно только для Standalone-приложений с ключом доступа, полученным по схеме Implicit Flow.).  random — возвращает друзей в случайном порядке.  mobile — возвращает выше тех друзей, у которых установлены мобильные приложения.  name — сортировать по имени. Данный тип сортировки работает медленно, так как сервер будет получать всех друзей а не только указанное количество count. (работает только при переданном параметре fields).   По умолчанию список сортируется в порядке возрастания идентификаторов пользователей. строка
        :param list_id: идентификатор списка друзей, полученный методом friends.getLists, друзей из которого необходимо получить. Данный параметр учитывается, только когда параметр user_id равен идентификатору текущего пользователя.  Этот параметр доступен только для Standalone-приложений с ключом доступа, полученным по схеме Implicit Flow. положительное число
        :param count: количество друзей, которое нужно вернуть. положительное число, по умолчанию 5000
        :param offset: смещение, необходимое для выборки определенного подмножества друзей. положительное число
        :param fields: список дополнительных полей, которые необходимо вернуть.  Доступные значения: nickname, domain, sex, bdate, city, country, timezone, photo_50, photo_100, photo_200_orig, has_mobile, contacts, education, online, relation, last_seen, status, can_write_private_message, can_see_all_posts, can_post, universities список слов, разделенных через запятую
        :param name_case: падеж для склонения имени и фамилии пользователя. Возможные значения: именительный – nom, родительный – gen, дательный – dat, винительный – acc, творительный – ins, предложный – abl. По умолчанию nom. строка
        :param ref: строка, максимальная длина 255
        :return: После успешного выполнения возвращает список идентификаторов (id) друзей пользователя, если параметр fields не использовался.  При использовании параметра fields  возвращает список объектов пользователей, но не более 5000.
        """

        params = locals()
        params.pop('self')
        return self._call('friends.get', params=params)

    def friends_getAppUsers(self):
        """
        Возвращает список идентификаторов друзей текущего пользователя, которые установили данное приложение.

        :return:    В ходе выполнения могут произойти общие ошибки. Их описание находится на отдельной странице.
        """

        params = locals()
        params.pop('self')
        return self._call('friends.getAppUsers', params=params)

    def friends_getByPhones(self, phones=None, fields=None):
        """
        Возвращает список друзей пользователя, у которых завалидированные или указанные в профиле телефонные номера входят в заданный список. Использование данного метода возможно только если у текущего пользователя завалидирован номер мобильного телефона. Для проверки этого условия можно использовать метод users.get c параметрами user_ids=API_USER и fields=has_mobile, где API_USER равен идентификатору текущего пользователя. Для доступа к этому методу приложение должно быть доверенным.

        :param phones: список телефонных номеров в формате MSISDN, разделеннных запятыми. Например +79219876543,+79111234567 Максимальное количество номеров в списке — 1000. список слов, разделенных через запятую
        :param fields: список дополнительных полей, которые необходимо вернуть.  Доступные значения: nickname, screen_name, sex, bdate, city, country, timezone, photo_50, photo_100, photo_200_orig, has_mobile, contacts, education, online, counters, relation, last_seen, status, can_write_private_message, can_see_all_posts, can_post, universities. список слов, разделенных через запятую
        :return: После успешного выполнения возвращает список объектов пользователей с дополнительным полем  phone, в котором содержится номер из списка заданных для поиска номеров.
        """

        params = locals()
        params.pop('self')
        return self._call('friends.getByPhones', params=params)

    def friends_getLists(self, user_id=None, return_system=None):
        """
        Возвращает список меток друзей пользователя.

        :param user_id: идентификатор пользователя. положительное число, по умолчанию идентификатор текущего пользователя
        :param return_system: возвращать ли системный список публичных меток друзей пользователя. флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает список объектов, каждый из которых содержит следующие поля:   name — название списка друзей;  id — идентификатор списка друзей.
        """

        params = locals()
        params.pop('self')
        return self._call('friends.getLists', params=params)

    def friends_getMutual(self, source_uid=None, target_uid=None, target_uids=None, order=None, count=None,
                          offset=None):
        """
        Возвращает список идентификаторов общих друзей между парой пользователей.

        :param source_uid: идентификатор пользователя, чьи друзья пересекаются с друзьями пользователя с идентификатором target_uid. Если параметр не задан, то считается, что source_uid равен идентификатору текущего пользователя. положительное число, по умолчанию идентификатор текущего пользователя
        :param target_uid: идентификатор пользователя, с которым необходимо искать общих друзей. положительное число
        :param target_uids: список идентификаторов пользователей, с которыми необходимо искать общих друзей. список положительных чисел, разделенных запятыми, количество элементов должно составлять не более 100
        :param order: порядок, в котором нужно вернуть список общих друзей. Допустимые значения: random - возвращает друзей в случайном порядке. По умолчанию — в порядке возрастания идентификаторов. строка
        :param count: количество общих друзей, которое нужно вернуть. (по умолчанию – все общие друзья) положительное число
        :param offset: смещение, необходимое для выборки определенного подмножества общих друзей. положительное число
        :return: После успешного выполнения возвращает список идентификаторов (id) общих друзей между пользователями с идентификаторами source_uid и target_uid.
        """

        params = locals()
        params.pop('self')
        return self._call('friends.getMutual', params=params)

    def friends_getOnline(self, user_id=None, list_id=None, online_mobile=None, order=None, count=None, offset=None):
        """
        Возвращает список идентификаторов друзей пользователя, находящихся на сайте.

        :param user_id: идентификатор пользователя, для которого необходимо получить список друзей онлайн. Если параметр не задан, то считается, что он равен идентификатору текущего пользователя. положительное число, по умолчанию идентификатор текущего пользователя
        :param list_id: идентификатор списка друзей. Если параметр не задан, возвращается информация обо всех друзьях, находящихся на сайте. положительное число
        :param online_mobile: 1 — будет возвращено дополнительное поле online_mobile.  По умолчанию — 0. флаг, может принимать значения 1 или 0
        :param order: порядок, в котором нужно вернуть список друзей, находящихся на сайте. Допустимые значения: random - возвращает друзей в случайном порядке, hints - сортировать по рейтингу, аналогично тому, как друзья сортируются в разделе Мои друзья (данный параметр доступен только для Desktop-приложений).  По умолчанию список сортируется в порядке возрастания идентификаторов пользователей. строка
        :param count: количество друзей онлайн, которое нужно вернуть. (по умолчанию – все друзья онлайн) положительное число
        :param offset: смещение, необходимое для выборки определенного подмножества друзей онлайн. положительное число
        :return: После успешного выполнения возвращает список идентификаторов (id) друзей, находящихся сейчас на сайте, у пользователя с идентификатором uid и входящих в список с идентификатором lid.  При использовании параметра online_mobile=1 также возвращается поле online_mobile, содержащее список идентификатор друзей, находящихся на сайте с мобильного устройства.
        """

        params = locals()
        params.pop('self')
        return self._call('friends.getOnline', params=params)

    def friends_getRecent(self, count=None):
        """
        Возвращает список идентификаторов недавно добавленных друзей текущего пользователя.

        :param count: максимальное количество недавно добавленных друзей, которое необходимо получить. положительное число, по умолчанию 100, максимальное значение 1000
        :return: После успешного выполнения возвращает отсортированный в антихронологическом порядке список идентификаторов (id) недавно добавленных друзей текущего пользователя.
        """

        params = locals()
        params.pop('self')
        return self._call('friends.getRecent', params=params)

    def friends_getRequests(self, offset=None, count=None, extended=None, need_mutual=None, out=None, sort=None,
                            need_viewed=None, suggested=None, ref=None, fields=None):
        """
        Возвращает информацию о полученных или отправленных заявках на добавление в друзья для текущего пользователя.

        :param offset: смещение, необходимое для выборки определенного подмножества заявок на добавление в друзья. положительное число
        :param count: максимальное количество заявок на добавление в друзья, которые необходимо получить (не более 1000).  По умолчанию — 100. положительное число, по умолчанию 100, максимальное значение 1000
        :param extended: определяет, требуется ли возвращать в ответе сообщения от пользователей, подавших заявку на добавление в друзья. И отправителя рекомендации при suggested = 1. флаг, может принимать значения 1 или 0
        :param need_mutual: определяет, требуется ли возвращать в ответе список общих друзей, если они есть. Обратите внимание, что при использовании need_mutual будет возвращено не более 2 заявок. флаг, может принимать значения 1 или 0
        :param out: 0 — возвращать полученные заявки в друзья (по умолчанию), 1 — возвращать отправленные пользователем заявки. флаг, может принимать значения 1 или 0
        :param sort: 0 — сортировать по дате добавления, 1 — сортировать по количеству общих друзей. (Если out = 1, этот параметр не учитывается). положительное число
        :param need_viewed: 0 - не возвращать просмотренные заявки, 1 — возвращать просмотренные заявки. (Если out = 1, данный параметр не учитывается). флаг, может принимать значения 1 или 0, по умолчанию 0
        :param suggested: 1 — возвращать рекомендованных другими пользователями друзей, 0 — возвращать заявки в друзья (по умолчанию). флаг, может принимать значения 1 или 0
        :param ref: строка, максимальная длина 255
        :param fields: список слов, разделенных через запятую
        :return:   Если не установлен параметр need_mutual, то в случае успеха возвращает отсортированный в антихронологическом порядке по времени подачи заявки список идентификаторов (id) пользователей (кому или от кого пришла заявка).     Если установлен параметр need_mutual, то в случае успеха возвращает отсортированный в антихронологическом порядке по времени подачи заявки массив объектов, содержащих информацию о заявках на добавление в друзья. Каждый из объектов содержит поле uid, являющийся идентификатором пользователя. При наличии общих друзей, в объекте будет содержаться поле mutual, в котором будет находиться список идентификаторов общих друзей.
        """

        params = locals()
        params.pop('self')
        return self._call('friends.getRequests', params=params)

    def friends_getSuggestions(self, filter=None, count=None, offset=None, fields=None, name_case=None):
        """
        Возвращает список профилей пользователей, которые могут быть друзьями текущего пользователя.

        :param filter: типы предлагаемых друзей, которые нужно вернуть, перечисленные через запятую.  Может принимать следующие значения:  mutual — пользователи, с которыми много общих друзей; По умолчанию будут возвращены все возможные друзья. список слов, разделенных через запятую
        :param count: количество рекомендаций, которое необходимо вернуть. положительное число, максимальное значение 500, по умолчанию 500
        :param offset: смещение, необходимое для выбора определённого подмножества списка. положительное число
        :param fields: список дополнительных полей, которые необходимо вернуть.  Доступные значения: nickname, screen_name, sex, bdate, city, country, timezone, photo_50, photo_100, photo_200_orig, has_mobile, contacts, education, online, counters, relation, last_seen, status, can_write_private_message, can_see_all_posts, can_post, universities список слов, разделенных через запятую
        :param name_case: падеж для склонения имени и фамилии пользователя. Возможные значения: именительный – nom, родительный – gen, дательный – dat, винительный – acc, творительный – ins, предложный – abl. По умолчанию nom. строка
        :return: После успешного выполнения возвращает список объектов пользователей с дополнительным полем found_with для пользователей, найденных через импорт контактов. Для некоторых пользователей, которые были найдены давно поле found_with может отсутствовать.
        """

        params = locals()
        params.pop('self')
        return self._call('friends.getSuggestions', params=params)

    def friends_search(self, user_id=None, q=None, fields=None, name_case=None, offset=None, count=None):
        """
        Позволяет искать по списку друзей пользователей. Для расширенного поиска по списку друзей можно использовать метод users.search с параметром from_list = friends.

        :param user_id: идентификатор пользователя, по списку друзей которого необходимо произвести поиск. положительное число, по умолчанию идентификатор текущего пользователя, обязательный параметр
        :param q: строка запроса. строка
        :param fields: список дополнительных полей, которые необходимо вернуть.  Доступные значения: nickname, screen_name, sex, bdate, city, country, timezone, photo_50, photo_100, photo_200_orig, has_mobile, contacts, education, online, relation, last_seen, status, can_write_private_message, can_see_all_posts, can_post, universities, domain список слов, разделенных через запятую
        :param name_case: падеж для склонения имени и фамилии пользователя. Возможные значения: именительный – nom, родительный – gen, дательный – dat, винительный – acc, творительный – ins, предложный – abl. По умолчанию nom. строка, по умолчанию Nom
        :param offset: смещение, необходимое для выборки определенного подмножества друзей. положительное число
        :param count: количество друзей, которое нужно вернуть. положительное число, по умолчанию 20, максимальное значение 1000
        :return: После успешного выполнения метод  возвращает список объектов пользователей.
        """

        params = locals()
        params.pop('self')
        return self._call('friends.search', params=params)

    def gifts_get(self, user_id=None, count=None, offset=None):
        """
        Возвращает список полученных подарков пользователя.

        :param user_id: идентификатор пользователя, для которого необходимо получить список подарков. положительное число, по умолчанию идентификатор текущего пользователя
        :param count: количество подарков, которое нужно вернуть. положительное число
        :param offset: смещение, необходимое для выборки определенного подмножества подарков. положительное число
        :return: Возвращает список объектов gift_item, каждый из которых содержит следующие поля:   id — идентификатор полученного подарка;  from_id — идентификатор пользователя, который отправил подарок, или 0, если отправитель скрыт;  message — текст сообщения, приложенного к подарку;  date — время отправки подарка в формате unixtime;  gift — объект подарка в следующем формате:   id — номер подарка;  thumb_256 — url изображения подарка размером 256x256px;  thumb_96 — url изображения подарка размером 96x96px;  thumb_48 — url изображения подарка размером 48x48px;   privacy — значение приватности подарка (только для текущего пользователя). Возможные значения:   0 — имя отправителя и сообщение видно всем;  1 — имя отправителя видно всем, сообщение видно только получателю;  2 — имя отправителя скрыто, сообщение видно только получателю.
        """

        params = locals()
        params.pop('self')
        return self._call('gifts.get', params=params)

    def groups_addAddress(self, group_id=None, title=None, address=None, additional_address=None, country_id=None,
                          city_id=None, metro_id=None, latitude=None, longitude=None, phone=None, work_info_status=None,
                          timetable=None, is_main_address=None):
        """
        Позволяет добавить адрес в сообщество.  Список адресов может быть получен методом groups.getAddresses.  Для того, чтобы воспользоваться этим методом, Вы должны быть администратором сообщества

        :param group_id: идентификатор сообщества, в которое добавляется адрес. положительное число, обязательный параметр
        :param title: заголовок адреса строка, обязательный параметр, максимальная длина 255
        :param address: строка адреса Невский проспект, дом 28 строка, обязательный параметр, максимальная длина 255
        :param additional_address: дополнительное описание адреса. Второй этаж, налево строка, максимальная длина 400
        :param country_id: идентификатор  страны. Для получения можно использовать database.getCountries положительное число, обязательный параметр, минимальное значение 1
        :param city_id: идентификатор города. Для получения можно использовать database.getCities положительное число, обязательный параметр, минимальное значение 1
        :param metro_id: идентификатор станции метро.  Для получения можно использовать database.getMetroStations положительное число, минимальное значение 0
        :param latitude: географическая широта отметки, заданная в градусах (от -90 до 90). дробное число, обязательный параметр, минимальное значение -90, максимальное значение 90
        :param longitude: географическая долгота отметки, заданная в градусах (от -180 до 180). дробное число, обязательный параметр, минимальное значение -180, максимальное значение 180
        :param phone: номер телефона строка
        :param work_info_status: тип расписания. Возможные значения:   no_information — нет информации о расписании;  temporarily_closed — временно закрыто;  always_opened — открыто круглосуточно;  forever_closed —  закрыто навсегда;  timetable —  открыто в указанные часы работы. Для этого типа расписания необходимо передать параметр timetable;  строка, по умолчанию no_information
        :param timetable: для типа timetable можно передать расписание в формате json. Время передается в минутах от 0 часов. Ключ по дню означает, что день рабочий. open_time, close_time - начало и конец рабочего дня. break_open_time, break_close_time —  время перерыва. {    "mon": {      "open_time": 1080,      "close_time": 1380    },    "tue": {      "open_time": 1080,      "close_time": 1380    },    "wed": {      "open_time": 1080,      "close_time": 1320    },    "thu": {      "open_time": 1080,      "close_time": 1320    },    "fri": {      "open_time": 1080,      "close_time": 1320    },    "sat": {      "open_time": 1080,      "close_time": 1320,      "break_open_time": 1200,      "break_close_time": 1230    }  }   данные в формате JSON
        :param is_main_address: установить адрес основным. Информация об основном адресе сразу показывается в сообществе. Для получения информации об остальных адресах нужно перейти к списку адресов. флаг, может принимать значения 1 или 0
        :return:
        """

        params = locals()
        params.pop('self')
        return self._call('groups.addAddress', params=params)

    def groups_addCallbackServer(self, group_id=None, url=None, title=None, secret_key=None):
        """
        Добавляет сервер для Callback API в сообщество.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :param url: URL сервера. строка, обязательный параметр
        :param title: название сервера. строка, обязательный параметр, максимальная длина 14
        :param secret_key: секретный ключ. строка, максимальная длина 50
        :return: После успешного выполнения возвращает идентификатор добавленного сервера в поле server_id (integer).
        """

        params = locals()
        params.pop('self')
        return self._call('groups.addCallbackServer', params=params)

    def groups_addLink(self, group_id=None, link=None, text=None):
        """
        Позволяет добавлять ссылки в сообщество. Список ссылок может быть получен методом groups.getById с параметром fields=links.  Для того, чтобы воспользоваться этим методом, Вы должны быть администратором группы

        :param group_id: идентификатор сообщества, в которое добавляется ссылка. положительное число, обязательный параметр
        :param link: адрес ссылки. строка, обязательный параметр
        :param text: текст ссылки. строка
        :return: В случае успешного выполнения возвращает объект со следующими полями:   id (integer) — идентификатор ссылки;  url (string) — URL ссылки;  name (string) — название ссылки в блоке сообщества;  edit_title (integer, [0,1]) — возвращается 1, если можно редактировать название ссылки (для внешних ссылок);  desc (string) — описание ссылки;  image_processing (integer, [1]) — возвращается 1, если превью находится в процессе обработки.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.addLink', params=params)

    def groups_approveRequest(self, group_id=None, user_id=None):
        """
        Позволяет одобрить заявку в группу от пользователя. Для того, чтобы воспользоваться этим методом Вы должны быть администратором группы

        :param group_id: идентификатор группы, заявку в которую необходимо одобрить. положительное число, обязательный параметр
        :param user_id: идентификатор пользователя, заявку которого необходимо одобрить. положительное число, обязательный параметр
        :return: В случае успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.approveRequest', params=params)

    def groups_ban(self, group_id=None, owner_id=None, end_date=None, reason=None, comment=None, comment_visible=None):
        """
        Добавляет пользователя или группу в черный список сообщества.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :param owner_id: идентификатор пользователя или сообщества, которое будет добавлено в черный список. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число
        :param end_date: дата завершения срока действия бана в формате unixtime. Максимальный возможный срок окончания бана, который можно указать, — один год с его начала. Если параметр не указан, пользователь будет заблокирован навсегда. положительное число
        :param reason: причина бана:   0 — другое (по умолчанию);  1 — спам;  2 — оскорбление участников;  3 — нецензурные выражения;  4 — сообщения не по теме.  положительное число
        :param comment: текст комментария к бану. строка
        :param comment_visible: 1 – текст комментария будет отображаться пользователю.  0 – текст комментария не доступен пользователю. (по умолчанию) флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.ban', params=params)

    def groups_create(self, title=None, description=None, type=None, public_category=None, subtype=None):
        """
        Создает новое сообщество.

        :param title: название сообщества. строка, обязательный параметр
        :param description: описание сообщества, (не учитывается при type=public). строка
        :param type: тип создаваемого сообщества:   group — группа;  event — мероприятие;  public — публичная страница.  строка, по умолчанию group
        :param public_category: категория публичной страницы (только для type = public). положительное число
        :param subtype: вид публичной страницы (только при type=public):   1 — место или небольшая компания;  2 — компания, организация или веб-сайт;  3 — известная личность или коллектив;  4 — произведение или продукция.  положительное число
        :return: После успешного выполнения возвращает объект, описывающий созданное сообщество.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.create', params=params)

    def groups_deleteAddress(self, group_id=None, address_id=None):
        """
        Удаляет адрес сообщества.

        :param group_id: Id группы положительное число, обязательный параметр
        :param address_id: Id адреса положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.deleteAddress', params=params)

    def groups_deleteCallbackServer(self, group_id=None, server_id=None):
        """
        Удаляет сервер для Callback API из сообщества.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :param server_id: идентификатор сервера, который нужно удалить. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.deleteCallbackServer', params=params)

    def groups_deleteLink(self, group_id=None, link_id=None):
        """
        Позволяет удалить ссылки из сообщества. Список ссылок может быть получен методом groups.getById, с параметром fields=links.  Для того, чтобы воспользоваться этим методом Вы должны быть администратором группы

        :param group_id: идентификатор сообщества, ссылки которого нужно удалить положительное число, обязательный параметр
        :param link_id: идентификатор ссылки, которую необходимо удалить положительное число, обязательный параметр
        :return: В случае успешного выполнения метод возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.deleteLink', params=params)

    def groups_disableOnline(self, group_id=None):
        """
        Выключает статус «онлайн» в сообществе.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.disableOnline', params=params)

    def groups_edit(self, group_id=None, title=None, description=None, screen_name=None, access=None, website=None,
                    subject=None, email=None, phone=None, rss=None, event_start_date=None, event_finish_date=None,
                    event_group_id=None, public_category=None, public_subcategory=None, public_date=None, wall=None,
                    topics=None, photos=None, video=None, audio=None, links=None, events=None, places=None,
                    contacts=None, docs=None, wiki=None, messages=None, articles=None, addresses=None, age_limits=None,
                    market=None, market_comments=None, market_country=None, market_city=None, market_currency=None,
                    market_contact=None, market_wiki=None, obscene_filter=None, obscene_stopwords=None,
                    obscene_words=None, main_section=None, secondary_section=None, country=None, city=None,
                    ads_easy_promote_disable_main_screen_button=None,
                    ads_easy_promote_disable_promote_post_button=None):
        """
        Редактирует сообщество. Для того, чтобы воспользоваться этим методом, Вы должны быть администратором сообщества.  Значения настроек можно получить методом groups.getSettings.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :param title: название сообщества. строка
        :param description: описание сообщества. строка
        :param screen_name: короткое имя сообщества. строка
        :param access: тип группы. Возможные значения:   0 — открытая;  1 — закрытая;  2 — частная.  положительное число
        :param website: адрес сайта, который будет указан в информации о группе. строка
        :param subject: тематика сообщества. Возможные значения:   1 — авто/мото;  2 — активный отдых;  3 — бизнес;  4 — домашние животные;  5 — здоровье;  6 — знакомство и общение;  7 — игры;  8 — ИТ (компьютеры и софт);  9 — кино;  10 — красота и мода;  11 — кулинария;  12 — культура и искусство;  13 — литература;  14 — мобильная связь и интернет;  15 — музыка;  16 — наука и техника;  17 — недвижимость;  18 — новости и СМИ;  19 — безопасность;  20 — образование;  21 — обустройство и ремонт;  22 — политика;  23 — продукты питания;  24 — промышленность;  25 — путешествия;  26 — работа;  27 — развлечения;  28 — религия;  29 — дом и семья;  30 — спорт;  31 — страхование;  32 — телевидение;  33 — товары и услуги;  34 — увлечения и хобби;  35 — финансы;  36 — фото;  37 — эзотерика;  38 — электроника и бытовая техника;  39 — эротика;  40 — юмор;  41 — общество, гуманитарные науки;  42 — дизайн и графика.  строка
        :param email: электронный адрес организатора (для мероприятий). строка
        :param phone: номер телефона организатора (для мероприятий). строка
        :param rss: адрес rss для импорта новостей (доступен только группам, получившим соответствующее разрешение, обратитесь в http://vk.com/support для получения разрешения). строка
        :param event_start_date: дата начала события. положительное число
        :param event_finish_date: дата окончания события. положительное число
        :param event_group_id: идентификатор группы, которая является организатором события (только для событий). положительное число
        :param public_category: категория публичной страницы. положительное число
        :param public_subcategory: подкатегория публичной станицы. Список подкатегорий можно получить методом groups.getSettings. положительное число
        :param public_date: дата основания компании, организации, которой посвящена публичная страница в виде строки формата "dd.mm.YYYY". строка
        :param wall: стена. Возможные значения:   0 — выключена;  1 — открытая;  2 — ограниченная (доступно только для групп и событий);  3 — закрытая (доступно только для групп и событий).  положительное число
        :param topics: обсуждения. Возможные значения:   0 — выключены;  1 — открытые;  2 — ограниченные (доступно только для групп и событий).  положительное число
        :param photos: фотографии. Возможные значения:   0 — выключены;  1 — открытые;  2 — ограниченные (доступно только для групп и событий).  положительное число
        :param video: видеозаписи. Возможные значения:   0 — выключены;  1 — открытые;  2 — ограниченные (доступно только для групп и событий).  положительное число
        :param audio: аудиозаписи. Возможные значения:   0 — выключены;  1 — открытые;  2 — ограниченные (доступно только для групп и событий).  положительное число
        :param links: ссылки (доступно только для публичных страниц).  Возможные значения:   0 — выключены;  1 — включены.  флаг, может принимать значения 1 или 0
        :param events: события (доступно только для публичных страниц).  Возможные значения:   0 — выключены;  1 — включены.  флаг, может принимать значения 1 или 0
        :param places: места (доступно только для публичных страниц). Возможные значения:   0 — выключены;  1 — включены.  флаг, может принимать значения 1 или 0
        :param contacts: контакты (доступно только для публичных страниц). Возможные значения:   0 — выключены;  1 — включены.  флаг, может принимать значения 1 или 0
        :param docs: документы сообщества. Возможные значения:   0 — выключены;  1 — открытые;  2 — ограниченные (доступно только для групп и событий).  положительное число
        :param wiki: wiki-материалы сообщества. Возможные значения:   0 — выключены;  1 — открытые;  2 — ограниченные (доступно только для групп и событий).  положительное число
        :param messages: сообщения сообщества. Возможные значения:   0 — выключены;  1 — включены.  флаг, может принимать значения 1 или 0
        :param articles: флаг, может принимать значения 1 или 0
        :param addresses: флаг, может принимать значения 1 или 0
        :param age_limits: возрастное ограничение для сообщества. Возможные значения:   1 — нет ограничений;  2 — 16+;  3 — 18+.  положительное число, по умолчанию 1
        :param market: товары. Возможные значения:   0 — выключены;  1 — включены.  флаг, может принимать значения 1 или 0
        :param market_comments: комментарии к товарам. Возможные значения:   0 — выключены;  1 — включены.  флаг, может принимать значения 1 или 0
        :param market_country: регионы доставки товаров. список целых чисел, разделенных запятыми
        :param market_city: города доставки товаров (в случае если указана одна страна). список положительных чисел, разделенных запятыми
        :param market_currency: идентификатор валюты магазина. Возможные значения:   643 — российский рубль;  980 — украинская гривна;  398 — казахстанский тенге;  978 — евро;  840 — доллар США.  положительное число
        :param market_contact: контакт для связи для продавцом.  Для использования сообщений сообщества следует включить их и передать значение 0. положительное число
        :param market_wiki: идентификатор wiki-страницы с описанием магазина. положительное число
        :param obscene_filter: фильтр нецензурных выражений в комментариях. Возможные значения:   0 — выключен;  1 — включен.  флаг, может принимать значения 1 или 0
        :param obscene_stopwords: фильтр по ключевым словам в комментариях. Возможные значения:   0 — выключен;  1 — включен.  флаг, может принимать значения 1 или 0
        :param obscene_words: ключевые слова для фильтра комментариев. список слов, разделенных через запятую
        :param main_section: положительное число
        :param secondary_section: положительное число
        :param country: положительное число
        :param city: положительное число
        :param ads_easy_promote_disable_main_screen_button: флаг, может принимать значения 1 или 0
        :param ads_easy_promote_disable_promote_post_button: флаг, может принимать значения 1 или 0
        :return: В случае успеха возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.edit', params=params)

    def groups_editAddress(self, group_id=None, address_id=None, title=None, address=None, additional_address=None,
                           country_id=None, city_id=None, metro_id=None, latitude=None, longitude=None, phone=None,
                           work_info_status=None, timetable=None, is_main_address=None):
        """
        Позволяет отредактировать адрес в сообществе.  Список адресов может быть получен методом groups.getAddresses.  Для того, чтобы воспользоваться этим методом, Вы должны быть администратором сообщества

        :param group_id: идентификатор сообщества, в которое добавляется адрес. положительное число, обязательный параметр
        :param address_id: идентификатор адреса положительное число, обязательный параметр
        :param title: заголовок адреса строка, максимальная длина 255
        :param address: строка адреса Невский проспект, дом 28 строка, максимальная длина 255
        :param additional_address: дополнительное описание адреса. Второй этаж, налево строка, максимальная длина 400
        :param country_id: идентификатор  страны. Для получения можно использовать database.getCountries положительное число, минимальное значение 0
        :param city_id: идентификатор города. Для получения можно использовать database.getCities положительное число, минимальное значение 0
        :param metro_id: идентификатор станции метро.  Для получения можно использовать database.getMetroStations положительное число, минимальное значение 0
        :param latitude: географическая широта отметки, заданная в градусах (от -90 до 90). дробное число, минимальное значение -90, максимальное значение 90
        :param longitude: географическая долгота отметки, заданная в градусах (от -180 до 180). дробное число, минимальное значение -180, максимальное значение 180
        :param phone: номер телефона строка
        :param work_info_status: тип расписания. Возможные значения:   no_information — нет информации о расписании;  temporarily_closed — временно закрыто;  always_opened — открыто круглосуточно;  forever_closed —  закрыто навсегда;  timetable —  открыто в указанные часы работы. Для этого типа расписания необходимо передать параметр timetable;  строка
        :param timetable: для типа timetable можно передать расписание в формате json. Время передается в минутах от 0 часов. Ключ по дню означает, что день рабочий. open_time, close_time —  начало и конец рабочего дня. break_open_time, break_close_time - время перерыва {    "mon": {      "open_time": 1080,      "close_time": 1380    },    "tue": {      "open_time": 1080,      "close_time": 1380    },    "wed": {      "open_time": 1080,      "close_time": 1320    },    "thu": {      "open_time": 1080,      "close_time": 1320    },    "fri": {      "open_time": 1080,      "close_time": 1320    },    "sat": {      "open_time": 1080,      "close_time": 1320,      "break_open_time": 1200,      "break_close_time": 1230    }  }   данные в формате JSON
        :param is_main_address: установить адрес основным. Информация об основном адресе сразу показывается в сообществе. Для получения информации об остальных адресах нужно перейти к списку адресов. флаг, может принимать значения 1 или 0
        :return:
        """

        params = locals()
        params.pop('self')
        return self._call('groups.editAddress', params=params)

    def groups_editCallbackServer(self, group_id=None, server_id=None, url=None, title=None, secret_key=None):
        """
        Редактирует данные сервера для Callback API в сообществе.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :param server_id: идентификатор сервера, данные которого нужно отредактировать. положительное число, обязательный параметр
        :param url: новый URL сервера. строка, обязательный параметр
        :param title: новое название сервера. строка, обязательный параметр, максимальная длина 14
        :param secret_key: новый секретный ключ сервера. строка, максимальная длина 50
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.editCallbackServer', params=params)

    def groups_editLink(self, group_id=None, link_id=None, text=None):
        """
        Позволяет редактировать ссылки в сообществе.

        :param group_id: идентификатор сообщества, в которое добавляется ссылка. положительное число, обязательный параметр
        :param link_id: идентификатор ссылки. положительное число, обязательный параметр
        :param text: новый текст описания для ссылки. строка
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.editLink', params=params)

    def groups_editManager(self, group_id=None, user_id=None, role=None, is_contact=None, contact_position=None,
                           contact_phone=None, contact_email=None):
        """
        Позволяет назначить/разжаловать руководителя в сообществе или изменить уровень его полномочий. Для того, чтобы воспользоваться этим методом, Вы должны быть администратором группы

        :param group_id: идентификатор сообщества (указывается без знака «минус»). положительное число, обязательный параметр
        :param user_id: идентификатор пользователя, чьи полномочия в сообществе нужно изменить. положительное число, обязательный параметр
        :param role: уровень полномочий:   moderator — модератор;  editor — редактор;  administrator — администратор.   Если параметр не задан, с пользователя user_id снимаются полномочия руководителя. строка
        :param is_contact: отображать ли пользователя в блоке контактов сообщества. флаг, может принимать значения 1 или 0
        :param contact_position: должность пользователя, отображаемая в блоке контактов. строка
        :param contact_phone: телефон пользователя, отображаемый в блоке контактов. строка
        :param contact_email: email пользователя, отображаемый в блоке контактов. строка
        :return: В случае успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.editManager', params=params)

    def groups_enableOnline(self, group_id=None):
        """
        Включает статус «онлайн» в сообществе.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :return: После успешного выполнения возращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.enableOnline', params=params)

    def groups_get(self, user_id=None, extended=None, filter=None, fields=None, offset=None, count=None):
        """
        Возвращает список сообществ указанного пользователя.

        :param user_id: идентификатор пользователя, информацию о сообществах которого требуется получить. положительное число, по умолчанию идентификатор текущего пользователя
        :param extended: если указать в качестве этого параметра 1, то будет возвращена полная информация о группах пользователя. По умолчанию 0. флаг, может принимать значения 1 или 0
        :param filter: список фильтров сообществ, которые необходимо вернуть, перечисленные через запятую. Доступны значения admin, editor, moder, advertiser, groups, publics, events, hasAddress. По умолчанию возвращаются все сообщества пользователя.  При указании фильтра hasAddress вернутся сообщества, в которых указаны адреса в соответствующем блоке, admin будут возвращены сообщества, в которых пользователь является администратором, editor — администратором или редактором, moder — администратором, редактором или модератором, advertiser — рекламодателем. Если передано несколько фильтров, то их результат объединяется. список слов, разделенных через запятую
        :param fields: список дополнительных полей, которые необходимо вернуть. Возможные значения: city, country, place, description, wiki_page, members_count, counters, start_date, finish_date, can_post, can_see_all_posts, activity, status, contacts, links, fixed_post, verified, site, can_create_topic. Подробнее см. описание полей объекта group.  Обратите внимание, этот параметр учитывается только при extended=1. список слов, разделенных через запятую
        :param offset: смещение, необходимое для выборки определённого подмножества сообществ. положительное число
        :param count: количество сообществ, информацию о которых нужно вернуть. положительное число, максимальное значение 1000
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count (integer) и массив идентификаторов сообщества в поле items ([integer]).  Если был задан параметр extended=1, возвращает объект, содержащий число результатов в поле count (integer) и массив объектов, описывающих сообщества, в поле items.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.get', params=params)

    def groups_getAddresses(self, group_id=None, address_ids=None, latitude=None, longitude=None, offset=None,
                            count=None, fields=None):
        """
        Возвращает адрес указанного сообщества.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :param address_ids: перечисленные через запятую идентификаторы адресов, информацию о которых необходимо вернуть. список положительных чисел, разделенных запятыми
        :param latitude: географическая широта отметки, заданная в градусах (от -90 до 90). дробное число, минимальное значение -90, максимальное значение 90
        :param longitude: географическая долгота отметки, заданная в градусах (от -180 до 180). дробное число, минимальное значение -180, максимальное значение 180
        :param offset: смещение, необходимое для выборки определенного подмножества черного списка. положительное число
        :param count: количество адресов, которое необходимо вернуть. положительное число, по умолчанию 10
        :param fields: список дополнительных полей сообществ, которые необходимо вернуть. список слов, разделенных через запятую
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count  и массив объектов адресов в поле items.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.getAddresses', params=params)

    def groups_getBanned(self, group_id=None, offset=None, count=None, fields=None, owner_id=None):
        """
        Возвращает список забаненных пользователей и сообществ в сообществе.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :param offset: смещение, необходимое для выборки определенного подмножества черного списка. положительное число
        :param count: количество пользователей, которое необходимо вернуть. положительное число, по умолчанию 20, максимальное значение 200
        :param fields: список дополнительных полей профилей и сообществ, которые необходимо вернуть. список слов, разделенных через запятую
        :param owner_id: идентификатор пользователя или сообщества из чёрного списка, информацию о котором нужно получить. целое число
        :return: После успешного выполнения возвращает общее число результатов в поле count (integer) и информацию о пользователях и сообществах из чёрного списка в поле items (array). Каждый объект в массиве items содержит поля:   type (string) — тип. Возможные значения:   group — сообщество;  profile — пользователь.   group (object) — информация о сообществе (для type = group). Объект сообщества.  profile (object) — информация о пользователе (для type = profile). Объект пользователя.  ban_info (object) — информация о блокировке в сообществе. Объект, который содержит поля:   Объект ban_info — информация о внесении в черный список сообщества.   admin_id идентификатор администратора, который добавил пользователя или сообщество в черный список.   integer date дата добавления в черный список в формате Unixtime.   integer reason причина добавления в черный список. Возможные значения:   0 — другое (по умолчанию);  1 — спам;  2 — оскорбление участников;  3 — нецензурные выражения;  4 — сообщения не по теме.    integer comment текст комментария.   string end_date дата окончания блокировки (0 — блокировка вечная).   integer
        """

        params = locals()
        params.pop('self')
        return self._call('groups.getBanned', params=params)

    def groups_getById(self, group_ids=None, group_id=None, fields=None):
        """
        Возвращает информацию о заданном сообществе или о нескольких сообществах.

        :param group_ids: идентификаторы или короткие имена сообществ. Максимальное число идентификаторов — 500. список слов, разделенных через запятую
        :param group_id: идентификатор или короткое имя сообщества. строка
        :param fields: список дополнительных полей, которые необходимо вернуть. Например: city, country, place, description, wiki_page, market, members_count, counters, start_date, finish_date, can_post, can_see_all_posts, activity, status, contacts, links, fixed_post, verified, site, ban_info, cover. Полный список полей доступен на этой странице.  Обратите внимание, для получения некоторых полей требуется право доступа groups. Подробнее см. описание полей объекта group список слов, разделенных через запятую
        :return: После успешного выполнения возвращает массив объектов, описывающих сообщества.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.getById', params=params)

    def groups_getCallbackConfirmationCode(self, group_id=None):
        """
        Позволяет получить строку, необходимую для подтверждения адреса сервера в Callback API. Для добавления или редактирования сервера используйте соответственно методы groups.addCallbackServer и groups.editCallbackServer.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :return: Возвращает строку, которую необходимо использовать в качестве ответа на уведомление с типом "confirmation" для подтверждения адреса сервера в Callback API.  Обратите внимание, что код, возвращаемый методом, можно использовать только для настройки сервера средствами API. В настройках Вашего сообщества на сайте ВКонтакте код будет отличаться.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.getCallbackConfirmationCode', params=params)

    def groups_getCallbackServers(self, group_id=None, server_ids=None):
        """
        Получает информацию о серверах для Callback API в сообществе.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :param server_ids: идентификаторы серверов, данные о которых нужно получить. По умолчанию возвращаются все серверы. список положительных чисел, разделенных запятыми
        :return: Возвращает число серверов в поле count (integer) и массив объектов items с данными о серверах. Каждый объект массива items содержит поля:   id (integer) — идентификатор сервера;  title (string) — название сервера;  creator_id (integer) — идентификатор пользователя, который добавил сервер (может содержать 0);  url (string) — URL сервера;  secret_key (string) — секретный ключ;  status (string) — статус сервера. Возможные значения:   unconfigured — адрес не задан;  failed — подтвердить адрес не удалось;  wait — адрес ожидает подтверждения;  ok — сервер подключен.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.getCallbackServers', params=params)

    def groups_getCallbackSettings(self, group_id=None, server_id=None):
        """
        Позволяет получить настройки уведомлений Callback API для сообщества.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :param server_id: идентификатор сервера. положительное число
        :return: После успешного выполнения возвращает объект, содержащий настройки уведомлений в формате «название события» : «статус» (0 — уведомления о событии выключены, 1 — уведомления о событии включены). Объект содержит следующие поля:  message_new новое сообщение  integer, [0,1] message_reply новое исходящее сообщение   integer, [0,1] message_edit редактирование сообщения   integer, [0,1] message_allow новая подписка на сообщения   integer, [0,1] message_deny новый запрет сообщений   integer, [0,1] photo_new добавление новой фотографии   integer, [0,1] audio_new добавление новой аудиозаписи   integer, [0,1] video_new добавление новой видеозаписи   integer, [0,1] wall_reply_new добавление нового комментария на стене   integer, [0,1] wall_reply_edit редактирование комментария на стене   integer, [0,1] wall_reply_delete удаление комментария на стене   integer, [0,1] wall_post_new добавление новой записи на стене   integer, [0,1] wall_repost новый репост записи на стене   integer, [0,1] board_post_new добавление нового комментария в обсуждении   integer, [0,1] board_post_edit редактирование комментария в обсуждении   integer, [0,1] board_post_delete удаление комментария в обсуждении   integer, [0,1] board_post_restore восстановление комментария в обсуждении   integer, [0,1] photo_comment_new добавление нового комментария к фото  integer, [0,1] photo_comment_edit редактирование комментария к фото  integer, [0,1] photo_comment_delete удаление комментария к фото  integer, [0,1] photo_comment_restore восстановление комментария к фото  integer, [0,1] video_comment_new добавление нового комментария к видео  integer, [0,1] video_comment_edit редактирование комментария к видео  integer, [0,1] video_comment_delete удаление комментария к видео  integer, [0,1] video_comment_restore восстановление комментария к видео  integer, [0,1] market_comment_new добавление нового комментария к товару  integer, [0,1] market_comment_edit редактирование комментария к товару  integer, [0,1] market_comment_delete удаление комментария к товару  integer, [0,1] market_comment_restore восстановление удалённого комментария к товару  integer, [0,1] poll_vote_new новый голос в публичном опросе  integer, [0,1] group_join вступление в сообщество   integer, [0,1] group_leave выход участника из сообщества   integer, [0,1] user_block занесение пользователя в черный список   integer, [0,1] user_unblock удаление пользователя из черного списка   integer, [0,1] group_change_settings изменение настроек сообщества   integer, [0,1] group_change_photo изменение главной фотографии   integer, [0,1] group_officers_edit изменение руководства сообщества   integer, [0,1]
        """

        params = locals()
        params.pop('self')
        return self._call('groups.getCallbackSettings', params=params)

    def groups_getCatalog(self, category_id=None, subcategory_id=None):
        """
        Возвращает список сообществ выбранной категории каталога.

        :param category_id: идентификатор категории, полученный в методе groups.getCatalogInfo. положительное число
        :param subcategory_id: идентификатор подкатегории, полученный в методе groups.getCatalogInfo. положительное число, максимальное значение 99
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count (integer) и массив объектов, описывающих сообщества, в поле items.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.getCatalog', params=params)

    def groups_getCatalogInfo(self, extended=None, subcategories=None):
        """
        Возвращает список категорий для каталога сообществ.

        :param extended: 1 — вернуть информацию о количестве сообществ в категории и три сообщества для предпросмотра.  По умолчанию 0. флаг, может принимать значения 1 или 0, по умолчанию 0, доступен начиная с версии 5.37
        :param subcategories: 1 — вернуть информацию об подкатегориях.  По умолчанию 0. флаг, может принимать значения 1 или 0, по умолчанию 0, доступен начиная с версии 5.37
        :return: После успешного выполнения возвращает поле enabled (0 — каталог недоступен для пользователя, 1 — каталог доступен), и, если enabled=1, массив categories объектов — категорий товаров.  Объект массива categories — категория каталога товаров.   id идентификатор категории.   положительное число name название категории.   строка subcategories поле возвращается при использовании subcategories=1. Массив объектов-подкатегорий. Каждый из объектов содержит поля id и name, содержащие идентификатор и название подкатегории.   Дополнительные поля (extended=1)   page_count количество сообществ в категории.   положительное число page_previews массив объектов сообществ для предпросмотра.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.getCatalogInfo', params=params)

    def groups_getInvitedUsers(self, group_id=None, offset=None, count=None, fields=None, name_case=None):
        """
        Возвращает список пользователей, которые были приглашены в группу.

        :param group_id: идентификатор группы, список приглашенных в которую пользователей нужно вернуть. положительное число, обязательный параметр
        :param offset: смещение, необходимое для выборки определённого подмножества пользователей. положительное число
        :param count: количество пользователей, информацию о которых нужно вернуть. положительное число, по умолчанию 20
        :param fields: список дополнительных полей, которые необходимо вернуть.  Доступные значения: nickname, domain, sex, bdate, city, country, timezone, photo_50, photo_100, photo_200_orig, has_mobile, contacts, education, online, relation, last_seen, status, can_write_private_message, can_see_all_posts, can_post, universities список слов, разделенных через запятую
        :param name_case: падеж для склонения имени и фамилии пользователя. Возможные значения: именительный – nom, родительный – gen, дательный – dat, винительный – acc, творительный – ins, предложный – abl. По умолчанию nom. строка
        :return: Возвращает список объектов пользователей.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.getInvitedUsers', params=params)

    def groups_getInvites(self, offset=None, count=None, extended=None):
        """
        Данный метод возвращает список приглашений в сообщества и встречи текущего пользователя.

        :param offset: смещение, необходимое для выборки определённого подмножества приглашений. положительное число
        :param count: количество приглашений, которое необходимо вернуть. положительное число, по умолчанию 20
        :param extended: 1 — вернуть дополнительную информацию о пользователях, отправлявших приглашения. По умолчанию — 0. флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает количество результатов в поле count и массив объектов, описывающих сообщества, с дополнительным полем invited_by, содержащим идентификатор пользователя, который отправил приглашение, в поле items.  Если был передан параметр extended=1, дополнительно будет возвращен список profiles пользователей, отправивших приглашения. Каждый объект в списке содержит поля id, first_name, last_name.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.getInvites', params=params)

    def groups_getLongPollServer(self, group_id=None):
        """
        Возвращает данные для подключения к Bots Longpoll API.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :return: Возвращает объект, который содержит следующие поля:   key (string) — ключ;  server (string) — url сервера;  ts (string) — timestamp.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.getLongPollServer', params=params)

    def groups_getLongPollSettings(self, group_id=None):
        """
        Получает настройки Bots Longpoll API для сообщества.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :return: Возвращает объект, который содержит следующие поля:   is_enabled (boolean) — true, если Bots Longpoll включен в сообществе.  events (object) — настройки Bots Longpoll. объект, содержащий настройки уведомлений в формате «название события» : «статус» (0 — уведомления о событии выключены, 1 — уведомления о событии включены). Объект содержит следующие поля:   message_new новое сообщение  integer, [0,1] message_reply новое исходящее сообщение   integer, [0,1] message_allow новая подписка на сообщения   integer, [0,1] message_deny новый запрет сообщений   integer, [0,1] photo_new добавление новой фотографии   integer, [0,1] audio_new добавление новой аудиозаписи   integer, [0,1] video_new добавление новой видеозаписи   integer, [0,1] wall_reply_new добавление нового комментария на стене   integer, [0,1] wall_reply_edit редактирование комментария на стене   integer, [0,1] wall_reply_delete удаление комментария на стене   integer, [0,1] wall_post_new добавление новой записи на стене   integer, [0,1] wall_repost новый репост записи на стене   integer, [0,1] board_post_new добавление нового комментария в обсуждении   integer, [0,1] board_post_edit редактирование комментария в обсуждении   integer, [0,1] board_post_delete удаление комментария в обсуждении   integer, [0,1] board_post_restore восстановление комментария в обсуждении   integer, [0,1] photo_comment_new добавление нового комментария к фото  integer, [0,1] photo_comment_edit редактирование комментария к фото  integer, [0,1] photo_comment_delete удаление комментария к фото  integer, [0,1] photo_comment_restore восстановление комментария к фото  integer, [0,1] video_comment_new добавление нового комментария к видео  integer, [0,1] video_comment_edit редактирование комментария к видео  integer, [0,1] video_comment_delete удаление комментария к видео  integer, [0,1] video_comment_restore восстановление комментария к видео  integer, [0,1] market_comment_new добавление нового комментария к товару  integer, [0,1] market_comment_edit редактирование комментария к товару  integer, [0,1] market_comment_delete удаление комментария к товару  integer, [0,1] market_comment_restore восстановление удалённого комментария к товару  integer, [0,1] poll_vote_new новый голос в публичном опросе  integer, [0,1] group_join вступление в сообщество   integer, [0,1] group_leave выход участника из сообщества   integer, [0,1] user_block занесение пользователя в черный список   integer, [0,1] user_unblock удаление пользователя из черного списка   integer, [0,1] group_change_settings изменение настроек сообщества   integer, [0,1] group_change_photo изменение главной фотографии   integer, [0,1] group_officers_edit изменение руководства сообщества   integer, [0,1]
        """

        params = locals()
        params.pop('self')
        return self._call('groups.getLongPollSettings', params=params)

    def groups_getMembers(self, group_id=None, sort=None, offset=None, count=None, fields=None, filter=None):
        """
        Возвращает список участников сообщества.

        :param group_id: идентификатор или короткое имя сообщества. строка
        :param sort: сортировка, с которой необходимо вернуть список участников. Может принимать значения:   id_asc — в порядке возрастания id;  id_desc — в порядке убывания id;  time_asc — в хронологическом порядке по вступлению в сообщество;  time_desc — в антихронологическом порядке по вступлению в сообщество.  Сортировка по time_asc и time_desc возможна только при вызове от модератора сообщества. строка, по умолчанию id_asc
        :param offset: смещение, необходимое для выборки определенного подмножества участников. По умолчанию 0. положительное число
        :param count: количество участников сообщества, информацию о которых необходимо получить. положительное число, по умолчанию 1000, максимальное значение 1000
        :param fields: список дополнительных полей, которые необходимо вернуть.  Доступные значения: sex, bdate, city, country, photo_50, photo_100, photo_200_orig, photo_200, photo_400_orig, photo_max, photo_max_orig, online, online_mobile, lists, domain, has_mobile, contacts, connections, site, education, universities, schools, can_post, can_see_all_posts, can_see_audio, can_write_private_message, status, last_seen, common_count, relation, relatives список слов, разделенных через запятую
        :param filter:   friends — будут возвращены только друзья в этом сообществе.  unsure — будут возвращены пользователи, которые выбрали «Возможно пойду» (если сообщество относится к мероприятиям).  managers — будут возвращены только руководители сообщества (доступно при запросе с передачей access_token от имени администратора сообщества).  строка
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count (integer) и массив идентификаторов пользователей в поле items ([integer]).  Если был передан параметр filter=managers, возвращается дополнительное поле role (string), которое содержит уровень полномочий руководителя:   moderator — модератор;  editor — редактор;  administrator — администратор;  creator — создатель сообщества.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.getMembers', params=params)

    def groups_getOnlineStatus(self, group_id=None):
        """
        Получает информацию о статусе «онлайн» в сообществе.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :return: Возвращает объект, который содержит поля:   status — статус сообщества. Возможные значения:   none — сообщество не онлайн;  online — сообщество онлайн (отвечает мгновенно);  answer_mark — сообщество отвечает быстро.   minutes — оценка времени ответа в минутах (для status = answer_mark).
        """

        params = locals()
        params.pop('self')
        return self._call('groups.getOnlineStatus', params=params)

    def groups_getRequests(self, group_id=None, offset=None, count=None, fields=None):
        """
        Возвращает список заявок на вступление в сообщество. Чтобы использовать этот метод, Вы должны быть руководителем сообщества

        :param group_id: идентификатор сообщества (указывается без знака «минус»). положительное число, обязательный параметр
        :param offset: смещение, необходимое для выборки определенного подмножества результатов. По умолчанию — 0. положительное число
        :param count: число результатов, которые необходимо вернуть. положительное число, по умолчанию 20, максимальное значение 200
        :param fields: список дополнительных полей профилей, которые необходимо вернуть. См. подробное описание.  Доступные значения: sex, bdate, city, country, photo_50, photo_100, photo_200_orig, photo_200, photo_400_orig, photo_max, photo_max_orig, online, online_mobile, domain, has_mobile, contacts, connections, site, education, universities, schools, can_post, can_see_all_posts, can_see_audio, can_write_private_message, status, last_seen, common_count, relation, relatives, counters, screen_name, maiden_name, timezone, occupation,activities, interests, music, movies, tv, books, games, about, quotes список слов, разделенных через запятую
        :return: Возвращает список идентификаторов пользователей, отправивших заявки на вступление в сообщество.  Если было передано значение в параметре fields, возвращается список объектов пользователей.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.getRequests', params=params)

    def groups_getSettings(self, group_id=None):
        """
        Позволяет получать данные, необходимые для отображения страницы редактирования данных сообщества. Чтобы использовать этот метод, Вы должны быть администратором сообщества

        :param group_id: идентификатор сообщества, данные которого необходимо получить. положительное число, обязательный параметр
        :return: В случае успешного выполнения метод вернет объект, содержащий данные сообщества, которые позволят отобразить форму редактирования для метода groups.edit.  В ответе содержится объект с полями:  title  stringназвание сообщества. description  stringописание сообщества. address  stringкороткий адрес. wall  integerнастройки стены. photos  integerнастройки фотографий. video  integerнастройки видеозаписей. audio  integerнастройки аудиозаписей. docs  integerнастройки документов. topics  integerнастройки обсуждений. wiki  integerнастройки вики-страниц. messages  integerсообщения сообщества. Возможные значения:   1  — включены;  0 — выключены. obscene_filter integerнастройки фильтра нецензурных слов. obscene_stopwords integerнастройки фильтра комментариев по ключевым словам. obscene_words stringсписок стоп-слов. public_category  integerкатегория публичной страницы. public_subcategory  integerподкатегория публичной страницы. public_category_list  arrayсписок возможных категорий для публичных страниц. access   integerтип сообщества. subject   integerидентификатор тематики. subject_list   arrayсписок возможных тематик. Массив объектов, каждый из которых содержит поля:   id (integer) — идентификатор тематики;  name (string) — название тематики. rss  stringадрес RSS-ленты. website  stringадрес веб-сайта. age_limits  integerвозрастные ограничения market    objectнастройки блока товаров. Объект, содержащий поля:   enabled (integer) — включен ли блок товаров (1 или 0);  comments_enabled (integer) — включены ли комментарии к товарам (1 или 0);  country_ids  (array) — идентификаторы стран;  city_ids  (array) — идентификаторы городов;  contact_id (integer) — идентификатор контактного лица;  currency (object) — объект, описывающий валюту.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.getSettings', params=params)

    def groups_getTokenPermissions(self):
        """
        Возвращает настройки прав для ключа доступа сообщества.

        :return:    В ходе выполнения могут произойти общие ошибки. Их описание находится на отдельной странице.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.getTokenPermissions', params=params)

    def groups_invite(self, group_id=None, user_id=None):
        """
        Позволяет приглашать друзей в группу.

        :param group_id: идентификатор группы, в которую необходимо выслать приглашение положительное число, обязательный параметр
        :param user_id: идентификатор пользователя, которому необходимо выслать приглашение положительное число, обязательный параметр
        :return: В случае успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.invite', params=params)

    def groups_isMember(self, group_id=None, user_id=None, user_ids=None, extended=None):
        """
        Возвращает информацию о том, является ли пользователь участником сообщества.

        :param group_id: идентификатор или короткое имя сообщества. строка, обязательный параметр
        :param user_id: идентификатор пользователя. положительное число
        :param user_ids: идентификаторы пользователей, не более 500. список положительных чисел, разделенных запятыми
        :param extended: 1  — вернуть ответ в расширенной форме. По умолчанию — 0. флаг, может принимать значения 1 или 0
        :return: Возвращает 1 в случае, если пользователь с идентификатором user_id является участником сообщества с идентификатором group_id, иначе 0.   При использовании параметра extended Возвращает объект, который содержит поля:   member (integer, [0,1]) — является ли пользователь участником сообщества;  и может содержать поля:   request  (integer, [0,1]) — есть ли непринятая заявка от пользователя на вступление в группу (такую заявку можно отозвать методом groups.leave);  invitation  (integer, [0,1]) — приглашён ли пользователь в группу или встречу;  can_invite (integer, [0,1]) — может ли автор запроса приглашать пользователя в группу;  can_recall (integer, [0,1]) — может ли автор отменить приглашение. Появляется, если invitation: 1.    При передаче нескольких идентификаторов Возвращает результат в виде массива объектов, в которых есть поля user_id (integer)  и member (integer, [0,1]).
        """

        params = locals()
        params.pop('self')
        return self._call('groups.isMember', params=params)

    def groups_join(self, group_id=None, not_sure=None):
        """
        Данный метод позволяет вступить в группу, публичную страницу, а также подтвердить участие во встрече.

        :param group_id: идентификатор сообщества. положительное число
        :param not_sure: опциональный параметр, учитываемый, если group_id принадлежит встрече. 1 — Возможно пойду. 0 — Точно пойду. По умолчанию 0. строка
        :return: В случае успешного вступления метод вернёт 1.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.join', params=params)

    def groups_leave(self, group_id=None):
        """
        Позволяет покинуть сообщество или отклонить приглашение в сообщество.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.leave', params=params)

    def groups_removeUser(self, group_id=None, user_id=None):
        """
        Позволяет исключить пользователя из группы или отклонить заявку на вступление. Для того, чтобы воспользоваться этим методом, Вы должны быть администратором группы

        :param group_id: идентификатор группы, из которой необходимо исключить пользователя. положительное число, обязательный параметр
        :param user_id: идентификатор пользователя, которого нужно исключить. положительное число, обязательный параметр
        :return: В случае успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.removeUser', params=params)

    def groups_reorderLink(self, group_id=None, link_id=None, after=None):
        """
        Позволяет менять местоположение ссылки в списке. Список ссылок может быть получен методом groups.getById, с параметром fields=links.  Для того, чтобы воспользоваться этим методом Вы должны быть администратором группы

        :param group_id: идентификатор группы, внутри которой перемещается ссылка положительное число, обязательный параметр
        :param link_id: идентификатор ссылки, которую необходимо переместить положительное число, обязательный параметр
        :param after: идентификатор ссылки после которой необходимо разместить перемещаемую ссылку. 0 – если ссылку нужно разместить в начале списка. положительное число
        :return: В случае успешного выполнение метод возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.reorderLink', params=params)

    def groups_search(self, q=None, type=None, country_id=None, city_id=None, future=None, market=None, sort=None,
                      offset=None, count=None):
        """
        Осуществляет поиск сообществ по заданной подстроке.

        :param q: текст поискового запроса. строка, обязательный параметр
        :param type: тип сообщества. Возможные значения: group, page, event. строка
        :param country_id: идентификатор страны. положительное число
        :param city_id: идентификатор города. При передаче этого параметра поле country_id игнорируется. положительное число
        :param future: при передаче значения 1 будут выведены предстоящие события. Учитывается только при передаче в качестве type значения event. флаг, может принимать значения 1 или 0
        :param market: при передаче значения 1 будут выведены сообщества с включенными товарами. флаг, может принимать значения 1 или 0
        :param sort:   0 — сортировать по умолчанию (аналогично результатам поиска в полной версии сайта);  1 — сортировать по скорости роста;  2 — сортировать по отношению дневной посещаемости к количеству пользователей;  3 — сортировать по отношению количества лайков к количеству пользователей;  4 — сортировать по отношению количества комментариев к количеству пользователей;  5 — сортировать по отношению количества записей в обсуждениях к количеству пользователей.  целое число
        :param offset: смещение, необходимое для выборки определённого подмножества результатов поиска. По умолчанию — 0. положительное число
        :param count: количество результатов поиска, которое необходимо вернуть. Обратите внимание — даже при использовании параметра offset для получения информации доступны только первые 1000 результатов.   положительное число, по умолчанию 20, максимальное значение 1000
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count (integer) и массив объектов group в поле items.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.search', params=params)

    def groups_setCallbackSettings(self, group_id=None, server_id=None, api_version=None, message_new=None,
                                   message_reply=None, message_allow=None, message_edit=None, message_deny=None,
                                   message_typing_state=None, photo_new=None, audio_new=None, video_new=None,
                                   wall_reply_new=None, wall_reply_edit=None, wall_reply_delete=None,
                                   wall_reply_restore=None, wall_post_new=None, wall_repost=None, board_post_new=None,
                                   board_post_edit=None, board_post_restore=None, board_post_delete=None,
                                   photo_comment_new=None, photo_comment_edit=None, photo_comment_delete=None,
                                   photo_comment_restore=None, video_comment_new=None, video_comment_edit=None,
                                   video_comment_delete=None, video_comment_restore=None, market_comment_new=None,
                                   market_comment_edit=None, market_comment_delete=None, market_comment_restore=None,
                                   poll_vote_new=None, group_join=None, group_leave=None, group_change_settings=None,
                                   group_change_photo=None, group_officers_edit=None, user_block=None,
                                   user_unblock=None, lead_forms_new=None):
        """
        Позволяет задать настройки уведомлений о событиях в Callback API.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :param server_id: идентификатор сервера. положительное число
        :param api_version: версия Callback API строка
        :param message_new: уведомления о новых сообщениях (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param message_reply: уведомления об исходящем сообщении (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param message_allow: уведомления о подписке на сообщения  (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param message_edit: уведомления о редактировании сообщения (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param message_deny: уведомления о запрете на сообщения (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param message_typing_state: уведомления о наборе текста сообщения ('0 — выключить, 1'' — включить). флаг, может принимать значения 1 или 0
        :param photo_new: уведомления о добавлении новой фотографии (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param audio_new: уведомления о добавлении новой аудиозаписи (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param video_new: уведомления о добавлении новой видеозаписи (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param wall_reply_new: уведомления о добавлении нового комментария на стене (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param wall_reply_edit: уведомления о редактировании комментария на стене (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param wall_reply_delete: уведомления об удалении комментария на стене (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param wall_reply_restore: уведомления о восстановлении комментария на стене (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param wall_post_new: уведомления о новой записи на стене (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param wall_repost: уведомления о репосте записи (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param board_post_new: уведомления о создании комментария в обсуждении (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param board_post_edit: уведомления о редактировании комментария в обсуждении (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param board_post_restore: уведомление о восстановлении комментария в обсуждении (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param board_post_delete: уведомления об удалении комментария в обсуждении (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param photo_comment_new: уведомления о добавлении нового комментария к фото (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param photo_comment_edit: уведомления о редактировании комментария к фото (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param photo_comment_delete: уведомления об удалении комментария к фото (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param photo_comment_restore: уведомления о восстановлении комментария к фото (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param video_comment_new: уведомления о добавлении нового комментария к видео (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param video_comment_edit: уведомления о редактировании комментария к видео (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param video_comment_delete: уведомления об удалении комментария к видео (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param video_comment_restore: уведомления о восстановлении комментария к видео (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param market_comment_new: уведомления о добавлении нового комментария к товару (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param market_comment_edit: уведомления о редактировании комментария к товару (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param market_comment_delete: уведомления об удалении комментария к товару (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param market_comment_restore: уведомления о восстановлении комментария к товару (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param poll_vote_new: уведомления о новом голосе в публичных опросах (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param group_join: уведомления о вступлении в сообщество (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param group_leave: уведомления о выходе из сообщества (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param group_change_settings: уведомления об изменении настроек (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param group_change_photo: уведомления об изменении главной фотографии (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param group_officers_edit: уведомления об изменении руководства (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param user_block: уведомления об внесении пользователя в чёрный список (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param user_unblock: уведомления об исключении пользователя из чёрного списка (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param lead_forms_new: уведомления о заполнении формы флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.setCallbackSettings', params=params)

    def groups_setLongPollSettings(self, group_id=None, enabled=None, api_version=None, message_new=None,
                                   message_reply=None, message_allow=None, message_deny=None, message_edit=None,
                                   message_typing_state=None, photo_new=None, audio_new=None, video_new=None,
                                   wall_reply_new=None, wall_reply_edit=None, wall_reply_delete=None,
                                   wall_reply_restore=None, wall_post_new=None, wall_repost=None, board_post_new=None,
                                   board_post_edit=None, board_post_restore=None, board_post_delete=None,
                                   photo_comment_new=None, photo_comment_edit=None, photo_comment_delete=None,
                                   photo_comment_restore=None, video_comment_new=None, video_comment_edit=None,
                                   video_comment_delete=None, video_comment_restore=None, market_comment_new=None,
                                   market_comment_edit=None, market_comment_delete=None, market_comment_restore=None,
                                   poll_vote_new=None, group_join=None, group_leave=None, group_change_settings=None,
                                   group_change_photo=None, group_officers_edit=None, user_block=None,
                                   user_unblock=None):
        """
        Задаёт настройки для Bots Long Poll API в сообществе.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :param enabled: 1 — включить Bots Long Poll, 0 — отключить. флаг, может принимать значения 1 или 0
        :param api_version: версия API строка
        :param message_new: уведомления о новых сообщениях (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param message_reply: уведомления об исходящем сообщении (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param message_allow: уведомления о подписке на сообщения  (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param message_deny: уведомления о запрете на сообщения (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param message_edit: уведомления о редактировании сообщения (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param message_typing_state: флаг, может принимать значения 1 или 0
        :param photo_new: уведомления о добавлении новой фотографии (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param audio_new: уведомления о добавлении новой аудиозаписи (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param video_new: уведомления о добавлении новой видеозаписи (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param wall_reply_new: уведомления о добавлении нового комментария на стене (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param wall_reply_edit: уведомления о редактировании комментария на стене (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param wall_reply_delete: уведомления об удалении комментария на стене (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param wall_reply_restore: уведомления о восстановлении комментария на стене (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param wall_post_new: уведомления о новой записи на стене (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param wall_repost: уведомления о репосте записи (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param board_post_new: уведомления о создании комментария в обсуждении (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param board_post_edit: уведомления о редактировании комментария в обсуждении (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param board_post_restore: уведомление о восстановлении комментария в обсуждении (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param board_post_delete: уведомления об удалении комментария в обсуждении (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param photo_comment_new: уведомления о добавлении нового комментария к фото (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param photo_comment_edit: уведомления о редактировании комментария к фото (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param photo_comment_delete: уведомления об удалении комментария к фото (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param photo_comment_restore: уведомления о восстановлении комментария к фото (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param video_comment_new: уведомления о добавлении нового комментария к видео (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param video_comment_edit: уведомления о редактировании комментария к видео (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param video_comment_delete: уведомления об удалении комментария к видео (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param video_comment_restore: уведомления о восстановлении комментария к видео (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param market_comment_new: уведомления о добавлении нового комментария к товару (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param market_comment_edit: уведомления о редактировании комментария к товару (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param market_comment_delete: уведомления об удалении комментария к товару (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param market_comment_restore: уведомления о восстановлении комментария к товару (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param poll_vote_new: уведомления о новом голосе в публичных опросах (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param group_join: уведомления о вступлении в сообщество (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param group_leave: уведомления о выходе из сообщества (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param group_change_settings: уведомления об изменении настроек (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param group_change_photo: уведомления об изменении главной фотографии (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param group_officers_edit: уведомления об изменении руководства (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param user_block: уведомления об внесении пользователя в чёрный список (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :param user_unblock: уведомления об исключении пользователя из чёрного списка (0 — выключить, 1 — включить). флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.setLongPollSettings', params=params)

    def groups_setSettings(self, group_id=None, messages=None, bots_capabilities=None, bots_start_button=None,
                           bots_add_to_chat=None):
        """
        Устанавливает настройки сообщества Для того, чтобы воспользоваться этим методом, Вы должны быть администратором сообщества.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :param messages: сообщения сообщества. Возможные значения:   0 — выключены;  1 — включены.  флаг, может принимать значения 1 или 0
        :param bots_capabilities: возможности ботов (использование клавиатуры, добавление в беседу). Возможные значения:   0 — выключены;  1 — включены.  флаг, может принимать значения 1 или 0
        :param bots_start_button: кнопка «Начать» в диалоге с сообществом.  Работает, в случае если bots_capabilities=1.  Если эта настройка включена, то при заходе в беседу с Вашим сообществом в первый раз пользователь увидит кнопку с командой «Начать», которая отправляет команду start. Payload этого сообщения будет выглядеть так: {"command":"start"}    Возможные значения:   0 — выключена;  1 — включена.  флаг, может принимать значения 1 или 0
        :param bots_add_to_chat: добавление бота в беседы.  Работает, в случае если bots_capabilities=1.  Возможные значения:   0 — запрещено;  1 — разрешено.  флаг, может принимать значения 1 или 0
        :return: В случае успеха возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.setSettings', params=params)

    def groups_unban(self, group_id=None, owner_id=None):
        """
        Убирает пользователя или группу из черного списка сообщества.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :param owner_id: идентификатор пользователя или группы, которого нужно убрать из черного списка. целое число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('groups.unban', params=params)

    def leadForms_create(self, group_id=None, name=None, title=None, description=None, questions=None,
                         policy_link_url=None, photo=None, confirmation=None, site_link_url=None, active=None,
                         once_per_user=None, pixel_code=None, notify_admins=None, notify_emails=None):
        """
        Создаёт форму сбора заявок.

        :param group_id: Идентификатор группы, в которой надо создать форму. обязательный параметр, целое число
        :param name: Название формы (видно только администратору). обязательный параметр, строка, максимальная длина 100
        :param title: Заголовок формы (видят пользователи). обязательный параметр, строка, максимальная длина 60
        :param description: Описание формы. обязательный параметр, строка, максимальная длина 600
        :param questions: Информация о вопросах формы. Массив структур следующего формата:   type — тип вопроса;  label — заголовок вопроса (только для нестандартных вопросов);  key — уникальный ключ вопроса (необязательно; только для нестандартных вопросов);  options — массив возможных ответов на вопрос (только для нестандартных вопросов типа radio, select, checkbox);   Типы стандартных вопросов:   first_name — имя;  patronymic_name — отчество;  last_name — фамилия;  email — адрес электронной почты;  phone_number — номер телефона;  age — возраст;  birthday — день рождения;  location — город и страна.   Типы нестандартных вопросов:   input — простое текстовое поле (строка);  textarea — большое текстовое поле (абзац);  radio — выбор одного из нескольких вариантов;  checkbox — выбор нескольких вариантов;  select — выбор одного варианта из выпадающего списка.   options должен быть массивом структур, описывающих варианты ответа:   label — текст ответа;  key — ключ ответа (необязательно).   Пример:[     {        "type":"first_name"     },     {        "type":"input",        "label":"Кличка кота"     },     {        "type":"select",        "key":"favorite_color",        "label":"Любимый цвет",        "options":[{          "key":"red",          "label":"Красный"        },{          "key":"green",          "label":"Зелёный"        }]     },     {        "type":"radio",        "label":"Я ношу часы...",        "options":[{           "key":"left",           "label":"на левой руке"        },{           "key":"right",           "label":"на правой руке"        }]     },     {        "type":"checkbox",        "key":"visited_cities",        "label":"Города, в которых я был",        "options":[{           "label":"Екатеринбург"        },{           "label":"Волгоград"        },{           "label":"Санкт-Петербург"        }]     }  ]   данные в формате JSON, обязательный параметр
        :param policy_link_url: Ссылка на политику конфиденциальности. строка, обязательный параметр, максимальная длина 200
        :param photo: Обложка формы.  Используйте значение, полученное после загрузки фотографии на сервер. См. метод leadForms.getUploadURL.  Также можно переиспользовать существующую фотографию из другой формы. Используйте значение поля photo, которое возвращает метод leadForms.get или leadForms.list. строка
        :param confirmation: Текст подтверждения, который увидит пользователь после отправки формы. строка, максимальная длина 300
        :param site_link_url: Ссылка на сайт или сообщество автора формы. строка, максимальная длина 200
        :param active: Передайте 1, чтобы активировать форму сразу после создания. Пользователи могут оставлять заявки только в активных формах. флаг, может принимать значения 1 или 0, по умолчанию 0
        :param once_per_user: Передайте 1, чтобы разрешить каждому пользователю заполнять форму только один раз. флаг, может принимать значения 1 или 0, по умолчанию 0
        :param pixel_code: Передайте код пикселя ретаргетинга ВКонтакте в виде VK-RTRG-000000-XXXXX. По этому пикселю будет собираться аудитория пользователей, открывших форму. Подробнее о пикселе см. здесь. строка
        :param notify_admins: Передайте список идентификаторов пользователей, которым будут отправлены оповещения о заполнении пользователями формы. Оповещения могут быть отправлены только администраторам сообщества. список положительных чисел, разделенных запятыми
        :param notify_emails: Передайте список email-адресов, разделённых запятой, на которые будут отправлены оповещения о заполнении пользователями формы. Можно указать до 10 адресов. список слов, разделенных через запятую
        :return: Возвращается структура с информацией о созданной форме:   form_id — идентификатор формы;  url — ссылка на форму.
        """

        params = locals()
        params.pop('self')
        return self._call('leadForms.create', params=params)

    def leadForms_delete(self, group_id=None, form_id=None):
        """
        Удаляет форму сбора заявок.

        :param group_id: Идентификатор группы, из которой надо удалить форму. обязательный параметр, целое число
        :param form_id: Идентификатор удаляемой формы. обязательный параметр, целое число
        :return: Возвращает идентификатор удалённой формы
        """

        params = locals()
        params.pop('self')
        return self._call('leadForms.delete', params=params)

    def leadForms_get(self, group_id=None, form_id=None):
        """
        Возвращает информацию о форме сбора заявок.

        :param group_id: Идентификатор группы, в которой находится форма. обязательный параметр, целое число
        :param form_id: Идентификатор формы. обязательный параметр, целое число
        :return: Возвращает структуру с информацией о форме. Значения полей см. в методе leadForms.create.  Дополнительно возвращает следующие поля:   form_id — идентификатор формы;  leads_count — количество заявок;  url — ссылка на форму.
        """

        params = locals()
        params.pop('self')
        return self._call('leadForms.get', params=params)

    def leadForms_getLeads(self, group_id=None, form_id=None, limit=None, next_page_token=None):
        """
        Возвращает заявки формы.

        :param group_id: идентификатор сообщества, в котором находится форма. обязательный параметр, целое число
        :param form_id: идентификатор формы. обязательный параметр, целое число
        :param limit: количество возвращаемых заявок за один запрос. положительное число, по умолчанию 10, максимальное значение 1000, минимальное значение 1
        :param next_page_token: строка, полученная из предыдущего запроса для получения следующей порции результатов. строка
        :return: Возвращает массив структур со следующими полями:   lead_id — идентификатор заявки, уникальный в рамках сообщества, в котором находится форма;  user_id — идентификатор пользователя, оставившего заявку;  date — дата и время оставления заявки в формате unix timestamp;  answers — информация об ответах на вопросы — массив структур со следующими полями:   key — ключ вопроса. Совпадает с типом стандартного вопроса (его полем type), или с ключом нестандартного вопроса (поле key). Для нестандартных вопросов, для которых не был задан ключ, будет использовано значение вида custom_0, где «0» — порядковый номер нестандартного вопроса, считая с 0.  answer — ответ на вопрос. Структура (для всех вопросов, кроме типа checkbox) или массив структур (для типа checkbox) со следующими полями:   key — ключ ответа (в случае, если был задан при создании формы);  value — текст ответа;    ad_id — идентификатор рекламного объявления, с которого пришла заявка (поле отсутствует в случае, если заявка пришла не из рекламного объявления).
        """

        params = locals()
        params.pop('self')
        return self._call('leadForms.getLeads', params=params)

    def leadForms_getUploadURL(self):
        """
        Возвращает URL для загрузки обложки для формы. Для форм сбора заявок используются прямоугольные изображения размером 1200х300. В случае загрузки изображения другого размера, оно будет автоматически обрезано до требуемого.  Загрузка изображения на сервер ВКонтакте осуществляется в 3 этапа:  1. С помощью метода leadForms.getUploadURL приложение узнает HTTP-адрес для загрузки изображения.  Один адрес можно использовать только для загрузки одного изображения.  2. Приложение формирует POST-запрос на полученный адрес для сохранения изображения на сервере ВКонтакте.  Запрос должен содержать поле file, которое содержит файл с изображением (JPG, PNG, BMP, TIFF или GIF). Максимальный объем файла: 5 Мб.  Результат загрузки изображения возвращается приложению в формате JSON.  В случае удачной загрузки: {"photo":"данные"}    В случае ошибки: {"errcode":код ошибки}    Если получен код ошибки ≥ 10, то загрузку изображения следует начать с первого пункта.  3. Полученные данные можно использовать в методах leadForms.create и leadForms.update.

        :return:    В ходе выполнения могут произойти общие ошибки. Их описание находится на отдельной странице.
        """

        params = locals()
        params.pop('self')
        return self._call('leadForms.getUploadURL', params=params)

    def leadForms_list(self, group_id=None):
        """
        Возвращает список форм сообщества.

        :param group_id: Идентификатор сообщества. обязательный параметр, целое число
        :return: Возвращает массив структур с описанием форм. Подробнее о структуре описания формы см. метод leadForms.get.
        """

        params = locals()
        params.pop('self')
        return self._call('leadForms.list', params=params)

    def leadForms_update(self, group_id=None, form_id=None, name=None, title=None, description=None, questions=None,
                         policy_link_url=None, photo=None, confirmation=None, site_link_url=None, active=None,
                         once_per_user=None, pixel_code=None, notify_admins=None, notify_emails=None):
        """
        Обновляет форму сбора заявок. При обновлении формы все указанные параметры заменяют собой предыдущие значения.

        :param group_id: Идентификатор группы, в которой надо обновить форму. обязательный параметр, целое число
        :param form_id: Идентификатор формы, которую надо обновить. обязательный параметр, целое число
        :param name: Новое название формы (видно только администратору). обязательный параметр, строка, максимальная длина 100
        :param title: Новый заголовок формы (видят пользователи). обязательный параметр, строка, максимальная длина 60
        :param description: Новое описание формы. обязательный параметр, строка, максимальная длина 600
        :param questions: Новые вопросы формы. Подробнее см. метод leadForms.create. данные в формате JSON, обязательный параметр
        :param policy_link_url: Новая ссылка на политику конфиденциальности. строка, обязательный параметр, максимальная длина 200
        :param photo: Новая обложка формы. строка
        :param confirmation: Новый текст подтверждения, который увидит пользователь после отправки формы. строка, максимальная длина 300
        :param site_link_url: Новая ссылка на сайт или сообщество автора формы. строка, максимальная длина 200
        :param active: Передайте 1, чтобы активировать форму. Пользователи могут оставлять заявки только в активных формах. флаг, может принимать значения 1 или 0, по умолчанию 0
        :param once_per_user: Передайте 1, чтобы разрешить каждому пользователю заполнять форму только один раз. флаг, может принимать значения 1 или 0, по умолчанию 0
        :param pixel_code: Новый код пикселя ретаргетинга ВКонтакте. строка
        :param notify_admins: Новый список идентификаторов пользователей, которым будут отправлены оповещения о заполнении пользователями формы. Оповещения могут быть отправлены только администраторам сообщества. список положительных чисел, разделенных запятыми
        :param notify_emails: Новый список email-адресов, разделённых запятой, на которые будут отправлены оповещения о заполнении пользователями формы. Можно указать до 10 адресов. список слов, разделенных через запятую
        :return: Возвращается структура с информацией об обновлённой форме:   form_id — идентификатор формы;  url — ссылка на форму.
        """

        params = locals()
        params.pop('self')
        return self._call('leadForms.update', params=params)

    def likes_add(self, type=None, owner_id=None, item_id=None, access_key=None):
        """
        Добавляет указанный объект в список Мне нравится текущего пользователя.

        :param type: тип объекта.  Возможные типы: post — запись на стене пользователя или группы; comment — комментарий к записи на стене; photo — фотография; audio — аудиозапись; video — видеозапись; note — заметка; market — товар; photo_comment — комментарий к фотографии; video_comment — комментарий к видеозаписи; topic_comment — комментарий в обсуждении; market_comment — комментарий к товару; строка, обязательный параметр
        :param owner_id: идентификатор владельца объекта. целое число, по умолчанию идентификатор текущего пользователя
        :param item_id: идентификатор объекта. положительное число, обязательный параметр
        :param access_key: ключ доступа в случае работы с приватными объектами. строка
        :return: В случае успеха возвращает объект с полем likes, в котором находится текущее количество пользователей, которые добавили данный объект в свой список Мне нравится.
        """

        params = locals()
        params.pop('self')
        return self._call('likes.add', params=params)

    def likes_delete(self, type=None, owner_id=None, item_id=None, access_key=None):
        """
        Удаляет указанный объект из списка Мне нравится текущего пользователя

        :param type: тип объекта.  Возможные типы: post — запись на стене пользователя или группы; comment — комментарий к записи на стене; photo — фотография; audio — аудиозапись; video — видеозапись; note — заметка; photo_comment — комментарий к фотографии; video_comment — комментарий к видеозаписи; topic_comment — комментарий в обсуждении; sitepage — страница сайта, на котором установлен виджет «Мне нравится». строка, обязательный параметр
        :param owner_id: идентификатор владельца объекта. целое число, по умолчанию идентификатор текущего пользователя
        :param item_id: идентификатор объекта. положительное число, обязательный параметр
        :param access_key: строка
        :return: В случае успеха возвращает объект с полем likes, в котором находится текущее количество пользователей, которые добавили данный объект в свой список Мне нравится.
        """

        params = locals()
        params.pop('self')
        return self._call('likes.delete', params=params)

    def likes_getList(self, type=None, owner_id=None, item_id=None, page_url=None, filter=None, friends_only=None,
                      extended=None, offset=None, count=None, skip_own=None):
        """
        Получает список идентификаторов пользователей, которые добавили заданный объект в свой список Мне нравится. Обратите внимание, данные о репостах доступны только для записей, созданных текущим пользователем или сообществом, в котором он является администратором.

        :param type: тип объекта.  Возможные типы: post — запись на стене пользователя или группы; comment — комментарий к записи на стене; photo — фотография; audio — аудиозапись; video — видеозапись; note — заметка; market — товар; photo_comment — комментарий к фотографии; video_comment — комментарий к видеозаписи; topic_comment — комментарий в обсуждении; market_comment — комментарий к товару; sitepage — страница сайта, на котором установлен виджет «Мне нравится». строка, обязательный параметр
        :param owner_id: идентификатор владельца Like-объекта: id пользователя, id сообщества (со знаком «минус») или id приложения. Если параметр type равен sitepage, то в качестве owner_id необходимо передавать id приложения. Если параметр не задан, то считается, что он равен либо идентификатору текущего пользователя, либо идентификатору текущего приложения (если type равен sitepage). целое число
        :param item_id: идентификатор Like-объекта. Если type равен sitepage, то параметр item_id может содержать значение параметра page_id, используемый при инициализации  виджета «Мне нравится». целое число
        :param page_url: url страницы, на которой установлен виджет «Мне нравится». Используется вместо параметра item_id, если при размещении виджета не был указан page_id. строка
        :param filter: указывает, следует ли вернуть всех пользователей, добавивших объект в список "Мне нравится" или только тех, которые рассказали о нем друзьям. Параметр может принимать следующие значения:   likes — возвращать информацию обо всех пользователях;  copies — возвращать информацию только о пользователях, рассказавших об объекте друзьям. По умолчанию возвращается информация обо всех пользователях.  строка
        :param friends_only: указывает, необходимо ли возвращать только пользователей, которые являются друзьями текущего пользователя. Параметр может принимать следующие значения:   0 — возвращать всех пользователей в порядке убывания времени добавления объекта;  1 — возвращать только друзей текущего пользователя в порядке убывания времени добавления объекта; Если метод был вызван без авторизации или параметр не был задан, то считается, что он равен 0.   целое число, по умолчанию 0
        :param extended: 1 — возвращать расширенную информацию о пользователях и сообществах из списка поставивших отметку «Мне нравится» или сделавших репост. По умолчанию — 0. флаг, может принимать значения 1 или 0
        :param offset: смещение, относительно начала списка, для выборки определенного подмножества. Если параметр не задан, то считается, что он равен 0. положительное число
        :param count: количество возвращаемых идентификаторов пользователей. Если параметр не задан, то считается, что он равен 100, если не задан параметр friends_only, в противном случае 10. Максимальное значение параметра 1000, если не задан параметр friends_only, в противном случае 100. положительное число
        :param skip_own: не возвращать самого пользователя. флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает список идентификаторов пользователей с учетом параметров offset и count, которые добавили заданный объект в свой список Мне нравится.  Если параметр type равен sitepage, то будет возвращён список пользователей, воспользовавшихся виджетом «Мне нравится» на внешнем сайте. Адрес страницы задаётся при помощи параметра page_url или item_id.  Если extended=1, дополнительно возвращается массив items, содержащий расширенную информацию о пользователях или сообществах.
        """

        params = locals()
        params.pop('self')
        return self._call('likes.getList', params=params)

    def likes_isLiked(self, user_id=None, type=None, owner_id=None, item_id=None):
        """
        Проверяет, находится ли объект в списке Мне нравится заданного пользователя.

        :param user_id: идентификатор пользователя, у которого необходимо проверить наличие объекта в списке «Мне нравится». Если параметр не задан, то считается, что он равен идентификатору текущего пользователя. положительное число, по умолчанию идентификатор текущего пользователя
        :param type: тип объекта.  Возможные типы: post — запись на стене пользователя или группы; comment — комментарий к записи на стене; photo — фотография; audio — аудиозапись; video — видеозапись; note — заметка; photo_comment — комментарий к фотографии; video_comment — комментарий к видеозаписи; topic_comment — комментарий в обсуждении; строка, обязательный параметр
        :param owner_id: идентификатор владельца Like-объекта. Если параметр не задан, то считается, что он равен идентификатору текущего пользователя.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param item_id: идентификатор объекта. положительное число, обязательный параметр
        :return: В случае успеха возвращает объект с полями:   liked — есть ли отметка «Мне нравится» от текущего пользователя (0 — отметки нет, 1 — отметка есть);  copied — сделан ли репост текущим пользователем (0 — не сделан, 1 — сделан).
        """

        params = locals()
        params.pop('self')
        return self._call('likes.isLiked', params=params)

    def market_add(self, owner_id=None, name=None, description=None, category_id=None, price=None, old_price=None,
                   deleted=None, main_photo_id=None, photo_ids=None, url=None):
        """
        Добавляет новый товар.

        :param owner_id: идентификатор владельца товара.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param name: название товара. Ограничение по длине считается в кодировке cp1251. строка, минимальная длина 4, максимальная длина 100, обязательный параметр
        :param description: описание товара. строка, минимальная длина 10, обязательный параметр
        :param category_id: идентификатор категории товара. положительное число, обязательный параметр
        :param price: цена товара. дробное число, минимальное значение 0.01
        :param old_price: старая цена товара. дробное число, минимальное значение 0.01
        :param deleted: статус товара (1 — товар удален, 0 — товар не удален). флаг, может принимать значения 1 или 0
        :param main_photo_id: идентификатор фотографии обложки товара.  Фотография должна быть загружена с помощью метода photos.getMarketUploadServer, передав параметр main_photo. См. подробную информацию о загрузке фотографии товаров положительное число, обязательный параметр
        :param photo_ids: идентификаторы дополнительных фотографий товара.  Фотография должна быть загружена с помощью метода photos.getMarketUploadServer. См. подробную информацию о загрузке фотографии товаров список положительных чисел, разделенных запятыми
        :param url: ссылка на сайт товара. строка, минимальная длина 0, максимальная длина 320
        :return: После успешного выполнения возвращает идентификатор добавленного товара.
        """

        params = locals()
        params.pop('self')
        return self._call('market.add', params=params)

    def market_addAlbum(self, owner_id=None, title=None, photo_id=None, main_album=None):
        """
        Добавляет новую подборку с товарами.

        :param owner_id: идентификатор владельца подборки.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param title: название подборки. строка, обязательный параметр, максимальная длина 128
        :param photo_id: идентификатор фотографии-обложки подборки.  Фотография должна быть загружена с помощью метода photos.getMarketAlbumUploadServer. См. подробную информацию о загрузке фотографии товаров. положительное число
        :param main_album: назначить подборку основной (1 — назначить, 0 — нет). флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает идентификатор созданной подборки.
        """

        params = locals()
        params.pop('self')
        return self._call('market.addAlbum', params=params)

    def market_addToAlbum(self, owner_id=None, item_id=None, album_ids=None):
        """
        Добавляет товар в одну или несколько выбранных подборок.

        :param owner_id: идентификатор владельца товара.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param item_id: идентификатор товара. положительное число, обязательный параметр
        :param album_ids: идентификаторы подборок, в которые нужно добавить товар. список положительных чисел, разделенных запятыми, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('market.addToAlbum', params=params)

    def market_createComment(self, owner_id=None, item_id=None, message=None, attachments=None, from_group=None,
                             reply_to_comment=None, sticker_id=None, guid=None):
        """
        Создает новый комментарий к товару.

        :param owner_id: идентификатор владельца товара. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param item_id: идентификатор товара. положительное число, обязательный параметр
        :param message: текст комментария (является обязательным, если не задан параметр attachments). строка
        :param attachments: список объектов, приложенных к комментарию и разделённых символом ",". Поле attachments представляется в формате: <type><owner_id>_<media_id>,<type><owner_id>_<media_id> <type> — тип медиа-вложения: photo — фотография  video — видеозапись  audio — аудиозапись  doc — документ <owner_id> — идентификатор владельца медиа-вложения  <media_id> — идентификатор медиа-вложения.   Например: photo100172_166443618,photo66748_265827614 Параметр является обязательным, если не задан параметр message. список слов, разделенных через запятую
        :param from_group: 1 — комментарий будет опубликован от имени группы, 0 — комментарий будет опубликован от имени пользователя (по умолчанию). флаг, может принимать значения 1 или 0
        :param reply_to_comment: идентификатор комментария, в ответ на который должен быть добавлен новый комментарий. положительное число
        :param sticker_id: идентификатор стикера. положительное число
        :param guid: уникальный идентификатор, предназначенный для предотвращения повторной отправки одинакового комментария. строка
        :return: После успешного выполнения возвращает идентификатор созданного комментария.
        """

        params = locals()
        params.pop('self')
        return self._call('market.createComment', params=params)

    def market_delete(self, owner_id=None, item_id=None):
        """
        Удаляет товар.

        :param owner_id: идентификатор владельца товара.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param item_id: идентификатор товара. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('market.delete', params=params)

    def market_deleteAlbum(self, owner_id=None, album_id=None):
        """
        Удаляет подборку с товарами.

        :param owner_id: идентификатор владельца подборки.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param album_id: идентификатор подборки. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('market.deleteAlbum', params=params)

    def market_deleteComment(self, owner_id=None, comment_id=None):
        """
        Удаляет комментарий к товару.

        :param owner_id: идентификатор владельца товара.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param comment_id: идентификатор комментария. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1 (0, если комментарий не найден).
        """

        params = locals()
        params.pop('self')
        return self._call('market.deleteComment', params=params)

    def market_edit(self, owner_id=None, item_id=None, name=None, description=None, category_id=None, price=None,
                    old_price=None, deleted=None, main_photo_id=None, photo_ids=None, url=None):
        """
        Редактирует товар.

        :param owner_id: идентификатор владельца товара.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param item_id: идентификатор товара. положительное число, обязательный параметр
        :param name: новое название товара. строка
        :param description: новое описание товара. строка
        :param category_id: идентификатор категории товара. положительное число
        :param price: цена товара. дробное число, минимальное значение 0.01
        :param old_price: старая цена. дробное число, минимальное значение 0.01
        :param deleted: статус товара (1 — товар удален, 0 — товар не удален). флаг, может принимать значения 1 или 0
        :param main_photo_id: идентификатор фотографии для обложки товара.  Фотография может быть загружена с помощью метода photos.getMarketUploadServer (параметр main_photo). См. подробную информацию о загрузке фотографии товаров.  Идентификатор фотографии может быть также получен методами market.get или market.getById (если Вы хотите использовать существующую фотографию товара). положительное число
        :param photo_ids: идентификаторы дополнительных фотографией товара.  Фотография должна быть загружена с помощью метода photos.getMarketUploadServer. См. подробную информацию о загрузке фотографии товаров. список положительных чисел, разделенных запятыми
        :param url: ссылка на сайт товара. строка, минимальная длина 0, максимальная длина 320
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('market.edit', params=params)

    def market_editAlbum(self, owner_id=None, album_id=None, title=None, photo_id=None, main_album=None):
        """
        Редактирует подборку с товарами.

        :param owner_id: идентификатор владельца подборки.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param album_id: идентификатор подборки. положительное число, обязательный параметр
        :param title: название подборки. строка, обязательный параметр, максимальная длина 128
        :param photo_id: идентификатор фотографии-обложки подборки.  Фотография должна быть загружена с помощью метода photos.getMarkeAlbumUploadServer. См. подробную информацию о загрузке фотографии товаров. положительное число
        :param main_album: назначить подборку основной (1 — назначить, 0 — нет). флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('market.editAlbum', params=params)

    def market_editComment(self, owner_id=None, comment_id=None, message=None, attachments=None):
        """
        Изменяет текст комментария к товару.

        :param owner_id: идентификатор владельца товара.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param comment_id: идентификатор комментария. положительное число, обязательный параметр
        :param message: новый текст комментария (является обязательным, если не задан параметр attachments).  Максимальное количество символов: 2048. строка
        :param attachments: новый список объектов, приложенных к комментарию и разделённых символом ",". Поле attachments представляется в формате: <type><owner_id>_<media_id>,<type><owner_id>_<media_id> <type> — тип медиа-вложения: photo — фотография  video — видеозапись  audio — аудиозапись  doc — документ <owner_id> — идентификатор владельца медиа-вложения  <media_id> — идентификатор медиа-вложения.   Например: photo100172_166443618,photo66748_265827614 Параметр является обязательным, если не задан параметр message. список слов, разделенных через запятую
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('market.editComment', params=params)

    def market_get(self, owner_id=None, album_id=None, count=None, offset=None, extended=None):
        """
        Возвращает список товаров в сообществе.

        :param owner_id: идентификатор владельца товаров.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param album_id: идентификатор подборки, товары из которой нужно вернуть. положительное число, по умолчанию 0
        :param count: количество возвращаемых товаров. положительное число, максимальное значение 200, по умолчанию 100
        :param offset: смещение относительно первого найденного товара для выборки определенного подмножества. положительное число
        :param extended: 1 — будут возвращены дополнительные поля likes, can_comment, can_repost, photos, views_count. По умолчанию эти поля не возвращается. флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов товаров в поле items.
        """

        params = locals()
        params.pop('self')
        return self._call('market.get', params=params)

    def market_getAlbumById(self, owner_id=None, album_ids=None):
        """
        Возвращает данные подборки с товарами.

        :param owner_id: идентификатор владельца подборки.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param album_ids: идентификаторы подборок, данные о которых нужно получить. список положительных чисел, разделенных запятыми, обязательный параметр
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов подборок в поле items.
        """

        params = locals()
        params.pop('self')
        return self._call('market.getAlbumById', params=params)

    def market_getAlbums(self, owner_id=None, offset=None, count=None):
        """
        Возвращает список подборок с товарами.

        :param owner_id: идентификатор владельца товаров.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param offset: смещение относительно первой найденной подборки для выборки определенного подмножества. положительное число
        :param count: количество возвращаемых подборок. положительное число, по умолчанию 50, максимальное значение 100
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов подборок в поле items.
        """

        params = locals()
        params.pop('self')
        return self._call('market.getAlbums', params=params)

    def market_getById(self, item_ids=None, extended=None):
        """
        Возвращает информацию о товарах по идентификаторам.

        :param item_ids: перечисленные через запятую идентификаторы — идущие через знак подчеркивания id владельцев товаров и id самих товаров. Если товар принадлежит сообществу, то в качестве первого параметра используется -id сообщества. Пример значения item_ids:  -4363_136089719,13245770_137352259 список слов, разделенных через запятую, обязательный параметр
        :param extended: 1 — будут возвращены дополнительные поля likes, can_comment, can_repost, photos, views_count. По умолчанию: 0. флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов товаров в поле items.
        """

        params = locals()
        params.pop('self')
        return self._call('market.getById', params=params)

    def market_getCategories(self, count=None, offset=None):
        """
        Возвращает список категорий для товаров.

        :param count: количество категорий, информацию о которых необходимо вернуть. положительное число, максимальное значение 1000, по умолчанию 10
        :param offset: смещение, необходимое для выборки определенного подмножества категорий. положительное число
        :return: После успешного выполнения возвращает список объектов category.  Объект category — категория товара.   id   (integer) идентификатор категории. name   (string) название метки. section   (object) секция. Объект, содержащий поля:   id (integer) — идентификатор секции;  name (string)  — название секции.
        """

        params = locals()
        params.pop('self')
        return self._call('market.getCategories', params=params)

    def market_getComments(self, owner_id=None, item_id=None, need_likes=None, start_comment_id=None, offset=None,
                           count=None, sort=None, extended=None, fields=None):
        """
        Возвращает список комментариев к товару.

        :param owner_id: идентификатор владельца товара.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param item_id: идентификатор товара. положительное число, обязательный параметр
        :param need_likes: 1 — возвращать информацию о лайках. флаг, может принимать значения 1 или 0
        :param start_comment_id: идентификатор комментария, начиная с которого нужно вернуть список (подробности см. ниже). положительное число
        :param offset: сдвиг, необходимый для получения конкретной выборки результатов. положительное число, по умолчанию 0
        :param count: число комментариев, которые необходимо получить. положительное число, по умолчанию 20, максимальное значение 100
        :param sort: порядок сортировки комментариев (asc — от старых к новым, desc - от новых к старым) строка, по умолчанию asc
        :param extended: 1 — комментарии в ответе будут возвращены в виде пронумерованных объектов, дополнительно будут возвращены списки объектов profiles, groups. флаг, может принимать значения 1 или 0
        :param fields: список дополнительных полей профилей, которые необходимо вернуть. См. подробное описание.  Доступные значения: sex, bdate, city, country, photo_50, photo_100, photo_200_orig, photo_200, photo_400_orig, photo_max, photo_max_orig, photo_id, online, online_mobile, domain, has_mobile, contacts, connections, site, education, universities, schools, can_post, can_see_all_posts, can_see_audio, can_write_private_message, status, last_seen, common_count, relation, relatives, counters, screen_name, maiden_name, timezone, occupation,activities, interests, music, movies, tv, books, games, about, quotes, personal, friend_status, military, career список слов, разделенных через запятую
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов комментариев в поле items.
        """

        params = locals()
        params.pop('self')
        return self._call('market.getComments', params=params)

    def market_removeFromAlbum(self, owner_id=None, item_id=None, album_ids=None):
        """
        Удаляет товар из одной или нескольких выбранных подборок.

        :param owner_id: идентификатор владельца товара.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param item_id: идентификатор товара. положительное число, обязательный параметр
        :param album_ids: идентификаторы подборок, из которых нужно удалить товар. список положительных чисел, разделенных запятыми, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('market.removeFromAlbum', params=params)

    def market_reorderAlbums(self, owner_id=None, album_id=None, before=None, after=None):
        """
        Изменяет положение подборки с товарами в списке.

        :param owner_id: идентификатор владельца альбома.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param album_id: идентификатор подборки. целое число, обязательный параметр
        :param before: идентификатор подборки, перед которой следует поместить текущую. положительное число
        :param after: идентификатор подборки, после которой следует поместить текущую. положительное число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('market.reorderAlbums', params=params)

    def market_reorderItems(self, owner_id=None, album_id=None, item_id=None, before=None, after=None):
        """
        Изменяет положение товара в подборке.

        :param owner_id: идентификатор владельца товара.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param album_id: идентификатор подборки, в которой находится товар.  0 — для сортировки общего списка товаров. целое число
        :param item_id: идентификатор товара. положительное число, обязательный параметр
        :param before: идентификатор товара, перед которым следует поместить текущий. положительное число
        :param after: идентификатор товара, после которого следует поместить текущий. положительное число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('market.reorderItems', params=params)

    def market_report(self, owner_id=None, item_id=None, reason=None):
        """
        Позволяет отправить жалобу на товар.

        :param owner_id: идентификатор владельца товаров.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param item_id: идентификатор товара положительное число, обязательный параметр
        :param reason: причина жалобы:   0 — спам;  1 — детская порнография;  2 — экстремизм;  3 — насилие;  4 — пропаганда наркотиков;  5 — материал для взрослых;  6 — оскорбление.  положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('market.report', params=params)

    def market_reportComment(self, owner_id=None, comment_id=None, reason=None):
        """
        Позволяет оставить жалобу на комментарий к товару.

        :param owner_id: идентификатор владельца товара.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param comment_id: идентификатор комментария. положительное число, обязательный параметр
        :param reason: причина жалобы:   0 — спам;  1 — детская порнография;  2 — экстремизм;  3 — насилие;  4 — пропаганда наркотиков;  5 — материал для взрослых;  6 — оскорбление.  положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('market.reportComment', params=params)

    def market_restore(self, owner_id=None, item_id=None):
        """
        Восстанавливает удаленный товар.

        :param owner_id: идентификатор владельца товара.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param item_id: идентификатор товара. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1 (0, если товар не найден среди удаленных).
        """

        params = locals()
        params.pop('self')
        return self._call('market.restore', params=params)

    def market_restoreComment(self, owner_id=None, comment_id=None):
        """
        Восстанавливает удаленный комментарий к товару.

        :param owner_id: идентификатор владельца товара.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param comment_id: идентификатор удаленного комментария. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1 (0, если комментарий с таким идентификатором не является удаленным).
        """

        params = locals()
        params.pop('self')
        return self._call('market.restoreComment', params=params)

    def market_search(self, owner_id=None, album_id=None, q=None, price_from=None, price_to=None, tags=None, sort=None,
                      rev=None, offset=None, count=None, extended=None, status=None):
        """
        Ищет товары в каталоге сообщества.

        :param owner_id: идентификатор сообщества, которому принадлежат товары. целое число, обязательный параметр
        :param album_id: идентификатор подборки, товары из которой нужно вернуть. положительное число, по умолчанию 0
        :param q: строка поискового запроса. Например, зеленые тапочки. строка
        :param price_from: минимальное значение цены товаров в сотых долях единицы валюты. Например, 100000. положительное число
        :param price_to: максимальное значение цены товаров  в сотых долях единицы валюты. Например, 1410000. положительное число
        :param tags: перечисленные через запятую идентификаторы меток. список целых чисел, разделенных запятыми
        :param sort: вид сортировки.  0 — пользовательская расстановка,  1 — по дате добавления товара, 2 — по цене, 3 — по популярности. целое число, по умолчанию 0
        :param rev: 0 — не использовать обратный порядок, 1 — использовать обратный порядок. положительное число, по умолчанию 1
        :param offset: смещение относительно первого найденного товара для выборки определенного подмножества. положительное число
        :param count: количество возвращаемых товаров. положительное число, по умолчанию 20, максимальное значение 200
        :param extended: 1 — будут возвращены дополнительные поля likes, can_comment, can_repost, photos, views_count. По умолчанию эти поля не возвращается. флаг, может принимать значения 1 или 0, по умолчанию 0
        :param status: положительное число, по умолчанию 0
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов товаров в поле items.
        """

        params = locals()
        params.pop('self')
        return self._call('market.search', params=params)

    def messages_addChatUser(self, chat_id=None, user_id=None, visible_messages_count=None):
        """
        Добавляет в мультидиалог нового пользователя. Чтобы пользователю вернуться в беседу, которую он сам покинул, достаточно отправить сообщение в неё (если есть свободные места), либо вызвать этот метод, передав свой идентификатор в параметре user_id.

        :param chat_id: идентификатор беседы. положительное число, обязательный параметр, максимальное значение 100000000
        :param user_id: идентификатор пользователя, которого необходимо включить в беседу. положительное число
        :param visible_messages_count: положительное число, максимальное значение 1000
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.addChatUser', params=params)

    def messages_allowMessagesFromGroup(self, group_id=None, key=None):
        """
        Позволяет разрешить отправку сообщений от сообщества текущему пользователю.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :param key: произвольная строка. Этот параметр можно использовать для идентификации пользователя. Его значение будет возвращено в событии message_allow Callback API. строка, максимальная длина 256
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.allowMessagesFromGroup', params=params)

    def messages_createChat(self, user_ids=None, title=None, group_id=None):
        """
        Создаёт беседу с несколькими участниками.

        :param user_ids: идентификаторы пользователей, которых нужно включить в мультидиалог. Должны быть в друзьях у текущего пользователя. список положительных чисел, разделенных запятыми
        :param title: название беседы. строка
        :param group_id: идентификатор сообщества (для сообщений сообщества с ключом доступа пользователя). положительное число
        :return: После успешного выполнения возвращает  идентификатор созданного чата (chat_id).
        """

        params = locals()
        params.pop('self')
        return self._call('messages.createChat', params=params)

    def messages_delete(self, message_ids=None, spam=None, group_id=None, delete_for_all=None):
        """
        Удаляет сообщение.

        :param message_ids: список идентификаторов сообщений, разделённых через запятую. список положительных чисел, разделенных запятыми
        :param spam: пометить сообщения как спам. флаг, может принимать значения 1 или 0
        :param group_id: идентификатор сообщества (для сообщений сообщества с ключом доступа пользователя). положительное число
        :param delete_for_all: 1 — если сообщение нужно удалить для получателей (если с момента отправки сообщения прошло не более 24 часов ). флаг, может принимать значения 1 или 0, по умолчанию
        :return: После успешного выполнения возвращает 1 для каждого удаленного сообщения.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.delete', params=params)

    def messages_deleteChatPhoto(self, chat_id=None, group_id=None):
        """
        Позволяет удалить фотографию мультидиалога.

        :param chat_id: идентификатор беседы. положительное число, обязательный параметр, максимальное значение 100000000
        :param group_id: идентификатор сообщества (для сообщений сообщества с ключом доступа пользователя). положительное число
        :return: После успешного выполнения возвращает объект, содержащий следующие поля:  message_id — идентификатор отправленного системного сообщения;  chat — объект мультидиалога.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.deleteChatPhoto', params=params)

    def messages_deleteConversation(self, user_id=None, peer_id=None, group_id=None, offset=None, count=None):
        """
        Удаляет беседу.

        :param user_id: идентификатор пользователя. Если требуется очистить историю беседы, используйте peer_id. строка
        :param peer_id: идентификатор назначения.  Для групповой беседы:  2000000000 + id беседы.   Для сообщества:  -id сообщества.   целое число, доступен начиная с версии 5.38
        :param group_id: идентификатор сообщества (для сообщений сообщества с ключом доступа пользователя). положительное число
        :param offset: начиная с какого сообщения нужно удалить переписку. (По умолчанию удаляются все сообщения начиная с первого). положительное число, по умолчанию 0, устаревший параметр, доступен только для версий меньше 5.100
        :param count: сколько сообщений нужно удалить. Обратите внимание, что на метод наложено ограничение, за один вызов нельзя удалить больше 10000 сообщений, поэтому если сообщений в переписке больше — метод нужно вызывать несколько раз. положительное число, максимальное значение 10000, устаревший параметр, доступен только для версий меньше 5.100
        :return: После успешного выполнения возвращает поле last_deleted_id, содержащее идентификатор последнего удалённого сообщения в переписке.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.deleteConversation', params=params)

    def messages_denyMessagesFromGroup(self, group_id=None):
        """
        Позволяет запретить отправку сообщений от сообщества текущему пользователю.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.denyMessagesFromGroup', params=params)

    def messages_edit(self, peer_id=None, message=None, message_id=None, lat=None, long=None, attachment=None,
                      keep_forward_messages=None, keep_snippets=None, group_id=None, dont_parse_links=None):
        """
        Редактирует сообщение.

        :param peer_id: идентификатор назначения.  Для пользователя:  id  пользователя.   Для групповой беседы:  2000000000 + id беседы.   Для сообщества:  -id сообщества.   целое число, обязательный параметр
        :param message: текст  сообщения. Обязательный параметр, если не задан параметр attachment. строка
        :param message_id: идентификатор сообщения. положительное число
        :param lat: географическая широта (от -90 до 90). дробное число
        :param long: географическая долгота (от -180 до 180). дробное число
        :param attachment: медиавложения к личному сообщению, перечисленные через запятую. Каждое прикрепление представлено в формате: <type><owner_id>_<media_id>  <type> — тип медиавложения:   photo — фотография;  video — видеозапись;  audio — аудиозапись;  doc — документ;  wall — запись на стене;  market — товар.     <owner_id> — идентификатор владельца медиавложения (обратите внимание, если объект находится в сообществе, этот параметр должен быть отрицательным). <media_id> — идентификатор медиавложения.  Например: photo100172_166443618  Параметр является обязательным, если не задан параметр message.  В случае, если прикрепляется объект, принадлежащий другому пользователю следует добавлять к вложению его access_key в формате <type><owner_id>_<media_id>_<access_key>, Например: video85635407_165186811_69dff3de4372ae9b6e строка
        :param keep_forward_messages: 1, чтобы сохранить прикреплённые пересланные сообщения. флаг, может принимать значения 1 или 0
        :param keep_snippets: 1, чтобы сохранить прикреплённые внешние ссылки (сниппеты). флаг, может принимать значения 1 или 0
        :param group_id: идентификатор сообщества (для сообщений сообщества с ключом доступа пользователя). положительное число
        :param dont_parse_links: 1 — не создавать сниппет ссылки из сообщения флаг, может принимать значения 1 или 0, по умолчанию
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.edit', params=params)

    def messages_editChat(self, chat_id=None, title=None):
        """
        Изменяет название беседы.

        :param chat_id: идентификатор беседы. положительное число, обязательный параметр, максимальное значение 100000000
        :param title: новое название для беседы. строка, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.editChat', params=params)

    def messages_getByConversationMessageId(self, peer_id=None, conversation_message_ids=None, extended=None,
                                            fields=None, group_id=None):
        """
        Возвращает сообщения по их идентификаторам в рамках беседы или диалога.

        :param peer_id: идентификаторы назначений, разделённые запятой.  Для пользователя:  id  пользователя.   Для групповой беседы:  2000000000 + id беседы (недоступно с ключом сообщества)   Для сообщества:  -id сообщества.   целое число, обязательный параметр
        :param conversation_message_ids: идентификаторы сообщений. Максимум 100 идентификаторов. список положительных чисел, разделенных запятыми, обязательный параметр
        :param extended: 1 — возвращать дополнительные поля. флаг, может принимать значения 1 или 0
        :param fields: дополнительные поля пользователей и сообществ, которые необходимо вернуть в ответе. список слов, разделенных через запятую
        :param group_id: идентификатор сообщества (для сообщений сообщества с ключом доступа пользователя). положительное число
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов, описывающих  сообщения, в поле items.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.getByConversationMessageId', params=params)

    def messages_getById(self, message_ids=None, preview_length=None, extended=None, fields=None, group_id=None):
        """
        Возвращает сообщения по их идентификаторам.

        :param message_ids: идентификаторы сообщений. Максимум 100 идентификаторов. список положительных чисел, разделенных запятыми, обязательный параметр
        :param preview_length: количество символов, по которому нужно обрезать сообщение. Укажите 0, если Вы не хотите обрезать сообщение. (по умолчанию сообщения не обрезаются). положительное число, по умолчанию 0
        :param extended: 1 — возвращать дополнительные поля. флаг, может принимать значения 1 или 0
        :param fields: список дополнительных полей для пользователей и сообществ. список слов, разделенных через запятую
        :param group_id: идентификатор сообщества (для сообщений сообщества с ключом доступа пользователя). положительное число
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов, описывающих  сообщения, в поле items.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.getById', params=params)

    def messages_getChat(self, chat_id=None, chat_ids=None, fields=None, name_case=None):
        """
        Возвращает информацию о беседе.

        :param chat_id: идентификатор беседы. положительное число, максимальное значение 100000000
        :param chat_ids: список идентификаторов бесед. список положительных чисел, разделенных запятыми
        :param fields: список дополнительных полей профилей, которые необходимо вернуть.  Доступные значения: nickname, screen_name, sex, bdate, city, country, timezone, photo_50, photo_100, photo_200_orig, has_mobile, contacts, education, online, counters, relation, last_seen, status, can_write_private_message, can_see_all_posts, can_post, universities список слов, разделенных через запятую
        :param name_case: падеж для склонения имени и фамилии пользователя. Возможные значения:   nom — именительный;  gen — родительный;  dat — дательный;  acc — винительный;  ins — творительный;  abl — предложный.  По умолчанию: nom. строка
        :return: После успешного выполнения возвращает объект (или список объектов) мультидиалога.  Если был задан параметр fields, поле users содержит список объектов пользователей с дополнительным полем invited_by, содержащим идентификатор пользователя, пригласившего в беседу.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.getChat', params=params)

    def messages_getChatPreview(self, peer_id=None, link=None, fields=None):
        """
        Получает данные для превью чата с приглашением по ссылке.

        :param peer_id: положительное число
        :param link: ссылка-приглашение. строка
        :param fields: список полей профилей, данные о которых нужно получить. Полный список смотрите на этой странице. список слов, разделенных через запятую
        :return: Возвращает объект, который содержит следующие поля:  preview  objectинформация о чате. Объект, который содержит поля:   admin_id (object) — идентификатор создателя чата;  members (array) — массив идентификаторов участников чата;  title (string) — название чата;  photo (object) — обложка чата. Объект, который содержит поля:   photo_50 (string) — URL копии фотографии с шириной 50 px;  photo_100 (string) — URL копии фотографии с шириной 100 px;  photo_200 (string) — URL копии фотографии с шириной 200px.   local_id (integer) — идентификатор чата для текущего пользователя. profiles (array)массив объектов пользователей; groups (array)массив объектов сообществ; emails (array)массив объектов, описывающих e-mail. Каждый из объектов содержит поля:   id (integer) — идентификатор e-mail;  address (string) — адрес e-mail.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.getChatPreview', params=params)

    def messages_getConversationMembers(self, peer_id=None, fields=None, group_id=None):
        """
        Позволяет получить список участников беседы.

        :param peer_id: идентификатор назначения.  Для пользователя:  id  пользователя.   Для групповой беседы:  2000000000 + id беседы.   Для сообщества:  -id сообщества.   целое число, обязательный параметр
        :param fields: дополнительные поля пользователей и сообществ, которые необходимо вернуть в ответе. список слов, разделенных через запятую
        :param group_id: идентификатор сообщества (для сообщений сообщества с ключом доступа пользователя). положительное число
        :return: Возвращает объект, который содержит следующие поля:  count integerчисло участников беседы. items arrayучастники беседы. Массив объектов, каждый из которых содержит поля:   member_id (integer) — идентификатор участника беседы;  invited_by (integer) — идентификатор пользователя, который пригласил участника;  join_date (integer) — дата добавления в беседу;  is_admin (boolean) — является ли пользователь администратором;  can_kick (boolean) — может ли текущий пользователь исключить участника. profiles  arrayмассив объектов пользователей. groups  arrayмассив объектов сообществ.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.getConversationMembers', params=params)

    def messages_getConversations(self, offset=None, count=None, filter=None, extended=None, start_message_id=None,
                                  fields=None, group_id=None, major_sort_id=None):
        """
        Возвращает список бесед пользователя.

        :param offset: смещение, необходимое для выборки определенного подмножества результатов. положительное число, по умолчанию 0
        :param count: максимальное число результатов, которые нужно получить. положительное число, по умолчанию 20, максимальное значение 200
        :param filter: фильтр. Возможные значения:   all — все беседы;  unread — беседы с непрочитанными сообщениями;  important — беседы, помеченные как важные (только для сообщений сообществ);  unanswered — беседы, помеченные как неотвеченные (только для сообщений сообществ).    По умолчанию: all. строка, по умолчанию all
        :param extended: 1 — возвращать дополнительные поля для пользователей и сообществ. флаг, может принимать значения 1 или 0
        :param start_message_id: идентификатор сообщения, начиная с которого нужно возвращать беседы. положительное число
        :param fields: список дополнительных полей для пользователей и сообществ. список слов, разделенных через запятую
        :param group_id: идентификатор сообщества (для сообщений сообщества с ключом доступа пользователя). положительное число
        :param major_sort_id: положительное число
        :return: Возвращает объект, который содержит следующие поля:  count integerчисло результатов. items arrayбеседы. Массив объектов, каждый из которых содержит поля:   conversation (object) — объект беседы.  last_message (object) — объект, описывающий последнее сообщение в беседе. unread_count integerчисло непрочитанных бесед. profiles  arrayмассив объектов пользователей. groups  arrayмассив объектов сообществ.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.getConversations', params=params)

    def messages_getConversationsById(self, peer_ids=None, extended=None, fields=None, group_id=None):
        """
        Позволяет получить беседу по её идентификатору.

        :param peer_ids: идентификаторы назначений, разделённые запятой.  Для пользователя:  id  пользователя.   Для групповой беседы:  2000000000 + id беседы.   Для сообщества:  -id сообщества.    Максимум – 100 идентификаторов. список целых чисел, разделенных запятыми, обязательный параметр
        :param extended: 1 — возвращать дополнительные поля. флаг, может принимать значения 1 или 0
        :param fields: дополнительные поля пользователей и сообществ, которые необходимо вернуть в ответе. список слов, разделенных через запятую
        :param group_id: идентификатор сообщества (для сообщений сообщества с ключом доступа пользователя). положительное число
        :return: Возвращает общее число результатов в поле count (integer) и массив объектов бесед в поле items.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.getConversationsById', params=params)

    def messages_getHistory(self, offset=None, count=None, user_id=None, peer_id=None, start_message_id=None, rev=None,
                            extended=None, fields=None, group_id=None):
        """
        Возвращает историю сообщений для указанного диалога.

        :param offset: смещение, необходимое для выборки определенного подмножества сообщений, должен быть >= 0, если не передан параметр start_message_id, и должен быть <= 0, если передан. целое число
        :param count: количество сообщений, которое необходимо получить (но не более 200). положительное число, по умолчанию 20, максимальное значение 200
        :param user_id: идентификатор пользователя, историю переписки с которым необходимо вернуть. строка
        :param peer_id: идентификатор назначения.  Для пользователя:  id  пользователя.   Для групповой беседы:  2000000000 + id беседы. (недоступно для вызовов от имени сообщества)   Для сообщества:  -id сообщества.   целое число, доступен начиная с версии 5.38
        :param start_message_id: если значение > 0, то это идентификатор сообщения, начиная с которого нужно вернуть историю переписки, если передано значение 0 то вернутся сообщения с самого начала переписки, если же передано значение -1, то к значению параметра offset прибавляется количество входящих непрочитанных сообщений в конце диалога (подробности см. ниже) целое число
        :param rev: 1 – возвращать сообщения в хронологическом порядке. 0 – возвращать сообщения в обратном хронологическом порядке (по умолчанию). целое число
        :param extended: Если указать в качестве этого параметра 1, то будет возвращена информация о пользователях, являющихся авторами сообщений. По умолчанию 0. флаг, может принимать значения 1 или 0
        :param fields: список дополнительных полей профилей, которые необходимо вернуть. См. подробное описание. список слов, разделенных через запятую
        :param group_id: идентификатор сообщества (для сообщений сообщества с ключом доступа пользователя). положительное число
        :return: После успешного выполнения возвращает список объектов сообщений.  В случае наличия в этом диалоге непрочитанных входящих сообщений, дополнительно (после поля count, перед полем items) в список будет добавлено поле unread с количеством непрочитанных входящих сообщений.  Поля in_read и out_read являются идентификаторами прочтения. in_read возвращает идентификатор последнего прочитанного сообщения текущим пользователем, а out_read возвращает идентификатор последнего прочитанного сообщения собеседником.  При переданном start_message_id в случае, если не было возвращено последнее сообщение в истории переписки (то есть сколько-то сообщений было пропущено), количество пропущенных сообщений (то есть реальное значение offset, которое использовалось для получения интервала истории) будет возвращено в поле skipped (вместе с count, items и иногда unread).
        """

        params = locals()
        params.pop('self')
        return self._call('messages.getHistory', params=params)

    def messages_getHistoryAttachments(self, peer_id=None, media_type=None, start_from=None, count=None,
                                       photo_sizes=None, fields=None, group_id=None, preserve_order=None,
                                       max_forwards_level=None):
        """
        Возвращает материалы диалога или беседы.

        :param peer_id: идентификатор назначения. Для групповой беседы:  2000000000 + id беседы.   Для пользователя:  id пользователя.   Для сообщества:  -id сообщества.   целое число, обязательный параметр
        :param media_type: тип материалов, который необходимо вернуть.  Доступные значения:   photo — фотографии;  video — видеозаписи;  audio — аудиозаписи;  doc — документы;  link — ссылки;  market — товары;  wall — записи;  share — ссылки, товары и записи.   Обратите внимание — существует ограничение по дате отправки вложений. Так, для получения доступны вложения типов photo,video,audio,doc , отправленные не ранее 25.03.2013, link — не ранее 20.05.13, market,wall — 01.02.2016. строка, по умолчанию photo
        :param start_from: смещение, необходимое для выборки определенного подмножества объектов. строка
        :param count: количество объектов, которое необходимо получить (но не более 200). положительное число, максимальное значение 200, по умолчанию 30
        :param photo_sizes: параметр, указывающий нужно ли возвращать ли доступные размеры фотографии в специальном формате. флаг, может принимать значения 1 или 0
        :param fields: дополнительные поля профилей пользователей и сообществ, которые необходимо вернуть в ответе. список слов, разделенных через запятую
        :param group_id: идентификатор сообщества (для сообщений сообщества с ключом доступа пользователя). положительное число
        :param preserve_order: параметр, указывающий нужно ли возвращать вложения в оригинальном порядке. флаг, может принимать значения 1 или 0
        :param max_forwards_level: максимальная глубина вложенности пересланных сообщений. положительное число, максимальное значение 45, по умолчанию 45
        :return: После успешного выполнения возвращает массив items, содержащий объекты, которые описывают вложения, а также дополнительное поле next_from, содержащее новое значение start_from. Каждый объект из массива items содержит следующие поля:  message_id  integerидентификатор сообщения, в котором было отправлено вложение. attachment  objectинформация о вложении. Объект, который содержит следующие поля:   type (string) — тип вложения. Возможные значения:   photo — фотография;  video — видеозапись;  audio — аудиозапись;  doc — документ;  link — ссылка;  market — товар;  wall — запись.   photo, video, audio, doc, link, market, wall (название поля зависит от значения поля type) — описание вложения. Содержит, соответственно, объект, описывающий фотографию, видеозапись, аудиозапись, документ, ссылку, товар или запись на стене.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.getHistoryAttachments', params=params)

    def messages_getImportantMessages(self, count=None, offset=None, start_message_id=None, preview_length=None,
                                      fields=None, extended=None, group_id=None):
        """
        Возвращает список важных сообщений пользователя.

        :param count: максимальное число результатов, которые нужно получить. положительное число, по умолчанию 20, максимальное значение 200
        :param offset: смещение, необходимое для выборки определенного подмножества результатов. положительное число
        :param start_message_id: идентификатор сообщения, начиная с которого нужно возвращать список. положительное число
        :param preview_length: положительное число
        :param fields: список дополнительных полей для пользователей и сообществ. список слов, разделенных через запятую
        :param extended: 1 — возвращать дополнительные поля для пользователей и сообществ. флаг, может принимать значения 1 или 0
        :param group_id: идентификатор сообщества (для сообщений сообщества с ключом доступа пользователя). положительное число
        :return:
        """

        params = locals()
        params.pop('self')
        return self._call('messages.getImportantMessages', params=params)

    def messages_getInviteLink(self, peer_id=None, reset=None, group_id=None):
        """
        Получает ссылку для приглашения пользователя в беседу. Только создатель беседы имеет доступ к ссылке на беседу.

        :param peer_id: идентификатор назначения.  Для групповой беседы:  2000000000 + id беседы.   положительное число, обязательный параметр
        :param reset: 1 — сгенерировать новую ссылку, сбросив предыдущую. 0 — получить предыдущую ссылку. флаг, может принимать значения 1 или 0, по умолчанию
        :param group_id: положительное число
        :return: Возвращает объект с единственным полем link (string), которое содержит ссылку для приглашения в беседу.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.getInviteLink', params=params)

    def messages_getLastActivity(self, user_id=None):
        """
        Возвращает текущий статус и дату последней активности указанного пользователя.

        :param user_id: идентификатор пользователя, информацию о последней активности которого требуется получить. целое число, обязательный параметр
        :return: Возвращает объект, содержащий следующие поля:   online — текущий статус пользователя (1 — в сети, 0 — не в сети);  time — дата последней активности пользователя в формате unixtime.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.getLastActivity', params=params)

    def messages_getLongPollHistory(self, ts=None, pts=None, preview_length=None, onlines=None, fields=None,
                                    events_limit=None, msgs_limit=None, max_msg_id=None, group_id=None, lp_version=None,
                                    last_n=None, credentials=None):
        """
        Возвращает обновления в личных сообщениях пользователя.  Подробнее о работе с Long Poll сервером Вы можете узнать на этой странице.  Для ускорения работы с личными сообщениями может быть полезно кешировать уже загруженные ранее сообщения на мобильном устройстве / ПК пользователя, чтобы не получать их повторно при каждом обращении. Этот метод помогает осуществить синхронизацию локальной копии списка сообщений с актуальной версией.

        :param ts: последнее значение параметра ts, полученное от Long Poll сервера или с помощью метода messages.getLongPollServer положительное число
        :param pts: последнее значение параметра new_pts, полученное от Long Poll сервера, используется для получения действий, которые хранятся всегда. положительное число
        :param preview_length: количество символов, по которому нужно обрезать сообщение. Укажите 0, если Вы не хотите обрезать сообщение. (по умолчанию сообщения не обрезаются). положительное число
        :param onlines: 1 — возвращать в числе прочих события 8 и 9 (пользователь стал онлайн/оффлайн). Учитывается только при использовании ts. флаг, может принимать значения 1 или 0
        :param fields: список дополнительных полей профилей, которые необходимо вернуть. См. подробное описание.  Доступные значения: sex, bdate, city, country, photo_50, photo_100, photo_200_orig, photo_200, photo_400_orig, photo_max, photo_max_orig, photo_id, online, online_mobile, domain, has_mobile, contacts, connections, site, education, universities, schools, can_post, can_see_all_posts, can_see_audio, can_write_private_message, status, last_seen, common_count, relation, relatives, counters, screen_name, maiden_name, timezone, occupation, activities, interests, music, movies, tv, books, games, about, quotes, personal, friend_status, military, career список слов, разделенных через запятую, по умолчанию photo,photo_medium_rec,sex,online,screen_name
        :param events_limit: лимит по количеству всех событий в истории. Обратите внимание, параметры events_limit и msgs_limit применяются совместно. Число результатов в ответе ограничивается первым достигнутым лимитом. положительное число, по умолчанию 1000, минимальное значение 1000
        :param msgs_limit: лимит по количеству событий с сообщениями в истории. Обратите внимание, параметры events_limit и msgs_limit применяются совместно. Число результатов в ответе ограничивается первым достигнутым лимитом. положительное число, по умолчанию 200, минимальное значение 200
        :param max_msg_id: максимальный идентификатор сообщения среди уже имеющихся в локальной копии. Необходимо учитывать как сообщения, полученные через методы API (например messages.getDialogs, messages.getHistory), так и данные, полученные из Long Poll сервера (события с кодом 4). положительное число
        :param group_id: идентификатор сообщества (для сообщений сообщества с ключом доступа пользователя). положительное число
        :param lp_version: версия Long Poll. положительное число
        :param last_n: положительное число, по умолчанию 0, максимальное значение 2000
        :param credentials: флаг, может принимать значения 1 или 0
        :return: Возвращает объект, который содержит поля history, messages, а также groups — массив объектов сообществ и profiles — массив объектов пользователей.  Поле history представляет из себя массив, аналогичный полю updates, получаемому от Long Poll сервера, за некоторыми исключениями: для событий с кодом 4 (добавление нового сообщения) отсутствуют все поля, кроме первых трёх, а также отсутствуют события с кодами 8, 9 (друг появился/пропал из сети) и 61, 62 (набор текста в диалоге/беседе).  Поле messages представляет из себя массив личных сообщений – объектов message, которые встречались среди событий с кодом 4 (добавление нового сообщения) из поля history. Каждый объект message содержит набор полей, описание которых доступно здесь. Первый элемент массива представляет собой общее количество сообщений.   Ограничения В случае, если ts слишком старый (больше суток), а max_msg_id не передан, метод может вернуть ошибку 10 (Internal server error).  Если количество событий превышает значение events_limit или количество событий с сообщениями превышает значение msgs_limit, ответ содержит дополнительное поле more со значением 1 — это означает, что нужно запросить оставшиеся данные с помощью запроса с параметром max_msg_id. Обратите внимание, что параметры events_limit и msgs_limit применяются совместно — число объектов в результате не превышает значения меньшего из этих параметров.  Ошибки с кодами 907 и 908 означают, что нужно получить новое значение pts (''ts''') и вызвать метод повторно с новыми значениями, поскольку данных для переданных значений не существует.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.getLongPollHistory', params=params)

    def messages_getLongPollServer(self, need_pts=None, group_id=None, lp_version=None):
        """
        Возвращает данные, необходимые для подключения к Long Poll серверу. Long Poll позволит Вам моментально узнавать о приходе новых сообщений и других событий.

        :param need_pts: 1 — возвращать поле pts, необходимое для работы метода messages.getLongPollHistory флаг, может принимать значения 1 или 0
        :param group_id: идентификатор сообщества (для сообщений сообщества с ключом доступа пользователя). положительное число
        :param lp_version: версия для подключения к Long Poll. Актуальная версия: 3.  Подробную информацию об изменениях в версиях Вы найдёте на этой странице. положительное число, по умолчанию 0, доступен начиная с версии 5.65
        :return: Возвращает объект, который содержит поля key, server, ts.  Используя эти данные, Вы можете подключиться к серверу быстрых сообщений для мгновенного получения приходящих сообщений и других событий.  Формат взаимодействия с LongPoll сервером.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.getLongPollServer', params=params)

    def messages_isMessagesFromGroupAllowed(self, group_id=None, user_id=None):
        """
        Возвращает информацию о том, разрешена ли отправка сообщений от сообщества пользователю.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :param user_id: идентификатор пользователя. положительное число, обязательный параметр
        :return: Возвращает объект с единственным полем is_allowed (integer, [0,1]). Если отправка сообщений разрешена, поле содержит 1, иначе — 0.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.isMessagesFromGroupAllowed', params=params)

    def messages_joinChatByInviteLink(self, link=None):
        """
        Позволяет присоединиться к чату по ссылке-приглашению.

        :param link: ссылка-приглашение. строка, обязательный параметр
        :return: Возвращает идентификатор чата в поле chat_id.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.joinChatByInviteLink', params=params)

    def messages_markAsAnsweredConversation(self, peer_id=None, answered=None, group_id=None):
        """
        Помечает беседу как отвеченную либо снимает отметку.

        :param peer_id: идентификатор беседы целое число, обязательный параметр
        :param answered: 1 - беседа отмечена отвеченной, 0 - неотвеченной флаг, может принимать значения 1 или 0, по умолчанию 1
        :param group_id: идентификатор сообщества (для сообщений сообщества с ключом доступа пользователя). положительное число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.markAsAnsweredConversation', params=params)

    def messages_markAsImportant(self, message_ids=None, important=None):
        """
        Помечает сообщения как важные либо снимает отметку.

        :param message_ids: список идентификаторов сообщений, которые необходимо пометить. список положительных чисел, разделенных запятыми
        :param important: 1, если сообщения необходимо пометить, как важные;  0, если необходимо снять пометку. положительное число
        :return: Возвращает массив идентификаторов успешно помеченных сообщений.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.markAsImportant', params=params)

    def messages_markAsImportantConversation(self, peer_id=None, important=None, group_id=None):
        """
        Помечает беседу как важную либо снимает отметку.

        :param peer_id: идентификатор беседы целое число, обязательный параметр
        :param important: 1, если сообщения необходимо пометить, как важные;  0, если необходимо снять пометку. флаг, может принимать значения 1 или 0, по умолчанию 1
        :param group_id: идентификатор сообщества (для сообщений сообщества с ключом доступа пользователя). положительное число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.markAsImportantConversation', params=params)

    def messages_markAsRead(self, message_ids=None, peer_id=None, start_message_id=None, group_id=None):
        """
        Помечает сообщения как прочитанные.

        :param message_ids: идентификаторы сообщений. список положительных чисел, разделенных запятыми
        :param peer_id: идентификатор назначения.  Для пользователя:  id  пользователя.   Для групповой беседы:  2000000000 + id беседы.   Для сообщества:  -id сообщества.   строка
        :param start_message_id: при передаче этого параметра будут помечены как прочитанные все сообщения, начиная с данного. положительное число
        :param group_id: идентификатор сообщества (для сообщений сообщества с ключом доступа пользователя). положительное число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.markAsRead', params=params)

    def messages_pin(self, peer_id=None, message_id=None):
        """
        Закрепляет сообщение.

        :param peer_id: идентификатор назначения.  Для пользователя:  id  пользователя.   Для групповой беседы:  2000000000 + id беседы.   Для сообщества:  -id сообщества.   целое число, обязательный параметр
        :param message_id: идентификатор сообщения, которое нужно закрепить. положительное число
        :return: Возвращает объект закрепленного сообщения.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.pin', params=params)

    def messages_removeChatUser(self, chat_id=None, user_id=None, member_id=None):
        """
        Исключает из мультидиалога пользователя, если текущий пользователь или сообщество является администратором беседы либо текущий пользователь пригласил исключаемого пользователя. Также может быть использован для выхода текущего пользователя из беседы, в которой он состоит. Чтобы пользователю вернуться в беседу, достаточно отправить сообщение в неё (если есть свободные места).

        :param chat_id: идентификатор беседы. положительное число, обязательный параметр, максимальное значение 100000000
        :param user_id: идентификатор пользователя, которого необходимо исключить из беседы. Может быть меньше нуля в случае, если пользователь приглашён по email. целое число
        :param member_id: идентификатор участника, которого необходимо исключить из беседы. Для сообществ — идентификатор сообщества со знаком «минус». целое число, доступен начиная с версии 5.81
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.removeChatUser', params=params)

    def messages_restore(self, message_id=None, group_id=None):
        """
        Восстанавливает удаленное сообщение.

        :param message_id: идентификатор сообщения, которое нужно восстановить. положительное число, обязательный параметр
        :param group_id: идентификатор сообщества (для сообщений сообщества с ключом доступа пользователя). положительное число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.restore', params=params)

    def messages_search(self, q=None, peer_id=None, date=None, preview_length=None, offset=None, count=None,
                        extended=None, fields=None, group_id=None):
        """
        Возвращает список найденных личных сообщений текущего пользователя по введенной строке поиска.

        :param q: подстрока, по которой будет производиться поиск. строка
        :param peer_id: фильтр по идентификатору назначения для поиска по отдельному диалогу.  Для пользователя:  id  пользователя.   Для групповой беседы:  2000000000 + id беседы.   Для сообщества:  -id сообщества.   целое число
        :param date: дата в формате DDMMYYYY — если параметр задан, в ответе будут только сообщения, отправленные до указанной даты. положительное число
        :param preview_length: количество символов, по которому нужно обрезать сообщение. Укажите 0, если Вы не хотите обрезать сообщение. (по умолчанию сообщения не обрезаются). положительное число, по умолчанию 0
        :param offset: смещение, необходимое для выборки определенного подмножества сообщений из списка найденных. положительное число, по умолчанию 0
        :param count: количество сообщений, которое необходимо получить. положительное число, по умолчанию 20, максимальное значение 100
        :param extended: 1 — возвращать дополнительные поля для пользователей и сообществ. В ответе будет содержаться массив объектов бесед. флаг, может принимать значения 1 или 0
        :param fields: список дополнительных полей для пользователей и сообществ. список слов, разделенных через запятую
        :param group_id: идентификатор сообщества (для сообщений сообщества с ключом доступа пользователя). положительное число
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов, описывающих личные сообщения, в поле items.  Обратите внимание, даже при использовании параметра offset максимальное число доступных результатов — 10000.  Если extended = 1, в поле items возвращается массив объектов бесед. Дополнительно возвращаются массивы profiles и groups, содержащие объекты пользователей и сообществ.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.search', params=params)

    def messages_searchConversations(self, q=None, count=None, extended=None, fields=None, group_id=None):
        """
        Позволяет искать диалоги.

        :param q: поисковой запрос. строка
        :param count: максимальное число результатов для получения. положительное число, по умолчанию 20
        :param extended: 1 — возвращать дополнительные поля. флаг, может принимать значения 1 или 0
        :param fields: дополнительные поля пользователей и сообществ, которые необходимо вернуть в ответе. список слов, разделенных через запятую
        :param group_id: идентификатор сообщества (для сообщений сообщества с ключом доступа пользователя). положительное число
        :return: Возвращает общее число результатов в поле count (integer) и массив объектов диалогов в поле items.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.searchConversations', params=params)

    def messages_send(self, user_id=None, random_id=None, peer_id=None, domain=None, chat_id=None, user_ids=None,
                      message=None, lat=None, long=None, attachment=None, reply_to=None, forward_messages=None,
                      sticker_id=None, group_id=None, keyboard=None, payload=None, dont_parse_links=None,
                      disable_mentions=None, intent=None, guid=None):
        """
        Отправляет сообщение.

        :param user_id: идентификатор пользователя, которому отправляется сообщение. целое число
        :param random_id: уникальный (в привязке к API_ID и ID отправителя) идентификатор, предназначенный для предотвращения повторной отправки одинакового сообщения. Сохраняется вместе с сообщением и доступен в истории сообщений.  Заданный random_id используется для проверки уникальности за всю историю сообщений, поэтому используйте большой диапазон (до int64). целое число, доступен начиная с версии 5.45
        :param peer_id: идентификатор назначения.  Для пользователя:  id  пользователя.   Для групповой беседы:  2000000000 + id беседы.   Для сообщества:  -id сообщества.   целое число, доступен начиная с версии 5.38
        :param domain: короткий адрес пользователя (например, illarionov). строка
        :param chat_id: идентификатор беседы, к которой будет относиться сообщение. положительное число, максимальное значение 100000000
        :param user_ids: идентификаторы получателей сообщения (при необходимости отправить сообщение сразу нескольким пользователям). Доступно только для ключа доступа сообщества. Максимальное количество идентификаторов: 100. список целых чисел, разделенных запятыми
        :param message: текст личного сообщения. Обязательный параметр, если не задан параметр attachment. строка
        :param lat: географическая широта (от -90 до 90). дробное число
        :param long: географическая долгота (от -180 до 180). дробное число
        :param attachment: медиавложения к личному сообщению, перечисленные через запятую. Каждое прикрепление представлено в формате: <type><owner_id>_<media_id>  <type> — тип медиавложения:   photo — фотография;  video — видеозапись;  audio — аудиозапись;  doc — документ;  wall — запись на стене;  market — товар.  poll — опрос.     <owner_id> — идентификатор владельца медиавложения (обратите внимание, если объект находится в сообществе, этот параметр должен быть отрицательным). <media_id> — идентификатор медиавложения.  Например: photo100172_166443618  Параметр является обязательным, если не задан параметр message.  В случае, если прикрепляется объект, принадлежащий другому пользователю следует добавлять к вложению его access_key в формате <type><owner_id>_<media_id>_<access_key>, Например: video85635407_165186811_69dff3de4372ae9b6e строка
        :param reply_to: идентификатор сообщения, на которое требуется ответить. целое число, доступен начиная с версии 5.92
        :param forward_messages: идентификаторы пересылаемых сообщений, перечисленные через запятую. Перечисленные сообщения отправителя будут отображаться в теле письма у получателя. Не более 100 значений на верхнем уровне, максимальный уровень вложенности: 45, максимальное количество пересылаемых сообщений 500  Например: 123,431,544 список целых чисел, разделенных запятыми
        :param sticker_id: идентификатор стикера. положительное число
        :param group_id: идентификатор сообщества (для сообщений сообщества с ключом доступа пользователя). положительное число
        :param keyboard: объект, описывающий клавиатуру бота.
        :param payload: Полезная нагрузка максимальная длина 1000
        :param dont_parse_links: 1 — не создавать сниппет ссылки из сообщения флаг, может принимать значения 1 или 0, по умолчанию
        :param disable_mentions: 1 - отключить уведомление об упоминании в сообщении флаг, может принимать значения 1 или 0, по умолчанию
        :param intent: строка, описывающая интенты строка
        :param guid: уникальный идентификатор, предназначенный для предотвращения повторной отправки одинакового сообщения. целое число, устаревший параметр, доступен только для версий меньше 5.46
        :return: После успешного выполнения возвращает идентификатор отправленного сообщения.  Если передан параметр user_ids, возвращает массив объектов, каждый из которых содержит поля:   peer_id — идентификатор назначения;  message_id — идентификатор сообщения;  error — сообщение об ошибке, если сообщение не было доставлено получателю.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.send', params=params)

    def messages_setActivity(self, user_id=None, type=None, peer_id=None, group_id=None):
        """
        Изменяет статус набора текста пользователем в диалоге.

        :param user_id: идентификатор пользователя. строка
        :param type: typing — пользователь начал набирать текст,  audiomessage — пользователь записывает голосовое сообщение строка
        :param peer_id: идентификатор назначения.  Для групповой беседы:  2000000000 + id беседы.   Для сообщества:  -id сообщества.   целое число, доступен начиная с версии 5.38
        :param group_id: идентификатор сообщества (для сообщений сообщества с ключом доступа пользователя). положительное число
        :return: После успешного выполнения возвращает 1.  Текст «N набирает сообщение...» отображается в течение 10 секунд после вызова метода, либо до момента отправки сообщения.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.setActivity', params=params)

    def messages_setChatPhoto(self, file=None):
        """
        Позволяет установить фотографию мультидиалога, загруженную с помощью метода photos.getChatUploadServer.

        :param file: Содержимое поля response из ответа специального upload сервера, полученного в результате загрузки изображения на адрес, полученный методом photos.getChatUploadServer. строка, обязательный параметр
        :return: После успешного выполнения возвращает объект, содержащий следующие поля:  message_id — идентификатор отправленного системного сообщения;  chat — объект мультидиалога.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.setChatPhoto', params=params)

    def messages_unpin(self, peer_id=None, group_id=None):
        """
        Открепляет сообщение.

        :param peer_id: идентификатор назначения.  Для пользователя:  id  пользователя.   Для групповой беседы:  2000000000 + id беседы.   Для сообщества:  -id сообщества.   целое число, обязательный параметр
        :param group_id: идентификатор сообщества (для сообщений сообщества с ключом доступа пользователя). положительное число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('messages.unpin', params=params)

    def newsfeed_addBan(self, user_ids=None, group_ids=None):
        """
        Запрещает показывать новости от заданных пользователей и групп в ленте новостей текущего пользователя.

        :param user_ids: перечисленные через запятую идентификаторы друзей пользователя, новости от которых необходимо скрыть из ленты новостей текущего пользователя. список целых чисел, разделенных запятыми
        :param group_ids: перечисленные через запятую идентификаторы групп пользователя, новости от которых необходимо скрыть из ленты новостей текущего пользователя. список целых чисел, разделенных запятыми
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('newsfeed.addBan', params=params)

    def newsfeed_deleteBan(self, user_ids=None, group_ids=None):
        """
        Разрешает показывать новости от заданных пользователей и групп в ленте новостей текущего пользователя.

        :param user_ids: идентификаторы пользователей, новости от которых необходимо вернуть в ленту. список положительных чисел, разделенных запятыми
        :param group_ids: идентификаторы сообществ, новости от которых необходимо вернуть в ленту. список положительных чисел, разделенных запятыми
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('newsfeed.deleteBan', params=params)

    def newsfeed_deleteList(self, list_id=None):
        """
        Метод позволяет удалить пользовательский список новостей

        :param list_id: числовой идентификатор списка положительное число, обязательный параметр
        :return:
        """

        params = locals()
        params.pop('self')
        return self._call('newsfeed.deleteList', params=params)

    def newsfeed_get(self, filters=None, return_banned=None, start_time=None, end_time=None, max_photos=None,
                     source_ids=None, start_from=None, count=None, fields=None, section=None, from_=None, offset=None):
        """
        Возвращает данные, необходимые для показа списка новостей для текущего пользователя.

        :param filters: перечисленные через запятую названия списков новостей, которые необходимо получить. В данный момент поддерживаются следующие списки новостей:   post — новые записи со стен;  photo — новые фотографии;  photo_tag — новые отметки на фотографиях;  wall_photo — новые фотографии на стенах;  friend — новые друзья;  note — новые заметки;  audio — записи сообществ и друзей, содержащие аудиозаписи, а также новые аудиозаписи, добавленные ими;  video — новые видеозаписи.   Если параметр не задан, то будут получены все возможные списки новостей. список слов, разделенных через запятую
        :param return_banned: 1 - включить в выдачу также скрытых из новостей пользователей. 0 - не возвращать скрытых пользователей. флаг, может принимать значения 1 или 0
        :param start_time: время в формате unixtime, начиная с которого следует получить новости для текущего пользователя. положительное число
        :param end_time: время в формате unixtime, до которого следует получить новости для текущего пользователя. Если параметр не задан, то он считается равным текущему времени. положительное число
        :param max_photos: Максимальное количество фотографий, информацию о которых необходимо вернуть. По умолчанию: 5, максимальное значение: 100. положительное число
        :param source_ids: перечисленные через запятую иcточники новостей, новости от которых необходимо получить.  Идентификаторы пользователей можно указывать в форматах <uid> или u<uid> где <uid> — идентификатор друга пользователя.  Идентификаторы сообществ можно указывать в форматах -<gid> или g<gid> где <gid> — идентификатор сообщества.  Помимо этого параметр может принимать строковые значения:   friends - список друзей пользователя  groups - список групп, на которые подписан текущий пользователь  pages - список публичных страниц, на который подписан тeкущий пользователь  following - список пользователей, на которых подписан текущий пользователь  list{идентификатор списка новостей} - список новостей. Вы можете найти все списки новостей пользователя используя метод newsfeed.getLists   Если параметр не задан, то считается, что он включает список всех друзей и групп пользователя, за исключением скрытых источников, которые можно получить методом newsfeed.getBanned.  Максимальное число символов — 5000. строка
        :param start_from: Идентификатор, необходимый для получения следующей страницы результатов. Значение, необходимое для передачи в этом параметре, возвращается в поле ответа next_from. строка, доступен начиная с версии 5.13
        :param count: указывает, какое максимальное число новостей следует возвращать, но не более 100. По умолчанию 50. положительное число
        :param fields: список дополнительных полей для профилей и  групп, которые необходимо вернуть. См. описание полей объекта user и описание полей объекта group. список слов, разделенных через запятую
        :param section: строка
        :param from: значение, полученное в поле new_from при последней загрузке новостей. Помогает избавляться от дубликатов при реализации автоподгрузки. строка, устаревший параметр, доступен только для версий меньше 5.13
        :param offset: указывает, начиная с какого элемента в данном промежутке времени необходимо получить новости. По умолчанию 0. Для автоподгрузки Вы можете использовать возвращаемый данным методом параметр new_offset. целое число, устаревший параметр, доступен только для версий меньше 5.13
        :return: После успешного выполнения возвращает объект, содержащий следующие поля:   items — массив новостей для текущего пользователя;  profiles — информация о пользователях, которые находятся в списке новостей;  groups — информация о группах, которые находятся в списке новостей;  new_offset — содержит offset, который необходимо передать, для того, чтобы получить следующую часть новостей;  next_from — содержит start_from, который необходимо передать, для того, чтобы получить следующую часть новостей. Позволяет избавиться от дубликатов, которые могут возникнуть при появлении новых новостей между вызовами этого метода.    Описание поля items Поле items содержит массив объектов, каждый из которых соответствует одной новости в ленте. Каждый из объектов содержит поля:   type — тип списка новости, соответствующий одному из значений параметра filters;  source_id — идентификатор источника новости (положительный — новость пользователя, отрицательный — новость группы);  date — время публикации новости в формате unixtime;  post_id — находится в записях со стен и содержит идентификатор записи на стене владельца;  post_type — находится в записях со стен, содержит тип новости (post или copy);  final_post — передается в случае, если этот пост сделан при удалении;  copy_owner_id — находится в записях со стен, если сообщение является копией сообщения с чужой стены, и содержит идентификатор владельца стены, у которого было скопировано сообщение;  copy_post_id — находится в записях со стен, если сообщение является копией сообщения с чужой стены, и содержит идентификатор скопированного сообщения на стене его владельца;  copy_history — массив, содержащий историю репостов для записи. Возвращается только в том случае, если запись является репостом. Каждый из объектов массива, в свою очередь, является объектом-записью стандартного формата.  copy_post_date — находится в записях со стен, если сообщение является копией сообщения с чужой стены, и содержит дату скопированного сообщения;  text — находится в записях со стен и содержит текст записи;  can_edit — содержит 1, если текущий пользователь может редактировать запись;  can_delete — возвращается, если пользователь может удалить новость, всегда содержит 1;  comments — находится в записях со стен и содержит информацию о комментариях к записи, содержит поля:   count — количество комментариев,  can_post — информация о том, может ли текущий пользователь комментировать запись  (1 — может, 0 — не может);   likes — находится в записях со стен и содержит информацию о числе людей, которым понравилась данная запись, содержит поля:   count — число пользователей, которым понравилась запись,  user_likes — наличие отметки «Мне нравится» от текущего пользователя  (1 — есть, 0 — нет),  can_like — информация о том, может ли текущий пользователь поставить отметку «Мне нравится»  (1 — может, 0 — не может),  can_publish — информация о том, может ли текущий пользователь сделать репост записи  (1 — может, 0 — не может);   reposts — находится в записях со стен и содержит информацию о числе людей, которые скопировали данную запись на свою страницу, содержит поля:   count — число пользователей, сделавших репост;  user_reposted — наличие репоста от текущего пользователя (0 — нет, 1 — есть);   attachments — находится в записях со стен и содержит массив объектов, которые прикреплены к текущей новости (фотография, ссылка и т.п.). Более подробная информация представлена на странице Описание поля attachments.  geo — находится в записях со стен, в которых имеется информация о местоположении, содержит поля:   place_id — идентификатор места;  title — название места;  type — тип места;  country_id — идентификатор страны;  city_id — идентификатор города;  address — строка с указанием адреса места в городе;  showmap — данный параметр указывается, если местоположение является прикреплённой картой.   photos, photo_tags, notes, friends — находятся в объектах соответствующих типов (кроме записей со стен) и содержат информацию о количестве объектов и до 5 последних объектов, связанных с данной новостью.  Каждый из элементов массива в полях photos и photo_tags содержит поля:    id — идентификатор фотографии  owner_id — идентификатор владельца фотографии  album_id — идентификатор альбома  src — адрес изображения для предпросмотра  src_big — адрес полноразмерного изображения  Каждый из элементов массива в поле notes содержит поля:   id — идентификатор заметки  owner_id — идентификатор владельца заметки  title — заголовок заметки  comments — количество комментариев к заметке  Каждый из элементов массива в поле friends содержит поля:   uid— идентификатор пользователя     Описание поля profiles Поле profiles содержит массив объектов пользователей с дополнительными полями:   photo — адрес фотографии пользователя размером 50x50px;  photo_medium_rec — адрес фотографии пользователя размером 100x100px;  screen_name — короткий адрес страницы пользователя (например, andrew или id6492).    Описание поля groups Поле groups содержит массив объектов сообществ, которые присутствуют в новостях.   Примечание Вы можете получить полный список фотографий, полученных с типом photo или photo_tag методом photos.get, указав параметры feed (дата события в поле date) и feed_type (тип события, photo или photo_tag) и uid — идентификатор пользователя, с которым произошло событие (поле source_id).
        """

        params = locals()
        params.pop('self')
        return self._call('newsfeed.get', params=params)

    def newsfeed_getBanned(self, extended=None, fields=None, name_case=None):
        """
        Возвращает список пользователей и групп, которые текущий пользователь скрыл из ленты новостей.

        :param extended: если этот параметр равен 1, возвращается дополнительная информация о пользователях и группах флаг, может принимать значения 1 или 0
        :param fields: список дополнительных полей профилей, которые необходимо вернуть. См. подробное описание.  Доступные значения: sex, bdate, city, country, photo_50, photo_100, photo_200_orig, photo_200, photo_400_orig, photo_max, photo_max_orig, online, online_mobile, domain, has_mobile, contacts, connections, site, education, universities, schools, can_post, can_see_all_posts, can_see_audio, can_write_private_message, status, last_seen, common_count, relation, relatives, counters, screen_name, maiden_name, timezone, occupation,activities, interests, music, movies, tv, books, games, about, quotes список слов, разделенных через запятую
        :param name_case: падеж для склонения имени и фамилии пользователя. Возможные значения: именительный – nom, родительный – gen, дательный – dat, винительный – acc, творительный – ins, предложный – abl. По умолчанию nom. строка
        :return: В случае успеха возвращает объект, в котором содержатся поля groups и members или profiles, в зависимости от параметра extended.  Если extended=0:   В поле groups содержится массив идентификаторов сообществ, которые пользователь скрыл из ленты новостей. В поле members содержится массив идентификаторов пользователей, которых пользователь скрыл из ленты новостей.   Если extended=1:   В поле groups содержится массив объектов с информацией о сообществах. В поле profiles содержится массив объектов с информацией о пользователях.
        """

        params = locals()
        params.pop('self')
        return self._call('newsfeed.getBanned', params=params)

    def newsfeed_getComments(self, count=None, filters=None, reposts=None, start_time=None, end_time=None,
                             last_comments_count=None, start_from=None, fields=None, from_ =None, last_comments=None):
        """
        Возвращает данные, необходимые для показа раздела комментариев в новостях пользователя.

        :param count: указывает, какое максимальное число новостей следует возвращать, но не более 100. По умолчанию 30. Для автоподгрузки Вы можете использовать возвращаемый данным методом параметр new_offset. положительное число, по умолчанию 30, максимальное значение 100
        :param filters: перечисленные через запятую типы объектов, изменения комментариев к которым нужно вернуть. В данный момент поддерживаются следующие списки новостей:    post — новые комментарии к записям со стен;  photo — новые комментарии к фотографиям;  video — новые комментарии к видеозаписям;  topic — новые сообщения в обсуждениях;  market — новые комментарии к товарам;  note — новые комментарии к заметкам. Если параметр не задан, то будут получены все возможные списки новостей. список слов, разделенных через запятую
        :param reposts: Идентификатор объекта, комментарии к репостам которого необходимо вернуть, например wall1_45486. Если указан данный параметр, параметр filters указывать необязательно. строка
        :param start_time: время в формате unixtime, начиная с которого следует получить новости для текущего пользователя. Если параметр не задан, то он считается равным значению времени, которое было сутки назад. положительное число
        :param end_time: время в формате unixtime, до которого следует получить новости для текущего пользователя. Если параметр не задан, то он считается равным текущему времени. положительное число
        :param last_comments_count: количество комментариев к записям, которые нужно получить. положительное число, доступен начиная с версии 5.23, по умолчанию 0, максимальное значение 10
        :param start_from: Идентификатор, необходимый для получения следующей страницы результатов. Значение, необходимое для передачи в этом параметре, возвращается в поле ответа next_from. строка, доступен начиная с версии 5.13
        :param fields: список дополнительных полей профилей, которые необходимо вернуть. См. подробное описание.  Доступные значения: sex, bdate, city, country, photo_50, photo_100, photo_200_orig, photo_200, photo_400_orig, photo_max, photo_max_orig, online, online_mobile, domain, has_mobile, contacts, connections, site, education, universities, schools, can_post, can_see_all_posts, can_see_audio, can_write_private_message, status, last_seen, common_count, relation, relatives, counters, screen_name, maiden_name, timezone, occupation,activities, interests, music, movies, tv, books, games, about, quotes список слов, разделенных через запятую
        :param from: значение, полученное в поле new_from при последней загрузке новостей. Помогает избавляться от дубликатов при реализации автоподгрузки. строка, устаревший параметр, доступен только для версий меньше 5.13
        :param last_comments: 1 - возвращать последние комментарии к записям. 0 - не возвращать последние комментарии. флаг, может принимать значения 1 или 0, устаревший параметр, доступен только для версий меньше 5.23
        :return: В случае успеха возвращает объект, содержащий поля:   items — содержит массив новостей для текущего пользователя  profiles — содержит информацию о пользователях, которые находятся в списке новостей  groups — содержит информацию о группах, которые находятся в списке новостей  next_from — содержит текстовое поле, которое необходимо передать при подгрузке следующего блока обсуждений    Описание поля items Поле items содержит массив объектов, каждый из которых соответствует одной новости в ленте. Каждый из объектов содержит поля:   type — тип списка новости, соответствующий одному из значений параметра filters  source_id — идентификатор источника новости (положительный — новость пользователя, отрицательный — новость сообщества)  date — время публикации новости в формате unixtime  post_id — находится в записях со стен и содержит идентификатор записи на стене владельца  copy_owner_id — находится в записях со стен, если сообщение является копией сообщения с чужой стены, и содержит идентификатор владельца стены, у которого было скопировано сообщение  copy_post_id — находится в записях со стен, если сообщение является копией сообщения с чужой стены, и содержит идентификатор скопированного сообщения на стене его владельца  text — находится в записях со стен и содержит текст записи  comments — находится в записях со стен и содержит информацию о комментариях к записи, содержит поля:   count — количество комментариев,  can_post — информация о том, может ли текущий пользователь комментировать запись (1 — может, 0 — не может);   likes — находится в записях со стен и содержит информацию о числе людей, которым понравилась данная запись, содержит поля:   count — число пользователей, которым понравилась запись,  user_likes — наличие отметки «Мне нравится» от текущего пользователя (1 — есть, 0 — нет),  can_like — информация о том, может ли текущий пользователь поставить отметку «Мне нравится» (1 — может, 0 — не может),  can_publish — информация о том, может ли текущий пользователь сделать репост записи (1 — может, 0 — не может);   attachment — находится в записях со стен и содержит объект, который присоединен к текущей новости (фотография, ссылка и т.п.). Более подробная информация представлена на странице Описание поля attachment.  geo — находится в записях со стен, в которых имеется информация о местоположении, содержит поля:   place_id — идентификатор места;  title — название места;  type — тип места;  country_id — идентификатор страны;  city_id — идентификатор города;  address — строка с указанием адреса места в городе;  showmap — данный параметр указывается, если местоположение является прикреплённой картой.   photos, photo_tags, notes, friends — находятся в объектах соответствующих типов (кроме записей со стен) и содержат информацию о количестве объектов и до 5 последних объектов, связанных с данной новостью.  Каждый из элементов массива в полях photos и photo_tags содержит поля:   id — идентификатор фотографии;  owner_id — идентификатор владельца фотографии;  album_id — идентификатор альбома;  photo_* — адреса различных размеров изображений для фотографии (75, 130, 604, 807, 1280, 2560 px)  Каждый из элементов массива в поле notes содержит поля:   id — идентификатор заметки;  owner_id — идентификатор владельца заметки;  title — заголовок заметки;  ncom — количество комментариев к заметке.  Каждый из элементов массива в поле friends содержит единственное поле   uid — идентификатор пользователя.     Описание поля profiles Поле profiles содержит массив объектов с информацией о данных пользователей, которые присутствуют в новостях.  Каждый из объектов содержит поля:   id — идентификатор пользователя;  first_name — имя пользователя;  last_name — фамилия пользователя;  photo — адрес фотографии пользователя размером 50x50 px.    Описание поля groups Поле groups содержит массив объектов с информацией о сообществах, которые присутствуют в новостях. Каждый из объектов содержит поля:   id — идентификатор сообщества;  name — название сообщества;  photo — адрес логотипа группы шириной 50px.
        """

        params = locals()
        params.pop('self')
        return self._call('newsfeed.getComments', params=params)

    def newsfeed_getLists(self, list_ids=None, extended=None):
        """
        Возвращает пользовательские списки новостей.

        :param list_ids: идентификаторы списков. список положительных чисел, разделенных запятыми
        :param extended: 1 — вернуть дополнительную информацию о списке (значения source_ids и no_reposts). флаг, может принимать значения 1 или 0, по умолчанию 0
        :return: Метод возвращает список объектов с полями:   id – идентификатор списка;  title – название списка, заданное пользователем.   Если был передан параметр extended=1, дополнительно возвращаются поля:   no_reposts — отключены ли копии постов;  source_ids — идентификаторы пользователей и сообществ, включенных в список.   Новости по отдельному списку могут быть получены методом newsfeed.get с использованием параметра source_ids.
        """

        params = locals()
        params.pop('self')
        return self._call('newsfeed.getLists', params=params)

    def newsfeed_getMentions(self, owner_id=None, start_time=None, end_time=None, offset=None, count=None):
        """
        Возвращает список записей пользователей на своих стенах, в которых упоминается указанный пользователь.

        :param owner_id: идентификатор пользователя или сообщества. целое число, по умолчанию идентификатор текущего пользователя
        :param start_time: время в формате unixtime начиная с которого следует получать упоминания о пользователе.  Если параметр не задан, то будут возвращены все упоминания о пользователе, если не задан параметр end_time, в противном случае упоминания с учетом параметра end_time. положительное число
        :param end_time: время, в формате unixtime, до которого следует получать упоминания о пользователе.  Если параметр не задан, то будут возвращены все упоминания о пользователе, если не задан параметр start_time, в противном случае упоминания с учетом параметра start_time. положительное число
        :param offset: смещение, необходимое для выборки определенного подмножества новостей. По умолчанию 0. положительное число
        :param count: количество возвращаемых записей. Если параметр не задан, то считается, что он равен 20. Максимальное значение параметра 50. положительное число, по умолчанию 20, максимальное значение 50
        :return: В случае успеха возвращает количество записей и массив соответствующих им объектов, каждый из которых содержит следующие поля:   id — идентификатор записи на стене пользователя;  from_id — идентификатор пользователя, написавшего запись;  date — время публикаии записи в формате unixtime;  text — текст записи;  likes — содержит информацию о числе людей, которым понравилась данная запись, и понравилась ли она текущему пользователю;  comments — содержит информацию о количестве комментариев к записи и возможности текущего пользователя оставлять комментарии к ней;  attachment — содержит объект, который присоединен к текущей новости ( фотография, ссылка и т.п.). Более подробная информация представлена на странице Описание поля attachment;  geo — находится в записях со стен, в которых имеется информация о местоположении, содержит поля:   place_id — идентификатор места;  title — название места;  type — тип места;  country_id — идентификатор страны;  city_id — идентификатор города;  address — строка с указанием адреса места в городе;  showmap — данный параметр указывается, если местоположение является прикреплённой картой.   copy_owner_id — если запись является копией записи с чужой стены, то в поле содержится идентификатор владельца стены у которого была скопирована запись;  copy_post_id — если запись является копией записи с чужой стены, то в поле содержится идентфикатор скопированной записи на стене ее владельца.
        """

        params = locals()
        params.pop('self')
        return self._call('newsfeed.getMentions', params=params)

    def newsfeed_getRecommended(self, start_time=None, end_time=None, max_photos=None, start_from=None, count=None,
                                fields=None, from_=None, offset=None):
        """
        Получает список новостей, рекомендованных пользователю.

        :param start_time: время в формате unixtime, начиная с которого следует получить новости для текущего пользователя. Если параметр не задан, то он считается равным значению времени, которое было сутки назад. положительное число
        :param end_time: время в формате unixtime, до которого следует получить новости для текущего пользователя. Если параметр не задан, то он считается равным текущему времени. положительное число
        :param max_photos: Максимальное количество фотографий, информацию о которых необходимо вернуть. По умолчанию 5. положительное число
        :param start_from: Идентификатор, необходимый для получения следующей страницы результатов. Значение, необходимое для передачи в этом параметре, возвращается в поле ответа next_from. строка, доступен начиная с версии 5.13
        :param count: указывает, какое максимальное число новостей следует возвращать, но не более 100. По умолчанию 50. положительное число
        :param fields: список дополнительных полей профилей, которые необходимо вернуть. См. подробное описание.  Доступные значения: sex, bdate, city, country, photo_50, photo_100, photo_200_orig, photo_200, photo_400_orig, photo_max, photo_max_orig, online, online_mobile, domain, has_mobile, contacts, connections, site, education, universities, schools, can_post, can_see_all_posts, can_see_audio, can_write_private_message, status, last_seen, common_count, relation, relatives, counters, screen_name, maiden_name, timezone, occupation,activities, interests, music, movies, tv, books, games, about, quotes список слов, разделенных через запятую
        :param from: значение, полученное в поле new_from при последней загрузке новостей. Помогает избавляться от дубликатов при реализации автоподгрузки. строка, устаревший параметр, доступен только для версий меньше 5.13
        :param offset: указывает, начиная с какого элемента в данном промежутке времени необходимо получить новости. По умолчанию 0. Для автоподгрузки Вы можете использовать возвращаемый данным методом параметр new_offset. целое число, устаревший параметр, доступен только для версий меньше 5.13
        :return: После успешного выполнения возвращает объект, содержащий следующие поля:   items — массив новостей для текущего пользователя;  profiles — информация о пользователях, которые находятся в списке новостей;  groups — информация о группах, которые находятся в списке новостей;  new_offset — содержит offset, который необходимо передать, для того, чтобы получить следующую часть новостей;  new_from — содержит from, который необходимо передать, для того, чтобы получить следующую часть новостей. Позволяет избавиться от дубликатов, которые могут возникнуть при появлении новых новостей между вызовами этого метода.    Описание поля items Поле items содержит массив объектов, каждый из которых соответствует одной новости в ленте. Каждый из объектов содержит поля:   type — тип списка новости, соответствующий одному из значений параметра filters;  source_id — идентификатор источника новости (положительный — новость пользователя, отрицательный — новость группы);  date — время публикации новости в формате unixtime;  post_id — находится в записях со стен и содержит идентификатор записи на стене владельца;  copy_owner_id — находится в записях со стен, если сообщение является копией сообщения с чужой стены, и содержит идентификатор владельца стены, у которого было скопировано сообщение;  copy_post_id — находится в записях со стен, если сообщение является копией сообщения с чужой стены, и содержит идентификатор скопированного сообщения на стене его владельца;  copy_post_date — находится в записях со стен, если сообщение является копией сообщения с чужой стены, и содержит дату скопированного сообщения;  text — находится в записях со стен и содержит текст записи;  comments — находится в записях со стен и содержит информацию о комментариях к записи, содержит поля:   count — количество комментариев,  can_post — информация о том, может ли текущий пользователь комментировать запись  (1 — может, 0 — не может);   likes — находится в записях со стен и содержит информацию о числе людей, которым понравилась данная запись, содержит поля:   count — число пользователей, которым понравилась запись,  user_likes — наличие отметки «Мне нравится» от текущего пользователя  (1 — есть, 0 — нет),  can_like — информация о том, может ли текущий пользователь поставить отметку «Мне нравится»  (1 — может, 0 — не может),  can_publish — информация о том, может ли текущий пользователь сделать репост записи  (1 — может, 0 — не может);   reposts — находится в записях со стен и содержит информацию о числе людей, которые скопировали данную запись на свою страницу, содержит поля:   count — число пользователей, сделавших репост;  user_reposted — наличие репоста от текущего пользователя (0 — нет, 1 — есть);   attachments — находится в записях со стен и содержит массив объектов, которые прикреплены к текущей новости (фотография, ссылка и т.п.). Более подробная информация представлена на странице Описание поля attachments.  geo — находится в записях со стен, в которых имеется информация о местоположении, содержит поля:   place_id — идентификатор места;  title — название места;  type — тип места;  country_id — идентификатор страны;  city_id — идентификатор города;  address — строка с указанием адреса места в городе;  showmap — данный параметр указывается, если местоположение является прикреплённой картой.      photos, photo_tags, notes, friends — находятся в объектах соответствующих типов (кроме записей со стен) и содержат информацию о количестве объектов и до 5 последних объектов, связанных с данной новостью.   Каждый из элементов массива в полях photos и photo_tags содержит поля:     id — идентификатор фотографии  owner_id — идентификатор владельца фотографии  album_id — идентификатор альбома  photo_75 - адрес изображения размером 75 пикселей  photo_130 - адрес изображения размером 130 пикселей  photo_604 - адрес изображения размером 604 пикселей  photo_807 - адрес изображения размером 807 пикселей  photo_1280 - адрес изображения размером 1280 пикселей    Каждый из элементов массива в поле notes содержит поля:     id — идентификатор заметки  owner_id — идентификатор владельца заметки  title — заголовок заметки  comments — количество комментариев к заметке    Каждый из элементов массива в поле friends содержит поля:     uid— идентификатор пользователя     Описание поля profiles Поле profiles содержит массив объектов user с информацией о данных пользователей, которые присутствуют в новостях. Каждый из объектов содержит следующие поля:   id — идентификатор пользователя;  first_name — имя пользователя;  last_name — фамилия пользователя;  photo — адрес фотографии пользователя размером 50x50px;  photo_medium_rec — адрес фотографии пользователя размером 100x100px;  screen_name — короткий адрес страницы пользователя (например, andrew или id6492).    Описание поля groups Поле groups содержит массив объектов group с информацией о сообществах, которые присутствуют в новостях. Каждый из объектов содержит следующие поля:   id — идентификатор сообщества;  name — название сообщества;  is_closed — флаг, определяющий является ли группа закрытой;  photo — адрес логотипа группы шириной 50px;  photo_medium — адрес логотипа группы шириной 100px;  photo_big — адрес логотипа группы шириной 200px;  screen_name — короткий адрес страницы сообщества (например, apiclub или club1).    Примечание Вы можете получить полный список фотографий, полученных с типом photo или photo_tag методом photos.get, указав параметры feed (дата события в поле date) и feed_type (тип события, photo или photo_tag) и uid — идентификатор пользователя, с которым произошло событие (поле source_id).
        """

        params = locals()
        params.pop('self')
        return self._call('newsfeed.getRecommended', params=params)

    def newsfeed_getSuggestedSources(self, offset=None, count=None, shuffle=None, fields=None):
        """
        Возвращает сообщества и пользователей, на которые текущему пользователю рекомендуется подписаться.

        :param offset: отступ, необходимый для выборки определенного подмножества сообществ или пользователей. положительное число
        :param count: количество сообществ или пользователей, которое необходимо вернуть. положительное число, максимальное значение 1000, по умолчанию 20
        :param shuffle: перемешивать ли возвращаемый список. флаг, может принимать значения 1 или 0
        :param fields: список дополнительных полей, которые необходимо вернуть. См. возможные поля для пользователей и сообществ. список слов, разделенных через запятую
        :return: Список объектов пользователей и групп.
        """

        params = locals()
        params.pop('self')
        return self._call('newsfeed.getSuggestedSources', params=params)

    def newsfeed_ignoreItem(self, type=None, owner_id=None, item_id=None):
        """
        Позволяет скрыть объект из ленты новостей.

        :param type: Тип объекта. Возможные значения:   wall — запись на стене;  tag — отметка на фотографии;  profilephoto — фотография профиля;  video — видеозапись;  photo — фотография;  audio — аудиозапись.  строка, обязательный параметр
        :param owner_id: Идентификатор владельца объекта (пользователь или сообщество). Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param item_id: Идентификатор объекта. положительное число, обязательный параметр
        :return: В результате успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('newsfeed.ignoreItem', params=params)

    def newsfeed_saveList(self, list_id=None, title=None, source_ids=None, no_reposts=None):
        """
        Метод позволяет создавать или редактировать пользовательские списки для просмотра новостей.

        :param list_id: числовой идентификатор списка (если не передан, будет назначен автоматически). положительное число
        :param title: название списка. строка, обязательный параметр
        :param source_ids: идентификаторы пользователей и сообществ, которые должны быть включены в список. Идентификаторы сообществ нужно указывать со знаком «минус». список целых чисел, разделенных запятыми
        :param no_reposts: нужно ли отображать копии постов в списке (1 — не нужно). флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает идентификатор созданного или отредактированного списка.
        """

        params = locals()
        params.pop('self')
        return self._call('newsfeed.saveList', params=params)

    def newsfeed_search(self, q=None, extended=None, count=None, latitude=None, longitude=None, start_time=None,
                        end_time=None, start_from=None, fields=None, start_id=None, offset=None):
        """
        Возвращает результаты поиска по статусам. Новости возвращаются в порядке от более новых к более старым. Обратите внимание, для этого метода действуют специальные ограничения на число запросов. Подробнее.

        :param q: поисковой запрос, например, "New Year". строка
        :param extended: 1, если необходимо получить информацию о пользователе или сообществе, разместившем запись. По умолчанию: 0. флаг, может принимать значения 1 или 0
        :param count: указывает, какое максимальное число записей следует возвращать. Обратите внимание — даже при использовании параметра offset для получения информации доступны только первые 1000 результатов.   положительное число, по умолчанию 30, максимальное значение 200
        :param latitude: географическая широта точки, в радиусе от которой необходимо производить поиск, заданная в градусах (от -90 до 90). дробное число
        :param longitude: географическая долгота точки, в радиусе от которой необходимо производить поиск, заданная в градусах (от -180 до 180). дробное число
        :param start_time: время в формате unixtime, начиная с которого следует получить новости для текущего пользователя. Если параметр не задан, то он считается равным значению времени, которое было сутки назад. положительное число
        :param end_time: время в формате unixtime, до которого следует получить новости для текущего пользователя. Если параметр не задан, то он считается равным текущему времени. положительное число
        :param start_from: Идентификатор, необходимый для получения следующей страницы результатов. Значение, необходимое для передачи в этом параметре, возвращается в поле ответа next_from. строка, доступен начиная с версии 5.13
        :param fields: список дополнительных полей для профилей и  групп, которые необходимо вернуть. См. описание полей объекта user и описание полей объекта group.  Обратите внимание, этот параметр учитывается только при extended=1. список слов, разделенных через запятую
        :param start_id: строковый идентификатор последней полученной записи. (Возвращается в результатах запроса (new_from) для того, чтобы исключить из выборки нового запроса уже полученные записи). строка, устаревший параметр, доступен только для версий меньше 5.13
        :param offset: смещение, необходимое для выборки определенного подмножества результатов поиска. положительное число, устаревший параметр, доступен только для версий меньше 5.13
        :return: В случае успеха возвращает общее количество записей и массив объектов, каждый из которых содержит поля:   id — локальный идентификатор записи (для конкретного владельца);  owner_id — идентификатор владельца стены, на которой размещена запись. Если стена принадлежит сообществу, то данный параметр равен -gid (идентификатор сообщества со знаком минус);  from_id —  идентификатор автора записи;  date — время публикации записи в формате unixtime;  text — текст записи;  comments — содержит информацию о количестве комментариев к записи. Для Desktop-приложений дополнительно содержит следующую информацию:   может ли текущий пользователь добавить комментарии к записи;   likes — содержит информацию о числе людей, которым понравилась данная запись. Для Desktop-приложений дополнительно содержит следующую информацию:   понравилась ли запись текущему пользователю;  может ли текущий пользователь добавить запись в спиcок Мне нравится;  может ли текущий пользователь опубликовать у себя запись.   attachments — находится в записях со стен и содержит массив объектов, которые прикреплены к текущей новости (фотография, ссылка и т.п.). Более подробная информация представлена на странице описания поля attachments.  geo — находится в записях со стен, в которых имеется информация о местоположении, содержит поля:   type — тип отметки;  coordinates — географические координаты места;  place — объект с полями:   pid — идентификатор места;  title — название места;  latitude — географическая широта;  longitude — географическая долгота;  created — дата добавления;  icon — значок места;  type — тип места;  address — строка с указанием адреса места в городе;  country — идентификатор страны;  city — идентификатор города;   showmap — данный параметр указывается, если местоположение является прикреплённой картой.    В случае, если передан параметр extended, дополнительно будут переданы поля profiles и groups, содержащие информацию о пользователях и сообществах.  Каждый из объектов в массиве profiles содержит следующие поля:   id — идентификатор пользователя;  first_name — имя пользователя;  last_name — фамилия пользователя;  photo — адрес фотографии пользователя размером 50x50px;  photo_medium_rec — адрес фотографии пользователя размером 100x100px;  screen_name — короткий адрес страницы пользователя (например, andrew или id6492).   Каждый из объектов в массиве groups содержит следующие поля:   id — идентификатор сообщества;  name — название сообщества;  is_closed — флаг, определяющий является ли группа закрытой;  photo — адрес логотипа группы шириной 50px;  photo_medium — адрес логотипа группы шириной 100px;  photo_big — адрес логотипа группы шириной 200px;  screen_name — короткий адрес страницы сообщества (например, apiclub или club1).
        """

        params = locals()
        params.pop('self')
        return self._call('newsfeed.search', params=params)

    def newsfeed_unignoreItem(self, type=None, owner_id=None, item_id=None):
        """
        Позволяет вернуть ранее скрытый объект в ленту новостей.

        :param type: Тип объекта. Возможные значения:   wall — запись на стене;  tag — отметка на фотографии;  profilephoto — фотография профиля;  video — видеозапись;  photo — фотография;  audio — аудиозапись.  строка, обязательный параметр
        :param owner_id: Идентификатор владельца объекта (пользователь или сообщество). Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param item_id: Идентификатор объекта. положительное число, обязательный параметр
        :return: В результате успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('newsfeed.unignoreItem', params=params)

    def newsfeed_unsubscribe(self, type=None, owner_id=None, item_id=None):
        """
        Отписывает текущего пользователя от комментариев к заданному объекту.

        :param type: тип объекта, от комментариев к которому необходимо отписаться.  Возможные типы: post — запись на стене пользователя или группы; photo — фотография; video — видеозапись; topic — обсуждение; note — заметка; market — товар. строка, обязательный параметр
        :param owner_id: идентификатор владельца объекта. целое число, по умолчанию идентификатор текущего пользователя
        :param item_id: идентификатор объекта. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('newsfeed.unsubscribe', params=params)

    def notes_add(self, title=None, text=None, privacy_view=None, privacy_comment=None, privacy=None,
                  comment_privacy=None):
        """
        Создает новую заметку у текущего пользователя.

        :param title: заголовок заметки. строка, обязательный параметр
        :param text: текст заметки. строка, обязательный параметр
        :param privacy_view: настройки приватности просмотра заметки в специальном формате. список слов, разделенных через запятую, по умолчанию all, доступен начиная с версии 5.30
        :param privacy_comment: настройки приватности комментирования заметки в специальном формате. список слов, разделенных через запятую, по умолчанию all, доступен начиная с версии 5.30
        :param privacy: уровень доступа к заметке. Возможные значения:  0 — все пользователи, 1 — только друзья, 2 — друзья и друзья друзей, 3 — только пользователь. целое число, устаревший параметр, доступен только для версий меньше 5.30
        :param comment_privacy: уровень доступа к комментированию заметки. Возможные значения:  0 — все пользователи, 1 — только друзья, 2 — друзья и друзья друзей, 3 — только пользователь. целое число, устаревший параметр, доступен только для версий меньше 5.30
        :return: После успешного выполнения возвращает идентификатор созданной заметки (nid).
        """

        params = locals()
        params.pop('self')
        return self._call('notes.add', params=params)

    def notes_createComment(self, note_id=None, owner_id=None, reply_to=None, message=None, guid=None):
        """
        Добавляет новый комментарий к заметке.

        :param note_id: идентификатор заметки. положительное число, обязательный параметр
        :param owner_id: идентификатор владельца заметки. положительное число, по умолчанию идентификатор текущего пользователя
        :param reply_to: идентификатор пользователя, ответом на комментарий которого является добавляемый комментарий (не передаётся, если комментарий не является ответом). положительное число
        :param message: текст комментария. строка, обязательный параметр
        :param guid: уникальный идентификатор, предназначенный для предотвращения повторной отправки одинакового комментария. строка
        :return: После успешного выполнения возвращает идентификатор созданного комментария (cid).
        """

        params = locals()
        params.pop('self')
        return self._call('notes.createComment', params=params)

    def notes_delete(self, note_id=None):
        """
        Удаляет заметку текущего пользователя.

        :param note_id: идентификатор заметки. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('notes.delete', params=params)

    def notes_deleteComment(self, comment_id=None, owner_id=None):
        """
        Удаляет комментарий к заметке.

        :param comment_id: идентификатор комментария. положительное число, обязательный параметр
        :param owner_id: идентификатор владельца заметки. положительное число, по умолчанию идентификатор текущего пользователя
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('notes.deleteComment', params=params)

    def notes_edit(self, note_id=None, title=None, text=None, privacy_view=None, privacy_comment=None, privacy=None,
                   comment_privacy=None):
        """
        Редактирует заметку текущего пользователя.

        :param note_id: идентификатор заметки. положительное число, обязательный параметр
        :param title: заголовок заметки. строка, обязательный параметр
        :param text: текст заметки. строка, обязательный параметр
        :param privacy_view: настройки приватности просмотра заметки в специальном формате. список слов, разделенных через запятую, по умолчанию all, доступен начиная с версии 5.30
        :param privacy_comment: настройки приватности комментирования заметки в специальном формате. список слов, разделенных через запятую, по умолчанию all, доступен начиная с версии 5.30
        :param privacy: уровень доступа к заметке. Возможные значения:  0 — все пользователи, 1 — только друзья, 2 — друзья и друзья друзей, 3 — только пользователь. целое число, устаревший параметр, доступен только для версий меньше 5.30
        :param comment_privacy: уровень доступа к комментированию заметки. Возможные значения:  0 — все пользователи, 1 — только друзья, 2 — друзья и друзья друзей, 3 — только пользователь. целое число, устаревший параметр, доступен только для версий меньше 5.30
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('notes.edit', params=params)

    def notes_editComment(self, comment_id=None, owner_id=None, message=None):
        """
        Редактирует указанный комментарий у заметки.

        :param comment_id: идентификатор комментария. положительное число, обязательный параметр
        :param owner_id: идентификатор владельца заметки. положительное число, по умолчанию идентификатор текущего пользователя
        :param message: новый текст комментария. строка, минимальная длина 2
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('notes.editComment', params=params)

    def notes_get(self, note_ids=None, user_id=None, offset=None, count=None, sort=None):
        """
        Возвращает список заметок, созданных пользователем.

        :param note_ids: идентификаторы заметок, информацию о которых необходимо получить. список положительных чисел, разделенных запятыми
        :param user_id: идентификатор пользователя, информацию о заметках которого требуется получить. положительное число, по умолчанию идентификатор текущего пользователя
        :param offset: смещение, необходимое для выборки определенного подмножества заметок. положительное число, по умолчанию 0
        :param count: количество заметок, информацию о которых необходимо получить. положительное число, по умолчанию 20, максимальное значение 100
        :param sort: сортировка результатов (0 — по дате создания в порядке убывания, 1 - по дате создания в порядке возрастания). положительное число, по умолчанию 0
        :return: После успешного выполнения возвращает список объектов заметок.
        """

        params = locals()
        params.pop('self')
        return self._call('notes.get', params=params)

    def notes_getById(self, note_id=None, owner_id=None, need_wiki=None):
        """
        Возвращает заметку по её id.

        :param note_id: идентификатор заметки. положительное число, обязательный параметр
        :param owner_id: идентификатор владельца заметки. положительное число, по умолчанию идентификатор текущего пользователя
        :param need_wiki: определяет, требуется ли в ответе wiki-представление заметки (работает, только если запрашиваются заметки текущего пользователя). флаг, может принимать значения 1 или 0, по умолчанию 0
        :return: После успешного выполнения возвращает список объектов заметок с дополнительными полями:   privacy — уровень доступа к заметке (0 — все пользователи, 1 — только друзья, 2 — друзья и друзья друзей, 3 — только пользователь);  comment_privacy — уровень доступа к комментированию заметки (0 — все пользователи, 1 — только друзья, 2 — друзья и друзья друзей, 3 — только пользователь);  can_comment — может ли текущий пользователь комментировать заметку (1 — может, 0 — не может).
        """

        params = locals()
        params.pop('self')
        return self._call('notes.getById', params=params)

    def notes_getComments(self, note_id=None, owner_id=None, sort=None, offset=None, count=None):
        """
        Возвращает список комментариев к заметке.

        :param note_id: идентификатор заметки. положительное число, обязательный параметр
        :param owner_id: идентификатор владельца заметки. положительное число, по умолчанию идентификатор текущего пользователя
        :param sort: сортировка результатов (0 — по дате добавления в порядке возрастания, 1 — по дате добавления в порядке убывания). положительное число, по умолчанию 0
        :param offset: смещение, необходимое для выборки определенного подмножества комментариев. положительное число, по умолчанию 0
        :param count: количество комментариев, которое необходимо получить. положительное число, по умолчанию 20, максимальное значение 100
        :return: Возвращает массив объектов comment, каждый из которых содержит следующие поля:   id — идентификатор комментария;  uid — идентификатор автора комментария;  nid — идентификатор заметки;  oid — идентификатор владельца заметки;  date — дата добавления комментария в формате unixtime;  message — текст комментария;  reply_to — идентификатор пользователя, в ответ на комментарий которого был оставлен текущий комментарий (если доступно).
        """

        params = locals()
        params.pop('self')
        return self._call('notes.getComments', params=params)

    def notes_restoreComment(self, comment_id=None, owner_id=None):
        """
        Восстанавливает удалённый комментарий.

        :param comment_id: идентификатор удаленного комментария. положительное число, обязательный параметр
        :param owner_id: идентификатор владельца заметки. положительное число, по умолчанию идентификатор текущего пользователя
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('notes.restoreComment', params=params)

    def notifications_get(self, count=None, start_from=None, filters=None, start_time=None, end_time=None, from_=None,
                          offset=None):
        """
        Возвращает список оповещений об ответах других пользователей на записи текущего пользователя.

        :param count: указывает, какое максимальное число оповещений следует возвращать. положительное число, по умолчанию 30, максимальное значение 100, минимальное значение 1
        :param start_from: строковый идентификатор оповещения, полученного последним в предыдущем вызове (см. описание поля next_from в результате). строка, доступен начиная с версии 5.27
        :param filters: перечисленные через запятую типы оповещений, которые необходимо получить. Возможные значения:   wall — записи на стене пользователя;  mentions — упоминания в записях на стене, в комментариях или в обсуждениях;  comments — комментарии к записям на стене, фотографиям и видеозаписям;  likes — отметки «Мне нравится»;  reposts — скопированные у текущего пользователя записи на стене, фотографии и видеозаписи;  followers — новые подписчики;  friends — принятые заявки в друзья.  Если параметр не задан, то будут получены все возможные типы оповещений. список слов, разделенных через запятую
        :param start_time: время в формате Unixtime, начиная с которого следует получить оповещения для текущего пользователя. Если параметр не задан, то он считается равным значению времени, которое было сутки назад. целое число
        :param end_time: время в формате Unixtime, до которого следует получить оповещения для текущего пользователя. Если параметр не задан, то он считается равным текущему времени. целое число
        :param from: строковый идентификатор последнего полученного предыдущим вызовом метода оповещения (см. описание поля new_from в результате). строка, устаревший параметр, доступен только для версий меньше 5.27
        :param offset: смещение, необходимое для получения определенного подмножества оповещений. положительное число, устаревший параметр, доступен только для версий меньше 5.27
        :return: После успешного выполнения возвращает объект, содержащий поля:  items  arrayмассив оповещений для текущего пользователя. profiles  arrayинформация о пользователях, которые находятся в списке оповещений. groups  arrayинформация о сообществах, которые находятся в списке оповещений. last_viewed  integerвремя последнего просмотра пользователем раздела оповещений в формате Unixtime.   Описание поля items Поле items содержит массив объектов, каждый из которых соответствует одному оповещению. Каждый из объектов содержит поля: type stringтип оповещения. Подробнее см. ниже. date  integerвремя появления ответа в формате Unixtime. parent  object (отсутствует, если type= follow, mention или wall)информация о материале, к которому появился ответ (запись на стене, комментарий, фотографию, видеозапись или обсуждение). Подробнее см. в описании поля type ниже. feedback  [object,array]объект (или массив объектов, если type равно follow, like_* или copy_*), описывающий поступивший ответ. Оповещения о новых подписчиках, пометках «Мне нравится» и скопированных записях могут быть сгруппированы в виде массива. reply  objectобъект, описывающий комментарий текущего пользователя, отправленный в ответ на данное оповещение. Отсутствует, если пользователь ещё не давал ответа.    Описание поля type Поле type описывает тип оповещения. От данного типа зависит наличие и содержание полей parent и feedback. В данный момент поддерживаются следующие типы оповещений:  Значение type Тип поля parentТип поля feedbackОписание follow-user[]У пользователя появился один или несколько новых подписчиков. friend_accepted-user[]Заявка в друзья, отправленная пользователем, была принята. mention-postБыла создана запись на чужой стене, содержащая упоминание пользователя. mention_commentspostcommentБыл оставлен комментарий, содержащий упоминание пользователя. wall-postБыла добавлена запись на стене пользователя. wall_publish-postБыла опубликована новость, предложенная пользователем в публичной странице. comment_postpostcommentБыл добавлен новый комментарий к записи, созданной пользователем. comment_photophotocommentБыл добавлен новый комментарий к фотографии пользователя. comment_videovideocommentБыл добавлен новый комментарий к видеозаписи пользователя. reply_commentcommentcommentБыл добавлен новый ответ на комментарий пользователя. reply_comment_photocommentcommentБыл добавлен новый ответ на комментарий пользователя к фотографии. reply_comment_videocommentcommentБыл добавлен новый ответ на комментарий пользователя к видеозаписи. reply_comment_marketcommentcommentБыл добавлен новый ответ на комментарий пользователя к товару. reply_topictopiccommentБыл добавлен новый ответ пользователю в обсуждении. like_postpostuser[]У записи пользователя появилась одна или несколько новых отметок «Мне нравится». like_commentcommentuser[]У комментария пользователя появилась одна или несколько новых отметок «Мне нравится». like_photophotouser[]У фотографии пользователя появилась одна или несколько новых отметок «Мне нравится». like_videovideouser[]У видеозаписи пользователя появилась одна или несколько новых отметок «Мне нравится». like_comment_photocommentuser[]У комментария пользователя к фотографии появилась одна или несколько новых отметок «Мне нравится». like_comment_videocommentuser[]У комментария пользователя к видеозаписи появилась одна или несколько новых отметок «Мне нравится». like_comment_topiccommentuser[]У комментария пользователя в обсуждении появилась одна или несколько новых отметок «Мне нравится». copy_postpostcopy[]Один или несколько пользователей скопировали запись пользователя copy_photophotocopy[]Один или несколько пользователей скопировали фотографию пользователя. copy_videovideocopy[]Один или несколько пользователей скопировали видеозапись пользователя. mention_comment_photophotocommentПод фотографией был оставлен комментарий, содержащий упоминание пользователя. mention_comment_videovideocommentПод видео был оставлен комментарий, содержащий упоминание пользователя.    Описание поля parent (post) Содержит информацию о записи на стене:   id — идентификатор записи;  to_id — идентификатор владельца записи;  from_id — идентификатор пользователя, создавшего запись;  date — время публикации записи в формате unixtime;  text — текст записи;  attachments — содержит массив объектов, которые присоединены к текущей записи (фотографии, ссылки и т.п.). Более подробная информация представлена на странице Описание поля attachments.  geo — находится в записях со стен, в которых имеется информация о местоположении, содержит поля:   place_id — идентификатор места;  title — название места;  type — тип места;  country_id — идентификатор страны;  city_id — идентификатор города;  address — строка с указанием адреса места в городе;  showmap — данный параметр указывается, если местоположение является прикреплённой картой.   copy_owner_id — если запись является копией записи с чужой стены, то в поле содержится идентификатор владельца стены у которого была скопирована запись;  copy_post_id — если запись является копией записи с чужой стены, то в поле содержится идентификатор скопированной записи на стене ее владельца    Описание поля parent (comment) Содержит информацию о комментарии:   id — идентификатор комментария;  owner_id — идентификатор автора комментария;  date — время публикации комментария в формате unixtime;  text — текст комментария;  post — запись, к которой оставлен комментарий (для типов reply_comment и like_comment);  photo — фотография, к которой оставлен комментарий (для типов reply_comment_photo и like_comment_photo);  video — видеозапись, к которой оставлен комментарий (для типов reply_comment_video и like_comment_video);  topic — обсуждение, в котором оставлен комментарий (для типа like_comment_topic).    Описание поля parent (photo) Содержит информацию о фотографии:   id — идентификатор фотографии;  owner_id — идентификатор владельца фотографии;  aid — идентификатор альбома фотографии;  src — ссылка на изображение в разрешении 130 пикселей по большей стороне;  src_big — ccылка на изображение в разрешении 604 пикселя по большей стороне;  src_small — ссылка на изображение в разрешении 75 пикселей по большей стороне;  text — текст описания фотографии;  created — дата загрузки фотографии в формате unixtime.    Описание поля parent (video) Содержит информацию о видеозаписи:   id — идентификатор видеозаписи;  owner_id — идентификатор владельца видеозаписи;  title — название видеозаписи;  description — описание видеозаписи;  duration — продолжительность видеозаписи в секундах;  link — ссылка на видео;  image — ссылка на изображение в разрешении 320 пикселей в ширину;  image_medium — ссылка на изображение в разрешении 160 пикселей в ширину;  date — дата добавления видеозаписи в формате unixtime;  views — количество просмотров;  player — ссылка на встраиваемый i_frame-видеоплеер.    Описание поля parent (topic) Содержит информацию о теме в обсуждениях сообщества:   id — идентификатор темы;  owner_id — идентификатор сообщества, содержащего тему в обсуждениях (со знаком "минус");  title — заголовок темы;  created — дата создания темы в формате unixtime;  created_by — идентификатор пользователя, создавшего тему;  updated — дата последнего сообщения в формате unixtime;  updated_by — идентификатор пользователя, оставившего последнее сообщение;  is_closed — 1, если тема является закрытой (в ней нельзя оставлять сообщения);  is_fixed — 1, если тема является прилепленной (находится в начале списка тем);  comments — число сообщений в теме.    Описание поля feedback Cодержит информацию об оповещении:   id (отсутствует, если type равно follow или like_*) — идентификатор записи-ответа;  to_id — идентификатор владельца стены, на которой размещена запись;  from_id — идентификатор автора ответа;  text (отсутствует, если type равно follow, like_* или copy_*) — текст ответа;  likes — находится в записях со стен и содержит информацию о числе людей, которым понравилась данная запись, содержит поля:   count — число пользователей, которым понравилась запись,  user_likes — наличие отметки «Мне нравится» от текущего пользователя  (1 — есть, 0 — нет),  can_like — информация о том, может ли текущий пользователь поставить отметку «Мне нравится»  (1 — может, 0 — не может),  can_publish — информация о том, может ли текущий пользователь сделать репост записи  (1 — может, 0 — не может);    В случае, если ответ является записью на стене (тип post в таблице выше), в объекте feedback также содержатся следующие поля:   attachments — содержит массив объектов, которые присоединены к текущей записи (фотографии, ссылки и т.п.). Более подробная информация представлена на странице Описание поля attachments.  geo — находится в записях со стен, в которых имеется информация о местоположении, содержит поля:   place_id — идентификатор места;  title — название места;  type — тип места;  country_id — идентификатор страны;  city_id — идентификатор города;  address — строка с указанием адреса места в городе;  showmap — данный параметр указывается, если местоположение является прикреплённой картой.     Описание поля reply Cодержит информацию об ответе пользователя на оповещение:   id — идентификатор комментария;  date — время публикации комментария в формате unixtime;  text — текст комментария.    Описание поля profiles Поле profiles содержит массив объектов user с информацией о данных пользователей, которые присутствуют в новостях. Каждый из объектов содержит следующие поля:   uid — идентификатор пользователя;  first_name — имя пользователя;  last_name — фамилия пользователя;  photo — адрес фотографии пользователя размером 50x50px;  photo_medium_rec — адрес фотографии пользователя размером 100x100px;  screen_name — короткий адрес страницы пользователя (например, andrew или id6492).
        """

        params = locals()
        params.pop('self')
        return self._call('notifications.get', params=params)

    def notifications_markAsViewed(self):
        """
        Сбрасывает счетчик непросмотренных оповещений об ответах других пользователей на записи текущего пользователя.

        :return:    В ходе выполнения могут произойти общие ошибки. Их описание находится на отдельной странице.
        """

        params = locals()
        params.pop('self')
        return self._call('notifications.markAsViewed', params=params)

    def notifications_sendMessage(self, user_ids=None, message=None, fragment=None, group_id=None):
        """
        Отправляет уведомление пользователю приложения VK Apps. Обратите внимание,  вы можете отправить пользователю одно уведомление в календарные сутки.  Отправка уведомлений администраторам приложения не ограничена для удобства тестирования.

        :param user_ids: список идентификаторов пользователей, которым нужно отправить уведомление (максимум 100 идентификаторов). список положительных чисел, разделенных запятыми, обязательный параметр
        :param message: текст уведомления. строка, обязательный параметр
        :param fragment: содержимое хэша (часть URL в ссылке на приложение вида https://vk.com/app123456#fragment). строка, максимальная длина 2047
        :param group_id: положительное число
        :return: После успешного выполнения возвращает массив объектов, каждый из которых содержит поля:   user_id (integer) —   идентификатор пользователя;  status  (boolean)  —  true, если уведомление отправлено успешно. Иначе false.  error (object) — в случае, если статус отправки имеет значение false, дополнительно вернётся объект ошибки, содержащий код ошибки в поле code (integer) и описание description (string).   Возможные значения code:   1 —  уведомления приложения отключены;  2 — отправлено слишком много уведомлений за последний час;  3 — отправлено слишком много уведомлений за последние сутки;  4 —  приложение не установлено.
        """

        params = locals()
        params.pop('self')
        return self._call('notifications.sendMessage', params=params)

    def pages_clearCache(self, url=None):
        """
        Позволяет очистить кеш отдельных внешних страниц, которые могут быть прикреплены к записям ВКонтакте. После очистки кеша при последующем прикреплении ссылки к записи, данные о странице будут обновлены. Внешние страницы – страницы которые прикрепляются к записям вместе с ссылкой и доступные по кнопке "Предпросмотр".

        :param url: Адрес страницы, закешированную версию которой необходимо очистить строка, обязательный параметр
        :return: При удачной очистке кеша – метод возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('pages.clearCache', params=params)

    def pages_get(self, owner_id=None, page_id=None, global_=None, site_preview=None, title=None, need_source=None,
                  need_html=None):
        """
        Возвращает информацию о вики-странице.

        :param owner_id: идентификатор владельца вики-страницы. целое число
        :param page_id: идентификатор вики-страницы. целое число
        :param global: 1 — требуется получить информацию о глобальной вики-странице. флаг, может принимать значения 1 или 0
        :param site_preview: 1 — получаемая wiki страница является предпросмотром для прикрепленной ссылки. флаг, может принимать значения 1 или 0
        :param title: название страницы. строка
        :param need_source: 1 —  требуется вернуть содержимое страницы в вики-формате. флаг, может принимать значения 1 или 0, доступен начиная с версии 5.20
        :param need_html: 1 —  требуется вернуть html-представление страницы. флаг, может принимать значения 1 или 0
        :return: Возвращает информацию о вики-странице в виде объекта page.  Если был задан параметр need_source равный 1, дополнительно будет возвращено поле source.  Если был задан параметр need_html равный 1, дополнительно будет возвращено поле html.
        """

        params = locals()
        params.pop('self')
        return self._call('pages.get', params=params)

    def pages_getHistory(self, page_id=None, group_id=None, user_id=None):
        """
        Возвращает список всех старых версий вики-страницы.

        :param page_id: идентификатор вики-страницы. целое число, обязательный параметр
        :param group_id: идентификатор сообщества, которому принадлежит вики-страница. целое число
        :param user_id: идентификатор пользователя, создавшего вики-страницу. целое число
        :return: Возвращает массив объектов page_version, имеющих следующую структуру:   id — идентификатор версии страницы;  length  длина версии страницы в байтах;  edited — дата редактирования страницы;  editor_id — идентификатор редактора;  editor_name — имя редактора.
        """

        params = locals()
        params.pop('self')
        return self._call('pages.getHistory', params=params)

    def pages_getTitles(self, group_id=None):
        """
        Возвращает список вики-страниц в группе.

        :param group_id: идентификатор сообщества, которому принадлежит вики-страница. целое число
        :return: Возвращает массив объектов вики-страниц.
        """

        params = locals()
        params.pop('self')
        return self._call('pages.getTitles', params=params)

    def pages_getVersion(self, version_id=None, group_id=None, user_id=None, need_html=None):
        """
        Возвращает текст одной из старых версий страницы.

        :param version_id: идентификатор версии. целое число, обязательный параметр
        :param group_id: идентификатор сообщества, которому принадлежит вики-страница. целое число
        :param user_id: идентификатор пользователя, который создал страницу. целое число
        :param need_html: определяет, требуется ли в ответе html-представление вики-страницы. флаг, может принимать значения 1 или 0
        :return: Возвращает объект вики-страницы.
        """

        params = locals()
        params.pop('self')
        return self._call('pages.getVersion', params=params)

    def pages_parseWiki(self, text=None, group_id=None):
        """
        Возвращает html-представление вики-разметки.

        :param text: текст в вики-формате. строка, обязательный параметр
        :param group_id: идентификатор группы, в контексте которой интерпретируется данная страница. положительное число
        :return: В случае успеха возвращает экранированный html, соответствующий вики-разметке.
        """

        params = locals()
        params.pop('self')
        return self._call('pages.parseWiki', params=params)

    def pages_save(self, text=None, page_id=None, group_id=None, user_id=None, title=None):
        """
        Сохраняет текст вики-страницы.

        :param text: новый текст страницы в вики-формате. строка
        :param page_id: идентификатор вики-страницы. Вместо page_id может быть передан параметр title. целое число
        :param group_id: идентификатор сообщества, которому принадлежит вики-страница. целое число
        :param user_id: идентификатор пользователя, создавшего вики-страницу. целое число
        :param title: название вики-страницы. строка
        :return: В случае успеха возвращает id созданной страницы.
        """

        params = locals()
        params.pop('self')
        return self._call('pages.save', params=params)

    def pages_saveAccess(self, page_id=None, group_id=None, user_id=None, view=None, edit=None):
        """
        Сохраняет новые настройки доступа на чтение и редактирование вики-страницы.

        :param page_id: идентификатор вики-страницы. целое число, обязательный параметр
        :param group_id: идентификатор сообщества, которому принадлежит вики-страница. целое число
        :param user_id: идентификатор пользователя, создавшего вики-страницу. целое число
        :param view: значение настройки доступа на чтение. Возможные значения:   2 — просматривать страницу могут все;  1 — только участники сообщества;  0 — только руководители сообщества.  целое число
        :param edit: значение настройки доступа на редактирование. Возможные значения:   2 — редактировать страницу могут все;  1 — только участники сообщества;  0 — только руководители сообщества.  целое число
        :return: В случае успеха возвращает id страницы, доступ к которой был отредактирован.
        """

        params = locals()
        params.pop('self')
        return self._call('pages.saveAccess', params=params)

    def photos_confirmTag(self, owner_id=None, photo_id=None, tag_id=None):
        """
        Подтверждает отметку на фотографии.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param photo_id: идентификатор фотографии. обязательный параметр
        :param tag_id: идентификатор отметки на фотографии. целое число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.confirmTag', params=params)

    def photos_copy(self, owner_id=None, photo_id=None, access_key=None):
        """
        Позволяет скопировать фотографию в альбом "Сохраненные фотографии"

        :param owner_id: идентификатор владельца фотографии целое число, обязательный параметр
        :param photo_id: индентификатор фотографии положительное число, обязательный параметр
        :param access_key: специальный код доступа для приватных фотографий строка
        :return: Возвращает идентификатор созданной фотографии.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.copy', params=params)

    def photos_createAlbum(self, title=None, group_id=None, description=None, privacy_view=None, privacy_comment=None,
                           upload_by_admins_only=None, comments_disabled=None, privacy=None, comment_privacy=None):
        """
        Создает пустой альбом для фотографий.

        :param title: название альбома. строка, обязательный параметр, минимальная длина 2
        :param group_id: идентификатор сообщества, в котором создаётся альбом. целое число
        :param description: текст описания альбома. строка
        :param privacy_view: настройки приватности просмотра альбома в специальном формате. список слов, разделенных через запятую, по умолчанию all, доступен начиная с версии 5.30
        :param privacy_comment: настройки приватности комментирования альбома в специальном формате. список слов, разделенных через запятую, по умолчанию all, доступен начиная с версии 5.30
        :param upload_by_admins_only: кто может загружать фотографии в альбом (только для альбома сообщества).  Возможные значения:   0 — фотографии могут добавлять все пользователи;  1 — фотографии могут добавлять только редакторы и администраторы.  флаг, может принимать значения 1 или 0
        :param comments_disabled: отключено ли комментирование альбома (только для альбома сообщества).  Возможные значения:   0 — комментирование включено;  1 — комментирование отключено.  флаг, может принимать значения 1 или 0
        :param privacy: уровень доступа к альбому (только для альбома пользователя).  Возможные значения:   0 — все пользователи,  1 — только друзья,  2 — друзья и друзья друзей,  3 — только я.  целое число, устаревший параметр, доступен только для версий меньше 5.30
        :param comment_privacy: уровень доступа к комментированию альбома (только для альбома пользователя).  Возможные значения:   0 — все пользователи,  1 — только друзья,  2 — друзья и друзья друзей,  3 — только я.  целое число, устаревший параметр, доступен только для версий меньше 5.30
        :return: После успешного выполнения возвращает объект, который содержит следующие поля:   id — идентификатор созданного альбома;  thumb_id — идентификатор фотографии, которая является обложкой альбома  (0, если обложка отсутствует);  owner_id  идентификатор пользователя или сообщества, которому принадлежит альбом;  title— название альбома;  description — описание альбома;  created — дата создания альбома в формате unixtime;  updated — дата обновления альбома в формате unixtime;  size — количество фотографий в альбоме;  privacy_view — настройки приватности для альбома в формате настроек приватности (только для альбома пользователя);  privacy_comment — настройки приватности для альбома в формате настроек приватности (только для альбома пользователя);  upload_by_admins_only — кто может загружать фотографии в альбом (только для альбома сообщества);  comments_disabled — отключено ли комментирование альбома (только для альбома сообщества);  can_upload — может ли текущий пользователь добавлять фотографии в альбом.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.createAlbum', params=params)

    def photos_createComment(self, owner_id=None, photo_id=None, message=None, attachments=None, from_group=None,
                             reply_to_comment=None, sticker_id=None, access_key=None, guid=None):
        """
        Создает новый комментарий к фотографии.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param photo_id: идентификатор фотографии. целое число, обязательный параметр
        :param message: текст комментария (является обязательным, если не задан параметр attachments).  Максимальное количество символов: 2048. строка
        :param attachments: список объектов, приложенных к комментарию и разделённых символом ",". Поле attachments представляется в формате: <type><owner_id>_<media_id>,<type><owner_id>_<media_id> <type> — тип медиа-вложения: photo — фотография  video — видеозапись  audio — аудиозапись  doc — документ <owner_id> — идентификатор владельца медиа-вложения  <media_id> — идентификатор медиа-вложения.   Например: photo100172_166443618,photo66748_265827614 Параметр является обязательным, если не задан параметр message. список слов, разделенных через запятую
        :param from_group: Данный параметр учитывается, если owner_id < 0 (комментарий к фотографии группы). Возможные значения:   1 — комментарий будет опубликован от имени группы;  0 — комментарий будет опубликован от имени пользователя.  По умолчанию: 0. флаг, может принимать значения 1 или 0
        :param reply_to_comment: идентификатор комментария, в ответ на который нужно оставить текущий. целое число
        :param sticker_id: идентификатор стикера, который нужно прикрепить к комментарию. положительное число
        :param access_key: ключ доступа. строка
        :param guid: уникальное значение для предотвращения повторной отправки одного и того же комментария. строка
        :return: После успешного выполнения возвращает идентификатор созданного комментария.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.createComment', params=params)

    def photos_delete(self, owner_id=None, photo_id=None):
        """
        Удаление фотографии на сайте.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param photo_id: идентификатор фотографии. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.delete', params=params)

    def photos_deleteAlbum(self, album_id=None, group_id=None):
        """
        Удаляет указанный альбом для фотографий у текущего пользователя

        :param album_id: идентификатор альбома. положительное число, обязательный параметр
        :param group_id: идентификатор сообщества, в котором размещен альбом. положительное число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.deleteAlbum', params=params)

    def photos_deleteComment(self, owner_id=None, comment_id=None):
        """
        Удаляет комментарий к фотографии.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param comment_id: идентификатор комментария. целое число, обязательный параметр
        :return: После успешного выполнения возвращает 1 (0, если комментарий не найден).
        """

        params = locals()
        params.pop('self')
        return self._call('photos.deleteComment', params=params)

    def photos_edit(self, owner_id=None, photo_id=None, caption=None, latitude=None, longitude=None, place_str=None,
                    foursquare_id=None, delete_place=None):
        """
        Редактирует описание или геометку у фотографии.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param photo_id: идентификатор фотографии. положительное число, обязательный параметр
        :param caption: новый текст описания к фотографии. Если параметр не задан, то считается, что он равен пустой строке. строка
        :param latitude: географическая широта. дробное число
        :param longitude: географическая долгота. дробное число
        :param place_str: название места. строка
        :param foursquare_id: id в Foursquare. строка
        :param delete_place: удалить место (0 — не удалять, 1 — удалить). флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.edit', params=params)

    def photos_editAlbum(self, album_id=None, title=None, description=None, owner_id=None, privacy_view=None,
                         privacy_comment=None, upload_by_admins_only=None, comments_disabled=None, privacy=None,
                         comment_privacy=None):
        """
        Редактирует данные альбома для фотографий.

        :param album_id: идентификатор альбома. целое число, положительное число, обязательный параметр
        :param title: новое название альбома. строка
        :param description: новый текст описания альбома. строка
        :param owner_id: идентификатор владельца альбома (пользователь или сообщество). Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param privacy_view: настройки приватности просмотра альбома в специальном формате. список слов, разделенных через запятую, доступен начиная с версии 5.30
        :param privacy_comment: настройки приватности комментирования альбома в специальном формате. список слов, разделенных через запятую, доступен начиная с версии 5.30
        :param upload_by_admins_only: кто может загружать фотографии в альбом (только для альбома сообщества).  Возможные значения:   0 — фотографии могут добавлять все пользователи;  1 — фотографии могут добавлять только редакторы и администраторы.  флаг, может принимать значения 1 или 0
        :param comments_disabled: отключено ли комментирование альбома (только для альбома сообщества).  Возможные значения:   0 — комментирование включено;  1 — комментирование отключено.  флаг, может принимать значения 1 или 0
        :param privacy: новый уровень доступа к альбому (только для альбома пользователя).  Возможные значения:   0 — все пользователи,  1 — только друзья,  2 — друзья и друзья друзей,  3 — только я.  целое число, устаревший параметр, доступен только для версий меньше 5.30
        :param comment_privacy: новый уровень доступа к комментированию альбома (только для альбома пользователя).  Возможные значения:   0 — все пользователи,  1 — только друзья,  2 — друзья и друзья друзей,  3 — только я.  целое число, устаревший параметр, доступен только для версий меньше 5.30
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.editAlbum', params=params)

    def photos_editComment(self, owner_id=None, comment_id=None, message=None, attachments=None):
        """
        Изменяет текст комментария к фотографии. Обратите внимание, что редактирование комментария доступно только в течение суток после его создания.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param comment_id: идентификатор комментария. целое число, обязательный параметр
        :param message: новый текст комментария. Обязательный параметр, если не задан параметр attachments.  Максимальное количество символов: 2048. строка
        :param attachments: новый список объектов, приложенных к комментарию и разделённых символом ",". Поле attachments представляется в формате: <type><owner_id>_<media_id>,<type><owner_id>_<media_id> <type> — тип медиа-вложения: photo — фотография  video — видеозапись  audio — аудиозапись  doc — документ <owner_id> — идентификатор владельца медиа-вложения  <media_id> — идентификатор медиа-вложения.   Например: photo100172_166443618,photo66748_265827614 Параметр является обязательным, если не задан параметр message. список слов, разделенных через запятую
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.editComment', params=params)

    def photos_get(self, owner_id=None, album_id=None, photo_ids=None, rev=None, extended=None, feed_type=None,
                   feed=None, photo_sizes=None, offset=None, count=None):
        """
        Возвращает список фотографий в альбоме.

        :param owner_id: идентификатор владельца альбома. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param album_id: идентификатор альбома. Для служебных альбомов используются следующие идентификаторы:   wall — фотографии со стены;  profile — фотографии профиля;  saved — сохраненные фотографии. Возвращается только с ключом доступа пользователя.  строка
        :param photo_ids: идентификаторы фотографий, информацию о которых необходимо вернуть. список слов, разделенных через запятую
        :param rev: порядок сортировки фотографий. Возможные значения:   1 — антихронологический;  0 — хронологический.  флаг, может принимать значения 1 или 0
        :param extended: 1 — будут возвращены дополнительные поля likes, comments, tags, can_comment, reposts. По умолчанию: 0. флаг, может принимать значения 1 или 0
        :param feed_type: тип новости, получаемый в поле type метода newsfeed.get, для получения только загруженных пользователем фотографий, либо только фотографий, на которых он был отмечен. Может принимать значения photo, photo_tag. строка
        :param feed: время в формате, который может быть получен методом newsfeed.get в поле date, для получения всех фотографий загруженных пользователем в определённый день либо на которых пользователь был отмечен. Также нужно указать параметр uid пользователя, с которым произошло событие.  Значение должно отличаться от текущего времени не более, чем на месяц. целое число
        :param photo_sizes: 1 — возвращать доступные размеры фотографии в специальном формате. По умолчанию: 0. флаг, может принимать значения 1 или 0
        :param offset: отступ, необходимый для получения определенного подмножества записей. положительное число
        :param count: количество записей, которое будет получено. положительное число, по умолчанию 50, максимальное значение 1000
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов фотографий в поле items.  Если к фотографии прикреплено местоположение, также возвращаются поля lat и long, содержащие географические координаты отметки.  Если был задан параметр extended=1, возвращаются дополнительные поля:   likes — количество отметок Мне нравится и информация о том, поставил ли лайк текущий пользователь;  comments — количество комментариев к фотографии;  tags — количество отметок на фотографии;  can_comment — может ли текущий пользователь комментировать фото (1 — может, 0 — не может);  reposts — число репостов фотографии.   Если был задан параметр photo_sizes=1, вместо полей width и height возвращаются размеры копий фотографии в специальном формате.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.get', params=params)

    def photos_getAlbums(self, owner_id=None, album_ids=None, offset=None, count=None, need_system=None,
                         need_covers=None, photo_sizes=None):
        """
        Возвращает список фотоальбомов пользователя или сообщества.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежат альбомы. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param album_ids: перечисленные через запятую идентификаторы альбомов. список целых чисел, разделенных запятыми
        :param offset: смещение, необходимое для выборки определенного подмножества альбомов. положительное число
        :param count: количество альбомов, которое нужно вернуть. (по умолчанию возвращаются все альбомы) положительное число
        :param need_system: 1 — будут возвращены системные альбомы, имеющие отрицательные идентификаторы.  Обратите внимание, что информация о системных альбомах возвращается даже в том случае, если они не содержат фотографий. флаг, может принимать значения 1 или 0
        :param need_covers: 1 — будет возвращено дополнительное поле thumb_src с адресом изображения-обложки. По умолчанию поле thumb_src не возвращается. флаг, может принимать значения 1 или 0
        :param photo_sizes: 1 — размеры фотографий будут возвращены  в специальном формате. флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов, описывающих альбомы, в поле items. Каждый из этих объектов содержит следующие поля:   id — идентификатор альбома;  thumb_id — идентификатор фотографии, которая является обложкой (0, если обложка отсутствует);  owner_id — идентификатор владельца альбома;  title — название альбома;  description — описание альбома; (не приходит для системных альбомов)  created — дата создания альбома в формате unixtime; (не приходит для системных альбомов);  updated — дата последнего обновления альбома в формате unixtime; (не приходит для системных альбомов);  size — количество фотографий в альбоме;  can_upload — 1, если текущий пользователь может загружать фотографии в альбом (при запросе информации об альбомах сообщества);  privacy_view* — настройки приватности для альбома в формате настроек приватности (только для альбома пользователя, не приходит для системных альбомов);  privacy_comment* — настройки приватности для альбома в формате настроек приватности (только для альбома пользователя, не приходит для системных альбомов);  upload_by_admins_only* — кто может загружать фотографии в альбом (только для альбома сообщества, не приходит для системных альбомов);  comments_disabled* — отключено ли комментирование альбома (только для альбома сообщества, не приходит для системных альбомов);  thumb_src — ссылка на изображение обложки альбома (если был указан параметр need_covers).   * — поля возвращаются только при запросе информации об альбомах текущего пользователя или альбомах администрируемых им сообществ.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.getAlbums', params=params)

    def photos_getAlbumsCount(self, user_id=None, group_id=None):
        """
        Возвращает количество доступных альбомов пользователя или сообщества.

        :param user_id: идентификатор пользователя, количество альбомов которого необходимо получить. целое число, по умолчанию идентификатор текущего пользователя
        :param group_id: идентификатор сообщества, количество альбомов которого необходимо получить. целое число
        :return: После успешного выполнения возвращает количество альбомов  с учетом настроек приватности.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.getAlbumsCount', params=params)

    def photos_getAll(self, owner_id=None, extended=None, offset=None, count=None, photo_sizes=None,
                      no_service_albums=None, need_hidden=None, skip_hidden=None):
        """
        Возвращает все фотографии пользователя или сообщества в антихронологическом порядке.

        :param owner_id: идентификатор пользователя или сообщества, фотографии которого нужно получить. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param extended: 1 — возвращать расширенную информацию о фотографиях. флаг, может принимать значения 1 или 0
        :param offset: смещение, необходимое для выборки определенного подмножества фотографий. По умолчанию — 0. положительное число
        :param count: число фотографий, информацию о которых необходимо получить. положительное число, по умолчанию 20, максимальное значение 200
        :param photo_sizes: 1 — будут возвращены размеры фотографий в специальном формате. флаг, может принимать значения 1 или 0
        :param no_service_albums: 0 — вернуть все фотографии, включая находящиеся в сервисных альбомах, таких как "Фотографии на моей стене" (по умолчанию);  1 — вернуть фотографии только из стандартных альбомов пользователя или сообщества. флаг, может принимать значения 1 или 0
        :param need_hidden: 1 — возвращает информацию от том, скрыта ли фотография из блока над стеной пользователя. флаг, может принимать значения 1 или 0
        :param skip_hidden: 1 — не возвращать фотографии, скрытые из блока над стеной пользователя (параметр учитывается только при owner_id > 0, параметр no_service_albums игнорируется). флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает список объектов photo.  Если был задан параметр extended=1, возвращаются дополнительные поля:   likes — количество отметок Мне нравится и информация о том, поставил ли лайк текущий пользователь;  reposts — число репостов фотографии.   Если был задан параметр photo_sizes=1, вместо полей width и height возвращаются размеры копий фотографии в специальном формате.  Если был задан параметр need_hidden=1, будет возвращено поле hidden = 1 в случае, если фотография скрыта из блока над стеной пользователя.  Если был задан параметр skip_hidden=1, для объекта ответа будет возвращено поле more, принимающее значения 1 или 0, обозначающее наличие или отсутствие следующих страниц с фотографиями пользователя. Также для объекта фотографии будет возвращено поле real_offset, обозначающее настоящую позицию фотографии из всех фотографий пользователя. Максимальное полученное значение этого поля необходимо передать в offset запроса для следующей страницы.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.getAll', params=params)

    def photos_getAllComments(self, owner_id=None, album_id=None, need_likes=None, offset=None, count=None):
        """
        Возвращает отсортированный в антихронологическом порядке список всех комментариев к конкретному альбому или ко всем альбомам пользователя.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежат фотографии. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param album_id: идентификатор альбома. Если параметр не задан, то считается, что необходимо получить комментарии ко всем альбомам пользователя или сообщества. положительное число
        :param need_likes: 1 — будет возвращено дополнительное поле likes. По умолчанию поле likes не возвращается. флаг, может принимать значения 1 или 0
        :param offset: смещение, необходимое для выборки определенного подмножества комментариев. Если параметр не задан, то считается, что он равен 0. положительное число
        :param count: количество комментариев, которое необходимо получить. Если параметр не задан, то считается что он равен 20. Максимальное значение параметра 100.  Обратите внимание, даже при использовании параметра offset для получения доступны только первые 10000 комментариев. положительное число
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов комментариев в поле items с дополнительным полем pid, содержащим идентификатор фотографии, к которой был оставлен комментарий.  Если был задан параметр need_likes=1, возвращается дополнительное поле likes:   count — число пользователей, которым понравился комментарий;  user_likes — наличие отметки «Мне нравится» от текущего пользователя  (1 — есть, 0 — нет);  can_like — информация о том, может ли текущий пользователь поставить отметку «Мне нравится»  (1 — может, 0 — не может).
        """

        params = locals()
        params.pop('self')
        return self._call('photos.getAllComments', params=params)

    def photos_getById(self, photos=None, extended=None, photo_sizes=None):
        """
        Возвращает информацию о фотографиях по их идентификаторам.

        :param photos: перечисленные через запятую идентификаторы, которые представляют собой идущие через знак подчеркивания id пользователей, разместивших фотографии, и id самих фотографий. Чтобы получить информацию о фотографии в альбоме группы, вместо id пользователя следует указать -id группы. Пример значения photos: 1_129207899,6492_135055734,  -20629724_271945303  Некоторые фотографии, идентификаторы которых могут быть получены через API, закрыты приватностью, и не будут получены. В этом случае следует использовать ключ доступа фотографии (access_key) в её идентификаторе. Пример значения photos: 1_129207899_220df2876123d3542f, 6492_135055734_e0a9bcc31144f67fbd  Поле access_key будет возвращено вместе с остальными данными фотографии в методах, которые возвращают фотографии, закрытые приватностью но доступные в данном контексте. Например данное поле имеют фотографии, возвращаемые методом newsfeed.get. список слов, разделенных через запятую, обязательный параметр
        :param extended: 1 — будут возвращены дополнительные поля likes, comments, tags, can_comment, can_repost. Поля comments и tags содержат только количество объектов. По умолчанию данные поля не возвращается. флаг, может принимать значения 1 или 0
        :param photo_sizes: возвращать ли доступные размеры фотографии в специальном формате. флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает массив объектов photo.  Если к фотографии прикреплено местоположение, также возвращаются поля lat и long, содержащие географические координаты отметки.  Если был задан параметр extended=1, возвращаются дополнительные поля:   likes — количество отметок Мне нравится и информация о том, поставил ли лайк текущий пользователь;  comments — количество комментариев к фотографии;  tags — количество отметок на фотографии;  can_comment — может ли текущий пользователь комментировать фото (1 — может, 0 — не может);  can_repost — может ли текущий пользователь сделать репост фотографии (1 — может, 0 — не может).   Если был задан параметр photo_sizes, вместо полей width и height возвращаются размеры копий фотографии в специальном формате.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.getById', params=params)

    def photos_getChatUploadServer(self, chat_id=None, crop_x=None, crop_y=None, crop_width=None):
        """
        Позволяет получить адрес для загрузки обложки чата.

        :param chat_id: идентификатор беседы, для которой нужно загрузить фотографию. положительное число, обязательный параметр
        :param crop_x: координата x для обрезки фотографии (верхний правый угол). положительное число
        :param crop_y: координата y для обрезки фотографии (верхний правый угол). положительное число
        :param crop_width: ширина фотографии после обрезки в px. положительное число, минимальное значение 200
        :return: После успешного выполнения возвращает объект с единственным полем upload_url.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.getChatUploadServer', params=params)

    def photos_getComments(self, owner_id=None, photo_id=None, need_likes=None, start_comment_id=None, offset=None,
                           count=None, sort=None, access_key=None, extended=None, fields=None, skip_before_id=None,
                           skip_after_id=None):
        """
        Возвращает список комментариев к фотографии.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param photo_id: идентификатор фотографии. целое число, обязательный параметр
        :param need_likes: 1 — будет возвращено дополнительное поле likes. По умолчанию: 0. флаг, может принимать значения 1 или 0
        :param start_comment_id: идентификатор комментария, начиная с которого нужно вернуть список (подробности см. ниже). положительное число, доступен начиная с версии 5.33
        :param offset: смещение, необходимое для выборки определенного подмножества комментариев. По умолчанию: 0. целое число
        :param count: количество комментариев, которое необходимо получить. положительное число, по умолчанию 20, максимальное значение 100
        :param sort: порядок сортировки комментариев. Возможные значения:   asc — от старых к новым;  desc — от новых к старым.  строка
        :param access_key: ключ доступа к фотографии. строка
        :param extended: 1 — в ответе будут возвращены дополнительные поля profiles и groups, содержащие информацию о пользователях и сообществах. По умолчанию: 0. флаг, может принимать значения 1 или 0
        :param fields: список дополнительных полей профилей, которые необходимо вернуть. См. подробное описание.  Доступные значения: photo_id, verified, sex, bdate, city, country, home_town, has_photo, photo_50, photo_100, photo_200_orig, photo_200, photo_400_orig, photo_max, photo_max_orig, online, lists, domain, has_mobile, contacts, site, education, universities, schools, status, last_seen, followers_count, common_count, occupation, nickname, relatives, relation, personal, connections, exports, wall_comments, activities, interests, music, movies, tv, books, games, about, quotes, can_post, can_see_all_posts, can_see_audio, can_write_private_message, can_send_friend_request, is_favorite, is_hidden_from_feed, timezone, screen_name, maiden_name, crop_photo, is_friend, friend_status, career, military, blacklisted, blacklisted_by_me. список слов, разделенных через запятую
        :param skip_before_id: идентификатор последнего полученного комментания, при подгрузке более новых комментариев. (этот и более старые комментарии получены не будут) положительное число, устаревший параметр, доступен только для версий меньше 5.33
        :param skip_after_id: идентификатор последнего полученного комментания, при подгрузке более старых комментариев. (этот и более новые комментарии получены не будут) положительное число, устаревший параметр, доступен только для версий меньше 5.33
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов комментариев в поле items.  Если был задан параметр extended=1, возвращает число результатов в поле count, отдельно массив объектов комментариев в поле items, пользователей в поле profiles и сообществ в поле groups.  Если был задан параметр need_likes=1, у объектов комментариев возвращается дополнительное поле likes:   count — число пользователей, которым понравился комментарий;  user_likes — наличие отметки «Мне нравится» от текущего пользователя  (1 — есть, 0 — нет);  can_like — информация о том, может ли текущий пользователь поставить отметку «Мне нравится»  (1 — может, 0 — не может).   Если был передан параметр start_comment_id, будет также возвращено поле real_offset – итоговое смещение данного подмножества комментариев (оно может быть отрицательным, если был указан отрицательный offset).
        """

        params = locals()
        params.pop('self')
        return self._call('photos.getComments', params=params)

    def photos_getMarketAlbumUploadServer(self, group_id=None):
        """
        Возвращает адрес сервера для загрузки фотографии подборки товаров в сообществе. После успешной загрузки Вы можете сохранить фотографию, воспользовавшись методом photos.saveMarketAlbumPhoto.

        :param group_id: идентификатор сообщества, для которого необходимо загрузить фотографию подборки товаров. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает объект с единственным полем upload_url.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.getMarketAlbumUploadServer', params=params)

    def photos_getMarketUploadServer(self, group_id=None, main_photo=None, crop_x=None, crop_y=None, crop_width=None):
        """
        Возвращает адрес сервера для загрузки фотографии товара. После успешной загрузки Вы можете сохранить фотографию, воспользовавшись методом photos.saveMarketPhoto.

        :param group_id: идентификатор сообщества, для которого необходимо загрузить фотографию товара. положительное число, обязательный параметр
        :param main_photo: является ли фотография обложкой товара  (1 — фотография для обложки, 0 — дополнительная фотография) флаг, может принимать значения 1 или 0
        :param crop_x: координата x для обрезки фотографии (верхний правый угол). положительное число
        :param crop_y: координата y для обрезки фотографии (верхний правый угол). положительное число
        :param crop_width: ширина фотографии после обрезки в px. положительное число, минимальное значение 400
        :return: После успешного выполнения возвращает объект с единственным полем upload_url.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.getMarketUploadServer', params=params)

    def photos_getMessagesUploadServer(self, peer_id=None):
        """
        Возвращает адрес сервера для загрузки фотографии в личное сообщение. После успешной загрузки Вы можете сохранить фотографию, воспользовавшись методом photos.saveMessagesPhoto.

        :param peer_id: идентификатор назначения (для загрузки фотографии в сообщениях сообществ). целое число
        :return: После успешного выполнения возвращает объект с полями upload_url, album_id (id альбома),  group_id (идентификатор сообщества, если используются сообщения сообщества).
        """

        params = locals()
        params.pop('self')
        return self._call('photos.getMessagesUploadServer', params=params)

    def photos_getNewTags(self, offset=None, count=None):
        """
        Возвращает список фотографий, на которых есть непросмотренные отметки.

        :param offset: смещение, необходимое для получения определённого подмножества фотографий. целое число
        :param count: количество фотографий, которые необходимо вернуть. положительное число, максимальное значение 100, по умолчанию 20
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов фотографий в поле items.  Объекты фотографий содержат дополнительные поля:   placer_id — идентификатор пользователя, сделавшего отметку;  tag_created — дата создания отметки в формате unixtime;  tag_id — идентификатор отметки.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.getNewTags', params=params)

    def photos_getOwnerCoverPhotoUploadServer(self, group_id=None, crop_x=None, crop_y=None, crop_x2=None,
                                              crop_y2=None):
        """
        Получает адрес для загрузки обложки сообщества.

        :param group_id: идентификатор сообщества. положительное число, обязательный параметр
        :param crop_x: координата X верхнего левого угла для обрезки изображения. положительное число, по умолчанию 0
        :param crop_y: координата Y верхнего левого угла для обрезки изображения. положительное число, по умолчанию 0
        :param crop_x2: координата X нижнего правого угла для обрезки изображения. положительное число, по умолчанию 795
        :param crop_y2: координата Y нижнего правого угла для обрезки изображения. положительное число, по умолчанию 200
        :return: Возвращает объект с единственным полем upload_url, содержащим адрес сервера для загрузки фото.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.getOwnerCoverPhotoUploadServer', params=params)

    def photos_getOwnerPhotoUploadServer(self, owner_id=None):
        """
        Возвращает адрес сервера для загрузки главной фотографии на страницу пользователя или сообщества. После удачной загрузки Вы можете воспользоваться методом photos.saveOwnerPhoto.

        :param owner_id: идентификатор сообщества или текущего пользователя. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :return: После успешного выполнения возвращает объект с единственным полем upload_url.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.getOwnerPhotoUploadServer', params=params)

    def photos_getTags(self, owner_id=None, photo_id=None, access_key=None):
        """
        Возвращает список отметок на фотографии.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param photo_id: идентификатор фотографии. целое число, обязательный параметр
        :param access_key: строковой ключ доступа, который может быть получен при получении объекта фотографии. строка
        :return: После успешного выполнения возвращает массив объектов tag, каждый из которых содержит следующие поля:   user_id — идентификатор пользователя, которому соответствует отметка;  id — идентификатор отметки;  placer_id — идентификатор пользователя, сделавшего отметку;  tagged_name — название отметки;  date — дата добавления отметки в формате unixtime;  x, y, x2, y2 — координаты прямоугольной области, на которой сделана отметка (верхний левый угол и нижний правый угол) в процентах;  viewed — статус отметки (1 — подтвержденная, 0 — неподтвержденная).
        """

        params = locals()
        params.pop('self')
        return self._call('photos.getTags', params=params)

    def photos_getUploadServer(self, album_id=None, group_id=None):
        """
        Возвращает адрес сервера для загрузки фотографий. После успешной загрузки фотография может быть сохранена с помощью метода photos.save.

        :param album_id: идентификатор альбома. целое число
        :param group_id: идентификатор сообщества, которому принадлежит альбом (если необходимо загрузить фотографию в альбом сообщества).  Если group_id не указан, возвращается адрес для загрузки на стену текущего пользователя. целое число
        :return: После успешного выполнения возвращает объект, содержащий следующие поля:   upload_url — адрес для загрузки фотографий;  album_id — идентификатор альбома, в который будет загружена фотография;  user_id — идентификатор пользователя, от чьего имени будет загружено фото.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.getUploadServer', params=params)

    def photos_getUserPhotos(self, user_id=None, offset=None, count=None, extended=None, sort=None):
        """
        Возвращает список фотографий, на которых отмечен пользователь

        :param user_id: идентификатор пользователя, список фотографий для которого нужно получить. положительное число, по умолчанию идентификатор текущего пользователя
        :param offset: смещение, необходимое для выборки определенного подмножества фотографий. По умолчанию — 0. положительное число
        :param count: количество фотографий, которое необходимо получить. положительное число, по умолчанию 20, максимальное значение 1000
        :param extended: 1 — будут возвращены дополнительные поля likes, comments, tags, can_comment. Поля comments и tags содержат только количество объектов. По умолчанию данные поля не возвращается. флаг, может принимать значения 1 или 0
        :param sort: сортировка результатов (0 — по дате добавления отметки в порядке убывания, 1 — по дате добавления отметки в порядке возрастания). строка
        :return: После успешного выполнения возвращает список объектов photo.  Если был задан параметр extended=1, возвращаются дополнительные поля:   likes — количество отметок Мне нравится и информация о том, поставил ли лайк текущий пользователь;  comments — количество комментариев к фотографии;  tags — количество отметок на фотографии;  can_comment — может ли текущий пользователь комментировать фото (1 — может, 0 — не может).   Если был задан параметр photo_sizes=1, вместо полей width и height возвращаются размеры копий фотографии в специальном формате.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.getUserPhotos', params=params)

    def photos_getWallUploadServer(self, group_id=None):
        """
        Возвращает адрес сервера для загрузки фотографии на стену пользователя или сообщества. После успешной загрузки Вы можете сохранить фотографию, воспользовавшись методом photos.saveWallPhoto.

        :param group_id: идентификатор сообщества, на стену которого нужно загрузить фото (без знака «минус»). целое число
        :return: После успешного выполнения возвращает объект с полями upload_url, album_id, user_id.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.getWallUploadServer', params=params)

    def photos_makeCover(self, owner_id=None, photo_id=None, album_id=None):
        """
        Делает фотографию обложкой альбома.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param photo_id: идентификатор фотографии. Фотография должна находиться в альбоме album_id. целое число, обязательный параметр
        :param album_id: идентификатор альбома. целое число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.makeCover', params=params)

    def photos_move(self, owner_id=None, target_album_id=None, photo_id=None):
        """
        Переносит фотографию из одного альбома в другой.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param target_album_id: идентификатор альбома, в который нужно переместить фотографию. целое число, обязательный параметр
        :param photo_id: идентификатор фотографии, которую нужно перенести. целое число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.move', params=params)

    def photos_putTag(self, owner_id=None, photo_id=None, user_id=None, x=None, y=None, x2=None, y2=None):
        """
        Добавляет отметку на фотографию.

        :param owner_id: идентификатор пользователя, которому принадлежит фотография. положительное число, по умолчанию идентификатор текущего пользователя
        :param photo_id: идентификатор фотографии. положительное число, обязательный параметр
        :param user_id: идентификатор пользователя, которого нужно отметить на фотографии. целое число, обязательный параметр
        :param x: координата верхнего левого угла области с отметкой в % от ширины фотографии. дробное число
        :param y: координата верхнего левого угла области с отметкой в % от высоты фотографии. дробное число
        :param x2: координата правого нижнего угла области с отметкой в % от ширины фотографии. дробное число
        :param y2: координата правого нижнего угла области с отметкой в % от высоты фотографии. дробное число
        :return: После успешного выполнения возвращает идентификатор созданной отметки (tag id).
        """

        params = locals()
        params.pop('self')
        return self._call('photos.putTag', params=params)

    def photos_removeTag(self, owner_id=None, photo_id=None, tag_id=None):
        """
        Удаляет отметку с фотографии.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param photo_id: идентификатор фотографии. целое число, обязательный параметр
        :param tag_id: идентификатор отметки. целое число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.removeTag', params=params)

    def photos_reorderAlbums(self, owner_id=None, album_id=None, before=None, after=None):
        """
        Меняет порядок альбома в списке альбомов пользователя.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит альбом. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param album_id: идентификатор альбома. целое число, обязательный параметр
        :param before: идентификатор альбома, перед которым следует поместить альбом. целое число
        :param after: идентификатор альбома, после которого следует поместить альбом. целое число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.reorderAlbums', params=params)

    def photos_reorderPhotos(self, owner_id=None, photo_id=None, before=None, after=None):
        """
        Меняет порядок фотографии в списке фотографий альбома пользователя.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param photo_id: идентификатор фотографии. целое число, обязательный параметр
        :param before: идентификатор фотографии, перед которой следует поместить фотографию. Если параметр не указан, фотография будет помещена последней. целое число
        :param after: идентификатор фотографии, после которой следует поместить фотографию. Если параметр не указан, фотография будет помещена первой. целое число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.reorderPhotos', params=params)

    def photos_report(self, owner_id=None, photo_id=None, reason=None):
        """
        Позволяет пожаловаться на фотографию.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. целое число, обязательный параметр
        :param photo_id: идентификатор фотографии. положительное число, обязательный параметр
        :param reason: причина жалобы:   0 — спам;  1 — детская порнография;  2 — экстремизм;  3 — насилие;  4 — пропаганда наркотиков;  5 — материал для взрослых;  6 — оскорбление.  положительное число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.report', params=params)

    def photos_reportComment(self, owner_id=None, comment_id=None, reason=None):
        """
        Позволяет пожаловаться на комментарий к фотографии.

        :param owner_id: идентификатор владельца фотографии к которой оставлен комментарий. целое число, обязательный параметр
        :param comment_id: идентификатор комментария. положительное число, обязательный параметр
        :param reason: причина жалобы:   0 — спам;  1 — детская порнография;  2 — экстремизм;  3 — насилие;  4 — пропаганда наркотиков;  5 — материал для взрослых;  6 — оскорбление.  положительное число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.reportComment', params=params)

    def photos_restore(self, owner_id=None, photo_id=None):
        """
        Восстанавливает удаленную фотографию.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param photo_id: идентификатор фотографии. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.restore', params=params)

    def photos_restoreComment(self, owner_id=None, comment_id=None):
        """
        Восстанавливает удаленный комментарий к фотографии.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит фотография. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param comment_id: идентификатор удаленного комментария. целое число, обязательный параметр
        :return: После успешного выполнения возвращает 1 (0, если комментарий с таким идентификатором не является удаленным).
        """

        params = locals()
        params.pop('self')
        return self._call('photos.restoreComment', params=params)

    def photos_save(self, album_id=None, group_id=None, server=None, photos_list=None, hash=None, latitude=None,
                    longitude=None, caption=None):
        """
        Сохраняет фотографии после успешной загрузки.

        :param album_id: идентификатор альбома, в который необходимо сохранить фотографии. целое число
        :param group_id: идентификатор сообщества, в которое необходимо сохранить фотографии. целое число
        :param server: параметр, возвращаемый в результате загрузки фотографий на сервер. целое число
        :param photos_list: параметр, возвращаемый в результате загрузки фотографий на сервер. строка
        :param hash: параметр, возвращаемый в результате загрузки фотографий на сервер. строка
        :param latitude: географическая широта, заданная в градусах (от -90 до 90); дробное число
        :param longitude: географическая долгота, заданная в градусах (от -180 до 180); дробное число
        :param caption: текст описания фотографии (максимум 2048 символов). строка
        :return: После успешного выполнения возвращает массив объектов фотографий.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.save', params=params)

    def photos_saveMarketAlbumPhoto(self, group_id=None, photo=None, server=None, hash=None):
        """
        Сохраняет фотографии после успешной загрузки на URI, полученный методом photos.getMarketAlbumUploadServer.

        :param group_id: идентификатор группы, для которой нужно загрузить фотографию. положительное число, обязательный параметр
        :param photo: параметр, возвращаемый в результате загрузки фотографии на сервер.  Минимальный размер фотографии — 1280x720 пикселей. строка, обязательный параметр
        :param server: параметр, возвращаемый в результате загрузки фотографии на сервер. положительное число, обязательный параметр
        :param hash: параметр, возвращаемый в результате загрузки фотографии на сервер. строка, обязательный параметр
        :return: После успешного выполнения возвращает массив, содержащий объект с загруженной фотографией.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.saveMarketAlbumPhoto', params=params)

    def photos_saveMarketPhoto(self, group_id=None, photo=None, server=None, hash=None, crop_data=None, crop_hash=None):
        """
        Сохраняет фотографии после успешной загрузки на URI, полученный методом photos.getMarketUploadServer.

        :param group_id: идентификатор группы, для которой нужно загрузить фотографию. положительное число
        :param photo: параметр, возвращаемый в результате загрузки фотографии на сервер. строка, обязательный параметр
        :param server: параметр, возвращаемый в результате загрузки фотографии на сервер. целое число, обязательный параметр
        :param hash: параметр, возвращаемый в результате загрузки фотографии на сервер. строка, обязательный параметр
        :param crop_data: параметр, возвращаемый в результате загрузки фотографии на сервер. строка
        :param crop_hash: параметр, возвращаемый в результате загрузки фотографии на сервер. строка
        :return: После успешного выполнения возвращает массив, содержащий объект с загруженной фотографией.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.saveMarketPhoto', params=params)

    def photos_saveMessagesPhoto(self, photo=None, server=None, hash=None):
        """
        Сохраняет фотографию после успешной загрузки на URI, полученный методом photos.getMessagesUploadServer.

        :param photo: параметр, возвращаемый в результате загрузки фотографии на сервер. строка, обязательный параметр
        :param server: параметр, возвращаемый в результате загрузки фотографии на сервер. целое число
        :param hash: параметр, возвращаемый в результате загрузки фотографии на сервер. строка
        :return: После успешного выполнения возвращает массив с загруженной фотографией, возвращённый объект имеет поля id, pid, aid, owner_id, src, src_big, src_small, created. В случае наличия фотографий в высоком разрешении также будут возвращены адреса с названиями src_xbig и src_xxbig.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.saveMessagesPhoto', params=params)

    def photos_saveOwnerCoverPhoto(self, hash=None, photo=None):
        """
        Сохраняет изображение для обложки сообщества после успешной загрузки.

        :param hash: параметр hash, полученный в результате загрузки фотографии на сервер. строка, обязательный параметр
        :param photo: параметр photo, полученный в результате загрузки фотографии на сервер. строка, обязательный параметр
        :return: Возвращает массив images объектов, описывающих копии загруженной фотографии. Каждый объект содержит следующие поля:   url (string) — URL копии фотографии;  width (integer) — ширина копии в px;  height (integer) — высота копии в px.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.saveOwnerCoverPhoto', params=params)

    def photos_saveOwnerPhoto(self, server=None, hash=None, photo=None):
        """
        Позволяет сохранить главную фотографию пользователя или сообщества. Адрес для загрузки фотографии Вы можете получить с помощью метода photos.getOwnerPhotoUploadServer.

        :param server: параметр, возвращаемый в результате загрузки фотографии на сервер. строка
        :param hash: параметр, возвращаемый в результате загрузки фотографии на сервер. строка
        :param photo: параметр, возвращаемый в результате загрузки фотографии на сервер. строка
        :return: После успешного выполнения возвращает объект, содержащий поля photo_hash и photo_src (при работе через VK.api метод вернёт поля photo_src, photo_src_big, photo_src_small). Параметр photo_hash необходим для подтверждения пользователем изменения его фотографии через вызов метода saveProfilePhoto Javascript API. Поле photo_src содержит путь к загруженной фотографии.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.saveOwnerPhoto', params=params)

    def photos_saveWallPhoto(self, user_id=None, group_id=None, photo=None, server=None, hash=None, latitude=None,
                             longitude=None, caption=None):
        """
        Сохраняет фотографии после успешной загрузки на URI, полученный методом photos.getWallUploadServer.

        :param user_id: идентификатор пользователя, на стену которого нужно сохранить фотографию. положительное число
        :param group_id: идентификатор сообщества, на стену которого нужно сохранить фотографию. положительное число
        :param photo: параметр, возвращаемый в результате загрузки фотографии на сервер. строка, обязательный параметр
        :param server: параметр, возвращаемый в результате загрузки фотографии на сервер. целое число
        :param hash: параметр, возвращаемый в результате загрузки фотографии на сервер. строка
        :param latitude: географическая широта, заданная в градусах (от -90 до 90); дробное число
        :param longitude: географическая долгота, заданная в градусах (от -180 до 180); дробное число
        :param caption: текст описания фотографии (максимум 2048 символов). строка
        :return: После успешного выполнения возвращает массив, содержащий объект с загруженной фотографией.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.saveWallPhoto', params=params)

    def photos_search(self, q=None, lat=None, long=None, start_time=None, end_time=None, sort=None, offset=None,
                      count=None, radius=None):
        """
        Осуществляет поиск изображений по местоположению или описанию.

        :param q: строка поискового запроса, например, Nature. строка
        :param lat: географическая широта отметки, заданная в градусах (от -90 до 90). дробное число
        :param long: географическая долгота отметки, заданная в градусах (от -180 до 180). дробное число
        :param start_time: время в формате unixtime, не раньше которого должны были быть загружены найденные фотографии. положительное число
        :param end_time: время в формате unixtime, не позже которого должны были быть загружены найденные фотографии. положительное число
        :param sort: сортировка результатов. Возможные значения:   1 — по количеству отметок «Мне нравится»;  0 — по дате добавления фотографии.  положительное число
        :param offset: смещение относительно первой найденной фотографии для выборки определенного подмножества. положительное число
        :param count: количество возвращаемых фотографий. положительное число, по умолчанию 100, максимальное значение 1000
        :param radius: радиус поиска в метрах. (работает очень приближенно, поэтому реальное расстояние до цели может отличаться от заданного). Может принимать значения: 10, 100, 800, 6000, 50000 положительное число, по умолчанию 5000
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов фотографий в поле items.
        """

        params = locals()
        params.pop('self')
        return self._call('photos.search', params=params)

    def podcasts_clearRecentSearches(self):
        """


        :return:
        """

        params = locals()
        params.pop('self')
        return self._call('podcasts.clearRecentSearches', params=params)

    def podcasts_getPopular(self):
        """


        :return:
        """

        params = locals()
        params.pop('self')
        return self._call('podcasts.getPopular', params=params)

    def podcasts_getRecentSearchRequests(self):
        """


        :return:
        """

        params = locals()
        params.pop('self')
        return self._call('podcasts.getRecentSearchRequests', params=params)

    def podcasts_search(self, search_string=None, offset=None, count=None):
        """


        :param search_string: обязательный параметр, строка
        :param offset: целое число
        :param count: целое число
        :return:
        """

        params = locals()
        params.pop('self')
        return self._call('podcasts.search', params=params)

    def polls_addVote(self, owner_id=None, poll_id=None, answer_ids=None, is_board=None):
        """
        Отдает голос текущего пользователя за выбранный вариант ответа в указанном опросе.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит опрос. целое число, по умолчанию идентификатор текущего пользователя
        :param poll_id: идентификатор опроса. положительное число, обязательный параметр
        :param answer_ids: список идентификаторов ответа (для опроса с мультивыбором). список положительных чисел, разделенных запятыми, обязательный параметр
        :param is_board: 1 – опрос находится в обсуждении, 0 – опрос прикреплен к стене. флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает одно из следующих значений:   1 — если голос текущего пользователя был отдан за выбранный вариант ответа;  0 — если текущий пользователь уже голосовал в указанном опросе
        """

        params = locals()
        params.pop('self')
        return self._call('polls.addVote', params=params)

    def polls_create(self, question=None, is_anonymous=None, is_multiple=None, end_date=None, owner_id=None,
                     add_answers=None, photo_id=None, background_id=None):
        """
        Позволяет создавать опросы, которые впоследствии можно прикреплять к записям на странице пользователя или сообщества.

        :param question: текст вопроса строка
        :param is_anonymous: 1 – анонимный опрос, список проголосовавших недоступен;  0 – опрос публичный, список проголосовавших доступен;  По умолчанию – 0. флаг, может принимать значения 1 или 0
        :param is_multiple: 1 — для создания опроса с мультивыбором. флаг, может принимать значения 1 или 0
        :param end_date: дата завершения опроса в Unixtime. положительное число, минимальное значение 1558003357
        :param owner_id: Если опрос будет добавлен в группу, необходимо передать отрицательный идентификатор группы. По умолчанию текущий пользователь. целое число, по умолчанию идентификатор текущего пользователя
        :param add_answers: список вариантов ответов, например: ["yes", "no", "maybe"]  Может быть не менее 1 и не более 10 вариантов ответа. данные в формате JSON
        :param photo_id: идентификатор фотографии для использования в качестве фона сниппета. положительное число
        :param background_id: идентификатор стандартного фона для сниппета.
        :return: В случае успешного создания опроса в качестве результата возвращается объект опроса.
        """

        params = locals()
        params.pop('self')
        return self._call('polls.create', params=params)

    def polls_deleteVote(self, owner_id=None, poll_id=None, answer_id=None, is_board=None):
        """
        Снимает голос текущего пользователя с выбранного варианта ответа в указанном опросе.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит опрос. целое число, по умолчанию идентификатор текущего пользователя
        :param poll_id: идентификатор опроса. положительное число, обязательный параметр
        :param answer_id: идентификатор варианта ответа. положительное число, обязательный параметр
        :param is_board: 1 – опрос находится в обсуждении, 0 – опрос прикреплен к стене. флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает одно из следующих значений:   1 — если голос текущего пользователя был снят с выбранного варианта ответа  0 — если текущий пользователь еще не голосовал в указанном опросе или указан не выбранный им вариант ответа
        """

        params = locals()
        params.pop('self')
        return self._call('polls.deleteVote', params=params)

    def polls_edit(self, owner_id=None, poll_id=None, question=None, add_answers=None, edit_answers=None,
                   delete_answers=None, end_date=None, photo_id=None, background_id=None):
        """
        Позволяет редактировать созданные опросы.

        :param owner_id: идентификатор владельца опроса целое число, по умолчанию идентификатор текущего пользователя
        :param poll_id: идентификатор редактируемого опроса положительное число, обязательный параметр
        :param question: новый текст редактируемого опроса строка
        :param add_answers: список вариантов ответов, например: ["yes", "no", "maybe"] данные в формате JSON
        :param edit_answers: объект, содержащий варианты ответов, которые необходимо отредактировать;  ключ – идентификатор ответа, значение – новый текст ответа  Пример: {"382967099":"option1", "382967103":"option2"} данные в формате JSON
        :param delete_answers: список идентификаторов ответов, которые необходимо удалить, например: [382967099, 382967103] данные в формате JSON
        :param end_date: дата завершения опроса в Unixtime. положительное число
        :param photo_id: идентификатор фотографии для сниппета. положительное число
        :param background_id: идентификатор стандартного фона для сниппета.
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('polls.edit', params=params)

    def polls_getBackgrounds(self):
        """
        Возвращает варианты фонового изображения для опросов.

        :return:    В ходе выполнения могут произойти общие ошибки. Их описание находится на отдельной странице.
        """

        params = locals()
        params.pop('self')
        return self._call('polls.getBackgrounds', params=params)

    def polls_getById(self, owner_id=None, is_board=None, poll_id=None, extended=None, friends_count=None, fields=None,
                      name_case=None):
        """
        Возвращает детальную информацию об опросе по его идентификатору.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит опрос. целое число, по умолчанию идентификатор текущего пользователя
        :param is_board: 1 — опрос находится в обсуждении, 0 — опрос прикреплен к стене.  По умолчанию — 0. флаг, может принимать значения 1 или 0
        :param poll_id: идентификатор опроса. положительное число, обязательный параметр
        :param extended: 1 — возвращать дополнительную информацию о профилях пользователей. флаг, может принимать значения 1 или 0
        :param friends_count: число идентификаторов проголосовавших друзей, которые необходимо вернуть в ответе. положительное число, по умолчанию 3, максимальное значение 100
        :param fields: список дополнительных полей профилей. См. список возможных значений список слов, разделенных через запятую
        :param name_case: падеж для склонения имени и фамилии пользователя. Возможные значения: именительный – nom, родительный – gen, дательный – dat, винительный – acc, творительный – ins, предложный – abl. По умолчанию nom. по умолчанию nom
        :return: После успешного выполнения возвращает объект опроса.
        """

        params = locals()
        params.pop('self')
        return self._call('polls.getById', params=params)

    def polls_getPhotoUploadServer(self, owner_id=None):
        """
        Возвращает адрес сервера для загрузки фоновой фотографии в опрос.

        :param owner_id: идентификатор владельца опроса. целое число, по умолчанию идентификатор текущего пользователя
        :return: Возвращает объект с единственным полем upload_url (string), содержащим URL для загрузки фотографии.
        """

        params = locals()
        params.pop('self')
        return self._call('polls.getPhotoUploadServer', params=params)

    def polls_getVoters(self, owner_id=None, poll_id=None, answer_ids=None, is_board=None, friends_only=None,
                        offset=None, count=None, fields=None, name_case=None):
        """
        Получает список идентификаторов пользователей, которые выбрали определенные варианты ответа в опросе.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит опрос. целое число
        :param poll_id: идентификатор опроса. положительное число, обязательный параметр
        :param answer_ids: идентификаторы вариантов ответа. список положительных чисел, разделенных запятыми, обязательный параметр
        :param is_board: 1 – опрос находится в обсуждении, 0 – опрос прикреплен к стене. флаг, может принимать значения 1 или 0
        :param friends_only: указывает, необходимо ли возвращать только пользователей, которые являются друзьями текущего пользователя. Параметр может принимать следующие значения: 0 — возвращать всех пользователей в порядке убывания времени голосования; 1 — возвращать только друзей текущего пользователя в порядке убывания времени голосования.  Если параметр не был задан, то считается, что он равен 0. флаг, может принимать значения 1 или 0
        :param offset: смещение относительно начала списка, для выборки определенного подмножества. Если параметр не задан, то считается, что он равен 0. положительное число
        :param count: количество возвращаемых идентификаторов пользователей. Если параметр не задан, то считается, что он равен 100, если не задан параметр friends_only, в противном случае 10. Максимальное значение параметра 1000, если не задан параметр friends_only, в противном случае 100. положительное число
        :param fields: перечисленные через запятую поля анкет, необходимые для получения. Доступные значения: nickname, screen_name, sex, bdate (birthdate), city, country, timezone, photo, photo_medium, photo_big, has_mobile, rate, contacts, education, online, counters. список слов, разделенных через запятую
        :param name_case: падеж для склонения имени и фамилии пользователя. Возможные значения: именительный – nom, родительный – gen, дательный – dat, винительный – acc, творительный – ins, предложный – abl. По умолчанию nom. строка
        :return: После успешного выполнения для каждого варианта ответа возвращает объект со следующими полями:   answer_id — идентификатор варианта ответа;  users — список идентификаторов пользователей с учетом параметров offset и count, которые проголосовали за данный вариант. Первый элемент списка - общее количество пользователей, проголосовавших за данный вариант.   При использовании параметра fields массив users будет состоять из объектов user, каждый из которых содержит набор полей, определенных в этом параметре. Значения uid, first_name и last_name возвращаются всегда, вне зависимости от выбранных полей и их количества.  см. Описание полей параметра user.
        """

        params = locals()
        params.pop('self')
        return self._call('polls.getVoters', params=params)

    def polls_savePhoto(self, photo=None, hash=None):
        """
        Сохраняет фотографию, загруженную в опрос.

        :param photo: параметр, полученный в результате загрузки фотографии. обязательный параметр, строка
        :param hash: параметр, полученный в результате загрузки фотографии. обязательный параметр, строка
        :return: Возвращает идентификатор загруженной фотографии.
        """

        params = locals()
        params.pop('self')
        return self._call('polls.savePhoto', params=params)

    def prettyCards_create(self, owner_id=None, photo=None, title=None, link=None, price=None, price_old=None,
                           button=None):
        """
        Создаёт карточку карусели. На текущий момент карусель поддерживается только в скрытых рекламных записях (см. метод wall.postAdsStealth)

        :param owner_id: Идентификатор владельца карточки. Необходимо указать идентификатор сообщества, взятый со знаком минус. Пример: -19542789. обязательный параметр, целое число
        :param photo: Фотография карточки.  Используйте значение, полученное после загрузки фотографии на сервер. См. метод prettyCards.getUploadURL.  Также можно переиспользовать существующую фотографию из другой карточки. Используйте значение поля photo, которое возвращает метод prettyCards.get или prettyCards.getById. обязательный параметр, строка
        :param title: Заголовок обязательный параметр, строка
        :param link: Ссылка. Кроме http(s)-ссылок также допускается указание телефонных номеров в виде tel:+79111234567 обязательный параметр, строка, максимальная длина 2000
        :param price: Цена. «0» будет отображён как «Бесплатно». Не передавайте этот параметр, чтобы не указывать цену. строка, максимальная длина 20
        :param price_old: Старая цена. Отображается зачёркнутой. «0» будет отображён как «Бесплатно». Не передавайте этот параметр, чтобы не указывать старую цену. строка, максимальная длина 20
        :param button: Кнопка. Не передавайте этот параметр, чтобы не использовать кнопку.  button Текст Действие Тип ссылок app_join Запустить Переход по ссылке Приложения и игры app_game_join Играть Переход по ссылке Игры open_url Перейти Переход по ссылке Внешние сайты, сообщества, приложения open Открыть Переход по ссылке Внешние сайты more Подробнее Переход по ссылке Сообщества call Позвонить Набор номера Телефонные номера book Забронировать Набор номера Телефонные номера enroll Записаться Переход по ссылке или набор номера Внешние сайты, телефонные номера register Зарегистрироваться Набор номера Телефонные номера buy Купить Переход по ссылке Внешние сайты buy_ticket Купить билет Переход по ссылке Внешние сайты to_shop В магазин Переход по ссылке Внешние сайты order Заказать Переход по ссылке Внешние сайты create Создать Переход по ссылке Внешние сайты install Установить Переход по ссылке Внешние сайты contact Связаться Переход по ссылке Внешние сайты fill Заполнить Переход по ссылке Внешние сайты choose Выбрать Переход по ссылке Внешние сайты try Попробовать Переход по ссылке Внешние сайты join_public Подписаться Подписка на публичную страницу Публичные страницы join_event Я пойду Участие в мероприятии События join_group Вступить Вступление в сообщество Сообщества im_group Связаться Переход к диалогу с сообществом Сообщества, публичные страницы, события im_group2 Написать Переход к диалогу с сообществом Сообщества, публичные страницы, события begin Начать Переход по ссылке Внешние сайты get Получить Переход по ссылке Внешние сайты  строка, максимальная длина 255
        :return: Возвращает структуру с информаций о созданной карточке.   owner_id — идентификатор владельца карточки  card_id — идентификатор карточки
        """

        params = locals()
        params.pop('self')
        return self._call('prettyCards.create', params=params)

    def prettyCards_delete(self, owner_id=None, card_id=None):
        """
        Удаляет карточку. Можно удалять только те карточки, которые не используются в записях.

        :param owner_id: Идентификатор владельца карточки. обязательный параметр, целое число
        :param card_id: Идентификатор карточки. обязательный параметр, строка
        :return: Возвращает структуру с информаций об удалённой карточке.   owner_id — идентификатор владельца карточки  card_id — идентификатор карточки
        """

        params = locals()
        params.pop('self')
        return self._call('prettyCards.delete', params=params)

    def prettyCards_edit(self, owner_id=None, card_id=None, photo=None, title=None, link=None, price=None,
                         price_old=None, button=None):
        """
        Редактирует карточку карусели.

        :param owner_id: Идентификатор владельца карточки. обязательный параметр, целое число
        :param card_id: Идентификатор карточки. обязательный параметр, строка
        :param photo: Новая фотография. Подробнее см. метод prettyCards.create. строка
        :param title: Новый заголовок. строка
        :param link: Новая ссылка. Подробнее см. метод prettyCards.create. строка, максимальная длина 2000
        :param price: Новая цена. Подробнее см. метод prettyCards.create. строка, максимальная длина 20
        :param price_old: Обновлённая старая цена. Подробнее см. метод prettyCards.create. строка, максимальная длина 20
        :param button: Новая кнопка. Подробнее см. метод prettyCards.create. строка, максимальная длина 255
        :return: Возвращает структуру с информаций об обновлённой карточке.   owner_id — идентификатор владельца карточки  card_id — идентификатор карточки
        """

        params = locals()
        params.pop('self')
        return self._call('prettyCards.edit', params=params)

    def prettyCards_get(self, owner_id=None, offset=None, count=None):
        """
        Возвращает неиспользованные карточки владельца.

        :param owner_id: Идентификатор владельца. обязательный параметр, целое число
        :param offset: Смещение относительно начала списка карточек. целое число, минимальное значение 0, по умолчанию 0
        :param count: Количество возвращаемых карточек. целое число, минимальное значение 0, по умолчанию 10, максимальное значение 100
        :return: Возвращает массив структур с описанием карточек (подробнее о структуре ответа см. метод prettyCards.getById). Возвращаются только те карточки, которые не были прикреплены к рекламной записи.
        """

        params = locals()
        params.pop('self')
        return self._call('prettyCards.get', params=params)

    def prettyCards_getById(self, owner_id=None, card_ids=None):
        """
        Возвращает информацию о карточке.

        :param owner_id: Идентификатор владельца карточки. обязательный параметр, целое число
        :param card_ids: Список идентификаторов карточек, которые необходимо вернуть. Строка с числами, разделёнными запятой. обязательный параметр
        :return: Возвращает массив структур со следующими полями:   card_id — идентификатор карточки,  link_url — целевая ссылка,  title — заголовок,  button — идентификатор кнопки (см. метод prettyCards.create),  button_text — текст кнопки,  photo — идентификатор фотографии,  images — массив структур с разными размерами фотографии:   url — url фотографии,  width — ширина фотографии,  height — высота фотографии,   price — цена,  price_old — старая цена.
        """

        params = locals()
        params.pop('self')
        return self._call('prettyCards.getById', params=params)

    def prettyCards_getUploadURL(self):
        """
        Возвращает URL для загрузки фотографии для карточки. Для карточек используются квадратные изображения минимальным размером 400х400. В случае загрузки неквадратного изображения, оно будет обрезано до квадратного.  Загрузка изображения для карточки на сервер ВКонтакте осуществляется в 3 этапа:  1. С помощью метода prettyCards.getUploadURL приложение узнает HTTP-адрес для загрузки изображения.  Один адрес можно использовать только для загрузки одного изображения.  2. Приложение формирует POST-запрос на полученный адрес для сохранения изображения на сервере ВКонтакте.  Запрос должен содержать поле file, которое содержит файл с изображением (JPG, PNG, BMP, TIFF или GIF). Максимальный объем файла: 5 Мб.  Результат загрузки изображения возвращается приложению в формате JSON.  В случае удачной загрузки: {"photo":"данные"}    В случае ошибки: {"errcode":код ошибки}    Если получен код ошибки ≥ 10, то загрузку изображения следует начать с первого пункта.  3. Полученные данные можно использовать в методах prettyCards.create и prettyCards.edit.

        :return:    В ходе выполнения могут произойти общие ошибки. Их описание находится на отдельной странице.
        """

        params = locals()
        params.pop('self')
        return self._call('prettyCards.getUploadURL', params=params)

    def search_getHints(self, q=None, offset=None, limit=None, filters=None, fields=None, search_global=None):
        """
        Метод позволяет получить результаты быстрого поиска по произвольной подстроке

        :param q: текст запроса, результаты которого нужно получить строка
        :param offset: смещение для выборки определённого подмножества результатов. положительное число, максимальное значение 200
        :param limit: ограничение на количество возвращаемых результатов. положительное число, по умолчанию 9, максимальное значение 200
        :param filters: Перечисленные через запятую типы данных, которые необходимо вернуть. Возможные значения:   friends – друзья пользователя;  idols – подписки пользователя;  publics – публичные страницы, на которые подписан пользователь;  groups – группы пользователя;  events – встречи пользователя;  correspondents – люди, с которыми пользователь имеет переписку;  mutual_friends – люди, у которых  есть общие друзья с текущим пользователем (этот фильтр позволяет получить не всех пользователей, имеющих общих друзей).    По умолчанию возвращаются все. список слов, разделенных через запятую
        :param fields: дополнительные поля профилей и сообществ для получения. список слов, разделенных через запятую
        :param search_global: 1 — к результатам поиска добавляются результаты глобального поиска по всем пользователям и группам. флаг, может принимать значения 1 или 0, по умолчанию 1
        :return: После успешного выполнения возвращает массив объектов, каждый из которых содержит стандартные поля type, section, description и соответствующий объект с набором дополнительных полей.  type integerтип объекта. Возможные значения:   group — сообщество;  profile — профиль.   group (для type = group) objectданные о сообществе. Объект, который содержит поля:   id (integer) — идентификатор сообщества;  name (string) — название сообщества;  screen_name (string) — короткий адрес;  is_closed (integer) — информация о том, является ли группа/встреча закрытой (0 — открытая, 1 — закрытая, 2 — частная);  is_admin (integer, [0,1]) — информация о том, является ли текущий пользователь администратором сообщества (1 — является, 0 — не является);  is_member (integer, [0,1]) — информация о том, является ли текущий пользователь участником сообщества (1 — является, 0 — не является);  type (string) — тип сообщества. Возможные значения:   group — группа;  page — публичная страница;  event — встреча);   photo (string) — URL квадратной фотографии сообщества с размером 50х50px;  photo_medium (string) — URL квадратной фотографии сообщества с размером 100х100px;  photo_big (string) — URL фотографии сообщества в максимальном доступном размере. profile (для type = profile) objectданные о профиле. Объект, который содержит поля:   id (integer) — идентификатор пользователя;  first_name (string) — имя пользователя;  last_name (string) — фамилия пользователя. section  stringтип объекта. Возможные значения для сообществ:   groups — группы;  events — мероприятия;  publics — публичные страницы.  Возможные значения для профилей:   correspondents — собеседники по переписке;  people — популярные пользователи;  friends — друзья;  mutual_friends — пользователи, имеющие общих друзей с текущим. description  stringописание объекта (для сообществ — тип и число участников, например, Group, 269,136 members, для профилей друзей или пользователями, которые не являются возможными друзьями — название университета или город, для профиля текущего пользователя — That's you, для профилей возможных друзей — N mutual friends). global  integer, [1]поле возвращается, если объект был найден в глобальном поиске, всегда содержит 1.
        """

        params = locals()
        params.pop('self')
        return self._call('search.getHints', params=params)

    def stats_get(self, group_id=None, app_id=None, timestamp_from=None, timestamp_to=None, interval=None,
                  intervals_count=None, filters=None, stats_groups=None, extended=None, date_from=None, date_to=None):
        """
        Возвращает статистику сообщества или приложения. Обратите внимание, если статистика скрыта (доступна только для администраторов сообщества/приложения), для получения данных в вызове необходимо передавать access_token с правами доступа stats.

        :param group_id: идентификатор сообщества. положительное число
        :param app_id: идентификатор приложения. положительное число
        :param timestamp_from: начало периода статистики в Unixtime. положительное число, доступен начиная с версии 5.86
        :param timestamp_to: окончание периода статистики в Unixtime. положительное число, доступен начиная с версии 5.86
        :param interval: временные интервалы. Возможные значения: day, week, month, year, all. По умолчанию: day. строка, по умолчанию day
        :param intervals_count: количество интервалов времени. положительное число
        :param filters: список слов, разделенных через запятую
        :param stats_groups: фильтр для получения данных по конкретному блоку статистики сообщества. Возможные значения: visitors, reach, activity. список слов, разделенных через запятую
        :param extended: 1 — возвращать дополнительно агрегированные данные в результатах. флаг, может принимать значения 1 или 0, по умолчанию 1
        :param date_from: Начальная дата выводимой статистики в формате YYYY-MM-DD. строка, устаревший параметр, доступен только для версий меньше 5.86
        :param date_to: Конечная дата выводимой статистики в формате YYYY-MM-DD. строка, устаревший параметр, доступен только для версий меньше 5.86
        :return: После успешного выполнения возвращает объект с данными статистики.
        """

        params = locals()
        params.pop('self')
        return self._call('stats.get', params=params)

    def stats_getPostReach(self, owner_id=None, post_ids=None):
        """
        Возвращает статистику для записи на стене. Статистика записей в настоящий момент доступна только для сообществ с количеством участников от 5000 и выше, а также официальных сообществ.  Пользователь, от имени которого вызывается метод, должен быть редактором или администратором сообщества, в котором размещена запись.

        :param owner_id: идентификатор сообщества — владельца записи. Указывается со знаком «минус». целое число, обязательный параметр
        :param post_ids: список положительных чисел, разделенных запятыми, обязательный параметр
        :return: После успешного выполнения возвращает данные о статистике записи в виде объекта, который содержит следующие поля:  reach_subscribers  integerохват подписчиков. reach_total  integerсуммарный охват. reach_ads  integerрекламный охват (если запись продвигалась с помощью таргетированной рекламы). reach_viral  integerвиральный охват (если запись продвигалась с помощью таргетированной рекламы).. links  integerпереходы по ссылке. to_group  integerпереходы в сообщество. join_group  integerвступления в сообщество. report  integerколичество жалоб на запись. hide  integerколичество скрывших запись. unsubscribe  integerколичество отписавшихся участников.
        """

        params = locals()
        params.pop('self')
        return self._call('stats.getPostReach', params=params)

    def stats_trackVisitor(self):
        """
        Добавляет данные о текущем сеансе в статистику посещаемости приложения. После первого вызова данного метода в разделе «Статистика» настроек Вашего приложения станет доступна вкладка «Посещаемость». В ней будет отображена информация о числе запусков и уникальных посетителей Вашего приложения.  Для получения достоверных данных о посещаемости приложения метод нужно вызывать при каждом запуске.

        :return:    В ходе выполнения могут произойти общие ошибки. Их описание находится на отдельной странице.
        """

        params = locals()
        params.pop('self')
        return self._call('stats.trackVisitor', params=params)

    def status_get(self, user_id=None, group_id=None):
        """
        Получает текст статуса пользователя или сообщества.

        :param user_id: идентификатор пользователя или сообщества, информацию о статусе которого нужно получить. целое число, по умолчанию идентификатор текущего пользователя
        :param group_id: идентификатор сообщества, статус которого необходимо получить. положительное число
        :return: В случае успеха возвращает объект, у которого в поле text содержится текст статуса пользователя или сообщества.  Если у пользователя включена трансляция играющей музыки в статус и в данный момент воспроизводится аудиозапись, то также будет возвращен объект audio.
        """

        params = locals()
        params.pop('self')
        return self._call('status.get', params=params)

    def status_set(self, text=None, group_id=None):
        """
        Устанавливает новый статус текущему пользователю или сообществу.

        :param text: текст нового статуса. строка
        :param group_id: идентификатор сообщества, в котором будет установлен статус. По умолчанию статус устанавливается текущему пользователю. положительное число
        :return: В случае успешной установки или очистки статуса возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('status.set', params=params)

    def storage_get(self, key=None, keys=None, user_id=None):
        """
        Возвращает значение переменной, название которой передано в параметре key. Задать значение позволяет метод storage.set.  Пользовательская переменная привязана к пользователю, другие пользователи не могут её прочитать.

        :param key: название переменной. строка, максимальная длина 100
        :param keys: список названий переменных, разделённых запятыми. Если указан этот параметр, то параметр key не учитывается. список слов, разделенных через запятую, количество элементов должно составлять не более 1000
        :param user_id: id пользователя, переменная которого устанавливается, в случае если данные запрашиваются серверным методом. положительное число
        :return: Возвращает значение одной или нескольких переменных. Если переменная на сервере отсутствует, то будет возвращена пустая строка.
        """

        params = locals()
        params.pop('self')
        return self._call('storage.get', params=params)

    def storage_getKeys(self, user_id=None, offset=None, count=None):
        """
        Возвращает названия всех переменных.

        :param user_id: id пользователя, названия переменных которого получаются, в случае если данные запрашиваются серверным методом. положительное число
        :param offset: смещение, необходимое для выборки определенного подмножества названий переменных. По умолчанию 0. положительное число, по умолчанию 0
        :param count: количество названий переменных, информацию о которых необходимо получить. положительное число, максимальное значение 1000, по умолчанию 100
        :return: Возвращает массив названий переменных.
        """

        params = locals()
        params.pop('self')
        return self._call('storage.getKeys', params=params)

    def storage_set(self, key=None, value=None, user_id=None):
        """
        Сохраняет значение переменной, название которой передано в параметре key. Получить сохранённое значение позволяет метод storage.get.  Пользовательская переменная привязана к пользователю, и только он или сервер приложения может получить к ней доступ. Может быть создано не более 1000 переменных для каждого пользователя.  Не более 1000 вызовов в час на каждого пользователя.

        :param key: название переменной. Может содержать символы латинского алфавита, цифры, знак тире, нижнее подчёркивание [a-zA-Z_\-0-9]. строка, максимальная длина 100, обязательный параметр
        :param value: значение переменной, сохраняются только первые 4096 байта. строка
        :param user_id: id пользователя, переменная которого устанавливается, в случае если данные запрашиваются серверным методом. положительное число
        :return: Возвращает 1 в случае удачного сохранения переменной.  Для удаления переменной необходимо передать пустое значение в параметре value.
        """

        params = locals()
        params.pop('self')
        return self._call('storage.set', params=params)

    def stories_banOwner(self, owners_ids=None):
        """
        Позволяет скрыть из ленты новостей истории от выбранных источников.

        :param owners_ids: список идентификаторов источников. список целых чисел, разделенных запятыми, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('stories.banOwner', params=params)

    def stories_delete(self, owner_id=None, story_id=None):
        """
        Удаляет историю.

        :param owner_id: идентификатор владельца истории. целое число, по умолчанию идентификатор текущего пользователя, обязательный параметр
        :param story_id: идентификатор истории. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('stories.delete', params=params)

    def stories_get(self, owner_id=None, extended=None, fields=None):
        """
        Возвращает истории, доступные для текущего пользователя.

        :param owner_id: идентификатор пользователя, истории которого необходимо получить. целое число
        :param extended: 1 — возвращать в ответе дополнительную информацию о профилях пользователей. флаг, может принимать значения 1 или 0, по умолчанию
        :param fields: список слов, разделенных через запятую
        :return: После успешного выполнения возвращает объект, содержащий число подборок в поле count и массив подборок историй  в поле items. Каждая подборка — массив историй от одного владельца.  Если был задан параметр extended=1, дополнительно возвращает массив объектов пользователей в поле profiles (array) и сообществ в поле groups (array).
        """

        params = locals()
        params.pop('self')
        return self._call('stories.get', params=params)

    def stories_getBanned(self, extended=None, fields=None):
        """
        Возвращает список источников историй, скрытых из ленты текущего пользователя.

        :param extended: 1 — возвращать расширенную информацию о пользователях и сообществах. флаг, может принимать значения 1 или 0
        :param fields: дополнительные поля пользователей и сообществ, которые необходимо вернуть. список слов, разделенных через запятую
        :return: После успешного выполнения возвращает общее количество скрытых источников в поле count (integer) и их идентификаторы в массиве items. Если extended = 1, items содержит два поля:   profiles (array) — массив объектов, описывающих пользователей;  groups (array) — массив объектов, описывающих сообщества.
        """

        params = locals()
        params.pop('self')
        return self._call('stories.getBanned', params=params)

    def stories_getById(self, stories=None, extended=None, fields=None):
        """
        Возвращает информацию об истории по её идентификатору.

        :param stories: перечисленные через запятую идентификаторы, которые представляют собой идущие через знак подчеркивания идентификаторы владельцев историй и идентификаторы самих историй.   Пример значения stories: 93388_21539,93388_20904 список слов, разделенных через запятую, обязательный параметр
        :param extended: 1 — возвращать в ответе дополнительную информацию о пользователях. флаг, может принимать значения 1 или 0, по умолчанию
        :param fields: дополнительные поля профилей и сообществ, которые необходимо вернуть в ответе. список слов, разделенных через запятую
        :return: После успешного выполнения возвращает объект, содержащий число историй в поле count и массив объектов историй  в поле items.  Если был задан параметр extended = 1, дополнительно возвращает массив объектов  пользователей в поле profiles и объектов сообществ в поле groups.
        """

        params = locals()
        params.pop('self')
        return self._call('stories.getById', params=params)

    def stories_getPhotoUploadServer(self, add_to_news=None, user_ids=None, reply_to_story=None, link_text=None,
                                     link_url=None, group_id=None, clickable_stickers=None):
        """
        Позволяет получить адрес для загрузки истории с фотографией. Обратите внимание, для работы с историями сообщество должно быть верифицированным или с меткой огня.

        :param add_to_news: 1 — разместить историю в новостях. Обязательно, если не указан user_ids флаг, может принимать значения 1 или 0
        :param user_ids: идентификаторы пользователей, которые будут видеть историю (для отправки в личном сообщении). Обязательно, если add_to_news не передан. список положительных чисел, разделенных запятыми
        :param reply_to_story: идентификатор истории, в ответ на которую создается новая. строка
        :param link_text: текст ссылки для перехода из истории (только для историй сообществ). Возможные значения:   to_store — «В магазин»;  vote — «Голосовать»;  more — «Ещё»;  book — «Забронировать»;  order — «Заказать»;  enroll — «Записаться»;  fill — «Заполнить»;  signup — «Зарегистрироваться»;  buy — «Купить»;  ticket — «Купить билет»;  write — «Написать»;  open — «Открыть»;  learn_more — «Подробнее» (по умолчанию);  view — «Посмотреть»;  go_to — «Перейти»;  contact — «Связаться»;  watch — «Смотреть»;  play — «Слушать»;  install — «Установить»;  read — «Читать».  строка
        :param link_url: адрес ссылки для перехода из истории. Допустимы только внутренние ссылки https://vk.com. строка, максимальная длина 2048
        :param group_id: идентификатор сообщества, в которое должна быть загружена история (при работе с ключом доступа пользователя).  Обратите внимание — загрузка историй доступна только для верифицированных сообществ и для сообществ, отмеченных «огоньком». положительное число
        :param clickable_stickers: объект кликабельного стикера. данные в формате JSON
        :return: После успешного выполнения возвращает объект, содержащий следующие поля:   upload_url (string) — адрес сервера для загрузки файла;  user_ids (array) — идентификаторы пользователей, которые могут видеть историю.
        """

        params = locals()
        params.pop('self')
        return self._call('stories.getPhotoUploadServer', params=params)

    def stories_getReplies(self, owner_id=None, story_id=None, access_key=None, extended=None, fields=None):
        """
        Позволяет получить ответы на историю.

        :param owner_id: идентификатор владельца истории. целое число, обязательный параметр
        :param story_id: идентификатор истории. положительное число, обязательный параметр
        :param access_key: ключ доступа для приватного объекта. строка
        :param extended: 1 — возвращать дополнительную информацию о профилях и сообществах. флаг, может принимать значения 1 или 0, по умолчанию
        :param fields: дополнительные поля профилей и сообществ, которые необходимо вернуть в ответе. список слов, разделенных через запятую
        :return: После успешного выполнения возвращает объект, содержащий число подборок в поле count и массив подборок историй  в поле items. Каждая подборка — массив историй от одного владельца.  Если был задан параметр extended=1, дополнительно возвращает массив объектов пользователей в поле profiles (array) и сообществ в поле groups (array).
        """

        params = locals()
        params.pop('self')
        return self._call('stories.getReplies', params=params)

    def stories_getStats(self, owner_id=None, story_id=None):
        """
        Возвращает статистику истории.

        :param owner_id: идентификатор владельца истории. целое число, обязательный параметр
        :param story_id: идентификатор истории. положительное число, обязательный параметр
        :return: Возвращает объект, который содержит следующие поля:   views (object) — просмотры. Содержит поля:   state (string) — доступность значения (on — доступно, off — недоступно);  count (integer) — значение счётчика;   replies  (object) — ответы на историю. Содержит поля:   state (string) — доступность значения (on — доступно, off — недоступно);  count (integer) — значение счётчика;   answer (object) — показывает сколько раз ответили на эту историю сообщением через поле под историей. Не путать с ответами на историю.   state (string) — доступность значения (on — доступно, off — недоступно);  count (integer) — значение счётчика;   shares  (object) — расшаривания истории. Содержит поля:   state (string) — доступность значения (on — доступно, off — недоступно);  count (integer) — значение счётчика;   subscribers (object) — новые подписчики. Содержит поля:   state (string) — доступность значения (on — доступно, off — недоступно);  count (integer) — значение счётчика;   bans  (object) — скрытия истории. Содержит поля:   state (string) — доступность значения (on — доступно, off — недоступно);  count (integer) — значение счётчика;   open_link (object) — переходы по ссылке. Содержит поля:   state (string) — доступность значения (on — доступно, hidden — недоступно);  count (integer) — значение счётчика.
        """

        params = locals()
        params.pop('self')
        return self._call('stories.getStats', params=params)

    def stories_getVideoUploadServer(self, add_to_news=None, user_ids=None, reply_to_story=None, link_text=None,
                                     link_url=None, group_id=None, clickable_stickers=None):
        """
        Позволяет получить адрес для загрузки видеозаписи в историю. Обратите внимание, для работы с историями сообщество должно быть верифицированным или с меткой огня.

        :param add_to_news: 1 — разместить историю в новостях. флаг, может принимать значения 1 или 0
        :param user_ids: идентификаторы пользователей, которые будут видеть историю (для отправки в личном сообщении). список положительных чисел, разделенных запятыми
        :param reply_to_story: идентификатор истории, в ответ на которую создается новая. строка
        :param link_text: текст ссылки для перехода из истории (только для историй сообществ). Возможные значения:   to_store — «В магазин»;  vote — «Голосовать»;  more — «Ещё»;  book — «Забронировать»;  order — «Заказать»;  enroll — «Записаться»;  fill — «Заполнить»;  signup — «Зарегистрироваться»;  buy — «Купить»;  ticket — «Купить билет»;  write — «Написать»;  open — «Открыть»;  learn_more — «Подробнее» (по умолчанию);  view — «Посмотреть»;  go_to — «Перейти»;  contact — «Связаться»;  watch — «Смотреть»;  play — «Слушать»;  install — «Установить»;  read — «Читать»;  game — «Играть».  строка
        :param link_url: адрес ссылки для перехода из истории. строка, максимальная длина 2048
        :param group_id: идентификатор сообщества, в которое должна быть загружена история (при работе с ключом доступа пользователя).  Обратите внимание — загрузка историй доступна только для верифицированных сообществ и для сообществ, отмеченных «огоньком». положительное число
        :param clickable_stickers: объект кликабельного стикера. данные в формате JSON
        :return: После успешного выполнения возвращает объект, содержащий следующие поля:   upload_url (string) — адрес сервера для загрузки файла;  user_ids (array) — идентификаторы пользователей, которые могут видеть историю.
        """

        params = locals()
        params.pop('self')
        return self._call('stories.getVideoUploadServer', params=params)

    def stories_getViewers(self, owner_id=None, story_id=None, count=None, offset=None, extended=None):
        """
        Возвращает список пользователей, просмотревших историю.

        :param owner_id: идентификатор владельца истории. целое число, по умолчанию идентификатор текущего пользователя, обязательный параметр
        :param story_id: идентификатор истории. положительное число, обязательный параметр
        :param count: максимальное число результатов в ответе. положительное число, по умолчанию 100
        :param offset: сдвиг для получения определённого подмножества результатов. положительное число, по умолчанию 0
        :param extended: 1 — возвращать в ответе расширенную информацию о пользователях. флаг, может принимать значения 1 или 0, по умолчанию 0
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и идентификаторы пользователей в поле items (array). Если extended = 1, поле items (array) содержит массив объектов пользователей.
        """

        params = locals()
        params.pop('self')
        return self._call('stories.getViewers', params=params)

    def stories_hideAllReplies(self, owner_id=None, group_id=None):
        """
        Скрывает все ответы автора за последние сутки на истории текущего пользователя.

        :param owner_id: идентификатор пользователя, ответы от которого нужно скрыть. целое число, обязательный параметр
        :param group_id: положительное число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('stories.hideAllReplies', params=params)

    def stories_hideReply(self, owner_id=None, story_id=None):
        """
        Скрывает ответ на историю.

        :param owner_id: идентификатор владельца истории (ответной). целое число, обязательный параметр
        :param story_id: идентификатор истории (ответной). положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('stories.hideReply', params=params)

    def stories_search(self, q=None, place_id=None, latitude=None, longitude=None, radius=None, mentioned_id=None,
                       count=None, extended=None, fields=None):
        """
        Возвращает результаты поиска по историям.

        :param q: поисковый запрос. строка, максимальная длина 255
        :param place_id: идентификатор места. положительное число
        :param latitude: географическая широта точки, в радиусе которой необходимо производить поиск, заданная в градусах (от -90 до 90). дробное число
        :param longitude: географическая долгота точки, в радиусе которой необходимо производить поиск, заданная в градусах (от -180 до 180). дробное число
        :param radius: радиус зоны поиска в метрах. положительное число
        :param mentioned_id: идентификатор упомянутого в истории пользователя или сообщества. целое число
        :param count: количество историй, информацию о которых необходимо вернуть. целое число, по умолчанию 20, минимальное значение 1, максимальное значение 1000
        :param extended: параметр, определяющий необходимость возвращать расширенную информацию о владельце истории. Возможные значения:   0 — возвращаются только идентификаторы;  1 — будут дополнительно возвращены имя и фамилия.   По умолчанию: 0. флаг, может принимать значения 1 или 0
        :param fields: список дополнительных полей профилей, которые необходимо вернуть. См. подробное описание. список слов, разделенных через запятую
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов историй в поле items.  Если был задан параметр extended=1, возвращает объекты profiles  и groups, содержащие массивы объектов, описывающих пользователей и сообщества
        """

        params = locals()
        params.pop('self')
        return self._call('stories.search', params=params)

    def stories_sendInteraction(self, access_key=None, message=None, is_broadcast=None, is_anonymous=None):
        """
        Отправляет фидбек на историю. Доступен приложениям с типом VK Mini Apps. Метод по умолчанию недоступен приложениям. Получить доступ к использованию метода можно подав заявку на размещение в каталоге приложений и подробно рассказав о механике, используемой в приложении.

        :param access_key: ключ доступа пользователя, полученный при подписке. Возвращает событие VKWebAppSubscribeStoryApp. строка, обязательный параметр
        :param message: текст фидбека. строка, максимальная длина 1000
        :param is_broadcast: Возможные значения:   0 —  фидбек виден только отправителю и автору истории;  1 —  фидбек виден всем зрителям истории и автору.  флаг, может принимать значения 1 или 0, по умолчанию
        :param is_anonymous: Возможные значения:   0 — автор фидбека не  анонимный;  1 —  автор фидбека  анонимный.  флаг, может принимать значения 1 или 0, по умолчанию
        :return:
        """

        params = locals()
        params.pop('self')
        return self._call('stories.sendInteraction', params=params)

    def stories_unbanOwner(self, owners_ids=None):
        """
        Позволяет вернуть пользователя или сообщество в список отображаемых историй в ленте.

        :param owners_ids: список идентификаторов владельцев историй, разделённых запятой. список целых чисел, разделенных запятыми, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('stories.unbanOwner', params=params)

    def streaming_getServerUrl(self):
        """
        Позволяет получить данные для подключения к Streaming API.

        :return:    В ходе выполнения могут произойти общие ошибки. Их описание находится на отдельной странице.
        """

        params = locals()
        params.pop('self')
        return self._call('streaming.getServerUrl', params=params)

    def streaming_getSettings(self):
        """
        Позволяет получить значение порога для Streaming API.

        :return:    В ходе выполнения могут произойти общие ошибки. Их описание находится на отдельной странице.
        """

        params = locals()
        params.pop('self')
        return self._call('streaming.getSettings', params=params)

    def streaming_getStats(self, type=None, interval=None, start_time=None, end_time=None):
        """
        Позволяет получить статистику для подготовленных и доставленных событий Streaming API. Руководствуясь статистикой, Вы можете отслеживать перебои в получении данных от ВКонтакте на стороне Вашего сервера — если значения prepared и received расходятся, значит, что-то пошло не так.

        :param type: тип статистики. Возможные значения:   received — события, полученные приложением;  prepared — события, сгенерированные со стороны ВКонтакте.  строка
        :param interval: интервалы статистики. Возможные значения:   5m — пять минут. Максимальный период — 3 дня между start_time и end_time;  1h — один час. Максимальный период — 7 дней между start_time и end_time;  24h — сутки. Максимальный период — 31 день между start_time и end_time.  строка, по умолчанию 5m
        :param start_time: время начала отсчёта в Unixtime. По умолчанию: end_time минус сутки. положительное число
        :param end_time: время окончания отсчёта в Unixtime. По умолчанию: текущее время. положительное число
        :return: Возвращает массив объектов, каждый из которых содержит поля:   event_type (string) — тип событий. Возможные значения:   post — записи на стене;  comment — комментарии;  share — репосты.   stats (array) — значения статистики. Массив объектов, каждый из которых содержит оля:   timestamp (integer) — время, соответствующее значению;  value (integer) — значение.
        """

        params = locals()
        params.pop('self')
        return self._call('streaming.getStats', params=params)

    def streaming_getStem(self, word=None):
        """
        Позволяет получить основу слова.

        :param word: слово, основу которого мы собираемся получить строка, обязательный параметр
        :return:
        """

        params = locals()
        params.pop('self')
        return self._call('streaming.getStem', params=params)

    def streaming_setSettings(self, monthly_tier=None):
        """
        Позволяет задать значение порога для Streaming API.

        :param monthly_tier: значение порога в месяц. Возможные значения:   tier_1;  tier_2;  tier_3;  tier_4;  tier_5;  tier_6;  unlimited.  строка
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('streaming.setSettings', params=params)

    def users_get(self, user_ids=None, fields=None, name_case=None):
        """
        Возвращает расширенную информацию о пользователях.

        :param user_ids: перечисленные через запятую идентификаторы пользователей или их короткие имена (screen_name). По умолчанию — идентификатор текущего пользователя. список слов, разделенных через запятую, количество элементов должно составлять не более 1000
        :param fields: список дополнительных полей профилей, которые необходимо вернуть. См. подробное описание.  Доступные значения: photo_id, verified, sex, bdate, city, country, home_town, has_photo, photo_50, photo_100, photo_200_orig, photo_200, photo_400_orig, photo_max, photo_max_orig, online, domain, has_mobile, contacts, site, education, universities, schools, status, last_seen, followers_count, common_count, occupation, nickname, relatives, relation, personal, connections, exports, activities, interests, music, movies, tv, books, games, about, quotes, can_post, can_see_all_posts, can_see_audio, can_write_private_message, can_send_friend_request, is_favorite, is_hidden_from_feed, timezone, screen_name, maiden_name, crop_photo, is_friend, friend_status, career, military, blacklisted, blacklisted_by_me, can_be_invited_group. список слов, разделенных через запятую
        :param name_case: падеж для склонения имени и фамилии пользователя. Возможные значения: именительный – nom, родительный – gen, дательный – dat, винительный – acc, творительный – ins, предложный – abl. По умолчанию nom. строка
        :return: После успешного выполнения возвращает массив объектов пользователей.
        """

        params = locals()
        params.pop('self')
        return self._call('users.get', params=params)

    def users_getFollowers(self, user_id=None, offset=None, count=None, fields=None, name_case=None):
        """
        Возвращает список идентификаторов пользователей, которые являются подписчиками пользователя.

        :param user_id: идентификатор пользователя. положительное число, по умолчанию идентификатор текущего пользователя
        :param offset: смещение, необходимое для выборки определенного подмножества подписчиков. положительное число
        :param count: количество подписчиков, информацию о которых нужно получить. положительное число, по умолчанию 100, максимальное значение 1000
        :param fields: список дополнительных полей профилей, которые необходимо вернуть. См. подробное описание.  Доступные значения: photo_id, verified, sex, bdate, city, country, home_town, has_photo, photo_50, photo_100, photo_200_orig, photo_200, photo_400_orig, photo_max, photo_max_orig, online, lists, domain, has_mobile, contacts, site, education, universities, schools, status, last_seen, followers_count, common_count, occupation, nickname, relatives, relation, personal, connections, exports, wall_comments, activities, interests, music, movies, tv, books, games, about, quotes, can_post, can_see_all_posts, can_see_audio, can_write_private_message, can_send_friend_request, is_favorite, is_hidden_from_feed, timezone, screen_name, maiden_name, crop_photo, is_friend, friend_status, career, military, blacklisted, blacklisted_by_me. список слов, разделенных через запятую
        :param name_case: падеж для склонения имени и фамилии пользователя. Возможные значения: именительный – nom, родительный – gen, дательный – dat, винительный – acc, творительный – ins, предложный – abl. По умолчанию nom. строка
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов user в поле items.  Идентификаторы пользователей в списке отсортированы в порядке убывания времени их добавления.
        """

        params = locals()
        params.pop('self')
        return self._call('users.getFollowers', params=params)

    def users_getSubscriptions(self, user_id=None, extended=None, offset=None, count=None, fields=None):
        """
        Возвращает список идентификаторов пользователей и публичных страниц, которые входят в список подписок пользователя.

        :param user_id: идентификатор пользователя, подписки которого необходимо получить. положительное число, по умолчанию идентификатор текущего пользователя
        :param extended: 1 – возвращает объединенный список, содержащий объекты group и user вместе. 0 – возвращает список идентификаторов групп и пользователей отдельно. (по умолчанию) флаг, может принимать значения 1 или 0
        :param offset: смещение необходимое для выборки определенного подмножества подписок. Этот параметр используется только если передан extended=1. положительное число
        :param count: количество подписок, которые необходимо вернуть. Этот параметр используется только если передан extended=1. положительное число, по умолчанию 20, максимальное значение 200
        :param fields: список дополнительных полей для объектов user и group, которые необходимо вернуть. список слов, разделенных через запятую
        :return: После успешного выполнения возвращает объекты users и groups, каждый из которых содержит поле count — количество результатов и items — список идентификаторов пользователей или публичных страниц, на которые подписан текущий пользователь  (из раздела «Интересные страницы»).  Если был задан параметр extended=1, возвращает объединенный массив объектов group и user в поле items, а также общее число результатов в поле count.
        """

        params = locals()
        params.pop('self')
        return self._call('users.getSubscriptions', params=params)

    def users_report(self, user_id=None, type=None, comment=None):
        """
        Позволяет пожаловаться на пользователя.

        :param user_id: идентификатор пользователя, на которого нужно подать жалобу. положительное число, обязательный параметр
        :param type: тип жалобы, может принимать следующие значения:   porn — порнография;  spam — рассылка спама;  insult — оскорбительное поведение;  advertisеment — рекламная страница, засоряющая поиск.  строка, обязательный параметр
        :param comment: комментарий к жалобе на пользователя. строка
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('users.report', params=params)

    def users_search(self, q=None, sort=None, offset=None, count=None, fields=None, city=None, country=None,
                     hometown=None, university_country=None, university=None, university_year=None,
                     university_faculty=None, university_chair=None, sex=None, status=None, age_from=None, age_to=None,
                     birth_day=None, birth_month=None, birth_year=None, online=None, has_photo=None,
                     school_country=None, school_city=None, school_class=None, school=None, school_year=None,
                     religion=None, company=None, position=None, group_id=None, from_list=None):
        """
        Возвращает список пользователей в соответствии с заданным критерием поиска.

        :param q: строка поискового запроса. Например, Вася Бабич. строка
        :param sort: сортировка результатов. Возможные значения:   1 — по дате регистрации;  0 — по популярности.  целое число
        :param offset: смещение относительно первого найденного пользователя для выборки определенного подмножества. положительное число
        :param count: количество возвращаемых пользователей. Обратите внимание — даже при использовании параметра offset для получения информации доступны только первые 1000 результатов.   положительное число, по умолчанию 20, максимальное значение 1000
        :param fields: список дополнительных полей профилей, которые необходимо вернуть. См. подробное описание.  Доступные значения: photo_id, verified, sex, bdate, city, country, home_town, has_photo, photo_50, photo_100, photo_200_orig, photo_200, photo_400_orig, photo_max, photo_max_orig, online, lists, domain, has_mobile, contacts, site, education, universities, schools, status, last_seen, followers_count, common_count, occupation, nickname, relatives, relation, personal, connections, exports, wall_comments, activities, interests, music, movies, tv, books, games, about, quotes, can_post, can_see_all_posts, can_see_audio, can_write_private_message, can_send_friend_request, is_favorite, is_hidden_from_feed, timezone, screen_name, maiden_name, crop_photo, is_friend, friend_status, career, military, blacklisted, blacklisted_by_me список слов, разделенных через запятую
        :param city: идентификатор города. положительное число
        :param country: идентификатор страны. положительное число
        :param hometown: название города строкой. строка
        :param university_country: идентификатор страны, в которой пользователи закончили ВУЗ. положительное число
        :param university: идентификатор ВУЗа. положительное число
        :param university_year: год окончания ВУЗа. положительное число
        :param university_faculty: идентификатор факультета. положительное число
        :param university_chair: идентификатор кафедры. положительное число
        :param sex: пол. Возможные значения:   1 —  женщина;  2 — мужчина;  0 — любой (по умолчанию).  положительное число
        :param status: семейное положение. Возможные значения:   1 — не женат (не замужем);  2 — встречается;  3 — помолвлен(-а);  4 — женат (замужем);  5 — всё сложно;  6 — в активном поиске;  7 — влюблен(-а);  8 — в гражданском браке.  положительное число
        :param age_from: возраст, от. положительное число
        :param age_to: возраст, до. положительное число
        :param birth_day: день рождения. положительное число
        :param birth_month: месяц рождения. положительное число
        :param birth_year: год рождения. положительное число, минимальное значение 1900, максимальное значение 2100
        :param online: учитывать ли статус «онлайн». Возможные значения:   1 — искать только пользователей онлайн;  0 — искать по всем пользователям.  флаг, может принимать значения 1 или 0
        :param has_photo: учитывать ли наличие фото. Возможные значения:   1 — искать только пользователей с фотографией;  0 — искать по всем пользователям.  флаг, может принимать значения 1 или 0
        :param school_country: идентификатор страны, в которой пользователи закончили школу. положительное число
        :param school_city: идентификатор города, в котором пользователи закончили школу. положительное число
        :param school_class: буква класса. положительное число
        :param school: идентификатор школы, которую закончили пользователи. положительное число
        :param school_year: год окончания школы. положительное число
        :param religion: религиозные взгляды. строка
        :param company: название компании, в которой работают пользователи. строка
        :param position: название должности. строка
        :param group_id: идентификатор группы, среди пользователей которой необходимо проводить поиск. положительное число
        :param from_list: Разделы среди которых нужно осуществить поиск, перечисленные через запятую. Возможные значения:   friends — искать среди друзей;  subscriptions — искать среди друзей и подписок пользователя.  список слов, разделенных через запятую
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов, описывающих пользователей в поле items.
        """

        params = locals()
        params.pop('self')
        return self._call('users.search', params=params)

    def utils_checkLink(self, url=None):
        """
        Возвращает информацию о том, является ли внешняя ссылка заблокированной на сайте ВКонтакте.

        :param url: внешняя ссылка, которую необходимо проверить.  Например, http://google.com строка, обязательный параметр
        :return: После успешного выполнения возвращает объект, который содержит следующие поля:    status (string) — статус ссылки. Возможные значения:   not_banned – ссылка не заблокирована;  banned – ссылка заблокирована;  processing – ссылка проверяется, необходимо выполнить повторный запрос через несколько секунд.   link (string) — исходная ссылка (url) либо полная ссылка (если в url была передана сокращенная ссылка).
        """

        params = locals()
        params.pop('self')
        return self._call('utils.checkLink', params=params)

    def utils_deleteFromLastShortened(self, key=None):
        """
        Удаляет сокращенную ссылку из списка пользователя.

        :param key: сокращенная ссылка (часть URL после "vk.cc/"). строка, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('utils.deleteFromLastShortened', params=params)

    def utils_getLastShortenedLinks(self, count=None, offset=None):
        """
        Получает список сокращенных ссылок для текущего пользователя.

        :param count: количество ссылок, которые необходимо получить. положительное число, по умолчанию 10
        :param offset: сдвиг для получения определенного подмножества ссылок. положительное число, по умолчанию 0
        :return: Возвращает количество ссылок в поле count (integer) и массив объектов items, описывающих ссылки. Каждый из объектов содержит следующие поля:   timestamp (integer) — время создания ссылки в Unixtime;  url (string) — URL ссылки до сокращения;  short_url (string) — сокращенный URL;  key (string) — содержательная часть (символы после "vk.cc");  views (integer) — число переходов;  access_key (string) — ключ доступа к приватной статистике.
        """

        params = locals()
        params.pop('self')
        return self._call('utils.getLastShortenedLinks', params=params)

    def utils_getLinkStats(self, key=None, source=None, access_key=None, interval=None, intervals_count=None,
                           extended=None):
        """
        Возвращает статистику переходов по сокращенной ссылке.

        :param key: сокращенная ссылка (часть URL после "vk.cc/"). строка, обязательный параметр
        :param source: строка, по умолчанию vk_cc
        :param access_key: ключ доступа к приватной статистике ссылки. строка
        :param interval: единица времени для подсчета статистики. Возможные значения:   hour — час;  day — день;  week — неделя;  month — месяц;  forever — все время с момента создания ссылки.  строка, по умолчанию day
        :param intervals_count: длительность периода для получения статистики в выбранных единицах (из параметра interval). положительное число, по умолчанию 1, максимальное значение 100
        :param extended: 1 — возвращать расширенную статистику (пол/возраст/страна/город), 0 — возвращать только количество переходов. флаг, может принимать значения 1 или 0, по умолчанию
        :return: Возвращает объект, содержащий поле key с входным параметром key и массив stats с данными о статистике. Каждый объект в массиве stats содержит поле timestamp с указанием времени начала отсчета, общее число переходов views и массивы, описывающие половозрастную/геостатистику: sex_age, countries, cities.  Каждый объект в массиве sex_age содержит следующие поля:   age_range (string) — обозначение возраста;  female (integer) — число переходов пользователей женского пола;  male (integer) — число переходов пользователей мужского пола;   Каждый объект в массиве countries (cities) содержит следующие поля:   country_id (city_id) (integer) — идентификатор страны (города);  views (integer) — число переходов из этой страны (города).
        """

        params = locals()
        params.pop('self')
        return self._call('utils.getLinkStats', params=params)

    def utils_getServerTime(self):
        """
        Возвращает текущее время на сервере ВКонтакте в unixtime.

        :return:    В ходе выполнения могут произойти общие ошибки. Их описание находится на отдельной странице.
        """

        params = locals()
        params.pop('self')
        return self._call('utils.getServerTime', params=params)

    def utils_getShortLink(self, url=None, private=None):
        """
        Позволяет получить URL, сокращенный с помощью vk.cc.

        :param url: URL, для которого необходимо получить сокращенный вариант. строка, обязательный параметр
        :param private: 1 — статистика ссылки приватная. 0 — статистика ссылки общедоступная. флаг, может принимать значения 1 или 0, по умолчанию
        :return: Возвращает объект, содержащий следующие поля:   short_url (string) — сокращённая ссылка.  access_key (string) — ключ для доступа к приватной статистике ссылки;  key (string) — содержательная часть ссылки (после "vk.cc");  url (string) — оригинальный URL.
        """

        params = locals()
        params.pop('self')
        return self._call('utils.getShortLink', params=params)

    def utils_resolveScreenName(self, screen_name=None):
        """
        Определяет тип объекта (пользователь, сообщество, приложение) и его идентификатор по короткому имени screen_name.

        :param screen_name: короткое имя пользователя, группы или приложения. Например, apiclub, andrew или rules_of_war. строка, обязательный параметр
        :return: После успешного выполнения возвращает объект, который содержит следующие поля:   type (string) — тип объекта. Возможные значения:   user — пользователь;  group — сообщество;  application — приложение.   object_id (integer)— идентификатор объекта.   Если короткое имя screen_name не занято, то будет возвращён пустой объект.
        """

        params = locals()
        params.pop('self')
        return self._call('utils.resolveScreenName', params=params)

    def video_add(self, target_id=None, video_id=None, owner_id=None):
        """
        Добавляет видеозапись в список пользователя.

        :param target_id: идентификатор пользователя или сообщества, в которое нужно добавить видео.  Обратите внимание, идентификатор сообщества в параметре target_id необходимо указывать со знаком "-" — например, target_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число
        :param video_id: идентификатор видеозаписи. положительное число, обязательный параметр
        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит видеозапись. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('video.add', params=params)

    def video_addAlbum(self, group_id=None, title=None, privacy=None):
        """
        Создает пустой альбом видеозаписей.

        :param group_id: идентификатор сообщества (если необходимо создать альбом в сообществе). положительное число
        :param title: название альбома. строка
        :param privacy: настройки доступа к альбому в специальном формате.  Приватность доступна для альбомов с видео в профиле пользователя. список слов, разделенных через запятую
        :return: После успешного выполнения возвращает  идентификатор созданного альбома в поле album_id.
        """

        params = locals()
        params.pop('self')
        return self._call('video.addAlbum', params=params)

    def video_addToAlbum(self, target_id=None, album_id=None, album_ids=None, owner_id=None, video_id=None):
        """
        Позволяет добавить видеозапись в альбом.

        :param target_id: идентификатор владельца альбома, в который нужно добавить видео.  Обратите внимание, идентификатор сообщества в параметре target_id необходимо указывать со знаком "-" — например, target_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param album_id: идентификатор альбома, в который нужно добавить видео.  Для добавления видео в общий альбом «Добавленные» передавайте -2. целое число
        :param album_ids: идентификаторы альбомов, в которые нужно добавить видео. список целых чисел, разделенных запятыми
        :param owner_id: идентификатор владельца видеозаписи.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param video_id: идентификатор видеозаписи. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('video.addToAlbum', params=params)

    def video_createComment(self, owner_id=None, video_id=None, message=None, attachments=None, from_group=None,
                            reply_to_comment=None, sticker_id=None, guid=None):
        """
        Cоздает новый комментарий к видеозаписи

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит видеозапись. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param video_id: идентификатор видеозаписи. целое число, обязательный параметр
        :param message: текст комментария. Обязательный параметр, если не задан параметр attachments. строка
        :param attachments: список объектов, приложенных к комментарию и разделённых символом ",". Поле attachments представляется в формате: <type><owner_id>_<media_id>,<type><owner_id>_<media_id> <type> — тип медиа-вложения: photo — фотография  video — видеозапись  audio — аудиозапись  doc — документ <owner_id> — идентификатор владельца медиа-вложения  <media_id> — идентификатор медиа-вложения.   Например: photo100172_166443618,photo66748_265827614 Параметр является обязательным, если не задан параметр message. список слов, разделенных через запятую
        :param from_group: этот параметр учитывается, если owner_id < 0 (комментарий к видеозаписи группы). 1 — комментарий будет опубликован от имени группы, 0 — комментарий будет опубликован от имени пользователя. по умолчанию: 0. флаг, может принимать значения 1 или 0
        :param reply_to_comment: идентификатор комментария, в ответ на который должен быть добавлен новый комментарий. положительное число
        :param sticker_id: идентификатор стикера. положительное число
        :param guid: уникальный идентификатор, предназначенный для предотвращения повторной отправки одинакового комментария. строка
        :return: После успешного выполнения возвращает идентификатор созданного комментария.
        """

        params = locals()
        params.pop('self')
        return self._call('video.createComment', params=params)

    def video_delete(self, video_id=None, owner_id=None, target_id=None):
        """
        Удаляет видеозапись со страницы пользователя.

        :param video_id: идентификатор видеозаписи. положительное число, обязательный параметр
        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит видеозапись. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param target_id: идентификатор пользователя или сообщества, для которого нужно удалить видеозапись.  Обратите внимание, идентификатор сообщества в параметре target_id необходимо указывать со знаком "-" — например, target_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('video.delete', params=params)

    def video_deleteAlbum(self, group_id=None, album_id=None):
        """
        Удаляет альбом видеозаписей.

        :param group_id: идентификатор сообщества (если альбом, который необходимо удалить, принадлежит сообществу). положительное число
        :param album_id: идентификатор альбома. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('video.deleteAlbum', params=params)

    def video_deleteComment(self, owner_id=None, comment_id=None):
        """
        Удаляет комментарий к видеозаписи.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит видеозапись. целое число, по умолчанию идентификатор текущего пользователя
        :param comment_id: идентификатор комментария. целое число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('video.deleteComment', params=params)

    def video_edit(self, owner_id=None, video_id=None, name=None, desc=None, privacy_view=None, privacy_comment=None,
                   no_comments=None, repeat=None):
        """
        Редактирует данные видеозаписи.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит видеозапись. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param video_id: идентификатор видеозаписи. целое число, обязательный параметр
        :param name: новое название для видеозаписи. строка
        :param desc: новое описание для видеозаписи. строка
        :param privacy_view: настройки приватности просмотра видеозаписи в специальном формате. Приватность доступна для видеозаписей, которые пользователь загрузил в профиль. список слов, разделенных через запятую
        :param privacy_comment: настройки приватности комментирования видеозаписи в специальном формате.  Приватность доступна для видеозаписей, которые пользователь загрузил в профиль. список слов, разделенных через запятую
        :param no_comments: закрыть комментарии (для видео из сообществ). флаг, может принимать значения 1 или 0
        :param repeat: зацикливание воспроизведения видеозаписи. флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('video.edit', params=params)

    def video_editAlbum(self, group_id=None, album_id=None, title=None, privacy=None):
        """
        Редактирует альбом с видео.

        :param group_id: идентификатор сообщества (если нужно отредактировать альбом, принадлежащий сообществу). положительное число
        :param album_id: идентификатор альбома. положительное число, обязательный параметр
        :param title: новое название для альбома. строка, обязательный параметр
        :param privacy: уровень доступа к альбому в специальном формате.  Приватность доступна для альбомов с видео в профиле пользователя. список слов, разделенных через запятую
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('video.editAlbum', params=params)

    def video_editComment(self, owner_id=None, comment_id=None, message=None, attachments=None):
        """
        Изменяет текст комментария к видеозаписи.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит видеозапись. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param comment_id: идентификатор комментария. целое число, обязательный параметр
        :param message: новый текст комментария. Обязательный параметр, если не задан параметр attachments. строка
        :param attachments: новый список объектов, приложенных к комментарию и разделённых символом ",". Поле attachments представляется в формате: <type><owner_id>_<media_id>,<type><owner_id>_<media_id> <type> — тип медиа-вложения: photo — фотография  video — видеозапись  audio — аудиозапись  doc — документ <owner_id> — идентификатор владельца медиа-вложения  <media_id> — идентификатор медиа-вложения.   Например: photo100172_166443618,photo66748_265827614 Обязательный параметр, если не задан параметр message. список слов, разделенных через запятую
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('video.editComment', params=params)

    def video_get(self, owner_id=None, videos=None, album_id=None, count=None, offset=None, extended=None):
        """
        Возвращает информацию о видеозаписях.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежат видеозаписи. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param videos: перечисленные через запятую идентификаторы — идущие через знак подчеркивания id пользователей, которым принадлежат видеозаписи, и id самих видеозаписей. Если видеозапись принадлежит сообществу, то в качестве первого параметра используется -id сообщества. Пример значения videos:  -4363_136089719,13245770_137352259  Некоторые видеозаписи, идентификаторы которых могут быть получены через API, закрыты приватностью, и не будут получены. В этом случае следует использовать ключ доступа access_key в её идентификаторе. Пример значения videos:   1_129207899_220df2876123d3542f, 6492_135055734_e0a9bcc31144f67fbd  Поле access_key будет возвращено вместе с остальными данными видеозаписи в методах, которые возвращают видеозаписи, закрытые приватностью, но доступные в данном контексте. Например, данное поле имеют видеозаписи, возвращаемые методом messages.get. список слов, разделенных через запятую
        :param album_id: идентификатор альбома, видеозаписи из которого нужно вернуть. целое число
        :param count: количество возвращаемых видеозаписей. положительное число, максимальное значение 200, по умолчанию 100
        :param offset: смещение относительно первой найденной видеозаписи для выборки определенного подмножества. положительное число
        :param extended: определяет, возвращать ли информацию о настройках приватности видео для текущего пользователя. флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов видеозаписей с дополнительным полем comments, содержащим число комментариев  у видеозаписи, в поле items.  Если был задан параметр extended=1, возвращаются дополнительные поля:   privacy_view — настройки приватности в формате настроек приватности; (приходит только для текущего пользователя)  privacy_comment — настройки приватности в формате настроек приватности; (приходит только для текущего пользователя)  can_comment — может ли текущий пользователь оставлять комментарии к ролику (1 — может, 0 — не может);  can_repost — может ли текущий пользователь скопировать ролик с помощью функции «Рассказать друзьям» (1 — может, 0 — не может);  likes — информация об отметках «Мне нравится»:   user_likes — есть ли отметка «Мне нравится» от текущего пользователя;  count — число отметок «Мне нравится»;   repeat — зацикливание воспроизведения видеозаписи (1 — зацикливается, 0 — не зацикливается).   Если в Вашем приложении используется  прямая авторизация, возвращается дополнительное поле files, содержащее ссылку на файл с видео (если ролик размещен на сервере ВКонтакте) или ссылку на внешний ресурс (если ролик встроен с какого-либо видеохостинга).
        """

        params = locals()
        params.pop('self')
        return self._call('video.get', params=params)

    def video_getAlbumById(self, owner_id=None, album_id=None):
        """
        Позволяет получить информацию об альбоме с видео.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит альбом. целое число, по умолчанию идентификатор текущего пользователя
        :param album_id: идентификатор альбома. целое число, обязательный параметр
        :return: После успешного выполнения возвращает объект, который содержит следующие поля:   id — идентификатор альбома;  owner_id — идентификатор владельца альбома;  title — название альбома;  count — число видеозаписей в альбоме;  photo_320 — url обложки альбома с шириной 320px;  photo_160 — url обложки альбома с шириной 160px;  updated_time — время последнего обновления в формате unixtime.
        """

        params = locals()
        params.pop('self')
        return self._call('video.getAlbumById', params=params)

    def video_getAlbums(self, owner_id=None, offset=None, count=None, extended=None, need_system=None):
        """
        Возвращает список альбомов видеозаписей пользователя или сообщества.

        :param owner_id: идентификатор владельца альбомов (пользователь или сообщество). По умолчанию — идентификатор текущего пользователя. целое число, по умолчанию идентификатор текущего пользователя
        :param offset: смещение, необходимое для выборки определенного подмножества альбомов. По умолчанию: 0. положительное число
        :param count: количество альбомов, информацию о которых нужно вернуть. По умолчанию: не больше 50, максимальное значение: 100. положительное число, по умолчанию 50, максимальное значение 100
        :param extended: 1— возвращать дополнительные поля count, photo_320, photo_160 и updated_time для каждого альбома. Если альбом пустой, то поля photo_320 и photo_160 возвращены не будут. По умолчанию: 0. флаг, может принимать значения 1 или 0
        :param need_system: 1 — возвращать системные альбомы. флаг, может принимать значения 1 или 0, по умолчанию 0
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов альбомов в поле items. Каждый из объектов, описывающих альбом, содержит следующие поля:   id — идентификатор альбома;  owner_id — идентификатор владельца альбома;  title — название альбома.
        """

        params = locals()
        params.pop('self')
        return self._call('video.getAlbums', params=params)

    def video_getAlbumsByVideo(self, target_id=None, owner_id=None, video_id=None, extended=None):
        """
        Возвращает список альбомов, в которых находится видеозапись.

        :param target_id: идентификатор пользователя или сообщества, для которого нужно получить список альбомов, содержащих видеозапись.  Обратите внимание, идентификатор сообщества в параметре target_id необходимо указывать со знаком "-" — например, target_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param owner_id: идентификатор владельца видеозаписи.  Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param video_id: идентификатор видеозаписи. положительное число, обязательный параметр
        :param extended: 1 — возвращать расширенную информацию об альбомах. флаг, может принимать значения 1 или 0, по умолчанию 0
        :return: Возвращает список идентификаторов альбомов, в которых видеозапись находится у пользователя или сообщества с идентификатором target_id. Если был передан параметр extended=1, возвращается список объектов альбомов с дополнительной информацией о каждом из них.
        """

        params = locals()
        params.pop('self')
        return self._call('video.getAlbumsByVideo', params=params)

    def video_getComments(self, owner_id=None, video_id=None, need_likes=None, start_comment_id=None, offset=None,
                          count=None, sort=None, extended=None, fields=None):
        """
        Возвращает список комментариев к видеозаписи.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит видеозапись. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param video_id: идентификатор видеозаписи. положительное число, обязательный параметр
        :param need_likes: 1 — будет возвращено дополнительное поле likes. По умолчанию поле likes не возвращается. флаг, может принимать значения 1 или 0
        :param start_comment_id: идентификатор комментария, начиная с которого нужно вернуть список (подробности см. ниже). положительное число, доступен начиная с версии 5.33
        :param offset: смещение, необходимое для выборки определенного подмножества комментариев. По умолчанию: 0. целое число
        :param count: количество комментариев, информацию о которых необходимо вернуть. положительное число, по умолчанию 20, максимальное значение 100
        :param sort: порядок сортировки комментариев. Возможные значения:   asc — от старых к новым;  desc — от новых к старым.  строка
        :param extended: 1 — в ответе будут возвращены дополнительные поля profiles и groups, содержащие информацию о пользователях и сообществах. По умолчанию: 0. флаг, может принимать значения 1 или 0
        :param fields: список слов, разделенных через запятую
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов комментариев в поле items. Каждый объект, описывающий комментарий, содержит следующие поля:   id — идентификатор комментария;  from_id — идентификатор автора комментария;  date — дата добавления комментария в формате unixtime;  text — текст комментария;  likes — информация об отметках «Мне нравится» текущего комментария (если был задан параметр need_likes), объект с полями:   count — число пользователей, которым понравился комментарий,  user_likes — наличие отметки «Мне нравится» от текущего пользователя  (1 — есть, 0 — нет),  can_like — информация о том, может ли текущий пользователь поставить отметку «Мне нравится»  (1 — может, 0 — не может).    Если был задан параметр extended=1, возвращает число результатов в поле count, отдельно массив объектов комментариев в поле items, пользователей в поле profiles и сообществ в поле groups.  Если был передан параметр start_comment_id, будет также возвращено поле real_offset – итоговое смещение данного подмножества комментариев (оно может быть отрицательным, если был указан отрицательный offset).
        """

        params = locals()
        params.pop('self')
        return self._call('video.getComments', params=params)

    def video_removeFromAlbum(self, target_id=None, album_id=None, album_ids=None, owner_id=None, video_id=None):
        """
        Позволяет убрать видеозапись из альбома.

        :param target_id: идентификатор владельца альбома. Обратите внимание, идентификатор сообщества в параметре target_id необходимо указывать со знаком "-" — например, target_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param album_id: идентификатор альбома, из которого нужно убрать видео. целое число
        :param album_ids: идентификаторы альбомов, из которых нужно убрать видео. список целых чисел, разделенных запятыми
        :param owner_id: идентификатор владельца видеозаписи. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, обязательный параметр
        :param video_id: идентификатор видеозаписи. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('video.removeFromAlbum', params=params)

    def video_reorderAlbums(self, owner_id=None, album_id=None, before=None, after=None):
        """
        Позволяет изменить порядок альбомов с видео.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит альбом. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param album_id: идентификатор альбома, который нужно переместить. положительное число, обязательный параметр
        :param before: идентификатор альбома, перед которым нужно поместить текущий. положительное число
        :param after: идентификатор альбома, после которого нужно поместить текущий. положительное число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('video.reorderAlbums', params=params)

    def video_reorderVideos(self, target_id=None, album_id=None, owner_id=None, video_id=None, before_owner_id=None,
                            before_video_id=None, after_owner_id=None, after_video_id=None):
        """
        Позволяет переместить видеозапись в альбоме.

        :param target_id: идентификатор пользователя или сообщества, в чьем альбоме нужно переместить видео. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param album_id: идентификатор альбома с видеозаписью, которую нужно переместить. целое число
        :param owner_id: идентификатор владельца видеозаписи, которую нужно переместить (пользователь или сообщество). целое число, обязательный параметр
        :param video_id: идентификатор видеозаписи, которую нужно переместить. положительное число, обязательный параметр
        :param before_owner_id: идентификатор владельца видеозаписи, перед которой следует поместить текущую (пользователь или сообщество). целое число
        :param before_video_id: идентификатор видеозаписи, перед которой следует поместить текущую. положительное число
        :param after_owner_id: идентификатор владельца видеозаписи, после которой следует поместить текущую (пользователь или сообщество). целое число
        :param after_video_id: идентификатор видеозаписи, после которой следует поместить текущую. положительное число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('video.reorderVideos', params=params)

    def video_report(self, owner_id=None, video_id=None, reason=None, comment=None, search_query=None):
        """
        Позволяет пожаловаться на видеозапись.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит видеозапись. целое число, обязательный параметр
        :param video_id: идентификатор видеозаписи. положительное число, обязательный параметр
        :param reason: тип жалобы. Возможные значения:   0 — это спам;  1 — детская порнография;  2 — экстремизм;  3 — насилие;  4 — пропаганда наркотиков;  5 — материал для взрослых;  6 — оскорбление.  положительное число
        :param comment: комментарий для жалобы. строка
        :param search_query: поисковой запрос, если видеозапись была найдена через поиск. строка
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('video.report', params=params)

    def video_reportComment(self, owner_id=None, comment_id=None, reason=None):
        """
        Позволяет пожаловаться на комментарий к видеозаписи.

        :param owner_id: идентификатор владельца видеозаписи, к которой оставлен комментарий. целое число, обязательный параметр
        :param comment_id: идентификатор комментария. положительное число, обязательный параметр
        :param reason: тип жалобы. Возможные значения:   0 — это спам;  1 — детская порнография;  2 — экстремизм;  3 — насилие;  4 — пропаганда наркотиков;  5 — материал для взрослых;  6 — оскорбление.  положительное число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('video.reportComment', params=params)

    def video_restore(self, video_id=None, owner_id=None):
        """
        Восстанавливает удаленную видеозапись.

        :param video_id: идентификатор видеозаписи. положительное число, обязательный параметр
        :param owner_id: идентификатор владельца видеозаписи (пользователь или сообщество). Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('video.restore', params=params)

    def video_restoreComment(self, owner_id=None, comment_id=None):
        """
        Восстанавливает удаленный комментарий к видеозаписи.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит видеозапись. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param comment_id: идентификатор удаленного комментария. целое число, обязательный параметр
        :return: После успешного выполнения возвращает 1 (0, если комментарий с таким идентификатором не является удаленным).
        """

        params = locals()
        params.pop('self')
        return self._call('video.restoreComment', params=params)

    def video_save(self, name=None, description=None, is_private=None, wallpost=None, link=None, group_id=None,
                   album_id=None, privacy_view=None, privacy_comment=None, no_comments=None, repeat=None,
                   compression=None):
        """
        Возвращает адрес сервера, необходимый для загрузки, и данные видеозаписи.

        :param name: название видеофайла. строка
        :param description: описание видеофайла. строка
        :param is_private: указывается 1, если видео загружается для отправки личным сообщением. После загрузки с этим параметром видеозапись не будет отображаться в списке видеозаписей пользователя и не будет доступна другим пользователям по ее идентификатору. По умолчанию: 0. флаг, может принимать значения 1 или 0
        :param wallpost: требуется ли после сохранения опубликовать запись с видео на стене (1 — требуется, 0 — не требуется).  Обратите внимание, для публикации записи на стене приложение должно иметь права wall. флаг, может принимать значения 1 или 0
        :param link: url для встраивания видео с внешнего сайта, например, с Youtube. В этом случае нужно вызвать полученный upload_url, не прикрепляя файл, достаточно просто обратиться по этому адресу. строка
        :param group_id: идентификатор сообщества, в которое будет сохранен видеофайл. По умолчанию файл сохраняется на страницу текущего пользователя. положительное число
        :param album_id: идентификатор альбома, в который будет загружен видео файл. положительное число
        :param privacy_view: настройки приватности просмотра видеозаписи в специальном формате. Приватность доступна для видеозаписей, которые пользователь загрузил в профиль. список слов, разделенных через запятую
        :param privacy_comment: настройки приватности комментирования видеозаписи в специальном формате.  Приватность доступна для видеозаписей, которые пользователь загрузил в профиль. список слов, разделенных через запятую
        :param no_comments: 1 — закрыть комментарии (для видео из сообществ). По умолчанию: 0. флаг, может принимать значения 1 или 0
        :param repeat: зацикливание воспроизведения видеозаписи. флаг, может принимать значения 1 или 0
        :param compression: флаг, может принимать значения 1 или 0
        :return: Возвращает объект, который имеет поля upload_url, video_id, title, description, owner_id.  Метод может быть вызван не более 5000 раз в сутки для одного сервиса.
        """

        params = locals()
        params.pop('self')
        return self._call('video.save', params=params)

    def video_search(self, q=None, sort=None, hd=None, adult=None, filters=None, search_own=None, offset=None,
                     longer=None, shorter=None, count=None, extended=None):
        """
        Возвращает список видеозаписей в соответствии с заданным критерием поиска.

        :param q: строка поискового запроса. Например, The Beatles. строка, обязательный параметр
        :param sort: сортировка результатов. Возможные значения:   0 — по дате добавления видеозаписи;  1 — по длительности;  2 — по релевантности.  целое число
        :param hd: если не равен нулю, то поиск производится только по видеозаписям высокого качества. целое число
        :param adult: фильтр «Безопасный поиск». Возможные значения:   1 — выключен;  0 — включен.  флаг, может принимать значения 1 или 0
        :param filters: список критериев, по которым требуется отфильтровать видео:   mp4 — искать только видео в формате mp4 (воспроизводимое на iOS);  youtube — возвращать только youtube видео;  vimeo — возвращать только vimeo видео;  short — возвращать только короткие видеозаписи;  long — возвращать только длинные видеозаписи.  список слов, разделенных через запятую
        :param search_own: 1 — искать по видеозаписям пользователя, 0 — не искать по видеозаписям пользователя. По умолчанию: 0. флаг, может принимать значения 1 или 0
        :param offset: смещение относительно первой найденной видеозаписи для выборки определенного подмножества. положительное число
        :param longer: количество секунд, видеозаписи длиннее которого необходимо вернуть. положительное число
        :param shorter: количество секунд, видеозаписи короче которого необходимо вернуть. положительное число
        :param count: количество возвращаемых видеозаписей. Обратите внимание — даже при использовании параметра offset для получения информации доступны только первые 1000 результатов.   положительное число, по умолчанию 20, максимальное значение 200
        :param extended: 1 — в ответе будут возвращены дополнительные поля profiles и groups, содержащие информацию о пользователях и сообществах. По умолчанию: 0. флаг, может принимать значения 1 или 0, по умолчанию 0
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов видеозаписей в поле items.  Если был задан параметр extended=1, возвращает число результатов в поле count, отдельно массив объектов видеозаписей в поле items, пользователей в поле profiles и сообществ в поле groups.  Если в Вашем приложении используется  прямая авторизация, возвращается дополнительное поле files, содержащее ссылку на файл с видео (если ролик размещен на сервере ВКонтакте) или ссылку на внешний ресурс (если ролик встроен с какого-либо видеохостинга).
        """

        params = locals()
        params.pop('self')
        return self._call('video.search', params=params)

    def wall_checkCopyrightLink(self, link=None):
        """
        Проверяет ссылку для указания источника.

        :param link: ссылка на источник. Поддерживаются внешние и внутренние ссылки. строка, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('wall.checkCopyrightLink', params=params)

    def wall_closeComments(self, owner_id=None, post_id=None):
        """
        Выключает комментирование записи

        :param owner_id: целое число, обязательный параметр
        :param post_id: положительное число, обязательный параметр
        :return: Возвращает 1 в случае успеха
        """

        params = locals()
        params.pop('self')
        return self._call('wall.closeComments', params=params)

    def wall_createComment(self, owner_id=None, post_id=None, from_group=None, message=None, reply_to_comment=None,
                           attachments=None, sticker_id=None, guid=None):
        """
        Добавляет комментарий к записи на стене.

        :param owner_id: идентификатор пользователя или сообщества, на чьей стене находится запись, к которой необходимо добавить комментарий. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param post_id: идентификатор записи на стене. положительное число, обязательный параметр
        :param from_group: идентификатор сообщества, от имени которого публикуется комментарий. По умолчанию: 0. положительное число
        :param message: текст комментария. Обязательный параметр, если не передан параметр attachments. строка, строка
        :param reply_to_comment: идентификатор комментария, в ответ на который должен быть добавлен новый комментарий. целое число
        :param attachments: список объектов, приложенных к комментарию и разделённых символом ",". Поле attachments представляется в формате: <type><owner_id>_<media_id>,<type><owner_id>_<media_id> <type> — тип медиа-вложения: photo — фотография  video — видеозапись  audio — аудиозапись  doc — документ <owner_id> — идентификатор владельца медиа-вложения  <media_id> — идентификатор медиа-вложения.   Например: photo100172_166443618,photo66748_265827614 Параметр является обязательным, если не задан параметр message. список слов, разделенных через запятую
        :param sticker_id: идентификатор стикера. положительное число
        :param guid: уникальный идентификатор, предназначенный для предотвращения повторной отправки одинакового комментария. строка
        :return: После успешного выполнения возвращает идентификатор добавленного комментария в поле comment_id (integer) и массив идентификаторов родительских комментариев в поле parent_stack (array).
        """

        params = locals()
        params.pop('self')
        return self._call('wall.createComment', params=params)

    def wall_delete(self, owner_id=None, post_id=None):
        """
        Удаляет запись со стены.

        :param owner_id: идентификатор пользователя или сообщества, на стене которого находится запись. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param post_id: идентификатор записи на стене. положительное число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('wall.delete', params=params)

    def wall_deleteComment(self, owner_id=None, comment_id=None):
        """
        Удаляет комментарий к записи на стене.

        :param owner_id: идентификатор пользователя, на чьей стене находится комментарий к записи. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param comment_id: идентификатор комментария. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('wall.deleteComment', params=params)

    def wall_edit(self, owner_id=None, post_id=None, friends_only=None, message=None, attachments=None, services=None,
                  signed=None, publish_date=None, lat=None, long=None, place_id=None, mark_as_ads=None,
                  close_comments=None, poster_bkg_id=None, poster_bkg_owner_id=None, poster_bkg_access_hash=None,
                  copyright=None):
        """
        Редактирует запись на стене.

        :param owner_id: идентификатор пользователя или сообщества, на стене которого находится запись. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param post_id: идентификатор записи на стене. положительное число, обязательный параметр
        :param friends_only: 1 — запись будет доступна только друзьям, 0 — всем пользователям.  Параметр учитывается только при редактировании отложенной записи. флаг, может принимать значения 1 или 0
        :param message: текст сообщения (является обязательным, если не задан параметр attachments) строка
        :param attachments: список объектов, приложенных к записи и разделённых символом ",". Поле attachments представляется в формате: <type><owner_id>_<media_id>,<type><owner_id>_<media_id>  <type> — тип медиа-приложения:    photo — фотография  video — видеозапись  audio — аудиозапись  doc — документ  graffiti — граффити  page — wiki-страница  note — заметка  poll — опрос  album — альбом.   <owner_id> — идентификатор владельца медиа-приложения; <media_id> — идентификатор медиа-приложения.  Например: photo100172_166443618,photo66748_265827614  Также в поле attachments может быть указана ссылка на внешнюю страницу, которую Вы хотите разместить в статусе, например: photo66748_265827614,http://habrahabr.ru  При попытке приложить больше одной ссылки будет возвращена ошибка.   Параметр является обязательным, если не задан параметр message. список слов, разделенных через запятую
        :param services: список сервисов или сайтов, на которые необходимо экспортировать запись, в случае если пользователь настроил соответствующую опцию. Например, twitter, facebook.  Параметр учитывается только при редактировании отложенной записи. строка
        :param signed: 1 — у записи, размещенной от имени сообщества будет добавлена подпись (имя пользователя, разместившего запись), 0 — подписи добавлено не будет.  Параметр учитывается только при редактировании записи на стене сообщества, опубликованной от имени группы. флаг, может принимать значения 1 или 0
        :param publish_date: дата публикации записи в формате unixtime. Если параметр не указан, отложенная запись будет опубликована.  Параметр учитывается только при редактировании отложенной записи. положительное число
        :param lat: географическая широта отметки, заданная в градусах (от -90 до 90). дробное число
        :param long: географическая долгота отметки, заданная в градусах (от -180 до 180). дробное число
        :param place_id: идентификатор места, в котором отмечен пользователь. положительное число
        :param mark_as_ads: 1 — у записи, размещенной от имени сообщества, будет добавлена метка "это реклама", 0 — метки добавлено не будет/снять установленную метку. Метка может быть снята в течение пяти минут после её установки. В сутки может быть опубликовано не более пяти рекламных записей, из которых не более трёх — вне Биржи ВКонтакте. флаг, может принимать значения 1 или 0
        :param close_comments: 1 — комментарии к записи отключены.  0 — комментарии к записи включены. флаг, может принимать значения 1 или 0
        :param poster_bkg_id: положительное число
        :param poster_bkg_owner_id: целое число
        :param poster_bkg_access_hash: строка
        :param copyright: источник материала. Поддерживаются внешние и внутренние ссылки. строка
        :return: После успешного выполнения возвращает post_id — идентификатор отредактированного поста.
        """

        params = locals()
        params.pop('self')
        return self._call('wall.edit', params=params)

    def wall_editAdsStealth(self, owner_id=None, post_id=None, message=None, attachments=None, signed=None, lat=None,
                            long=None, place_id=None, link_button=None, link_title=None, link_image=None,
                            link_video=None):
        """
        Позволяет отредактировать скрытую запись. Создание скрытых записей возможно только в сообществах от имени группы, публичной страницы или мероприятия; пользователь должен обладать правами администратора или редактора.

        :param owner_id: идентификатор владельца стены (идентификатор сообщества нужно указывать со знаком «минус»). целое число, по умолчанию идентификатор текущего пользователя
        :param post_id: идентификатор записи. положительное число, обязательный параметр
        :param message: текст записи. строка
        :param attachments: список объектов, приложенных к записи и разделённых символом ",". Поле attachments представляется в формате: <type><owner_id>_<media_id>,<type><owner_id>_<media_id>   <type> — тип медиа-приложения:   photo — фотография;  video — видеозапись ;  audio — аудиозапись;  doc — документ;  page — wiki-страница;  note — заметка;  poll — опрос;  album — альбом;  market — товар;  market_album — подборка товаров.   <owner_id> — идентификатор владельца медиа-приложения (обратите внимание, если объект находится в сообществе, этот параметр должен быть отрицательным).  <media_id> — идентификатор медиа-приложения.   Например: photo100172_166443618,photo-1_265827614  Также в поле attachments может быть указана ссылка на внешнюю страницу, которую Вы хотите разместить в записи, например: photo66748_265827614,http://habrahabr.ru  В качестве ссылки может быть использован телефонный номер  в формате tel:+71234567890  При попытке приложить больше одной ссылки будет возвращена ошибка.   Параметр является обязательным, если не задан параметр message. список слов, разделенных через запятую
        :param signed: 1 — у записи, размещенной от имени сообщества, будет добавлена подпись (имя пользователя, разместившего запись), 0 — без подписи. флаг, может принимать значения 1 или 0
        :param lat: географическая широта отметки, заданная в градусах (от -90 до 90). дробное число
        :param long: географическая долгота отметки, заданная в градусах (от -180 до 180). дробное число
        :param place_id: идентификатор места. положительное число
        :param link_button: идентификатор кнопки, которую необходимо добавить к сниппету для ссылки. Подробнее см. документацию метода wall.postAdsStealth. строка
        :param link_title: заголовок, который должен быть использован для сниппета. Если не указан, будет автоматически получен с целевой ссылки. Обязательно указывать в случае, если ссылка является номером телефона. строка
        :param link_image: ссылка на изображение, которое должно быть использовано для сниппета. Минимальное разрешение: 537x240. Если не указана, будет автоматически загружена с целевой ссылки. Обязательно указывать в случае, если ссылка является номером телефона. Одновременно может быть указан либо параметр link_image, либо параметр link_video. строка
        :param link_video: идентификатор видео в формате "<owner_id>_<media_id>". Одновременно может быть указан либо параметр link_image, либо параметр link_video. Кроме того, параметр link_video может быть указан только вместе с параметрами  link_button, link_title. строка
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('wall.editAdsStealth', params=params)

    def wall_editComment(self, owner_id=None, comment_id=None, message=None, attachments=None):
        """
        Редактирует комментарий на стене.

        :param owner_id: идентификатор владельца стены. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param comment_id: идентификатор комментария, который необходимо отредактировать. положительное число, обязательный параметр
        :param message: новый текст комментария. Обязательный параметр, если не передан параметр attachments. строка
        :param attachments: новые вложения к комментарию. Список объектов, приложенных к комментарию и разделённых символом ",". Поле attachments представляется в формате: <type><owner_id>_<media_id>,<type><owner_id>_<media_id> <type> — тип медиа-вложения: photo — фотография  video — видеозапись  audio — аудиозапись  doc — документ <owner_id> — идентификатор владельца медиа-вложения  <media_id> — идентификатор медиа-вложения.   Например: photo100172_166443618,photo66748_265827614 Параметр является обязательным, если не задан параметр message. список слов, разделенных через запятую
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('wall.editComment', params=params)

    def wall_get(self, owner_id=None, domain=None, offset=None, count=None, filter=None, extended=None, fields=None):
        """
        Возвращает список записей со стены пользователя или сообщества. Обратите внимание, для этого метода действуют специальные ограничения на число запросов. Подробнее.

        :param owner_id: идентификатор пользователя или сообщества, со стены которого необходимо получить записи (по умолчанию — текущий пользователь). Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число
        :param domain: короткий адрес пользователя или сообщества. строка
        :param offset: смещение, необходимое для выборки определенного подмножества записей. положительное число
        :param count: количество записей, которое необходимо получить. Максимальное значение: 100. положительное число
        :param filter: определяет, какие типы записей на стене необходимо получить. Возможные значения:   suggests — предложенные записи на стене сообщества (доступно только при вызове с передачей access_token);  postponed — отложенные записи (доступно только при вызове с передачей access_token);  owner — записи владельца стены;  others — записи не от владельца стены;  all — все записи на стене (owner + others).  По умолчанию: all. строка
        :param extended: 1 — в ответе будут возвращены дополнительные поля profiles и groups, содержащие информацию о пользователях и сообществах. По умолчанию: 0. флаг, может принимать значения 1 или 0
        :param fields: список дополнительных полей для профилей и сообществ, которые необходимо вернуть.  Обратите внимание, этот параметр учитывается только при extended=1. список слов, разделенных через запятую
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов записей на стене в поле items.  Если был задан параметр extended=1, возвращает число результатов в поле count, отдельно массив объектов записей на стене в поле items, пользователей в поле profiles и сообществ в поле groups.
        """

        params = locals()
        params.pop('self')
        return self._call('wall.get', params=params)

    def wall_getById(self, posts=None, extended=None, copy_history_depth=None, fields=None):
        """
        Возвращает список записей со стен пользователей или сообществ по их идентификаторам.

        :param posts: перечисленные через запятую идентификаторы, которые представляют собой идущие через знак подчеркивания id владельцев стен и id самих записей на стене. Максимум 100 идентификаторов. Пример значения posts: 93388_21539,93388_20904,-1_340364 список слов, разделенных через запятую, обязательный параметр
        :param extended: 1 — в ответе будут возвращены дополнительные поля profiles и groups, содержащие информацию о пользователях и сообществах. По умолчанию: 0. флаг, может принимать значения 1 или 0
        :param copy_history_depth: определяет размер массива copy_history, возвращаемого в ответе, если запись является репостом записи с другой стены.  Например, copy_history_depth=1 — copy_history будет содержать один элемент с информацией о записи, прямым репостом которой является текущая.  copy_history_depth=2 — copy_history будет содержать два элемента, добавляется информация о записи, репостом которой является первый элемент, и так далее (при условии, что иерархия репостов требуемой глубины для текущей записи существует). целое число, по умолчанию 2
        :param fields: список дополнительных полей для профилей и  групп, которые необходимо вернуть. См. описание полей объекта user и описание полей объекта group.  Обратите внимание, этот параметр учитывается только при extended=1. список слов, разделенных через запятую
        :return: После успешного выполнения возвращает объект, содержащий  массив объектов записей на стене.  Если был задан параметр extended=1, возвращает отдельно массив объектов записей на стене в поле items, пользователей в поле profiles и сообществ в поле groups.
        """

        params = locals()
        params.pop('self')
        return self._call('wall.getById', params=params)

    def wall_getComment(self, owner_id=None, comment_id=None, extended=None, fields=None):
        """
        Получает информацию о комментарии на стене.

        :param owner_id: идентификатор владельца стены (для сообществ — со знаком «минус»). целое число, по умолчанию идентификатор текущего пользователя
        :param comment_id: идентификатор комментария. положительное число, обязательный параметр
        :param extended: 1 — в ответе будут возвращены дополнительные поля profiles и groups, содержащие информацию о пользователях и сообществах. По умолчанию: 0. флаг, может принимать значения 1 или 0
        :param fields: список дополнительных полей для профилей и сообществ, которые необходимо вернуть.  Обратите внимание, этот параметр учитывается только при extended=1. список слов, разделенных через запятую
        :return: Возвращает объект комментария на стене. Если был передан параметр extended = 1, дополнительно возвращает поля groups (array) и profiles (array), содержащие информацию о пользователях и сообществах.
        """

        params = locals()
        params.pop('self')
        return self._call('wall.getComment', params=params)

    def wall_getComments(self, owner_id=None, post_id=None, need_likes=None, start_comment_id=None, offset=None,
                         count=None, sort=None, preview_length=None, extended=None, fields=None, comment_id=None,
                         thread_items_count=None):
        """
        Возвращает список комментариев к записи на стене.

        :param owner_id: идентификатор владельца страницы (пользователь или сообщество). Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param post_id: идентификатор записи на стене. положительное число
        :param need_likes: 1 — возвращать информацию о лайках. флаг, может принимать значения 1 или 0
        :param start_comment_id: идентификатор комментария, начиная с которого нужно вернуть список (подробности см. ниже). положительное число, доступен начиная с версии 5.33
        :param offset: сдвиг, необходимый для получения конкретной выборки результатов. целое число
        :param count: число комментариев, которые необходимо получить. По умолчанию: 10, максимальное значение: 100. положительное число
        :param sort: порядок сортировки комментариев. Возможные значения:   asc — от старых к новым;  desc — от новых к старым.  строка
        :param preview_length: количество символов, по которому нужно обрезать текст комментария. Укажите 0, если Вы не хотите обрезать текст. положительное число
        :param extended: 1 — в ответе будут возвращены дополнительные поля profiles и groups, содержащие информацию о пользователях и сообществах. По умолчанию: 0. флаг, может принимать значения 1 или 0
        :param fields: список дополнительных полей для профилей и сообществ, которые необходимо вернуть.  Обратите внимание, этот параметр учитывается только при extended=1. список слов, разделенных через запятую
        :param comment_id: идентификатор комментария, ветку которого нужно получить. положительное число, доступен начиная с версии 5.91
        :param thread_items_count: максимальное число элементов в поле thread. целое число, максимальное значение 10, минимальное значение 0, по умолчанию 0, доступен начиная с версии 5.91
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов комментариев в поле items, а также поля:   current_level_count (integer) — количество комментариев в ветке.  can_post (boolean) – может ли текущий пользователь оставлять комментарии в этой ветке.  show_reply_button (boolean) – нужно ли отображать кнопку «ответить» в ветке.  groups_can_post (boolean)  – могут ли сообщества оставлять комментарии в ветке.   Если был задан параметр need_likes=1, у объектов комментариев возвращается дополнительное поле likes, содержащее объект с полями:   count — число пользователей, которым понравился комментарий;  user_likes — наличие отметки «Мне нравится» от текущего пользователя  (1 — есть, 0 — нет);  can_like — информация о том, может ли текущий пользователь поставить отметку «Мне нравится»  (1 — может, 0 — не может).   Если был передан параметр start_comment_id, будет также возвращено поле real_offset – итоговое смещение данного подмножества комментариев (оно может быть отрицательным, если был указан отрицательный offset).  Если был задан параметр extended=1, возвращает число результатов в поле count, отдельно массив объектов комментариев в поле items, пользователей в поле profiles и сообществ в поле groups.
        """

        params = locals()
        params.pop('self')
        return self._call('wall.getComments', params=params)

    def wall_getReposts(self, owner_id=None, post_id=None, offset=None, count=None):
        """
        Позволяет получать список репостов заданной записи. Обратите внимание, получить список репостов можно только для записи, созданной текущим пользователем, или в сообществе, где текущий пользователь является администратором.

        :param owner_id: идентификатор пользователя или сообщества, на стене которого находится запись. Если параметр не задан, то он считается равным идентификатору текущего пользователя. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param post_id: идентификатор записи на стене. положительное число
        :param offset: смещение, необходимое для выборки определенного подмножества записей. положительное число
        :param count: количество записей, которое необходимо получить. положительное число, по умолчанию 20, максимальное значение 1000
        :return: После успешного выполнения возвращает объект, содержащий поля:   items — содержит массив записей-репостов;  profiles — содержит объекты пользователей с дополнительными полями sex, online, photo, photo_medium_rec, screen_name;  groups — содержит информацию о сообществах.
        """

        params = locals()
        params.pop('self')
        return self._call('wall.getReposts', params=params)

    def wall_openComments(self, owner_id=None, post_id=None):
        """
        Включает комментирование записи Работает только с конкретными записями, комментирование которых было выключено с помощью wall.closeComments

        :param owner_id: целое число, обязательный параметр
        :param post_id: положительное число, обязательный параметр
        :return: Возвращает 1 в случае успеха
        """

        params = locals()
        params.pop('self')
        return self._call('wall.openComments', params=params)

    def wall_pin(self, owner_id=None, post_id=None):
        """
        Закрепляет запись на стене (запись будет отображаться выше остальных).

        :param owner_id: идентификатор пользователя или сообщества, на стене которого находится запись. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param post_id: идентификатор записи на стене. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('wall.pin', params=params)

    def wall_post(self, owner_id=None, friends_only=None, from_group=None, message=None, attachments=None,
                  services=None, signed=None, publish_date=None, lat=None, long=None, place_id=None, post_id=None,
                  guid=None, mark_as_ads=None, close_comments=None, mute_notifications=None, copyright=None):
        """
        Позволяет создать запись на стене, предложить запись на стене публичной страницы, опубликовать существующую отложенную запись. Чтобы создать предложенную запись, необходимо передать в owner_id идентификатор публичной страницы, в которой текущий пользователь не является руководителем.  Для публикации предложенных и отложенных записей используйте параметр post_id, значение для которого можно получить методом wall.get с filter=suggests и postponed соответственно.

        :param owner_id: идентификатор пользователя или сообщества, на стене которого должна быть опубликована запись. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param friends_only: 1 — запись будет доступна только друзьям, 0 — всем пользователям. По умолчанию публикуемые записи доступны всем пользователям. флаг, может принимать значения 1 или 0
        :param from_group: данный параметр учитывается, если owner_id < 0 (запись публикуется на стене группы). 1 — запись будет опубликована от имени группы, 0 — запись будет опубликована от имени пользователя (по умолчанию). флаг, может принимать значения 1 или 0
        :param message: текст сообщения (является обязательным, если не задан параметр attachments) строка
        :param attachments: список объектов, приложенных к записи и разделённых символом ",". Поле attachments представляется в формате: <type><owner_id>_<media_id>,<type><owner_id>_<media_id>   <type> — тип медиа-приложения:   photo — фотография;  video — видеозапись ;  audio — аудиозапись;  doc — документ;  page — wiki-страница;  note — заметка;  poll — опрос;  album — альбом;  market — товар;  market_album — подборка товаров;  audio_playlist — плейлист с аудио.   <owner_id> — идентификатор владельца медиа-приложения (обратите внимание, если объект находится в сообществе, этот параметр должен быть отрицательным).  <media_id> — идентификатор медиа-приложения.   Например: photo100172_166443618,photo-1_265827614  Также в поле attachments может быть указана ссылка на внешнюю страницу, которую Вы хотите разместить в записи, например: photo66748_265827614,http://habrahabr.ru  При попытке приложить больше одной ссылки будет возвращена ошибка.   Параметр является обязательным, если не задан параметр message. список слов, разделенных через запятую
        :param services: список сервисов или сайтов, на которые необходимо экспортировать запись, в случае если пользователь настроил соответствующую опцию. Например, twitter, facebook. строка
        :param signed: 1 — у записи, размещенной от имени сообщества, будет добавлена подпись (имя пользователя, разместившего запись), 0 — подписи добавлено не будет. Параметр учитывается только при публикации на стене сообщества и указании параметра from_group. По умолчанию подпись не добавляется. флаг, может принимать значения 1 или 0
        :param publish_date: дата публикации записи в формате unixtime. Если параметр указан, публикация записи будет отложена до указанного времени. положительное число
        :param lat: географическая широта отметки, заданная в градусах (от -90 до 90). дробное число
        :param long: географическая долгота отметки, заданная в градусах (от -180 до 180). дробное число
        :param place_id: идентификатор места, в котором отмечен пользователь. положительное число
        :param post_id: идентификатор записи, которую необходимо опубликовать. Данный параметр используется для публикации отложенных записей и предложенных новостей. положительное число
        :param guid: уникальный идентификатор, предназначенный для предотвращения повторной отправки одинаковой записи. Действует в течение одного часа. строка
        :param mark_as_ads: 1 — у записи, размещенной от имени сообщества, будет добавлена метка "это реклама", 0 — метки добавлено не будет. В сутки может быть опубликовано не более пяти рекламных записей, из которых не более трёх — вне Биржи ВКонтакте. флаг, может принимать значения 1 или 0, по умолчанию
        :param close_comments: 1 — комментарии к записи отключены.  0 — комментарии к записи включены. флаг, может принимать значения 1 или 0
        :param mute_notifications: 1 — уведомления к записи  отключены.  0 — уведомления к записи включены. флаг, может принимать значения 1 или 0
        :param copyright: источник материала. Поддерживаются внешние и внутренние ссылки. строка
        :return: После успешного выполнения возвращает идентификатор созданной записи (post_id).
        """

        params = locals()
        params.pop('self')
        return self._call('wall.post', params=params)

    def wall_postAdsStealth(self, owner_id=None, message=None, attachments=None, signed=None, lat=None, long=None,
                            place_id=None, guid=None, link_button=None, link_title=None, link_image=None,
                            link_video=None):
        """
        Позволяет создать скрытую запись, которая не попадает на стену сообщества и в дальнейшем может быть использована  для создания рекламного объявления типа "Запись в сообществе". Создание скрытых записей возможно только в сообществах от имени группы, публичной страницы или мероприятия; пользователь должен обладать правами администратора или редактора. Обратите внимание  — в сутки можно создать не более 100 скрытых записей.

        :param owner_id: идентификатор владельца стены (идентификатор сообщества нужно указывать со знаком «минус»). целое число, обязательный параметр
        :param message: текст записи. строка
        :param attachments: список объектов, приложенных к записи и разделённых символом ",".   В поле может быть указана ссылка на страницу, которую Вы хотите разместить в записи, например: photo66748_265827614,http://habrahabr.ru   Либо поле attachments представляется в формате: <type><owner_id>_<media_id>,<type><owner_id>_<media_id>   <type> — тип медиа-приложения:   photo — фотография;  video — видеозапись;  audio — аудиозапись;  doc — документ;  page — wiki-страница;  note — заметка;  poll — опрос;  album — альбом;  market — товар;  market_album — подборка товаров,  pretty_card — карточка карусели.   <owner_id> — идентификатор владельца медиа-приложения (обратите внимание, если объект находится в сообществе, этот параметр должен быть отрицательным).  <media_id> — идентификатор медиа-приложения.   Например: photo100172_166443618,photo-1_265827614  В качестве ссылки может быть использован телефонный номер  в формате tel:+71234567890  При попытке приложить больше одной ссылки будет возвращена ошибка.   Параметр является обязательным, если не задан параметр message. список слов, разделенных через запятую
        :param signed: 1 — у записи, размещенной от имени сообщества, будет добавлена подпись (имя пользователя, разместившего запись), 0 — без подписи. флаг, может принимать значения 1 или 0
        :param lat: географическая широта отметки, заданная в градусах (от -90 до 90). дробное число
        :param long: географическая долгота отметки, заданная в градусах (от -180 до 180). дробное число
        :param place_id: идентификатор места. положительное число
        :param guid: уникальный идентификатор, предназначенный для предотвращения повторной отправки одинаковой записи. строка
        :param link_button: идентификатор кнопки, которую необходимо добавить к сниппету для ссылки.  link_button Текст Действие Тип ссылок auto (автовыбор) (автовыбор) Все app_join Запустить Переход по ссылке Приложения и игры app_game_join Играть Переход по ссылке Игры open_url Перейти Переход по ссылке Внешние сайты, сообщества, приложения open Открыть Переход по ссылке Внешние сайты more Подробнее Переход по ссылке Сообщества call Позвонить Набор номера Телефонные номера book Забронировать Набор номера Телефонные номера enroll Записаться Переход по ссылке или набор номера Внешние сайты, телефонные номера register Зарегистрироваться Набор номера Телефонные номера buy Купить Переход по ссылке Внешние сайты buy_ticket Купить билет Переход по ссылке Внешние сайты order Заказать Переход по ссылке Внешние сайты create Создать Переход по ссылке Внешние сайты install Установить Переход по ссылке Внешние сайты contact Связаться Переход по ссылке Внешние сайты fill Заполнить Переход по ссылке Внешние сайты join_public Подписаться Подписка на публичную страницу Публичные страницы join_event Я пойду Участие в мероприятии События join Вступить Вступление в сообщество Сообщества im Связаться Переход к диалогу с сообществом Сообщества, публичные страницы, события im2 Написать Переход к диалогу с сообществом Сообщества, публичные страницы, события begin Начать Переход по ссылке Внешние сайты get Получить Переход по ссылке Внешние сайты watch Смотреть Переход по ссылке Внешние сайты download Скачать Переход по ссылке Внешние сайты participate Участвовать Переход по ссылке Внешние сайты play Играть Переход по ссылке Внешние сайты apply Подать заявку Переход по ссылке Внешние сайты get_an_offer Получить предложение Переход по ссылке Внешние сайты to_write Написать Переход по ссылке Внешние сайты reply Откликнуться Переход по ссылке Внешние сайты  строка
        :param link_title: заголовок, который должен быть использован для сниппета. Если не указан, будет автоматически получен с целевой ссылки. Обязательно указывать в случае, если ссылка является номером телефона. строка
        :param link_image: ссылка на изображение, которое должно быть использовано для сниппета. Минимальное разрешение: 537x240. Если не указана, будет автоматически загружена с целевой ссылки. Обязательно указывать в случае, если ссылка является номером телефона. Одновременно может быть указан либо параметр link_image, либо параметр link_video. строка
        :param link_video: идентификатор видео в формате "<owner_id>_<media_id>". Одновременно может быть указан либо параметр link_image, либо параметр link_video. Кроме того, параметр link_video может быть указан только вместе с параметрами  link_button, link_title. строка
        :return: Идентификатор созданной записи
        """

        params = locals()
        params.pop('self')
        return self._call('wall.postAdsStealth', params=params)

    def wall_reportComment(self, owner_id=None, comment_id=None, reason=None):
        """
        Позволяет пожаловаться на комментарий к записи.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит комментарий. целое число, обязательный параметр
        :param comment_id: идентификатор комментария. положительное число, обязательный параметр
        :param reason: причина жалобы:   0 — спам;  1 — детская порнография;  2 — экстремизм;  3 — насилие;  4 — пропаганда наркотиков;  5 — материал для взрослых;  6 — оскорбление;  8 — призывы к суициду.  положительное число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('wall.reportComment', params=params)

    def wall_reportPost(self, owner_id=None, post_id=None, reason=None):
        """
        Позволяет пожаловаться на запись.

        :param owner_id: идентификатор пользователя или сообщества, которому принадлежит запись. целое число, обязательный параметр
        :param post_id: идентификатор записи. положительное число, обязательный параметр
        :param reason: причина жалобы:   0 — спам;  1 — детская порнография;  2 — экстремизм;  3 — насилие;  4 — пропаганда наркотиков;  5 — материал для взрослых;  6 — оскорбление;  8 — призывы к суициду.  положительное число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('wall.reportPost', params=params)

    def wall_repost(self, object=None, message=None, group_id=None, mark_as_ads=None, mute_notifications=None):
        """
        Копирует объект на стену пользователя или сообщества.

        :param object: строковый идентификатор объекта, который необходимо разместить на стене, например, wall66748_3675 или wall-1_340364.  Формируется из типа объекта (wall, photo, video и т.п.), идентификатора владельца объекта и идентификатора самого объекта. строка, обязательный параметр
        :param message: сопроводительный текст, который будет добавлен к записи с объектом. строка
        :param group_id: идентификатор сообщества, на стене которого будет размещена запись с объектом. Если не указан, запись будет размещена на стене текущего пользователя. положительное число
        :param mark_as_ads: 1 — пометить запись как рекламную. флаг, может принимать значения 1 или 0, по умолчанию
        :param mute_notifications: флаг, может принимать значения 1 или 0
        :return: После успешного выполнения возвращает объект со следующими полями:   success — всегда содержит 1;  post_id — идентификатор созданной записи;  reposts_count — количество репостов объекта с учетом осуществленного;  likes_count — число отметок «Мне нравится» у объекта.
        """

        params = locals()
        params.pop('self')
        return self._call('wall.repost', params=params)

    def wall_restore(self, owner_id=None, post_id=None):
        """
        Восстанавливает удаленную запись на стене пользователя или сообщества.

        :param owner_id: идентификатор пользователя или сообщества, на стене которого находилась удаленная запись. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param post_id: идентификатор записи на стене. положительное число
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('wall.restore', params=params)

    def wall_restoreComment(self, owner_id=None, comment_id=None):
        """
        Восстанавливает удаленный комментарий к записи на стене.

        :param owner_id: идентификатор пользователя или сообщества, на стене которого находится комментарий к записи. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param comment_id: идентификатор комментария на стене. целое число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('wall.restoreComment', params=params)

    def wall_search(self, owner_id=None, domain=None, query=None, owners_only=None, count=None, offset=None,
                    extended=None, fields=None):
        """
        Позволяет искать записи на стене в соответствии с заданными критериями. Обратите внимание, для этого метода действуют специальные ограничения на число запросов. Подробнее.

        :param owner_id: идентификатор пользователя или сообщества. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число
        :param domain: короткий адрес пользователя или сообщества. строка
        :param query: поисковой запрос. Для точного результата запрос необходимо передавать в двойных кавычках. строка
        :param owners_only: 1 — возвращать только записи от имени владельца стены. флаг, может принимать значения 1 или 0
        :param count: количество записей, которые необходимо вернуть. положительное число, по умолчанию 20, максимальное значение 100
        :param offset: смещение, необходимо для получения определенного подмножества результатов. положительное число, по умолчанию 0
        :param extended: 1 — в ответе будут возвращены дополнительные поля profiles и groups, содержащие информацию о пользователях и сообществах. По умолчанию: 0. флаг, может принимать значения 1 или 0
        :param fields: список дополнительных полей для профилей и  групп, которые необходимо вернуть. См. описание полей объекта user и описание полей объекта group.  Обратите внимание, этот параметр учитывается только при extended=1. список слов, разделенных через запятую
        :return: После успешного выполнения возвращает объект, содержащий число результатов в поле count и массив объектов записей на стене в поле items.  Если был задан параметр extended=1, возвращает число результатов в поле count, отдельно массив объектов записей на стене в поле items, пользователей в поле profiles и сообществ в поле groups.
        """

        params = locals()
        params.pop('self')
        return self._call('wall.search', params=params)

    def wall_unpin(self, owner_id=None, post_id=None):
        """
        Отменяет закрепление записи на стене.

        :param owner_id: идентификатор пользователя или сообщества, на стене которого находится запись. Обратите внимание, идентификатор сообщества в параметре owner_id необходимо указывать со знаком "-" — например, owner_id=-1 соответствует идентификатору сообщества ВКонтакте API (club1)  целое число, по умолчанию идентификатор текущего пользователя
        :param post_id: идентификатор записи на стене. положительное число, обязательный параметр
        :return: После успешного выполнения возвращает 1.
        """

        params = locals()
        params.pop('self')
        return self._call('wall.unpin', params=params)

    def widgets_getComments(self, widget_api_id=None, url=None, page_id=None, order=None, fields=None, offset=None,
                            count=None):
        """
        Получает список комментариев к странице, оставленных через Виджет комментариев.

        :param widget_api_id: Идентификатор приложения/сайта, с которым инициализируются виджеты. целое число
        :param url: URL-адрес страницы. строка
        :param page_id: внутренний идентификатор страницы в приложении/сайте (в случае, если для инициализации виджетов использовался параметр page_id). строка
        :param order: тип сортировки комментариев. Возможные значения: date, likes, last_comment. Значение по умолчанию - date. строка, по умолчанию date
        :param fields: перечисленные через запятую поля анкет необходимые для получения. Если среди полей присутствует replies, будут возвращены последние комментарии второго уровня для каждого комментария первого уровня. список слов, разделенных через запятую
        :param offset: смещение необходимое для выборки определенного подмножества комментариев. По умолчанию 0. положительное число, по умолчанию 0
        :param count: количество возвращаемых записей. положительное число, по умолчанию 10, минимальное значение 10, максимальное значение 200
        :return: В случае успеха возвращает объект со следующими полями:   count — общее количество комментариев первого уровня к странице (без учета ограничений из входного параметра count).  posts — список комментариев первого уровня. Каждый элемент списка имеет структуру, схожую с объектами post из результатов метода wall.get. Помимо этого:   В случае, если среди необходимых полей было указано replies, в поле comments.replies будет содержаться список комментариев второго уровня. Каждый элемент этого списка имеет структуру, схожую с объектами comment из результатов метода wall.getComments.  В случае, если были указаны любые другие необходимые поля анкет, в каждом элементе post и comment будет поле user, содержащее соответствующую информацию об авторе комментария.
        """

        params = locals()
        params.pop('self')
        return self._call('widgets.getComments', params=params)

    def widgets_getPages(self, widget_api_id=None, order=None, period=None, offset=None, count=None):
        """
        Получает список страниц приложения/сайта, на которых установлен Виджет комментариев или «Мне нравится».

        :param widget_api_id: идентификатор приложения/сайта, с которым инициализируются виджеты. целое число
        :param order: Тип сортировки страниц. Возможные значения: date, comments, likes, friend_likes. строка, по умолчанию friend_likes
        :param period: Период выборки. Возможные значения: day, week, month, alltime. строка, по умолчанию week
        :param offset: смещение необходимое для выборки определенного подмножества комментариев. По умолчанию 0. положительное число, по умолчанию 0
        :param count: количество возвращаемых записей. положительное число, по умолчанию 10, минимальное значение 10, максимальное значение 200
        :return: В случае успеха возвращает объект со следующими полями:   count — общее количество страниц (без учета ограничений входного параметра count)  pages — список объектов-страниц Каждый объект, описывающий страницу, имеет следующие поля:   id — идентификатор страницы в системе;  title — заголовок страницы (берется из мета-тегов на странице или задается параметром pageTitle при инициализации)  description — краткое описание страницы (берется из мета-тегов на странице или задается параметром pageDescription при инициализации);  photo — объект, содержащий фотографию-миниатюру страницы (берется из мета-тегов на странице или задается параметром pageImage при инициализации)  url — абсолютный адрес страницы;  likes — объект, содержащий поле count — количество отметок «Мне нравится» к странице. Для получения списка пользователей, отметивших страницу можно использовать метод likes.getList с параметром type равным site_page;  comments — объект, содержащий поле count — количество комментариев к странице внутри виджета. Для получения списка комментариев можно использовать метод widgets.getComments;  date — дата первого обращения к виджетам на странице  page_id — внутренний идентификатор страницы в приложении/на сайте (в случае, если при инициализации виджетов использовался параметр page_id);
        """

        params = locals()
        params.pop('self')
        return self._call('widgets.getPages', params=params)