# -*- coding: utf-8 -*-
from setuptools import setup

modules = \
['multicommand']
setup_kwargs = {
    'name': 'multicommand',
    'version': '0.0.8',
    'description': 'Simple subcommand CLIs with argparse',
    'long_description': '# multicommand\n\nSimple subcommand CLIs with argparse.\n\n[![PyPI Version](https://img.shields.io/pypi/v/multicommand.svg)](https://pypi.org/project/multicommand/)\n\n`multicommand` uses only the standard library and is ~100 lines of code (modulo comments and whitespace)\n\n## Installation\n\n```bash\npip install multicommand\n```\n\n## Overview\n\nMulticommand enables you to easily write CLIs with deeply nested commands using vanilla argparse. You provide it with a package, it searches that package for parsers (`ArgumentParser` objects), and connects, names, and converts those parsers into subcommands based on the package structure.\n\n```text\n        Package                       ->                    CLI\n\n\ncommands/unary/negate.py                            mycli unary negate ...\ncommands/binary/add.py                              mycli binary add ...\ncommands/binary/divide.py             ->            mycli binary divide ...\ncommands/binary/multiply.py                         mycli binary multiply ...\ncommands/binary/subtract.py                         mycli binary subtract ...\n```\n\nAll it needs is for each module to define a module-level `parser` variable which points to an instance of `argparse.ArgumentParser`.\n\n## Motivation\n\nI like `argparse`. It\'s flexible, full-featured and it\'s part of the standard library, so if you have python you probably have argparse. I also like the "subcommand" pattern, i.e. one root command that acts as an entrypoint and subcommands to group related functionality. Of course, argparse can handle adding subcommands to parsers, but it\'s always felt a bit cumbersome, especially when there are many subcommands with lots of nesting.\n\nIf you\'ve ever worked with technologies like `Next.js` or `oclif` (or even if you haven\'t) there\'s a duality between files and "objects". For Next.js each file under `pages/` maps to a webpage, in oclif each module under `commands/` maps to a CLI command. And that\'s the basic premise for multicommand: A light-weight package that let\'s you write one parser per file, pretty much in isolation, and it handles the wiring, exploiting the duality between command structure and file system structure.\n\n## Getting Started\n\nSee the [simple example](https://github.com/andrewrosss/multicommand/tree/master/examples/01_simple), or for the impatient:\n\nCreate a directory to work in, for example:\n\n```bash\nmkdir ~/multicommand-sample && cd ~/multicommand-sample\n```\n\nInstall `multicommand`:\n\n```bash\npython3 -m venv ./venv\nsource ./venv/bin/activate\n\npython3 -m pip install multicommand\n```\n\nCreate the subpackage to house our parsers:\n\n```bash\nmkdir -p mypkg/parsers/topic/cmd/subcmd\n```\n\nCreate the `*.py` files required for the directories to be packages\n\n```bash\ntouch mypkg/__init__.py\ntouch mypkg/parsers/__init__.py\ntouch mypkg/parsers/topic/__init__.py\ntouch mypkg/parsers/topic/cmd/__init__.py\ntouch mypkg/parsers/topic/cmd/subcmd/{__init__.py,greet.py}\n```\n\nAdd a `parser` to `greet.py`:\n\n```python\n# file: mypkg/parsers/topic/cmd/subcmd/greet.py\nimport argparse\n\n\ndef handler(args):\n    greeting = f\'Hello, {args.name}!\'\n    print(greeting.upper() if args.shout else greeting)\n\n\nparser = argparse.ArgumentParser(\n    description=\'My first CLI with multicommand\',\n    formatter_class=argparse.ArgumentDefaultsHelpFormatter\n)\nparser.add_argument(\'name\', help=\'Name to use in greeting\')\nparser.add_argument(\'--shout\', action=\'store_true\', help=\'Yell the greeting\')\nparser.set_defaults(handler=handler)\n```\n\nlastly, add an entrypoint:\n\n```bash\ntouch mypkg/cli.py\n```\n\nwith the following content:\n\n```python\n# file: mypkg/cli.py\nimport multicommand\nfrom . import parsers\n\n\ndef main():\n    parser = multicommand.create_parser(parsers)\n    args = parser.parse_args()\n    if hasattr(args, \'handler\'):\n        args.handler(args)\n        return\n    parser.print_help()\n\n\nif __name__ == "__main__":\n    exit(main())\n```\n\nTry it out!\n\n```bash\n$ python3 -m mypkg.cli\nusage: cli.py [-h] {topic} ...\n\noptional arguments:\n  -h, --help  show this help message and exit\n\nsubcommands:\n\n  {topic}\n```\n\nTake a look at our `greet` command:\n\n```bash\n$ python3 -m mypkg.cli topic cmd subcmd greet --help\nusage: cli.py topic cmd subcmd greet [-h] [--shout] name\n\nMy first CLI with multicommand\n\npositional arguments:\n  name        Name to use in greeting\n\noptional arguments:\n  -h, --help  show this help message and exit\n  --shout     Yell the greeting (default: False)\n```\n\nFrom this we get:\n\n```bash\n$ python3 -m mypkg.cli topic cmd subcmd greet "World"\nHello, World!\n\n$ python3 -m mypkg.cli topic cmd subcmd greet --shout "World"\nHELLO, WORLD!\n```\n\n### Bonus\n\nWant to add the command `topic cmd ungreet ...` to say goodbye?\n\nAdd the module:\n\n```bash\ntouch mypkg/parsers/topic/cmd/ungreet.py\n```\n\nwith contents:\n\n```python\n# file: mypkg/parsers/topic/cmd/ungreet.py\nimport argparse\n\n\ndef handler(args):\n    print(f\'Goodbye, {args.name}!\')\n\n\nparser = argparse.ArgumentParser(description=\'Another subcommand with multicommand\')\nparser.add_argument(\'name\', help=\'Name to use in un-greeting\')\nparser.set_defaults(handler=handler)\n```\n\nThe new command is automatically added!:\n\n```bash\n$ python3 -m mypkg.cli topic cmd --help\nusage: cli.py cmd [-h] {subcmd,ungreet} ...\n\noptional arguments:\n  -h, --help        show this help message and exit\n\nsubcommands:\n\n  {subcmd,ungreet}\n```\n\nTry it out:\n\n```bash\n$ python3 -m mypkg.cli topic cmd ungreet "World"\nGoodbye, World!\n```\n',
    'author': 'Andrew Ross',
    'author_email': 'andrew.ross.mail@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/andrewrosss/multicommand',
    'py_modules': modules,
    'python_requires': '>=3.6.1,<4.0.0',
}


setup(**setup_kwargs)
