"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const cdk = require("aws-cdk-lib");
const lib_1 = require("../lib");
const api = require("aws-cdk-lib/aws-apigateway");
const waf = require("aws-cdk-lib/aws-wafv2");
const defaults = require("@aws-solutions-constructs/core");
const assertions_1 = require("aws-cdk-lib/assertions");
function deployConstruct(stack, constructProps) {
    const restApi = new api.RestApi(stack, 'test-api', {});
    restApi.root.addMethod('ANY');
    const props = constructProps ?
        { webaclProps: constructProps, existingApiGatewayInterface: restApi }
        : { existingApiGatewayInterface: restApi };
    return new lib_1.WafwebaclToApiGateway(stack, 'test-wafwebacl-apigateway', props);
}
// --------------------------------------------------------------
// Test error handling for existing WAF web ACL and user provided web ACL props
// --------------------------------------------------------------
test('Test error handling for existing WAF web ACL and user provider web ACL props', () => {
    const stack = new cdk.Stack();
    const props = {
        defaultAction: {
            allow: {}
        },
        scope: 'REGIONAL',
        visibilityConfig: {
            cloudWatchMetricsEnabled: false,
            metricName: 'webACL',
            sampledRequestsEnabled: true
        },
    };
    const wafAcl = new waf.CfnWebACL(stack, 'test-waf', props);
    const restApi = new api.RestApi(stack, 'empty-api', {});
    expect(() => {
        new lib_1.WafwebaclToApiGateway(stack, 'test-waf-gateway', {
            existingApiGatewayInterface: restApi,
            existingWebaclObj: wafAcl,
            webaclProps: props
        });
    }).toThrowError();
});
// --------------------------------------------------------------
// Test default deployment
// --------------------------------------------------------------
test('Test default deployment', () => {
    const stack = new cdk.Stack();
    const construct = deployConstruct(stack);
    expect(construct.webacl !== null);
    expect(construct.apiGateway !== null);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::WAFv2::WebACL", {
        Rules: [
            {
                Name: "AWS-AWSManagedRulesBotControlRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 0,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesBotControlRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesBotControlRuleSet",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesKnownBadInputsRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 1,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesKnownBadInputsRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesKnownBadInputsRuleSet",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesCommonRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 2,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesCommonRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesCommonRuleSet",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesAnonymousIpList",
                OverrideAction: {
                    None: {}
                },
                Priority: 3,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesAnonymousIpList",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesAnonymousIpList",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesAmazonIpReputationList",
                OverrideAction: {
                    None: {}
                },
                Priority: 4,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesAmazonIpReputationList",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesAmazonIpReputationList",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesAdminProtectionRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 5,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesAdminProtectionRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesAdminProtectionRuleSet",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesSQLiRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 6,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesSQLiRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesSQLiRuleSet",
                    SampledRequestsEnabled: true
                }
            }
        ]
    });
});
// --------------------------------------------------------------
// Test web acl with user provided acl props
// --------------------------------------------------------------
test('Test user provided acl props', () => {
    const stack = new cdk.Stack();
    const webaclProps = {
        defaultAction: {
            allow: {}
        },
        scope: 'REGIONAL',
        visibilityConfig: {
            cloudWatchMetricsEnabled: false,
            metricName: 'webACL',
            sampledRequestsEnabled: true
        },
        rules: [
            defaults.wrapManagedRuleSet("AWSManagedRulesCommonRuleSet", "AWS", 0),
            defaults.wrapManagedRuleSet("AWSManagedRulesWordPressRuleSet", "AWS", 1),
        ]
    };
    deployConstruct(stack, webaclProps);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::WAFv2::WebACL", {
        VisibilityConfig: {
            CloudWatchMetricsEnabled: false,
            MetricName: "webACL",
            SampledRequestsEnabled: true
        },
        Rules: [
            {
                Name: "AWS-AWSManagedRulesCommonRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 0,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesCommonRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesCommonRuleSet",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesWordPressRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 1,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesWordPressRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesWordPressRuleSet",
                    SampledRequestsEnabled: true
                }
            }
        ]
    });
});
// --------------------------------------------------------------
// Test existing web ACL
// --------------------------------------------------------------
test('Test existing web ACL', () => {
    const stack = new cdk.Stack();
    const webacl = new waf.CfnWebACL(stack, 'test-webacl', {
        defaultAction: {
            allow: {}
        },
        scope: 'REGIONAL',
        visibilityConfig: {
            cloudWatchMetricsEnabled: true,
            metricName: 'webACL',
            sampledRequestsEnabled: true
        },
    });
    const restApi = new api.RestApi(stack, 'test-api', {});
    restApi.root.addMethod('ANY');
    new lib_1.WafwebaclToApiGateway(stack, 'test-wafwebacl-apigateway', {
        existingWebaclObj: webacl,
        existingApiGatewayInterface: restApi
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::WAFv2::WebACL", {
        VisibilityConfig: {
            CloudWatchMetricsEnabled: true,
            MetricName: "webACL",
            SampledRequestsEnabled: true
        }
    });
    template.resourceCountIs("AWS::WAFv2::WebACL", 1);
});
//# sourceMappingURL=data:application/json;base64,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