from __future__ import annotations

import inspect
from pathlib import Path, PosixPath
from typing import Any, Callable

from jinja2 import Environment, FileSystemLoader, select_autoescape

from ..env import dump_env, read_env
from ..host import Host
from ..manifest_object import ManifestObject
from ..path import AppPath, ManifestPath
from ..settings import DIR_ASSETS, FILENAME_COMPOSE, FILENAME_ENV
from .names import normalise_name, pascal_to_snake


# Abstract app registry for type lookups
abstract_app_registry: dict[str, type[BaseApp]] = {}


class AppsTemplateContext:
    """
    Lazy context getter for use in template context `apps`
    """

    apps: dict[str, BaseApp]

    def __init__(self, apps: dict[str, BaseApp]):
        self.apps = apps

    def __getitem__(self, name: str) -> dict[str, Any]:
        return self.get(name)

    def __getattr__(self, name: str) -> dict[str, Any]:
        return self.get(name)

    def get(self, name: str) -> dict[str, Any]:
        normalised = normalise_name(name)
        if normalised not in self.apps:
            raise ValueError(f"Unknown app {name} ({normalised})")
        return self.apps[normalised].get_compose_context()

    def __contains__(self, name: str) -> bool:
        normalised = normalise_name(name)
        return normalised in self.apps


class EnvTemplateContext:
    """
    Lazy context getter for use in template context `env`
    """

    app: BaseApp

    def __init__(self, app: BaseApp):
        self.app = app

    def __getitem__(self, name: str) -> Any:
        return self.get(name)

    def __getattr__(self, name: str) -> Any:
        return self.get(name)

    def get(self, name: str) -> Any:
        env_data = self.app.get_host_env_data()
        return env_data[name]

    def __contains__(self, name: str) -> bool:
        env_data = self.app.get_host_env_data()
        return name in env_data


class BaseApp(ManifestObject, abstract=True):
    #: Path to the directory containing the app
    #:
    #: For access see ``.get_path``
    #:
    #: Default: same dir as manifest
    path: str = ""

    #: Path to a base docker0s manifest for this app.
    #:
    #: If the path ends ``::<name>`` it will look for an app definition with that name,
    #: eg ``app://bases.py::Store``. Otherwise it will look for an app with the same
    #: name as this.
    #:
    #: The base manifest must not define a host.
    #:
    #: This referenced manifest will will act as the base manifest. That in turn can
    #: reference an additional base manifest.
    #:
    #: Default: ``app://d0s-manifest.py``, then ``app://d0s-manifest.yml``
    extends: str | None = None

    # Defaults for ``extends`` - first found will be used
    default_extends: list[str] = [
        "app://d0s-manifest.py",
        "app://d0s-manifest.yml",
    ]

    #: Path to the app's docker compose file. This will be pushed to the host.
    #:
    #: This can be a ``.yml`` file, or a ``.jinja2`` template.
    #:
    #: For access see ``.get_compose_path``
    #:
    #: Default: ``app://docker-compose.jinja2``, then ``app://docker-compose.yml``
    compose: str | None = None

    # Defaults for ``compose`` - first found will be used
    default_compose: list[str] = [
        "app://docker-compose.jinja2",
        "app://docker-compose.yml",
    ]

    #: Context for docker-compose Jinja2 template rendering
    #:
    #: To add instance data, override ``.get_compose_context``
    compose_context: dict[str, Any] | None = None

    #: File containing environment variables for docker-compose
    #:
    #: Path to an env file, or a list of paths
    #:
    #: For access see ``.get_env_data``
    env_file: str | list[str] | None = None

    #: Environment variables for docker-compose
    #:
    #: For access see ``.get_env_data``
    env: dict[str, (str | int)] | None = None

    #: If True, COMPOSE_PROJECT_NAME will be automatically added to the env if not
    #: set by ``env_file`` or ``env``
    set_project_name: bool = True

    # Host this app instance is bound to on initialisation
    host: Host

    # All app instances defined in the manifest which defines this app, including self
    manifest_apps: dict[str, BaseApp]

    def __init_subclass__(
        cls, abstract: bool = False, name: str | None = None, **kwargs
    ):
        """
        Set abstract flag and register abstract classes with the registry
        """
        super().__init_subclass__(abstract=abstract, name=name, **kwargs)

        if abstract:
            global abstract_app_registry  # not required, for clarity
            if cls.__name__ in abstract_app_registry:
                raise ValueError(
                    f"Abstract class names must be unique, {cls.__name__} is duplicate"
                )
            abstract_app_registry[cls.__name__] = cls

    def __init__(self, host: Host):
        self.host = host
        self.other_apps: dict[str, BaseApp] = {}

    def __str__(self):
        return self.get_name()

    @classmethod
    def get_name(cls) -> str:
        """
        The docker0s name of this app in PascalCase
        """
        return cls.__name__

    @classmethod
    def get_docker_name(cls) -> str:
        """
        The docker container name of this app in snake_case
        """
        return pascal_to_snake(cls.get_name())

    @classmethod
    def get_manifest_path(cls) -> Path:
        """
        Find the path of the manifest file which defined this app.

        If this was pulled from a git repository, this will be the local path.
        """
        cls_module = inspect.getmodule(cls)
        if cls_module is None or cls_module.__file__ is None:
            raise ValueError(f"Cannot find module path for app {cls}")
        return Path(cls_module.__file__)

    @classmethod
    def get_manifest_dir(cls) -> Path:
        return cls.get_manifest_path().parent

    @classmethod
    def get_path(cls) -> ManifestPath:
        """
        Resolve ``cls.path`` to a ``ManifestPath``
        """
        # TODO: Add a test to check this is a dir, not a file?

        path = ManifestPath(cls.path, manifest_dir=cls.get_manifest_dir())
        return path

    @classmethod
    def _mk_app_path(cls, path: str | AppPath) -> AppPath:
        """
        Internal helper for building an AppPath where this is the app
        """
        return AppPath(path, manifest_dir=cls.get_manifest_dir(), app=cls)

    @classmethod
    def _get_base_manifest(cls) -> AppPath | None:
        """
        Find the path to the base manifest if one exists, otherwise return None
        """
        # Find paths to seek
        extends: list[str]
        if cls.extends:
            extends = [cls.extends]
        else:
            extends = cls.default_extends

        # Return the first which exists
        for base_path in extends:
            if "::" in base_path:
                base_path = base_path.split("::", 1)[0]

            extends_path = cls._mk_app_path(base_path)
            if extends_path.exists():
                if extends_path.absolute == cls.get_manifest_path().absolute():
                    # Make sure we didn't find ourselves and get stuck in a loop
                    # TODO: Check for infinite loop between multiple manifests
                    if cls.extends:
                        raise ValueError(
                            f"Resolving base manifest {cls.extends} for app"
                            f" {cls.get_name()} led back to current manifest"
                        )
                    return None
                return extends_path

        return None

    @classmethod
    def apply_base_manifest(cls, history: list[ManifestPath] | None = None):
        """
        If a base manifest can be found by _get_base_manifest, load it and look for a
        BaseApp subclass with the same name as this. If found, add it to the base
        classes for this class.
        """
        path = cls._get_base_manifest()
        if path is None:
            if cls.extends is not None:
                raise ValueError(
                    f"Could not find base manifest {cls.extends} in {cls.path}"
                )
            # Just looking for defaults, ignore
            return

        from ..manifest import Manifest

        base_manifest = Manifest.load(path, history)
        if base_manifest.host is not None:
            raise ValueError("A base manifest cannot define a host")

        base_name = cls.get_name()
        if cls.extends and "::" in cls.extends:
            base_name = cls.extends.split("::", 1)[1]

        base_app = base_manifest.get_app(base_name)
        if base_app is None:
            raise ValueError(
                f"Base manifest {path} does not define an app called {cls.get_name()}"
            )

        if not issubclass(cls, base_app):
            cls.__bases__ = (base_app,) + cls.__bases__

    @classmethod
    def get_compose_path(cls) -> AppPath:
        """
        Return an AppPath to the compose file or template
        """
        # Find paths to seek
        composes: list[str]
        if cls.compose:
            composes = [cls.compose]
        else:
            composes = cls.default_compose

        for compose in composes:
            compose_app_path = cls._mk_app_path(compose)
            if compose_app_path.exists():
                return compose_app_path
        raise ValueError("Compose path not found")

    @classmethod
    def get_env_data(cls) -> dict[str, str | int | None]:
        """
        Load env files in order (for key conflicts last wins), and then merge in the env
        dict, if defined
        """

        def collect_without_inheritance(mro_cls):
            # Build list of files
            raw_env_files: list[str] = []

            # Get attributes directly from the class without inheritance
            env_file: str | list[str] = mro_cls.__dict__.get("env_file", None)
            env_dict: dict[str, (str | int)] = mro_cls.__dict__.get("env", None)

            if env_file is not None:
                if isinstance(env_file, (tuple, list)):
                    raw_env_files = env_file
                else:
                    raw_env_files = [env_file]
            env_files: list[AppPath] = [
                cls._mk_app_path(env_file) for env_file in raw_env_files
            ]

            # Prepare dict
            if env_dict is None:
                env_dict = {}

            env: dict[str, str | int | None] = read_env(*env_files, **env_dict)
            return env

        env = {}
        for mro_cls in reversed(cls.mro()):
            env.update(collect_without_inheritance(mro_cls))

        if cls.set_project_name and "COMPOSE_PROJECT_NAME" not in env:
            env["COMPOSE_PROJECT_NAME"] = cls.get_docker_name()
        return env

    @staticmethod
    def command(fn):
        fn.is_command = True
        return fn

    def get_command(self, name: str) -> Callable:
        """
        Return the specified command
        """
        attr = getattr(self, name)
        if callable(attr) and hasattr(attr, "is_command"):
            return attr
        raise ValueError(f"Command {name} not found")

    @property
    def remote_path(self) -> PosixPath:
        """
        The remote path for this app
        """
        return self.host.path(self.get_docker_name())

    @property
    def remote_compose(self) -> PosixPath:
        """
        A PosixPath to the remote compose file
        """
        return self.remote_path / FILENAME_COMPOSE

    @property
    def remote_env(self) -> PosixPath:
        """
        A PosixPath for the remote env file
        """
        return self.remote_path / FILENAME_ENV

    @property
    def remote_assets(self) -> PosixPath:
        """
        A PosixPath for the remote assets dir
        """
        return self.remote_path / DIR_ASSETS

    def get_compose_context(self, **kwargs: Any) -> dict[str, Any]:
        """
        Build the template context for the compose template
        """
        context = {
            "host": self.host,
            "env": EnvTemplateContext(self),
            "apps": AppsTemplateContext(self.manifest_apps),
            # Reserved for future expansion
            "docker0s": NotImplemented,
            "globals": NotImplemented,
            **kwargs,
        }

        if self.compose_context is not None:
            context.update(self.compose_context)

        return context

    def get_compose_content(self, context: dict[str, Any] | None = None) -> str:
        """
        Return the content for the docker-compose file

        This will either be rendered from ``compose_template`` if it exists, otherwise
        it will be read from ``compose``
        """
        compose_app_path = self.get_compose_path()
        if compose_app_path.filetype == ".yml":
            compose_path = compose_app_path.get_local_path()
            return compose_path.read_text()

        elif compose_app_path.filetype == ".jinja2":
            template_path = compose_app_path.get_local_path()

            env = Environment(
                loader=FileSystemLoader(template_path.parent),
                autoescape=select_autoescape(),
            )

            context = self.get_compose_context(**(context or {}))
            template = env.get_template(template_path.name)
            return template.render(context)

        raise ValueError(f"Unrecognised compose filetype {compose_app_path.filetype}")

    def get_host_env_data(self) -> dict[str, str | int | None]:
        """
        Build the env data dict to be sent to the server
        """
        env_data = self.get_env_data()
        env_data.update(
            {
                "ENV_FILE": str(self.remote_env),
                "ASSETS": str(self.remote_assets),
            }
        )
        return env_data

    def deploy(self):
        """
        Deploy the env file for this app
        """
        print(f"Deploying {self} to {self.host}")
        self.write_env_to_host()

    def write_env_to_host(self):
        env_dict = self.get_env_data()
        env_str = dump_env(env_dict)
        self.host.write(self.remote_env, env_str)

    def call_compose(self, cmd: str, args: dict[str, Any] | None = None):
        """
        Run a docker-compose command on the host
        """
        self.host.call_compose(
            compose=self.remote_compose,
            env=self.remote_env,
            cmd=cmd,
            cmd_args=args,
        )

    def up(self, *services: str):
        """
        Bring up one or more services in this app

        If no services are specified, all services are selected
        """
        if services:
            for service in services:
                self.call_compose("up --build --detach {service}", {"service": service})
        else:
            self.call_compose("up --build --detach")

    def down(self, *services: str):
        """
        Take down one or more containers in this app

        If no services are specified, all services are selected
        """
        if services:
            for service in services:
                self.call_compose(
                    "rm --force --stop -v {service}", {"service": service}
                )
        else:
            self.call_compose("down")

    def restart(self, *services: str):
        """
        Restart one or more services in this app

        If no services are specified, all services are selected
        """
        if services:
            for service in services:
                self.call_compose("restart {service}", {"service": service})
        else:
            self.call_compose("restart")

    def exec(self, service: str, command: str):
        """
        Execute a command in the specified service

        Command is passed as it arrives, values are not escaped
        """
        self.call_compose(f"exec {{service}} {command}", {"service": service})
