import * as cdk from 'aws-cdk-lib';
import { aws_ec2 as ec2, aws_ecr as ecr, aws_iam as iam, aws_logs as logs, aws_s3_assets as s3_assets, Duration, RemovalPolicy } from 'aws-cdk-lib';
import { Construct } from 'constructs';
import { Architecture, IImageBuilder, Os, RunnerImage, RunnerVersion } from '../common';
/**
 * Properties for ContainerImageBuilder construct.
 */
export interface ContainerImageBuilderProps {
    /**
     * Image architecture.
     *
     * @default Architecture.X86_64
     */
    readonly architecture?: Architecture;
    /**
     * Image OS.
     *
     * @default OS.LINUX
     */
    readonly os?: Os;
    /**
     * Version of GitHub Runners to install.
     *
     * @default latest version available
     */
    readonly runnerVersion?: RunnerVersion;
    /**
     * Schedule the image to be rebuilt every given interval. Useful for keeping the image up-do-date with the latest GitHub runner version and latest OS updates.
     *
     * Set to zero to disable.
     *
     * @default Duration.days(7)
     */
    readonly rebuildInterval?: Duration;
    /**
     * VPC to launch the runners in.
     *
     * @default default account VPC
     */
    readonly vpc?: ec2.IVpc;
    /**
     * Security Group to assign to this instance.
     *
     * @default default account security group
     */
    readonly securityGroup?: ec2.ISecurityGroup;
    /**
     * Where to place the network interfaces within the VPC.
     *
     * @default default VPC subnet
     */
    readonly subnetSelection?: ec2.SubnetSelection;
    /**
     * The instance type used to build the image.
     *
     * @default m5.large
     */
    readonly instanceType?: ec2.InstanceType;
    /**
     * The number of days log events are kept in CloudWatch Logs. When updating
     * this property, unsetting it doesn't remove the log retention policy. To
     * remove the retention policy, set the value to `INFINITE`.
     *
     * @default logs.RetentionDays.ONE_MONTH
     */
    readonly logRetention?: logs.RetentionDays;
    /**
     * Removal policy for logs of image builds. If deployment fails on the custom resource, try setting this to `RemovalPolicy.RETAIN`. This way the CodeBuild logs can still be viewed, and you can see why the build failed.
     *
     * We try to not leave anything behind when removed. But sometimes a log staying behind is useful.
     *
     * @default RemovalPolicy.DESTROY
     */
    readonly logRemovalPolicy?: RemovalPolicy;
}
declare abstract class ImageBuilderObjectBase extends cdk.Resource {
    protected constructor(scope: Construct, id: string);
    protected version(type: 'Component' | 'ImageRecipe' | 'ContainerRecipe', name: string, data: any): string;
    private versionFunction;
}
/**
 * An asset including file or directory to place inside the built image.
 */
export interface ImageBuilderAsset {
    /**
     * Path to place asset in the image.
     */
    readonly path: string;
    /**
     * Asset to place in the image.
     */
    readonly asset: s3_assets.Asset;
}
/**
 * Properties for ImageBuilderComponent construct.
 */
export interface ImageBuilderComponentProperties {
    /**
     * Component platform. Must match the builder platform.
     */
    readonly platform: 'Linux' | 'Windows';
    /**
     * Component display name.
     */
    readonly displayName: string;
    /**
     * Component description.
     */
    readonly description: string;
    /**
     * Shell commands to run when adding this component to the image.
     *
     * On Linux, these are bash commands. On Windows, there are PowerShell commands.
     */
    readonly commands: string[];
    /**
     * Optional assets to add to the built image.
     */
    readonly assets?: ImageBuilderAsset[];
}
/**
 * Components are a set of commands to run and optional files to add to an image. Components are the building blocks of images built by Image Builder.
 *
 * Example:
 *
 * ```
 * new ImageBuilderComponent(this, 'AWS CLI', {
 *   platform: 'Windows',
 *   displayName: 'AWS CLI',
 *   description: 'Install latest version of AWS CLI',
 *   commands: [
 *     '$ErrorActionPreference = \'Stop\'',
 *     'Start-Process msiexec.exe -Wait -ArgumentList \'/i https://awscli.amazonaws.com/AWSCLIV2.msi /qn\'',
 *   ],
 * }
 * ```
 */
export declare class ImageBuilderComponent extends ImageBuilderObjectBase {
    /**
     * Component ARN.
     */
    readonly arn: string;
    /**
     * Supported platform for the component.
     */
    readonly platform: 'Windows' | 'Linux';
    private readonly assets;
    constructor(scope: Construct, id: string, props: ImageBuilderComponentProperties);
    /**
     * Grants read permissions to the principal on the assets buckets.
     *
     * @param grantee
     */
    grantAssetsRead(grantee: iam.IGrantable): void;
}
/**
 * An image builder that uses Image Builder to build Docker images pre-baked with all the GitHub Actions runner requirements. Builders can be used with runner providers.
 *
 * The CodeBuild builder is better and faster. Only use this one if you have no choice. For example, if you need Windows containers.
 *
 * Each builder re-runs automatically at a set interval to make sure the images contain the latest versions of everything.
 *
 * You can create an instance of this construct to customize the image used to spin-up runners. Some runner providers may require custom components. Check the runner provider documentation. The default components work with CodeBuild.
 *
 * For example, to set a specific runner version, rebuild the image every 2 weeks, and add a few packages for the Fargate provider, use:
 *
 * ```
 * const builder = new ContainerImageBuilder(this, 'Builder', {
 *     runnerVersion: RunnerVersion.specific('2.293.0'),
 *     rebuildInterval: Duration.days(14),
 * });
 * new CodeBuildRunner(this, 'Fargate provider', {
 *     label: 'windows-codebuild',
 *     imageBuilder: builder,
 * });
 * ```
 */
export declare class ContainerImageBuilder extends Construct implements IImageBuilder {
    readonly architecture: Architecture;
    readonly os: Os;
    readonly platform: 'Windows' | 'Linux';
    readonly description: string;
    readonly runnerVersion: RunnerVersion;
    readonly repository: ecr.IRepository;
    private components;
    private boundImage?;
    readonly subnetId: string | undefined;
    readonly securityGroupIds: string[] | undefined;
    readonly instanceTypes: string[];
    readonly rebuildInterval: Duration;
    readonly logRetention: logs.RetentionDays;
    readonly logRemovalPolicy: cdk.RemovalPolicy;
    constructor(scope: Construct, id: string, props?: ContainerImageBuilderProps);
    private addBaseWindowsComponents;
    /**
     * Add a component to be installed before any other components. Useful for required system settings like certificates or proxy settings.
     * @param component
     */
    prependComponent(component: ImageBuilderComponent): void;
    /**
     * Add a component to be installed.
     * @param component
     */
    addComponent(component: ImageBuilderComponent): void;
    /**
     * Add extra trusted certificates. This helps deal with self-signed certificates for GitHub Enterprise Server.
     *
     * All first party Dockerfiles support this. Others may not.
     *
     * @param path path to directory containing a file called certs.pem containing all the required certificates
     */
    addExtraCertificates(path: string): void;
    /**
     * Called by IRunnerProvider to finalize settings and create the image builder.
     */
    bind(): RunnerImage;
    private infrastructure;
    private imageCleaner;
}
export {};
