"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeBuildImageBuilder = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_ecr_1 = require("aws-cdk-lib/aws-ecr");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
const utils_1 = require("../../utils");
const common_1 = require("../common");
/**
 * An image builder that uses CodeBuild to build Docker images pre-baked with all the GitHub Actions runner requirements. Builders can be used with runner providers.
 *
 * Each builder re-runs automatically at a set interval to make sure the images contain the latest versions of everything.
 *
 * You can create an instance of this construct to customize the image used to spin-up runners. Each provider has its own requirements for what an image should do. That's why they each provide their own Dockerfile.
 *
 * For example, to set a specific runner version, rebuild the image every 2 weeks, and add a few packages for the Fargate provider, use:
 *
 * ```
 * const builder = new CodeBuildImageBuilder(this, 'Builder', {
 *     dockerfilePath: FargateProvider.LINUX_X64_DOCKERFILE_PATH,
 *     runnerVersion: RunnerVersion.specific('2.293.0'),
 *     rebuildInterval: Duration.days(14),
 * });
 * builder.setBuildArg('EXTRA_PACKAGES', 'nginx xz-utils');
 * new FargateRunner(this, 'Fargate provider', {
 *     label: 'customized-fargate',
 *     imageBuilder: builder,
 * });
 * ```
 */
class CodeBuildImageBuilder extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.preBuild = [];
        this.postBuild = [];
        this.buildArgs = new Map();
        this.policyStatements = [];
        this.secondaryAssets = new Map();
        if (props.subnetSelection?.subnetType == aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_ISOLATED) {
            aws_cdk_lib_1.Annotations.of(this).addWarning('Private isolated subnets cannot pull from public ECR and VPC endpoint is not supported yet. ' +
                'See https://github.com/aws/containers-roadmap/issues/1160');
        }
        // set platform
        this.architecture = props.architecture ?? common_1.Architecture.X86_64;
        this.os = props.os ?? common_1.Os.LINUX;
        // create repository that only keeps one tag
        this.repository = new aws_cdk_lib_1.aws_ecr.Repository(this, 'Repository', {
            imageScanOnPush: true,
            imageTagMutability: aws_ecr_1.TagMutability.MUTABLE,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            lifecycleRules: [
                {
                    description: 'Remove untagged images that have been replaced by CodeBuild',
                    tagStatus: aws_ecr_1.TagStatus.UNTAGGED,
                    maxImageAge: aws_cdk_lib_1.Duration.days(1),
                },
            ],
        });
        // upload Dockerfile to S3 as an asset
        this.dockerfile = new aws_cdk_lib_1.aws_s3_assets.Asset(this, 'Dockerfile', {
            path: props.dockerfilePath,
        });
        // choose build image
        this.buildImage = this.getBuildImage();
    }
    /**
     * Uploads a folder to the build server at a given folder name.
     *
     * @param sourcePath path to source directory
     * @param destName name of destination folder
     */
    addFiles(sourcePath, destName) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        const asset = new aws_cdk_lib_1.aws_s3_assets.Asset(this, destName, { path: sourcePath });
        this.secondaryAssets.set(destName, asset);
        this.preBuild.push(`rm -rf "${destName}" && cp -r "$CODEBUILD_SRC_DIR_${destName}" "${destName}"`); // symlinks don't work with docker
    }
    /**
     * Adds a command that runs before `docker build`.
     *
     * @param command command to add
     */
    addPreBuildCommand(command) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        this.preBuild.push(command);
    }
    /**
     * Adds a command that runs after `docker build` and `docker push`.
     *
     * @param command command to add
     */
    addPostBuildCommand(command) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        this.postBuild.push(command);
    }
    /**
     * Adds a build argument for Docker. See the documentation for the Dockerfile you're using for a list of supported build arguments.
     *
     * @param name build argument name
     * @param value build argument value
     */
    setBuildArg(name, value) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        this.buildArgs.set(name, value);
    }
    /**
     * Add a policy statement to the builder to access resources required to the image build.
     *
     * @param statement IAM policy statement
     */
    addPolicyStatement(statement) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        this.policyStatements.push(statement);
    }
    /**
     * Add extra trusted certificates. This helps deal with self-signed certificates for GitHub Enterprise Server.
     *
     * All first party Dockerfiles support this. Others may not.
     *
     * @param path path to directory containing a file called certs.pem containing all the required certificates
     */
    addExtraCertificates(path) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        this.addFiles(path, 'extra_certs');
    }
    /**
     * Called by IRunnerProvider to finalize settings and create the image builder.
     */
    bind() {
        if (this.boundImage) {
            return this.boundImage;
        }
        // log group for the image builds
        const logGroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Logs', {
            retention: this.props.logRetention ?? aws_logs_1.RetentionDays.ONE_MONTH,
            removalPolicy: this.props.logRemovalPolicy ?? aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        // generate buildSpec
        const buildSpec = this.getBuildSpec(this.repository, logGroup, this.props.runnerVersion);
        // create CodeBuild project that builds Dockerfile and pushes to repository
        const project = new aws_cdk_lib_1.aws_codebuild.Project(this, 'CodeBuild', {
            description: `Build docker image for self-hosted GitHub runner ${this.node.path} (${this.os.name}/${this.architecture.name})`,
            buildSpec: aws_cdk_lib_1.aws_codebuild.BuildSpec.fromObject(buildSpec),
            source: aws_cdk_lib_1.aws_codebuild.Source.s3({
                bucket: this.dockerfile.bucket,
                path: this.dockerfile.s3ObjectKey,
            }),
            vpc: this.props.vpc,
            securityGroups: this.props.securityGroup ? [this.props.securityGroup] : undefined,
            subnetSelection: this.props.subnetSelection,
            timeout: this.props.timeout ?? aws_cdk_lib_1.Duration.hours(1),
            environment: {
                buildImage: this.buildImage,
                computeType: this.props.computeType ?? aws_codebuild_1.ComputeType.SMALL,
                privileged: true,
            },
            logging: {
                cloudWatch: {
                    logGroup,
                },
            },
        });
        // permissions
        this.repository.grantPullPush(project);
        this.policyStatements.forEach(project.addToRolePolicy);
        // call CodeBuild during deployment and delete all images from repository during destruction
        const cr = this.customResource(project);
        // rebuild image on a schedule
        this.rebuildImageOnSchedule(project, this.props.rebuildInterval);
        for (const [assetPath, asset] of this.secondaryAssets.entries()) {
            project.addSecondarySource(aws_cdk_lib_1.aws_codebuild.Source.s3({
                identifier: assetPath,
                bucket: asset.bucket,
                path: asset.s3ObjectKey,
            }));
        }
        this.boundImage = {
            imageRepository: aws_cdk_lib_1.aws_ecr.Repository.fromRepositoryAttributes(this, 'Dependable Image', {
                repositoryName: this.repository.repositoryName,
                // There are simpler ways to get the ARN, but we want an image object that depends on the custom resource.
                // We want whoever is using this image to automatically wait for CodeBuild to start and finish through the custom resource.
                repositoryArn: cr.ref,
            }),
            imageTag: 'latest',
            imageDigest: cr.getAtt('Digest').toString(),
            architecture: this.architecture,
            os: this.os,
            logGroup,
        };
        return this.boundImage;
    }
    getBuildImage() {
        if (this.os.is(common_1.Os.LINUX)) {
            if (this.architecture.is(common_1.Architecture.X86_64)) {
                return aws_cdk_lib_1.aws_codebuild.LinuxBuildImage.STANDARD_4_0;
            }
            else if (this.architecture.is(common_1.Architecture.ARM64)) {
                return aws_cdk_lib_1.aws_codebuild.LinuxArmBuildImage.AMAZON_LINUX_2_STANDARD_2_0;
            }
        }
        if (this.os.is(common_1.Os.WINDOWS)) {
            throw new Error('CodeBuild cannot be used to build Windows Docker images https://github.com/docker-library/docker/issues/49');
        }
        throw new Error(`Unable to find CodeBuild image for ${this.os.name}/${this.architecture.name}`);
    }
    getBuildSpec(repository, logGroup, runnerVersion) {
        // don't forget to change BUILDSPEC_VERSION when the buildSpec changes, and you want to trigger a rebuild on deploy
        let buildArgs = '';
        for (const [name, value] of this.buildArgs.entries()) {
            buildArgs += ` --build-arg "${name}"="${value}"`;
        }
        buildArgs += ` --build-arg RUNNER_VERSION="${runnerVersion ? runnerVersion.version : common_1.RunnerVersion.latest().version}"`;
        return {
            version: '0.2',
            env: {
                variables: {
                    REPO_ARN: repository.repositoryArn,
                    REPO_URI: repository.repositoryUri,
                    STACK_ID: 'unspecified',
                    REQUEST_ID: 'unspecified',
                    LOGICAL_RESOURCE_ID: 'unspecified',
                    RESPONSE_URL: 'unspecified',
                    RUNNER_VERSION: runnerVersion ? runnerVersion.version : common_1.RunnerVersion.latest().version,
                },
            },
            phases: {
                pre_build: {
                    commands: this.preBuild.concat([
                        'mkdir -p extra_certs',
                        '$(aws ecr get-login --no-include-email --region "$AWS_DEFAULT_REGION")',
                    ]),
                },
                build: {
                    commands: [
                        `docker build . -t "$REPO_URI" ${buildArgs}`,
                        'docker push "$REPO_URI"',
                    ],
                },
                post_build: {
                    commands: this.postBuild.concat([
                        'STATUS="SUCCESS"',
                        'DIGEST="UNKNOWN"',
                        'if [ $CODEBUILD_BUILD_SUCCEEDING -ne 1 ]; then STATUS="FAILED"; else DIGEST=`docker inspect "$REPO_URI" | jq -r \'.[0].RepoDigests[0] | split("@")[1] | split(":")[1]\'`; fi',
                        'cat <<EOF > /tmp/payload.json\n' +
                            '{\n' +
                            '  "StackId": "$STACK_ID",\n' +
                            '  "RequestId": "$REQUEST_ID",\n' +
                            '  "LogicalResourceId": "$LOGICAL_RESOURCE_ID",\n' +
                            '  "PhysicalResourceId": "$REPO_ARN",\n' +
                            '  "Status": "$STATUS",\n' +
                            `  "Reason": "See logs in ${logGroup.logGroupName}/$CODEBUILD_LOG_PATH (deploy again with \'cdk deploy -R\' or logRemovalPolicy=RemovalPolicy.RETAIN if they are already deleted)",\n` +
                            '  "Data": {"Digest": "$DIGEST"}\n' + // include the digest to mark the resource updated so the runner providers get updated with the latest digest too (specifically Lambda)
                            '}\n' +
                            'EOF',
                        'if [ "$RESPONSE_URL" != "unspecified" ]; then jq . /tmp/payload.json; curl -fsSL -X PUT -H "Content-Type:" -d "@/tmp/payload.json" "$RESPONSE_URL"; fi',
                    ]),
                },
            },
        };
    }
    customResource(project) {
        const crHandler = utils_1.BundledNodejsFunction.singleton(this, 'build-image', {
            description: 'Custom resource handler that triggers CodeBuild to build runner images, and cleans-up images on deletion',
            timeout: cdk.Duration.minutes(3),
        });
        const policy = new aws_cdk_lib_1.aws_iam.Policy(this, 'CR Policy', {
            statements: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['codebuild:StartBuild'],
                    resources: [project.projectArn],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['ecr:BatchDeleteImage', 'ecr:ListImages'],
                    resources: [this.repository.repositoryArn],
                }),
            ],
        });
        crHandler.role?.attachInlinePolicy(policy);
        const cr = new aws_cdk_lib_1.CustomResource(this, 'Builder', {
            serviceToken: crHandler.functionArn,
            resourceType: 'Custom::ImageBuilder',
            properties: {
                RepoName: this.repository.repositoryName,
                ProjectName: project.projectName,
                // We include a hash so the image is built immediately on changes, and we don't have to wait for its scheduled build.
                // This also helps make sure the changes are good. If they have a bug, the deployment will fail instead of just the scheduled build.
                BuildHash: this.hashBuildSettings(),
            },
        });
        // add dependencies to make sure resources are there when we need them
        cr.node.addDependency(project);
        cr.node.addDependency(policy);
        cr.node.addDependency(crHandler);
        return cr;
    }
    /**
     * Return hash of all settings that can affect the result image so we can trigger the build when it changes.
     * @private
     */
    hashBuildSettings() {
        // main Dockerfile
        let components = [this.dockerfile.assetHash];
        // all additional files
        for (const [name, asset] of this.secondaryAssets.entries()) {
            components.push(name);
            components.push(asset.assetHash);
        }
        // buildspec.yml version
        components.push(`v${CodeBuildImageBuilder.BUILDSPEC_VERSION}`);
        // user commands
        components = components.concat(this.preBuild);
        components = components.concat(this.postBuild);
        for (const [name, value] of this.buildArgs.entries()) {
            components.push(name);
            components.push(value);
        }
        // hash it
        const all = components.join('-');
        return crypto.createHash('md5').update(all).digest('hex');
    }
    rebuildImageOnSchedule(project, rebuildInterval) {
        rebuildInterval = rebuildInterval ?? aws_cdk_lib_1.Duration.days(7);
        if (rebuildInterval.toMilliseconds() != 0) {
            const scheduleRule = new aws_cdk_lib_1.aws_events.Rule(this, 'Build Schedule', {
                description: `Rebuild runner image for ${this.repository.repositoryName}`,
                schedule: aws_cdk_lib_1.aws_events.Schedule.rate(rebuildInterval),
            });
            scheduleRule.addTarget(new aws_cdk_lib_1.aws_events_targets.CodeBuildProject(project));
        }
    }
}
exports.CodeBuildImageBuilder = CodeBuildImageBuilder;
_a = JSII_RTTI_SYMBOL_1;
CodeBuildImageBuilder[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.CodeBuildImageBuilder", version: "0.4.0" };
CodeBuildImageBuilder.BUILDSPEC_VERSION = 1;
//# sourceMappingURL=data:application/json;base64,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