"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProjectType = exports.Project = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const os_1 = require("os");
const path = require("path");
const glob = require("glob");
const cleanup_1 = require("./cleanup");
const common_1 = require("./common");
const dependencies_1 = require("./dependencies");
const file_1 = require("./file");
const gitattributes_1 = require("./gitattributes");
const ignore_file_1 = require("./ignore-file");
const render_options_1 = require("./javascript/render-options");
const json_1 = require("./json");
const logger_1 = require("./logger");
const object_file_1 = require("./object-file");
const project_build_1 = require("./project-build");
const projenrc_json_1 = require("./projenrc-json");
const renovatebot_1 = require("./renovatebot");
const tasks_1 = require("./tasks");
const util_1 = require("./util");
/**
 * Base project
 */
class Project {
    constructor(options) {
        this._components = new Array();
        this.subprojects = new Array();
        this.tips = new Array();
        this.initProject = render_options_1.resolveInitProject(options);
        this.name = options.name;
        this.parent = options.parent;
        this.excludeFromCleanup = [];
        this._ejected = util_1.isTruthy(process.env.PROJEN_EJECTING);
        if (this.ejected) {
            this.projenCommand = "scripts/run-task";
        }
        else {
            this.projenCommand = options.projenCommand ?? "npx projen";
        }
        this.outdir = this.determineOutdir(options.outdir);
        this.root = this.parent ? this.parent.root : this;
        // must happen after this.outdir, this.parent and this.root are initialized
        this.parent?._addSubProject(this);
        // ------------------------------------------------------------------------
        this.gitattributes = new gitattributes_1.GitAttributesFile(this);
        this.annotateGenerated("/.projen/**"); // contents  of the .projen/ directory are generated by projen
        this.annotateGenerated(`/${this.gitattributes.path}`); // the .gitattributes file itself is generated
        if (options.gitOptions?.lfsPatterns) {
            for (const pattern of options.gitOptions.lfsPatterns) {
                this.gitattributes.addAttributes(pattern, "filter=lfs", "diff=lfs", "merge=lfs", "-text");
            }
        }
        this.gitignore = new ignore_file_1.IgnoreFile(this, ".gitignore");
        this.gitignore.exclude("node_modules/"); // created by running `npx projen`
        this.gitignore.include(`/${this.gitattributes.path}`);
        // oh no: tasks depends on gitignore so it has to be initialized after
        // smells like dep injectionn but god forbid.
        this.tasks = new tasks_1.Tasks(this);
        if (!this.ejected) {
            this.defaultTask = this.tasks.addTask(Project.DEFAULT_TASK, {
                description: "Synthesize project files",
            });
            this.ejectTask = this.tasks.addTask("eject", {
                description: "Remove projen from the project",
                env: {
                    PROJEN_EJECTING: "true",
                },
            });
            this.ejectTask.spawn(this.defaultTask);
        }
        this.projectBuild = new project_build_1.ProjectBuild(this);
        this.deps = new dependencies_1.Dependencies(this);
        this.logger = new logger_1.Logger(this, options.logging);
        const projenrcJson = options.projenrcJson ?? false;
        if (projenrcJson) {
            new projenrc_json_1.Projenrc(this, options.projenrcJsonOptions);
        }
        if (options.renovatebot) {
            new renovatebot_1.Renovatebot(this, options.renovatebotOptions);
        }
        this.commitGenerated = options.commitGenerated ?? true;
        if (!this.ejected) {
            new json_1.JsonFile(this, cleanup_1.FILE_MANIFEST, {
                omitEmpty: true,
                obj: () => ({
                    // replace `\` with `/` to ensure paths match across platforms
                    files: this.files
                        .filter((f) => f.readonly)
                        .map((f) => f.path.replace(/\\/g, "/")),
                }),
                // This file is used by projen to track the generated files, so must be committed.
                committed: true,
            });
        }
    }
    /**
     * Returns all the components within this project.
     */
    get components() {
        return [...this._components];
    }
    /**
     * All files in this project.
     */
    get files() {
        const isFile = (c) => c instanceof file_1.FileBase;
        return this._components
            .filter(isFile)
            .sort((f1, f2) => f1.path.localeCompare(f2.path));
    }
    /**
     * Adds a new task to this project. This will fail if the project already has
     * a task with this name.
     *
     * @param name The task name to add
     * @param props Task properties
     */
    addTask(name, props = {}) {
        return this.tasks.addTask(name, props);
    }
    /**
     * Removes a task from a project.
     *
     * @param name The name of the task to remove.
     *
     * @returns The `Task` that was removed, otherwise `undefined`.
     */
    removeTask(name) {
        return this.tasks.removeTask(name);
    }
    get buildTask() {
        return this.projectBuild.buildTask;
    }
    get compileTask() {
        return this.projectBuild.compileTask;
    }
    get testTask() {
        return this.projectBuild.testTask;
    }
    get preCompileTask() {
        return this.projectBuild.preCompileTask;
    }
    get postCompileTask() {
        return this.projectBuild.postCompileTask;
    }
    get packageTask() {
        return this.projectBuild.packageTask;
    }
    /**
     * Finds a file at the specified relative path within this project and all
     * its subprojects.
     *
     * @param filePath The file path. If this path is relative, it will be resolved
     * from the root of _this_ project.
     * @returns a `FileBase` or undefined if there is no file in that path
     */
    tryFindFile(filePath) {
        const absolute = path.isAbsolute(filePath)
            ? filePath
            : path.resolve(this.outdir, filePath);
        for (const file of this.files) {
            if (absolute === file.absolutePath) {
                return file;
            }
        }
        for (const child of this.subprojects) {
            const file = child.tryFindFile(absolute);
            if (file) {
                return file;
            }
        }
        return undefined;
    }
    /**
     * Finds a json file by name.
     * @param filePath The file path.
     * @deprecated use `tryFindObjectFile`
     */
    tryFindJsonFile(filePath) {
        const file = this.tryFindObjectFile(filePath);
        if (!file) {
            return undefined;
        }
        if (!(file instanceof json_1.JsonFile)) {
            throw new Error(`found file ${filePath} but it is not a JsonFile. got: ${file.constructor.name}`);
        }
        return file;
    }
    /**
     * Finds an object file (like JsonFile, YamlFile, etc.) by name.
     * @param filePath The file path.
     */
    tryFindObjectFile(filePath) {
        const file = this.tryFindFile(filePath);
        if (!file) {
            return undefined;
        }
        if (!(file instanceof object_file_1.ObjectFile)) {
            throw new Error(`found file ${filePath} but it is not a ObjectFile. got: ${file.constructor.name}`);
        }
        return file;
    }
    /**
     * Finds a file at the specified relative path within this project and removes
     * it.
     *
     * @param filePath The file path. If this path is relative, it will be
     * resolved from the root of _this_ project.
     * @returns a `FileBase` if the file was found and removed, or undefined if
     * the file was not found.
     */
    tryRemoveFile(filePath) {
        const absolute = path.isAbsolute(filePath)
            ? filePath
            : path.resolve(this.outdir, filePath);
        const isFile = (c) => c instanceof file_1.FileBase;
        const index = this._components.findIndex((c) => isFile(c) && c.absolutePath === absolute);
        if (index !== -1) {
            return this._components.splice(index, 1)[0];
        }
        for (const child of this.subprojects) {
            const file = child.tryRemoveFile(absolute);
            if (file) {
                return file;
            }
        }
        return undefined;
    }
    /**
     * Prints a "tip" message during synthesis.
     * @param message The message
     * @deprecated - use `project.logger.info(message)` to show messages during synthesis
     */
    addTip(message) {
        this.tips.push(message);
    }
    /**
     * Exclude the matching files from pre-synth cleanup. Can be used when, for example, some
     * source files include the projen marker and we don't want them to be erased during synth.
     *
     * @param globs The glob patterns to match
     */
    addExcludeFromCleanup(...globs) {
        this.excludeFromCleanup.push(...globs);
    }
    /**
     * Returns the shell command to execute in order to run a task.
     *
     * By default, this is `npx projen@<version> <task>`
     *
     * @param task The task for which the command is required
     */
    runTaskCommand(task) {
        return `npx projen@${common_1.PROJEN_VERSION} ${task.name}`;
    }
    /**
     * Exclude these files from the bundled package. Implemented by project types based on the
     * packaging mechanism. For example, `NodeProject` delegates this to `.npmignore`.
     *
     * @param _pattern The glob pattern to exclude
     */
    addPackageIgnore(_pattern) {
        // nothing to do at the abstract level
    }
    /**
     * Adds a .gitignore pattern.
     * @param pattern The glob pattern to ignore.
     */
    addGitIgnore(pattern) {
        this.gitignore.addPatterns(pattern);
    }
    /**
     * Consider a set of files as "generated". This method is implemented by
     * derived classes and used for example, to add git attributes to tell GitHub
     * that certain files are generated.
     *
     * @param _glob the glob pattern to match (could be a file path).
     */
    annotateGenerated(_glob) {
        // nothing to do at the abstract level
    }
    /**
     * Synthesize all project files into `outdir`.
     *
     * 1. Call "this.preSynthesize()"
     * 2. Delete all generated files
     * 3. Synthesize all sub-projects
     * 4. Synthesize all components of this project
     * 5. Call "postSynthesize()" for all components of this project
     * 6. Call "this.postSynthesize()"
     */
    synth() {
        const outdir = this.outdir;
        this.logger.debug("Synthesizing project...");
        this.preSynthesize();
        for (const comp of this._components) {
            comp.preSynthesize();
        }
        // we exclude all subproject directories to ensure that when subproject.synth()
        // gets called below after cleanup(), subproject generated files are left intact
        for (const subproject of this.subprojects) {
            this.addExcludeFromCleanup(subproject.outdir + "/**");
        }
        // delete orphaned files before we start synthesizing new ones
        cleanup_1.cleanup(outdir, this.files.map((f) => f.path.replace(/\\/g, "/")), this.excludeFromCleanup);
        for (const subproject of this.subprojects) {
            subproject.synth();
        }
        for (const comp of this._components) {
            comp.synthesize();
        }
        if (!util_1.isTruthy(process.env.PROJEN_DISABLE_POST)) {
            for (const comp of this._components) {
                comp.postSynthesize();
            }
            // project-level hook
            this.postSynthesize();
        }
        if (this.ejected) {
            this.logger.debug("Ejecting project...");
            // Backup projenrc files
            const files = glob.sync(".projenrc.*", {
                cwd: this.outdir,
                dot: true,
                nodir: true,
                absolute: true,
            });
            for (const file of files) {
                fs_1.renameSync(file, `${file}.bak`);
            }
        }
        this.logger.debug("Synthesis complete");
    }
    /**
     * Whether or not the project is being ejected.
     */
    get ejected() {
        return this._ejected;
    }
    /**
     * Called before all components are synthesized.
     */
    preSynthesize() { }
    /**
     * Called after all components are synthesized. Order is *not* guaranteed.
     */
    postSynthesize() { }
    /**
     * Adds a component to the project.
     * @internal
     */
    _addComponent(component) {
        this._components.push(component);
    }
    /**
     * Adds a sub-project to this project.
     *
     * This is automatically called when a new project is created with `parent`
     * pointing to this project, so there is no real need to call this manually.
     *
     * @param sub-project The child project to add.
     * @internal
     */
    _addSubProject(subproject) {
        if (subproject.parent !== this) {
            throw new Error('"parent" of child project must be this project');
        }
        // check that `outdir` is exclusive
        for (const p of this.subprojects) {
            if (path.resolve(p.outdir) === path.resolve(subproject.outdir)) {
                throw new Error(`there is already a sub-project with "outdir": ${subproject.outdir}`);
            }
        }
        this.subprojects.push(subproject);
    }
    /**
     * Resolves the project's output directory.
     */
    determineOutdir(outdirOption) {
        if (this.parent && outdirOption && path.isAbsolute(outdirOption)) {
            throw new Error('"outdir" must be a relative path');
        }
        // if this is a subproject, it is relative to the parent
        if (this.parent) {
            if (!outdirOption) {
                throw new Error('"outdir" must be specified for subprojects');
            }
            return path.resolve(this.parent.outdir, outdirOption);
        }
        // if this is running inside a test and outdir is not explicitly set
        // use a temp directory (unless cwd is aleady under tmp)
        if (common_1.IS_TEST_RUN && !outdirOption) {
            const realCwd = fs_1.realpathSync(process.cwd());
            const realTmp = fs_1.realpathSync(os_1.tmpdir());
            if (realCwd.startsWith(realTmp)) {
                return path.resolve(realCwd, outdirOption ?? ".");
            }
            return fs_1.mkdtempSync(path.join(os_1.tmpdir(), "projen."));
        }
        return path.resolve(outdirOption ?? ".");
    }
}
exports.Project = Project;
_a = JSII_RTTI_SYMBOL_1;
Project[_a] = { fqn: "projen.Project", version: "0.67.87" };
/**
 * The name of the default task (the task executed when `projen` is run without arguments). Normally
 * this task should synthesize the project files.
 */
Project.DEFAULT_TASK = "default";
/**
 * Which type of project this is.
 *
 * @deprecated no longer supported at the base project level
 */
var ProjectType;
(function (ProjectType) {
    /**
     * This module may be a either a library or an app.
     */
    ProjectType["UNKNOWN"] = "unknown";
    /**
     * This is a library, intended to be published to a package manager and
     * consumed by other projects.
     */
    ProjectType["LIB"] = "lib";
    /**
     * This is an app (service, tool, website, etc). Its artifacts are intended to
     * be deployed or published for end-user consumption.
     */
    ProjectType["APP"] = "app";
})(ProjectType = exports.ProjectType || (exports.ProjectType = {}));
//# sourceMappingURL=data:application/json;base64,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