"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.synth = void 0;
const child_process_1 = require("child_process");
const os = require("os");
const path = require("path");
const fs = require("fs-extra");
const common_1 = require("../common");
const logging = require("../logging");
const project_1 = require("../project");
const projenModule = path.dirname(require.resolve("../../package.json"));
async function synth(runtime, options) {
    const workdir = runtime.workdir;
    const rcfile = path.resolve(workdir, options.rcfile ?? common_1.PROJEN_RC); // TODO: support non javascript projenrc (e.g. java projects)
    // if --rc points to .projenrc.js, then behave as if it wasn't specified.
    if (rcfile === path.resolve(workdir, common_1.PROJEN_RC)) {
        delete options.rcfile;
    }
    // if there are no tasks, we assume this is not a projen project (modern
    // projects must at least have the "default" task).
    if (runtime.tasks.length === 0 && !fs.existsSync(rcfile)) {
        logging.error('Unable to find projen project. Use "projen new" to create a new project.');
        process.exit(1);
    }
    // run synth once
    const success = await trySynth();
    if (options.watch) {
        // if we are in watch mode, start the watch loop
        watchLoop();
    }
    else if (!success) {
        // make sure exit code is non-zero if we are not in watch mode
        process.exit(1);
    }
    async function trySynth() {
        // determine if post synthesis tasks should be executed (e.g. "yarn install").
        process.env.PROJEN_DISABLE_POST = (!options.post).toString();
        try {
            const defaultTask = runtime.tasks.find((t) => t.name === project_1.Project.DEFAULT_TASK);
            // if "--rc" is specified, ignore the default task
            if (defaultTask) {
                if (!options.rcfile) {
                    runtime.runTask(defaultTask.name);
                    return true;
                }
                else {
                    logging.warn("Default task skipped. Trying legacy synthesis since --rc is specified");
                }
            }
            // for backwards compatibility, if there is a .projenrc.js file, default to "node .projenrc.js"
            if (tryLegacySynth()) {
                return true;
            }
            throw new Error('Unable to find a task named "default"');
        }
        catch (e) {
            logging.error(`Synthesis failed: ${e.message}`);
            return false;
        }
    }
    function watchLoop() {
        logging.info(`Watching for changes in ${workdir}...`);
        const watch = fs.watch(workdir, { recursive: true });
        watch.on("change", (event) => {
            // we only care about "change" events
            if (event !== "change") {
                return;
            }
            process.stdout.write("\x1Bc"); // clear screen
            watch.close();
            trySynth()
                .then(() => watchLoop())
                .catch(() => watchLoop());
        });
    }
    function tryLegacySynth() {
        const rcdir = path.dirname(rcfile);
        if (!fs.existsSync(rcfile)) {
            return false;
        }
        // if node_modules/projen is not a directory or does not exist, create a
        // temporary symlink to the projen that we are currently running in order to
        // allow .projenrc.js to `require()` it.
        const nodeModules = path.resolve(rcdir, "node_modules");
        const projenModulePath = path.resolve(nodeModules, "projen");
        if (!fs.existsSync(path.join(projenModulePath, "package.json")) ||
            !fs.statSync(projenModulePath).isDirectory()) {
            fs.removeSync(projenModulePath);
            fs.mkdirpSync(nodeModules);
            fs.symlinkSync(projenModule, projenModulePath, os.platform() === "win32" ? "junction" : null);
        }
        const ret = child_process_1.spawnSync(process.execPath, [rcfile], {
            stdio: ["inherit", "inherit", "pipe"],
        });
        if (ret.error) {
            throw new Error(`Synthesis failed: ${ret.error}`);
        }
        else if (ret.status !== 0) {
            logging.error(ret.stderr.toString());
            throw new Error(`Synthesis failed: calling "${process.execPath} ${rcfile}" exited with status=${ret.status}`);
        }
        return true;
    }
}
exports.synth = synth;
//# sourceMappingURL=data:application/json;base64,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