"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ColumnEncoding = exports.TableSortStyle = exports.TableDistStyle = exports.Table = exports.TableAction = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/* eslint-disable import/no-extraneous-dependencies */
const cdk = require("aws-cdk-lib");
const cx_api_1 = require("aws-cdk-lib/cx-api");
const constructs_1 = require("constructs");
const database_query_1 = require("./private/database-query");
const handler_name_1 = require("./private/database-query-provider/handler-name");
const util_1 = require("./private/database-query-provider/util");
/**
 * An action that a Redshift user can be granted privilege to perform on a table.
 */
var TableAction;
(function (TableAction) {
    /**
     * Grants privilege to select data from a table or view using a SELECT statement.
     */
    TableAction[TableAction["SELECT"] = 0] = "SELECT";
    /**
     * Grants privilege to load data into a table using an INSERT statement or a COPY statement.
     */
    TableAction[TableAction["INSERT"] = 1] = "INSERT";
    /**
     * Grants privilege to update a table column using an UPDATE statement.
     */
    TableAction[TableAction["UPDATE"] = 2] = "UPDATE";
    /**
     * Grants privilege to delete a data row from a table.
     */
    TableAction[TableAction["DELETE"] = 3] = "DELETE";
    /**
     * Grants privilege to drop a table.
     */
    TableAction[TableAction["DROP"] = 4] = "DROP";
    /**
     * Grants privilege to create a foreign key constraint.
     *
     * You need to grant this privilege on both the referenced table and the referencing table; otherwise, the user can't create the constraint.
     */
    TableAction[TableAction["REFERENCES"] = 5] = "REFERENCES";
    /**
     * Grants all available privileges at once to the specified user or user group.
     */
    TableAction[TableAction["ALL"] = 6] = "ALL";
})(TableAction = exports.TableAction || (exports.TableAction = {}));
class TableBase extends constructs_1.Construct {
    grant(user, ...actions) {
        user.addTablePrivileges(this, ...actions);
    }
}
/**
 * A table in a Redshift cluster.
 */
class Table extends TableBase {
    /**
     * Specify a Redshift table using a table name and schema that already exists.
     */
    static fromTableAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_TableAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromTableAttributes);
            }
            throw error;
        }
        return new class extends TableBase {
            constructor() {
                super(...arguments);
                this.tableName = attrs.tableName;
                this.tableColumns = attrs.tableColumns;
                this.cluster = attrs.cluster;
                this.databaseName = attrs.databaseName;
            }
        }(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_TableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Table);
            }
            throw error;
        }
        this.addColumnIds(props.tableColumns);
        this.validateDistKeyColumns(props.tableColumns);
        if (props.distStyle) {
            this.validateDistStyle(props.distStyle, props.tableColumns);
        }
        if (props.sortStyle) {
            this.validateSortStyle(props.sortStyle, props.tableColumns);
        }
        this.tableColumns = props.tableColumns;
        this.cluster = props.cluster;
        this.databaseName = props.databaseName;
        const useColumnIds = !!cdk.FeatureFlags.of(this).isEnabled(cx_api_1.REDSHIFT_COLUMN_ID);
        this.resource = new database_query_1.DatabaseQuery(this, 'Resource', {
            removalPolicy: cdk.RemovalPolicy.RETAIN,
            ...props,
            handler: handler_name_1.HandlerName.Table,
            properties: {
                tableName: {
                    prefix: props.tableName ?? cdk.Names.uniqueId(this),
                    generateSuffix: !props.tableName ? 'true' : 'false',
                },
                tableColumns: this.tableColumns,
                distStyle: props.distStyle,
                sortStyle: props.sortStyle ?? this.getDefaultSortStyle(props.tableColumns),
                tableComment: props.tableComment,
                useColumnIds,
            },
        });
        this.tableName = this.resource.ref;
    }
    /**
     * Apply the given removal policy to this resource
     *
     * The Removal Policy controls what happens to this resource when it stops
     * being managed by CloudFormation, either because you've removed it from the
     * CDK application or because you've made a change that requires the resource
     * to be replaced.
     *
     * The resource can be destroyed (`RemovalPolicy.DESTROY`), or left in your AWS
     * account for data recovery and cleanup later (`RemovalPolicy.RETAIN`).
     *
     * This resource is retained by default.
     */
    applyRemovalPolicy(policy) {
        this.resource.applyRemovalPolicy(policy);
    }
    validateDistKeyColumns(columns) {
        try {
            (0, util_1.getDistKeyColumn)(columns);
        }
        catch {
            throw new Error('Only one column can be configured as distKey.');
        }
    }
    validateDistStyle(distStyle, columns) {
        const distKeyColumn = (0, util_1.getDistKeyColumn)(columns);
        if (distKeyColumn && distStyle !== TableDistStyle.KEY) {
            throw new Error(`Only 'TableDistStyle.KEY' can be configured when distKey is also configured. Found ${distStyle}`);
        }
        if (!distKeyColumn && distStyle === TableDistStyle.KEY) {
            throw new Error('distStyle of "TableDistStyle.KEY" can only be configured when distKey is also configured.');
        }
    }
    validateSortStyle(sortStyle, columns) {
        const sortKeyColumns = (0, util_1.getSortKeyColumns)(columns);
        if (sortKeyColumns.length === 0 && sortStyle !== TableSortStyle.AUTO) {
            throw new Error(`sortStyle of '${sortStyle}' can only be configured when sortKey is also configured.`);
        }
        if (sortKeyColumns.length > 0 && sortStyle === TableSortStyle.AUTO) {
            throw new Error(`sortStyle of '${TableSortStyle.AUTO}' cannot be configured when sortKey is also configured.`);
        }
    }
    getDefaultSortStyle(columns) {
        const sortKeyColumns = (0, util_1.getSortKeyColumns)(columns);
        return (sortKeyColumns.length === 0) ? TableSortStyle.AUTO : TableSortStyle.COMPOUND;
    }
    addColumnIds(columns) {
        const columnIds = new Set();
        for (const column of columns) {
            if (column.id) {
                if (columnIds.has(column.id)) {
                    throw new Error(`Column id '${column.id}' is not unique.`);
                }
                columnIds.add(column.id);
            }
        }
    }
}
_a = JSII_RTTI_SYMBOL_1;
Table[_a] = { fqn: "@aws-cdk/aws-redshift-alpha.Table", version: "2.72.1-alpha.0" };
exports.Table = Table;
/**
 * The data distribution style of a table.
 */
var TableDistStyle;
(function (TableDistStyle) {
    /**
     *  Amazon Redshift assigns an optimal distribution style based on the table data
     */
    TableDistStyle["AUTO"] = "AUTO";
    /**
     * The data in the table is spread evenly across the nodes in a cluster in a round-robin distribution.
     */
    TableDistStyle["EVEN"] = "EVEN";
    /**
     * The data is distributed by the values in the DISTKEY column.
     */
    TableDistStyle["KEY"] = "KEY";
    /**
     * A copy of the entire table is distributed to every node.
     */
    TableDistStyle["ALL"] = "ALL";
})(TableDistStyle = exports.TableDistStyle || (exports.TableDistStyle = {}));
/**
 * The sort style of a table.
 */
var TableSortStyle;
(function (TableSortStyle) {
    /**
     * Amazon Redshift assigns an optimal sort key based on the table data.
     */
    TableSortStyle["AUTO"] = "AUTO";
    /**
     * Specifies that the data is sorted using a compound key made up of all of the listed columns,
     * in the order they are listed.
     */
    TableSortStyle["COMPOUND"] = "COMPOUND";
    /**
     * Specifies that the data is sorted using an interleaved sort key.
     */
    TableSortStyle["INTERLEAVED"] = "INTERLEAVED";
})(TableSortStyle = exports.TableSortStyle || (exports.TableSortStyle = {}));
/**
 * The compression encoding of a column.
 *
 * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Compression_encodings.html
 */
var ColumnEncoding;
(function (ColumnEncoding) {
    /**
     * Amazon Redshift assigns an optimal encoding based on the column data.
     * This is the default.
     */
    ColumnEncoding["AUTO"] = "AUTO";
    /**
     * The column is not compressed.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Raw_encoding.html
     */
    ColumnEncoding["RAW"] = "RAW";
    /**
     * The column is compressed using the AZ64 algorithm.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/az64-encoding.html
     */
    ColumnEncoding["AZ64"] = "AZ64";
    /**
     * The column is compressed using a separate dictionary for each block column value on disk.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Byte_dictionary_encoding.html
     */
    ColumnEncoding["BYTEDICT"] = "BYTEDICT";
    /**
     * The column is compressed based on the difference between values in the column.
     * This records differences as 1-byte values.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Delta_encoding.html
     */
    ColumnEncoding["DELTA"] = "DELTA";
    /**
     * The column is compressed based on the difference between values in the column.
     * This records differences as 2-byte values.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Delta_encoding.html
     */
    ColumnEncoding["DELTA32K"] = "DELTA32K";
    /**
     * The column is compressed using the LZO algorithm.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/lzo-encoding.html
     */
    ColumnEncoding["LZO"] = "LZO";
    /**
     * The column is compressed to a smaller storage size than the original data type.
     * The compressed storage size is 1 byte.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_MostlyN_encoding.html
     */
    ColumnEncoding["MOSTLY8"] = "MOSTLY8";
    /**
     * The column is compressed to a smaller storage size than the original data type.
     * The compressed storage size is 2 bytes.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_MostlyN_encoding.html
     */
    ColumnEncoding["MOSTLY16"] = "MOSTLY16";
    /**
     * The column is compressed to a smaller storage size than the original data type.
     * The compressed storage size is 4 bytes.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_MostlyN_encoding.html
     */
    ColumnEncoding["MOSTLY32"] = "MOSTLY32";
    /**
     * The column is compressed by recording the number of occurrences of each value in the column.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Runlength_encoding.html
     */
    ColumnEncoding["RUNLENGTH"] = "RUNLENGTH";
    /**
     * The column is compressed by recording the first 245 unique words and then using a 1-byte index to represent each word.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Text255_encoding.html
     */
    ColumnEncoding["TEXT255"] = "TEXT255";
    /**
     * The column is compressed by recording the first 32K unique words and then using a 2-byte index to represent each word.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Text255_encoding.html
     */
    ColumnEncoding["TEXT32K"] = "TEXT32K";
    /**
     * The column is compressed using the ZSTD algorithm.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/zstd-encoding.html
     */
    ColumnEncoding["ZSTD"] = "ZSTD";
})(ColumnEncoding = exports.ColumnEncoding || (exports.ColumnEncoding = {}));
//# sourceMappingURL=data:application/json;base64,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