"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const redshift_data_1 = require("./redshift-data");
const types_1 = require("./types");
const util_1 = require("./util");
async function handler(props, event) {
    const tableNamePrefix = props.tableName.prefix;
    const tableNameSuffix = props.tableName.generateSuffix === 'true' ? `${event.RequestId.substring(0, 8)}` : '';
    const tableColumns = props.tableColumns;
    const tableAndClusterProps = props;
    const useColumnIds = props.useColumnIds;
    if (event.RequestType === 'Create') {
        const tableName = await createTable(tableNamePrefix, tableNameSuffix, tableColumns, tableAndClusterProps);
        return { PhysicalResourceId: tableName };
    }
    else if (event.RequestType === 'Delete') {
        await dropTable(event.PhysicalResourceId, tableAndClusterProps);
        return;
    }
    else if (event.RequestType === 'Update') {
        const tableName = await updateTable(event.PhysicalResourceId, tableNamePrefix, tableNameSuffix, tableColumns, useColumnIds, tableAndClusterProps, event.OldResourceProperties);
        return { PhysicalResourceId: tableName };
    }
    else {
        /* eslint-disable-next-line dot-notation */
        throw new Error(`Unrecognized event type: ${event['RequestType']}`);
    }
}
exports.handler = handler;
async function createTable(tableNamePrefix, tableNameSuffix, tableColumns, tableAndClusterProps) {
    const tableName = tableNamePrefix + tableNameSuffix;
    const tableColumnsString = tableColumns.map(column => `${column.name} ${column.dataType}${getEncodingColumnString(column)}`).join();
    let statement = `CREATE TABLE ${tableName} (${tableColumnsString})`;
    if (tableAndClusterProps.distStyle) {
        statement += ` DISTSTYLE ${tableAndClusterProps.distStyle}`;
    }
    const distKeyColumn = (0, util_1.getDistKeyColumn)(tableColumns);
    if (distKeyColumn) {
        statement += ` DISTKEY(${distKeyColumn.name})`;
    }
    const sortKeyColumns = (0, util_1.getSortKeyColumns)(tableColumns);
    if (sortKeyColumns.length > 0) {
        const sortKeyColumnsString = getSortKeyColumnsString(sortKeyColumns);
        statement += ` ${tableAndClusterProps.sortStyle} SORTKEY(${sortKeyColumnsString})`;
    }
    await (0, redshift_data_1.executeStatement)(statement, tableAndClusterProps);
    for (const column of tableColumns) {
        if (column.comment) {
            await (0, redshift_data_1.executeStatement)(`COMMENT ON COLUMN ${tableName}.${column.name} IS '${column.comment}'`, tableAndClusterProps);
        }
    }
    if (tableAndClusterProps.tableComment) {
        await (0, redshift_data_1.executeStatement)(`COMMENT ON TABLE ${tableName} IS '${tableAndClusterProps.tableComment}'`, tableAndClusterProps);
    }
    return tableName;
}
async function dropTable(tableName, clusterProps) {
    await (0, redshift_data_1.executeStatement)(`DROP TABLE ${tableName}`, clusterProps);
}
async function updateTable(tableName, tableNamePrefix, tableNameSuffix, tableColumns, useColumnIds, tableAndClusterProps, oldResourceProperties) {
    const alterationStatements = [];
    const oldClusterProps = oldResourceProperties;
    if (tableAndClusterProps.clusterName !== oldClusterProps.clusterName || tableAndClusterProps.databaseName !== oldClusterProps.databaseName) {
        return createTable(tableNamePrefix, tableNameSuffix, tableColumns, tableAndClusterProps);
    }
    const oldTableNamePrefix = oldResourceProperties.tableName.prefix;
    if (tableNamePrefix !== oldTableNamePrefix) {
        return createTable(tableNamePrefix, tableNameSuffix, tableColumns, tableAndClusterProps);
    }
    const oldTableColumns = oldResourceProperties.tableColumns;
    const columnDeletions = oldTableColumns.filter(oldColumn => (tableColumns.every(column => {
        if (useColumnIds) {
            return oldColumn.id ? oldColumn.id !== column.id : oldColumn.name !== column.name;
        }
        return oldColumn.name !== column.name;
    })));
    if (columnDeletions.length > 0) {
        alterationStatements.push(...columnDeletions.map(column => `ALTER TABLE ${tableName} DROP COLUMN ${column.name}`));
    }
    const columnAdditions = tableColumns.filter(column => {
        return !oldTableColumns.some(oldColumn => {
            if (useColumnIds) {
                return oldColumn.id ? oldColumn.id === column.id : oldColumn.name === column.name;
            }
            return oldColumn.name === column.name;
        });
    }).map(column => `ADD ${column.name} ${column.dataType}`);
    if (columnAdditions.length > 0) {
        alterationStatements.push(...columnAdditions.map(addition => `ALTER TABLE ${tableName} ${addition}`));
    }
    const columnEncoding = tableColumns.filter(column => {
        return oldTableColumns.some(oldColumn => column.name === oldColumn.name && column.encoding !== oldColumn.encoding);
    }).map(column => `ALTER COLUMN ${column.name} ENCODE ${column.encoding || 'AUTO'}`);
    if (columnEncoding.length > 0) {
        alterationStatements.push(`ALTER TABLE ${tableName} ${columnEncoding.join(', ')}`);
    }
    const columnComments = tableColumns.filter(column => {
        return oldTableColumns.some(oldColumn => column.name === oldColumn.name && column.comment !== oldColumn.comment);
    }).map(column => `COMMENT ON COLUMN ${tableName}.${column.name} IS ${column.comment ? `'${column.comment}'` : 'NULL'}`);
    if (columnComments.length > 0) {
        alterationStatements.push(...columnComments);
    }
    if (useColumnIds) {
        const columnNameUpdates = tableColumns.reduce((updates, column) => {
            const oldColumn = oldTableColumns.find(oldCol => oldCol.id && oldCol.id === column.id);
            if (oldColumn && oldColumn.name !== column.name) {
                updates[oldColumn.name] = column.name;
            }
            return updates;
        }, {});
        if (Object.keys(columnNameUpdates).length > 0) {
            alterationStatements.push(...Object.entries(columnNameUpdates).map(([oldName, newName]) => (`ALTER TABLE ${tableName} RENAME COLUMN ${oldName} TO ${newName}`)));
        }
    }
    const oldDistStyle = oldResourceProperties.distStyle;
    if ((!oldDistStyle && tableAndClusterProps.distStyle) ||
        (oldDistStyle && !tableAndClusterProps.distStyle)) {
        return createTable(tableNamePrefix, tableNameSuffix, tableColumns, tableAndClusterProps);
    }
    else if (oldDistStyle !== tableAndClusterProps.distStyle) {
        alterationStatements.push(`ALTER TABLE ${tableName} ALTER DISTSTYLE ${tableAndClusterProps.distStyle}`);
    }
    const oldDistKey = (0, util_1.getDistKeyColumn)(oldTableColumns)?.name;
    const newDistKey = (0, util_1.getDistKeyColumn)(tableColumns)?.name;
    if ((!oldDistKey && newDistKey) || (oldDistKey && !newDistKey)) {
        return createTable(tableNamePrefix, tableNameSuffix, tableColumns, tableAndClusterProps);
    }
    else if (oldDistKey !== newDistKey) {
        alterationStatements.push(`ALTER TABLE ${tableName} ALTER DISTKEY ${newDistKey}`);
    }
    const oldSortKeyColumns = (0, util_1.getSortKeyColumns)(oldTableColumns);
    const newSortKeyColumns = (0, util_1.getSortKeyColumns)(tableColumns);
    const oldSortStyle = oldResourceProperties.sortStyle;
    const newSortStyle = tableAndClusterProps.sortStyle;
    if ((oldSortStyle === newSortStyle && !(0, util_1.areColumnsEqual)(oldSortKeyColumns, newSortKeyColumns))
        || (oldSortStyle !== newSortStyle)) {
        switch (newSortStyle) {
            case types_1.TableSortStyle.INTERLEAVED:
                // INTERLEAVED sort key addition requires replacement.
                // https://docs.aws.amazon.com/redshift/latest/dg/r_ALTER_TABLE.html
                return createTable(tableNamePrefix, tableNameSuffix, tableColumns, tableAndClusterProps);
            case types_1.TableSortStyle.COMPOUND: {
                const sortKeyColumnsString = getSortKeyColumnsString(newSortKeyColumns);
                alterationStatements.push(`ALTER TABLE ${tableName} ALTER ${newSortStyle} SORTKEY(${sortKeyColumnsString})`);
                break;
            }
            case types_1.TableSortStyle.AUTO: {
                alterationStatements.push(`ALTER TABLE ${tableName} ALTER SORTKEY ${newSortStyle}`);
                break;
            }
        }
    }
    const oldComment = oldResourceProperties.tableComment;
    const newComment = tableAndClusterProps.tableComment;
    if (oldComment !== newComment) {
        alterationStatements.push(`COMMENT ON TABLE ${tableName} IS ${newComment ? `'${newComment}'` : 'NULL'}`);
    }
    await Promise.all(alterationStatements.map(statement => (0, redshift_data_1.executeStatement)(statement, tableAndClusterProps)));
    return tableName;
}
function getSortKeyColumnsString(sortKeyColumns) {
    return sortKeyColumns.map(column => column.name).join();
}
function getEncodingColumnString(column) {
    if (column.encoding) {
        return ` ENCODE ${column.encoding}`;
    }
    return '';
}
//# sourceMappingURL=data:application/json;base64,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