# -*- coding: utf-8 -*-
from setuptools import setup

modules = \
['flake8_mock_spec']
install_requires = \
['flake8>=5']

entry_points = \
{'flake8.extension': ['TMS = flake8_mock_spec:Plugin']}

setup_kwargs = {
    'name': 'flake8-mock-spec',
    'version': '1.4.0',
    'description': 'A linter that checks mocks are constructed with the spec argument',
    'long_description': '# flake8-mock-spec\n\nAre you using mocks in your code and want to ensure that you are not accessing\nor calling methods that the mocked objects don\'t have? Using mocks incorrectly\ncan lead to bugs in your code and falsely passing tests. To avoid this,\nflake8-mock-spec linter has been created to enforce the use of the spec\nargument when creating mocks. This ensures that your use of mocks is compliant\nwith the interface of the actual object being mocked, and helps you catch\nerrors early on. Using this linter can save you time and help you write more\nrobust and maintainable code.\n\n## Getting Started\n\nTo start using `flake8-mock-spec`, you need to install the package and run it\non your source code. Here are the steps to get started:\n\n1. Create a virtual environment and activate it:\n\n  ```shell\n  python -m venv venv\n  source ./venv/bin/activate\n  ```\n\n2. Install `flake8-mock-spec`:\n\n  ```shell\n  pip install flake8-mock-spec\n  ```\n\n3. Run `flake8` on your source code:\n\n  ```shell\n  flake8 test_source.py\n  ```\n\nFor example, consider the following code:\n\n```Python\n# test_source.py\nfrom unittest import mock\n\ndef test_foo():\n    mocked_foo = mock.Mock()\n```\n\nRunning `flake8` on this code will produce the following warning:\n\n```shell\nflake8 test_source.py\ntest_source.py:5:22: TMS010 unittest.mock.Mock instances should be constructed with the spec or spec_set argument, more information: https://github.com/jdkandersson/flake8-mock-spec#fix-tms010\n```\n\nTo resolve this warning, you need to specify the `spec` or `spec_set` argument\nwhen creating the mock object:\n\n```Python\n# test_source.py\nfrom unittest import mock\n\nfrom foo import Foo\n\ndef test_foo():\n    mocked_foo = mock.Mock(spec=Foo)\n```\n\n## Rules\n\nA set of linting rules have been defined to ensure best practices are followed\nwhen using unittest.mock library. These rules allow for selective suppression,\nmeaning that specific rules can be ignored in certain scenarios. The following\nrules have been defined:\n\n* `TMS010`: checks that `unittest.mock.Mock` instances are constructed with the\n  `spec` or `spec_set` argument.\n* `TMS011`: checks that `unittest.mock.MagicMock` instances are constructed with\n  the `spec` or `spec_set` argument.\n* `TMS012`: checks that `unittest.mock.NonCallableMock` instances are\n  constructed with the `spec` or `spec_set` argument.\n* `TMS013`: checks that `unittest.mock.AsyncMock` instances are constructed\n  with the `spec` or `spec_set` argument.\n* `TMS020`: checks that `unittest.mock.patch` is called with any one or more of\n  the `new`, `spec`, `spec_set`, `autospec` or `new_callable` arguments\n* `TMS021`: checks that `unittest.mock.patch.object` is called with any one or\n  more of the `new`, `spec`, `spec_set`, `autospec` or `new_callable` arguments\n* `TMS022`: checks that `unittest.mock.patch.multiple` is called with any one\n  or more of the `spec`, `spec_set`, `autospec` or `new_callable` arguments\n\n### Fix TMS010\n\nThis linting rule is triggered when a `unittest.mock.Mock` instance is created\nwithout the `spec` or `spec_set` argument. For example:\n\n```Python\nfrom unittest import mock\n\ndef test_foo():\n    mocked_foo = mock.Mock()\n```\n\nTo fix this issue, you need to provide the `spec` or `spec_set` argument when\ncreating the mock object. Here are a few examples:\n\n```Python\nfrom unittest import mock\n\nfrom foo import Foo\n\ndef test_foo():\n    mocked_foo = mock.Mock(spec=Foo)\n```\n\n```Python\nfrom unittest import mock\n\nfrom foo import Foo\n\ndef test_foo():\n    mocked_foo = mock.Mock(spec_set=Foo)\n```\n\nFor more information about `mock.Mock` and how to use it, please refer to the\nofficial documentation:\nhttps://docs.python.org/3/library/unittest.mock.html#unittest.mock.Mock\n\n### Fix TMS011\n\nThis linting rule is triggered when a `unittest.mock.MagicMock` instance is\ncreated without the `spec` or `spec_set` argument. For example:\n\n```Python\nfrom unittest import mock\n\ndef test_foo():\n    mocked_foo = mock.MagicMock()\n```\n\nTo fix this issue, you need to provide the `spec` or `spec_set` argument when\ncreating the mock object. Here are a few examples:\n\n```Python\nfrom unittest import mock\n\nfrom foo import Foo\n\ndef test_foo():\n    mocked_foo = mock.MagicMock(spec=Foo)\n```\n\n```Python\nfrom unittest import mock\n\nfrom foo import Foo\n\ndef test_foo():\n    mocked_foo = mock.MagicMock(spec_set=Foo)\n```\n\nFor more information about `mock.MagicMock` and how to use it, please refer to\nthe official documentation:\nhttps://docs.python.org/3/library/unittest.mock.html#unittest.mock.MagicMock\n\n### Fix TMS012\n\nThis linting rule is triggered when a `unittest.mock.NonCallableMock` instance\nis created without the `spec` or `spec_set` argument. For example:\n\n```Python\nfrom unittest import mock\n\ndef test_foo():\n    mocked_foo = mock.NonCallableMock()\n```\n\nTo fix this issue, you need to provide the `spec` or `spec_set` argument when\ncreating the mock object. Here are a few examples:\n\n```Python\nfrom unittest import mock\n\nfrom foo import Foo\n\ndef test_foo():\n    mocked_foo = mock.NonCallableMock(spec=Foo)\n```\n\n```Python\nfrom unittest import mock\n\nfrom foo import Foo\n\ndef test_foo():\n    mocked_foo = mock.NonCallableMock(spec_set=Foo)\n```\n\nFor more information about `mock.NonCallableMock` and how to use it, please\nrefer to the official documentation:\nhttps://docs.python.org/3/library/unittest.mock.html#unittest.mock.NonCallableMock\n\n### Fix TMS013\n\nThis linting rule is triggered when a `unittest.mock.AsyncMock` instance is\ncreated without the `spec` or `spec_set` argument. For example:\n\n```Python\nfrom unittest import mock\n\ndef test_foo():\n    mocked_foo = mock.AsyncMock()\n```\n\nTo fix this issue, you need to provide the `spec` or `spec_set` argument when\ncreating the mock object. Here are a few examples:\n\n```Python\nfrom unittest import mock\n\nfrom foo import Foo\n\ndef test_foo():\n    mocked_foo = mock.AsyncMock(spec=Foo)\n```\n\n```Python\nfrom unittest import mock\n\nfrom foo import Foo\n\ndef test_foo():\n    mocked_foo = mock.AsyncMock(spec_set=Foo)\n```\n\nFor more information about `mock.AsyncMock` and how to use it, please refer to the\nofficial documentation:\nhttps://docs.python.org/3/library/unittest.mock.html#unittest.mock.AsyncMock\n\n### Fix TMS020\n\nThis linting rule is triggered when calling `unittest.mock.patch` without\nincluding one or more of the following arguments: `new`, `spec`, `spec_set`,\n`autospec`, or `new_callable`.\n\nFor example, this code will trigger the rule:\n\n```Python\nfrom unittest import mock\n\n@mock.patch("Foo")\ndef test_foo():\n    pass\n\nwith mock.patch("Foo") as mocked_foo:\n    pass\n\nfoo_patcher = patch("Foo")\n```\n\nTo fix this issue, include one or more of the aforementioned arguments when\ncalling `mock.patch`. For example:\n\n```Python\nfrom unittest import mock\n\nfrom foo import Foo\n\n@mock.patch("Foo", spec=Foo)\ndef test_foo():\n    pass\n\nwith mock.patch("Foo", spec_set=Foo) as mocked_foo:\n    pass\n\nfoo_patcher = patch("Foo", autospec=True)\n```\n\nFor more information about `mock.patch` and how to use it, please refer to the\nofficial documentation:\nhttps://docs.python.org/3/library/unittest.mock.html#unittest.mock.patch\n\n### Fix TMS021\n\nThis linting rule is triggered when calling `unittest.mock.patch.object`\nwithout including one or more of the following arguments: `new`, `spec`,\n`spec_set`, `autospec`, or `new_callable`.\n\nFor example, this code will trigger the rule:\n\n```Python\nfrom unittest import mock\n\nfrom foo import Foo\n\n@mock.patch.object(Foo, "bar")\ndef test_foo():\n    pass\n\nwith mock.patch.object(Foo, "bar") as mocked_foo:\n    pass\n\nfoo_patcher = patch(Foo, "bar")\n```\n\nTo fix this issue, include one or more of the aforementioned arguments when\ncalling `mock.patch.object`. For example:\n\n```Python\nfrom unittest import mock\n\nfrom foo import Foo\n\n@mock.patch.object(Foo, "bar", spec=Foo.bar)\ndef test_foo():\n    pass\n\nwith mock.patch.object(Foo, "bar", spec_set=Foo.bar) as mocked_foo:\n    pass\n\nfoo_patcher = patch(Foo, "bar", autospec=True)\n```\n\nFor more information about `mock.patch.object` and how to use it, please refer\nto the official documentation:\nhttps://docs.python.org/3/library/unittest.mock.html#unittest.mock.patch.object\n\n### Fix TMS022\n\nThis linting rule is triggered when calling `unittest.mock.patch.multiple`\nwithout including one or more of the following arguments: `spec`, `spec_set`,\n`autospec`, or `new_callable`.\n\nFor example, this code will trigger the rule:\n\n```Python\nfrom unittest import mock\n\n@mock.patch.multiple("Foo", FIRST_PATCH=\'bar\', SECOND_PATCH=\'baz\')\ndef test_foo():\n    pass\n\nwith mock.patch.object("Foo", FIRST_PATCH=\'bar\', SECOND_PATCH=\'baz\') as mocked_foo:\n    pass\n\nfoo_patcher = patch("Foo", FIRST_PATCH=\'bar\', SECOND_PATCH=\'baz\')\n```\n\nTo fix this issue, include one or more of the aforementioned arguments when\ncalling `mock.patch.multiple`. For example:\n\n```Python\nfrom unittest import mock\n\nfrom foo import Foo\n\n@mock.patch.multiple("Foo", spec=Foo, FIRST_PATCH=\'bar\', SECOND_PATCH=\'baz\')\ndef test_foo():\n    pass\n\nwith mock.patch.object("Foo", spec_set=Foo, FIRST_PATCH=\'bar\', SECOND_PATCH=\'baz\') as mocked_foo:\n    pass\n\nfoo_patcher = patch("Foo", autospec=True, FIRST_PATCH=\'bar\', SECOND_PATCH=\'baz\')\n```\n\nFor more information about `mock.patch.multiple` and how to use it, please\nrefer to the official documentation:\nhttps://docs.python.org/3/library/unittest.mock.html#unittest.mock.patch.multiple\n',
    'author': 'David Andersson',
    'author_email': 'david@jdkandersson.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'py_modules': modules,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8.1,<4.0.0',
}


setup(**setup_kwargs)
